/*	Parameter_Pane

PIRL CVS ID: Parameter_Pane.java,v 1.21 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2004-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import	PIRL.PVL.*;
import	PIRL.TreeTable.*;

import	javax.swing.*;
import	javax.swing.border.*;
import	javax.swing.event.*;
import	javax.swing.table.*;
import	javax.swing.tree.*;
import	java.awt.*;
import	java.awt.event.*;
import	java.io.File;
import	java.util.Hashtable;
import	java.util.Vector;
import	java.util.Enumeration;
import	java.util.Iterator;

//	Used the by the testing stub:
import	PIRL.Utilities.Streams;
import	javax.swing.JFrame;
import	java.io.InputStream;

/**	A <I>Parameter_Pane</I> provides a graphical view of a <I>Parameter</I>.
<P>
	The display uses a Parameter_Model to wrap the Parameter data and a
	JTreeTable controller to represent the Parameters and its Values in
	two columns. Any Parameter that is an aggregate of other Parameters
	may be expanded to display its contents. The name of each Parameter
	is listed in the first column of the table. In the second column the
	Parameter's Value is shown as the String provided by the Value's
	<CODE>toString</CODE> method. Those Values that are array types
	(<CODE>SET</CODE> or <CODE>SEQUENCE</CODE>) may be shown in various
	display {@link #Array_Viewing_Mode(int) modes}.
<P>
	@see		Parameter
	@see		Value
	@see		JTreeTable

	@author		Bradford Castalia, UA/PIRL
	@version	1.21
*/
public class Parameter_Pane
	extends JPanel
{
private static final String
	ID = "PIRL.Viewers.Parameter_Pane (1.21 2012/04/16 06:22:59)";

/**	Array Value elements are viewed as comma separated values (CSV) text.
*/ 			
public static final int		CSV_ARRAY_VIEWING = 0;

/**	Array Value elements are viewed in a scrollable subtree cell pane.
*/
public static final int		SUBTREE_ARRAY_VIEWING = 1;

/**	Array Value elements are viewed in a separate window.
*/
public static final int		WINDOW_ARRAY_VIEWING = 2;

/**	Array Value elements are viewed in multiple separate windows.
*/
public static final int		MULTIPLE_WINDOW_ARRAY_VIEWING = 3;

/**	The default Array Value viewing mode.
*/
public static int			Default_Array_Viewing_Mode = WINDOW_ARRAY_VIEWING;

/**	Current Array Value viewing mode.
*/
protected int				Array_Viewing_Mode = Default_Array_Viewing_Mode; 

/**	Default tree line style.
*/
public static String		Default_Line_Style = "None";


private JScrollPane			Parameters_Pane;

/**	The JTreeTable used to view the Parameter.
*/
protected JTreeTable		The_TreeTable;

/**	The Parameter being viewed.
*/
private Parameter			The_Parameter;

//	A Value to Value_View table for current Value Array displays.
private Hashtable			Value_Views = new Hashtable ();

//	Indicates that a new Value_View has just been created.
private boolean				New_Value = false;

//	A flag to wave over a Value_View.
private static final Blinker
	Value_View_Blinker =
		new Blinker
			(
			null,
			Color.YELLOW,
			8,		//	Repeats.
			500,	//	Initial delay (ms).
			150		//	Blink interval (ms).
			);

//	The last path expanded in the values column.
private TreePath			Last_Value_Path_Expanded = null;

//	Parameter comments pane.
private JTextArea			Comments_Text = null;

//	Array Value viewing control menu.
private JPopupMenu			Popup_Menu;
private	JMenu				Array_Values_Menu;


//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_UI		= 1 << 1,
	DEBUG_EDITOR	= 1 << 3,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Creates a Parameter_Pane from the specified Parameter.
<P>
	@param	parameter	The Parameter to be displayed.
*/
public Parameter_Pane
	(
	Parameter	parameter
	)
{
setLayout (new BorderLayout ());

//	Popup menu.
Popup_Menu = new JPopupMenu ("View");
Popup_Menu.add (Array_Values_Menu = Popup_Menu ());

//	Parameters table.
Parameters_Pane = new JScrollPane ();
Parameter (parameter);

//	Comments pane.
JSplitPane
	split_pane = new JSplitPane (JSplitPane.VERTICAL_SPLIT, true,
		Parameters_Pane, Comments_Pane ());
split_pane.setResizeWeight (1.0);
split_pane.setDividerLocation (-1);
add (split_pane);
}

/*==============================================================================
	Accessors
*/
/**	Sets the Parameter to be viewed.
<P>
	<B>N.B.</b>: If the Name of the Parameter is the {@link
	Parser#CONTAINER_NAME} then the root node will not be displayed.
<p>
	@param	parameter	The Parameter to View.
	@return	This Parameter_Pane.
*/
public Parameter_Pane Parameter
	(
	Parameter	parameter
	)
{
if (parameter == null)
	parameter = new Parameter ();
//	The JTreeTable for the Parameter.
The_TreeTable = Tree_Table (new Parameter_Model (The_Parameter = parameter));
if (The_Parameter.Name ().equals (Parser.CONTAINER_NAME))
	{
	//	Don't show The Container group node.
	The_TreeTable.getTree ().setRootVisible (false);
	The_TreeTable.getTree ().setShowsRootHandles (true);
	}
//	The JTree line style.
Line_Style (Default_Line_Style);

//	Take down our Value_Views.
Take_Down_Value_Views (Value_Views);

Parameters_Pane.setViewportView (The_TreeTable);
return this;
}

/**	Gets the Parameter being viewed in the display.
<P>
	@return	The Parameter viewed in this Parameter_View.
*/
public Parameter Parameter ()
{return The_Parameter;}

/**	Gets the "JTree.lineStyle" property currently being used.
<P>
	@return	The "JTree.lineStyle" property String in effect.
*/
public String Line_Style ()
{return (String)The_TreeTable.getTree ().getClientProperty ("JTree.lineStyle");}

/**	Sets the "JTree.lineStyle" property to the specified style.
<P>
	@param	style	The String describing the style to use.
	@return	This Parameter_Pane.
	@see	javax.swing.JComponent#putClientProperty(Object, Object)
*/
public Parameter_Pane Line_Style
	(
	String		style
	)
{
The_TreeTable.getTree ().putClientProperty ("JTree.lineStyle", style);
return this;
}

/**	Gets the JTreeTable being used to view the Parameter in the display.
<P>
	@return	The JTreeTable viewed in this Parameter_View.
*/
public JTreeTable TreeTable ()
{return The_TreeTable;}

/**	Gets the Array Value viewing mode.
<P>
	@return	The Value Array viewing mode. This will be one of {@link
		#CSV_ARRAY_VIEWING}, {@link #SUBTREE_ARRAY_VIEWING},  {@link
		#WINDOW_ARRAY_VIEWING}, or {@link #MULTIPLE_WINDOW_ARRAY_VIEWING}.
*/
public int Array_Viewing_Mode ()
{return Array_Viewing_Mode;}

/**	Sets the Array Value viewing mode.
<P>
	The current view will be redisplayed as appropriate.
<p>
	@param	mode	The Value Array viewing mode. This should be one of
		{@link #CSV_ARRAY_VIEWING}, {@link #SUBTREE_ARRAY_VIEWING}, or
		{@link #WINDOW_ARRAY_VIEWING}. A value less than 0 will be set to
		0; a value greater than {@link #MULTIPLE_WINDOW_ARRAY_VIEWING}
		will be set to {@link #MULTIPLE_WINDOW_ARRAY_VIEWING}.
	@return	This Parameter_Pane.
*/
public Parameter_Pane Array_Viewing_Mode
	(
	int		mode
	)
{
if (mode < 0)
	mode = 0;
else
if (mode > MULTIPLE_WINDOW_ARRAY_VIEWING)
	mode = MULTIPLE_WINDOW_ARRAY_VIEWING;
if (Array_Viewing_Mode != mode)
	{
	Array_Viewing_Mode = mode;
	Refresh ();
	}
return this;
}

/**	Sets the path to the last value expanded.
<P>
	<B>NOT FOR PUB USE!</B>
<P>
	<B>N.B.</B>: This method is an implementation artifact that is
	only for use by a Value_Pane created by this Parameter_Pane.
<P>
	@param	last_path_expanded	The TreePath for the last
		expanded Array Value row.
*/
public void Last_Value_Path_Expanded
	(
	TreePath	last_path_expanded
	)
{Last_Value_Path_Expanded = last_path_expanded;}

/*==============================================================================
	Display Elements
*/
/**	Refresh the view.
<p>
	Any change to the {@link #Array_Viewing_Mode(int) array viewing mode}
	is applied.
*/
public void Refresh ()
{
JScrollBar
	scroll_bar = Parameters_Pane.getVerticalScrollBar ();
int
	total_rows = The_TreeTable.getTree ().getRowCount (),
	select_row = The_TreeTable.getTree ()
		.getRowForPath (The_TreeTable.getTree ().getSelectionPath ());
boolean[]
	expanded_rows = new boolean [total_rows];
for (int row = 0;
		 row < total_rows;
		 row++)
	expanded_rows[row] = The_TreeTable.getTree ().isExpanded (row);

Take_Down_Value_Views (Value_Views);

The_TreeTable.removeEditor ();
if (Array_Viewing_Mode == SUBTREE_ARRAY_VIEWING)
   The_TreeTable.setDefaultEditor (Value.class,  new Value_Cell_Editor ());
else
if (Array_Viewing_Mode == WINDOW_ARRAY_VIEWING ||
	Array_Viewing_Mode == MULTIPLE_WINDOW_ARRAY_VIEWING)
	{
	The_TreeTable.getSelectionModel ()
		.addListSelectionListener (new List_Selector ());
	Parameter_Selected ();
	}
//	Update the Comments.
Update_Comments ();

Parameters_Pane.setViewportView (The_TreeTable);

for (int row = 0;
		 row < total_rows;
		 row++)
	if (expanded_rows[row])
		The_TreeTable.getTree ().expandRow (row);
The_TreeTable.getTree ().addSelectionRow (select_row);
The_TreeTable.editCellAt (select_row, 1);
Parameters_Pane.setVerticalScrollBar (scroll_bar);
}

/**	Creates the JTreeTable that will display the Parameter's name and
	Value.
<P>
	The JTreeTable is constructed using the specified Parameter_Model.
<P>
	A ListSelectionListener is provided for the JTreeTable that will
	create a Value_View when a row in the table is selected that contains
	an Array Value.
<P>
	The Value_View displays that are being viewed are maintained in the
	Value_Views Hashtable. When the Array_Viewing_Mode is
	WINDOW_ARRAY_VIEWING any Value_View for a non-selected Value is
	disposed of. If the Array_Viewing_Mode is
	MULTIPLE_WINDOW_ARRAY_VIEWING and a previously selected Value has a
	<CODE>Value_Views</CODE> entry, then that display is flashed
	with a Blinker to bring attention to the reselected Value display.
<P>
	@param	model	The Parameter_Model to use in constructing
		the JTreeTable.
	@return	The JTreeTable that was created.
*/
protected JTreeTable Tree_Table
	(
	Parameter_Model	model
	)
{
JTreeTable
	tree_table = new JTreeTable (model);

//	Supress the icon for leaf nodes.
TreeCellRenderer
	renderer = tree_table.getTree ().getCellRenderer (); 
if (renderer instanceof DefaultTreeCellRenderer)
	((DefaultTreeCellRenderer)renderer).setLeafIcon (null);

//	Set the Array Value viewing mode.
if (Array_Viewing_Mode == SUBTREE_ARRAY_VIEWING)
   tree_table.setDefaultEditor (Value.class,  new Value_Cell_Editor ());
else if (Array_Viewing_Mode == WINDOW_ARRAY_VIEWING ||
		 Array_Viewing_Mode == MULTIPLE_WINDOW_ARRAY_VIEWING)
	{
	tree_table
		.getSelectionModel ()
		.addListSelectionListener (new List_Selector());
	}
//	No cell editor for CSV mode.

if (Popup_Menu != null)
	tree_table.addMouseListener (new Popup_Listener ());

return tree_table;
}

/**	Handle a selected Parameter.
<P>
	The Value_View displays that are being viewed are maintained in the
	Value_Views Hashtable. When the Array_Viewing_Mode is
	WINDOW_ARRAY_VIEWING any Value_View for a non-selected Value is
	disposed of. If the Array_Viewing_Mode is
	MULTIPLE_WINDOW_ARRAY_VIEWING and a previously selected Value has a
	<CODE>Value_Views</CODE> entry, then that display is flashed
	with a Blinker to bring attention to the reselected Value display.
*/
private void Parameter_Selected ()
{
//	Determine the Parameters that were selected.
int[]
	selected_rows = The_TreeTable.getSelectedRows ();
Hashtable
	old_Value_Views = Value_Views;
if (Array_Viewing_Mode == WINDOW_ARRAY_VIEWING)
	//	Re-assemble the list of Value displays.
	Value_Views = new Hashtable ();
Point
	view_location = getLocationOnScreen ();
view_location.x += 4 * (getSize ().width / 5);

for (int selection = 0;
		 selection < selected_rows.length;
		 selection++)
	{
	Parameter
		parameter =
			(Parameter)The_TreeTable.getTreeNode (selected_rows[selection]);
	Value
		value;
	try {value = parameter.Value ();}
	catch (PVL_Exception exception) {continue;}
	if (parameter.Is_Assignment () &&
		value.Is_Array ())
		{
		//	Value Array selected.
		Value_View
			view;
		if (! old_Value_Views.containsKey (value))
			{
			//	New view.
			try
				{
				view = new Value_View (parameter.Path_Name (), value, this);
				if (! view.Value_Pane ().Line_Style ().equals (Line_Style ()))
					view.Value_Pane ().Line_Style (Line_Style ());
    			view.addWindowListener (new WindowAdapter ()
					{public void windowClosing (WindowEvent event)
					{
					Window
						window = event.getWindow ();
					Enumeration
						views = Value_Views.keys ();
					while (views.hasMoreElements ())
						{
						Value
							view_value = (Value)views.nextElement ();
						if (Value_Views.get (view_value) == window)
							Value_Views.remove (view_value);
						}
					window.dispose ();
        			}});

				Rectangle
					cell_bounds = The_TreeTable.getCellRect
						(selected_rows[selection], 0, false);
				view_location.y =
					The_TreeTable.getLocationOnScreen ().y +
					cell_bounds.y + (cell_bounds.height / 2);
 				view.setLocation (view_location);
				view.setVisible (true);
				view.Tree ().expandRow (0);
				}
			catch (PVL_Exception exception)
				{
				Dialog_Box.Error ("Unable to create the Value_View.\n\n"
					+ ID + '\n'
					+ exception.Message ());
				continue;
				}
			Value_Views.put (value, view);
			New_Value = true;
			}
		else
			{
			//	Previously selected view.
			if (Array_Viewing_Mode == WINDOW_ARRAY_VIEWING)
				//	Copy over the old Value_View.
				Value_Views.put (value, old_Value_Views.remove (value));
			else if (New_Value)
				New_Value = false;
			else
				{
				//	Wave a flag over the view.
				Value_View_Blinker.setComponent
					(((Value_View)Value_Views.get (value)).Tree ());
				Value_View_Blinker.restart ();
				}
			}
		}
	}
if (Array_Viewing_Mode == WINDOW_ARRAY_VIEWING)
	//	Take down any deselected views.
	Take_Down_Value_Views (old_Value_Views);
}

/**	Any windows displaying Array Values are taken down.
*/
public void Take_Down_Value_Views ()
{Take_Down_Value_Views (Value_Views);}


private void Take_Down_Value_Views
	(
	Hashtable	value_views
	)
{
Enumeration
	views = value_views.keys ();
while (views.hasMoreElements ())
	((Window)value_views.get (views.nextElement ())).dispose ();
value_views.clear ();
}


private JScrollPane Comments_Pane ()
{
Comments_Text = new JTextArea ();
Comments_Text.setTabSize (4);
JScrollPane
	scroll_pane = new JScrollPane (Comments_Text);
scroll_pane.setViewportBorder (BorderFactory.createTitledBorder ("Comments"));
scroll_pane.setMinimumSize (new Dimension (0, 100));
return scroll_pane;
}


private void Update_Comments ()
{
String
	comments = null;
//	Determine the Parameters that were selected.
int[]
	selected = The_TreeTable.getSelectedRows ();
if (selected.length == 1)
	comments = ((Parameter)The_TreeTable.getTreeNode (selected[0])).Comments ();
Comments_Text.setText (comments);
}

/*------------------------------------------------------------------------------
	Popup Menu
*/
/**	Get the menu that controls how Array Values are viewed.
<p>
	@return	The Array Values JMenu.
*/
public JMenu Array_Values_Menu ()
{return Array_Values_Menu;}


private JMenu Popup_Menu ()
{
JMenu
	menu = new JMenu ("Array Values"),
	submenu;
ButtonGroup
	button_group = new ButtonGroup ();
JRadioButtonMenuItem
	radio_button;
radio_button = new JRadioButtonMenuItem ("CSV");
radio_button.setMnemonic ('C');
radio_button.setAccelerator (KeyStroke.getKeyStroke ('C', Event.CTRL_MASK));
radio_button.addActionListener (new ActionListener ()
    {public void actionPerformed (ActionEvent event)
  	 {Array_Viewing_Mode (CSV_ARRAY_VIEWING);}});
radio_button.setSelected
	(Default_Array_Viewing_Mode == CSV_ARRAY_VIEWING);
button_group.add (radio_button);
menu.add (radio_button);

radio_button = new JRadioButtonMenuItem ("Subtree");
radio_button.setMnemonic ('T');
radio_button.setAccelerator (KeyStroke.getKeyStroke ('T', Event.CTRL_MASK));
radio_button.addActionListener (new ActionListener ()
    {public void actionPerformed (ActionEvent event)
	 {Array_Viewing_Mode (SUBTREE_ARRAY_VIEWING);}});
radio_button.setSelected
	(Default_Array_Viewing_Mode == SUBTREE_ARRAY_VIEWING);
button_group.add (radio_button);
menu.add (radio_button);

submenu = new JMenu ("Windows");
radio_button = new JRadioButtonMenuItem ("Single");
radio_button.setMnemonic ('L');
radio_button.setAccelerator (KeyStroke.getKeyStroke ('L', Event.CTRL_MASK));
radio_button.addActionListener (new ActionListener ()
    {public void actionPerformed (ActionEvent event)
    {Array_Viewing_Mode (WINDOW_ARRAY_VIEWING);}});
radio_button.setSelected
	(Default_Array_Viewing_Mode == WINDOW_ARRAY_VIEWING);
button_group.add (radio_button);
submenu.add (radio_button);

radio_button = new JRadioButtonMenuItem ("Multiple");
radio_button.setMnemonic ('M');
radio_button.setAccelerator (KeyStroke.getKeyStroke ('M', Event.CTRL_MASK));
radio_button.addActionListener (new ActionListener ()
    {public void actionPerformed (ActionEvent event)
    {Array_Viewing_Mode (MULTIPLE_WINDOW_ARRAY_VIEWING);}});
radio_button.setSelected
	(Default_Array_Viewing_Mode == MULTIPLE_WINDOW_ARRAY_VIEWING);
button_group.add (radio_button);
submenu.add (radio_button);
menu.add (submenu);
return menu;
}

/*..............................................................................
	Popup Menu Listener
*/
class Popup_Listener
	extends MouseAdapter
{
public void mousePressed (MouseEvent event)
{maybeShowPopup (event);}

public void mouseReleased (MouseEvent event)
{maybeShowPopup (event);}

private void maybeShowPopup
	(
	MouseEvent	event
	)
{
if (event.isPopupTrigger ())
	Popup_Menu.show (event.getComponent (), event.getX (), event.getY ());
}
}

/*=*****************************************************************************
	List_Selector
*/
/**	A <I>List_Selector</I> to handle the selection of values on the
	Parameter_Pane.
<p>
	Though it will listen for ListSelectionEvents in any viewing mode,
	it only takes action if one of the Value_View viewing modes is in
	effect.
*/
private class List_Selector
	implements ListSelectionListener
{
public void valueChanged
	(
	ListSelectionEvent	event
	)
{
if (! event.getValueIsAdjusting ())
	{
	if (Array_Viewing_Mode == WINDOW_ARRAY_VIEWING ||
		Array_Viewing_Mode == MULTIPLE_WINDOW_ARRAY_VIEWING)
		Parameter_Selected ();
	//	Update the Comments.
	Update_Comments ();
	}
}
}	//	End class List_Selector

/*=*****************************************************************************
	Parameter_Model
*/
/**	Names of the columns in the JTreeTable display.

	The first column is represented by the Parameter; this must be a
	form of TreeTableModel. The second column is represented by the
	Value.
<P>
@see		Value
*/
private static final String[]
	COLUMN_NAMES =
		{
		"Name",
		"Value"
		};

/**	Types of the columns.

	The entries correspond to the <CODE>COLUMN_NAMES</CODE> array.
*/
private static final Class[]
	COLUMN_CLASSES =
		{
		TreeTableModel.class,
		Value.class
		};
private static final int
	TREE_TABLE_COLUMN	= 0,
	VALUE_COLUMN		= 1;

/**
A <I>Parameter_Model</I> provides the data model of a <I>Parameter</I>
for use with a <I>JTreeTable</I> controller.
<P>
@see		Parameter
@see		Value
@see		Parameter_Pane

@author		Bradford Castalia, UA/PIRL
@version	1.21
*/
class Parameter_Model
	extends AbstractTreeTableModel
{
/**	Creates a Parameter_Model rooted at the Parameter.
<P>
	@param	parameter	The Parameter to be modeled.
*/
public Parameter_Model
	(
	Parameter	parameter
	)
{super (parameter);}

/*==============================================================================
	The TreeModel interface
*/
/**	Gets the number of children of a Parameter's node.
<P>
	@param	node	A node of the Parameter being modeled.
	@return	The number of Parameters in the node's aggregate list.
	@see	Parameter#getChildCount()
*/
public int getChildCount
	(
	Object		node
	)
{return ((Parameter)node).getChildCount ();}

/**	Gets the child of a Parameter's node at the specified index
	of the aggregate list.
<P>
	@param	node	A node of the Parameter being modeled.
	@param	index	The index into the Parameter node's aggregate list
		where the child will be found.
	@return	The Parameter Object from the aggregate list.
	@see	Parameter#getChildAt(int)
 */
public Object getChild
	(
	Object		node,
	int			index
	)
{return ((Parameter)node).getChildAt (index);}

/**	Tests if a node is a leaf.
<P>
	@param	node	A node of the Parameter being modeled.
	@see	Parameter#isLeaf()
*/
public boolean isLeaf
	(
	Object		node
	)
{return ((Parameter)node).isLeaf ();}

/*==============================================================================
	The TreeTableNode interface. 
*/
/**	Gets the number of columns in the table.
<P>
	@return	The length of the COLUMN_NAMES array.
*/
public int getColumnCount ()
{return COLUMN_NAMES.length;}

/**	Gets the name for a particular column.
<P>
	@param	column	The column number for which to get a name.
	@return	The corresponding entry in the COLUMN_NAMES array.
*/
public String getColumnName (int column)
{return COLUMN_NAMES[column];}

/**	Gets the class for a particular column.
<P>
	@param	column	The column number for which to get a class type.
	@return	The corresponding entry in the COLUMN_CLASSES array.
*/
public Class getColumnClass (int column)
{return COLUMN_CLASSES[column];}

/**	Tests if a table column for a Parameter is editable.
<P>
	@param	node	A node of the Parameter being modeled.
	@param	column	A table column number.
	@return	true, it the cell is editable; false otherwise.
*/
public boolean isCellEditable
	(
	Object	node,
	int		column
	)
{
Parameter
	parameter = (Parameter)node;

switch (column)
	{
	case 0:
		return parameter.Is_Aggregate ();
	case 1:	
		try
			{
			return parameter.Value().Is_Array ();
			}
		catch (PVL_Exception exception) {return false;}
	}
return false;
}

/**	Gets the value of a particular column for a specified
	Parameter node representing a table record.
<P>
	The first column value is the Parameter's name String.
<P>
	The second column value is the Parameter's Value if it has
	one. If the Parameter is an Aggregate, however, the value is
	its the name of the Parameter's specific classification. For
	all other Assignment Parameters a new Value with empty
	String Data is provided.
<P>
	@param	node	A node of the Parameter being modeled.
	@param	column	The column number for which to get a value.
	@return	An Object representing the value of the column.
 */
public Object getValueAt
	(
	Object		node,
	int			column
	)
{
Parameter
	parameter = (Parameter)node;

switch (column)
	{
	case 0:
		return parameter.Name ();
	case 1:
		if (parameter.Is_Aggregate ())
			return parameter.Classification_Name ();
		try
			{
			Value
				value = (Value)parameter.Value ();
			if (value == null)
				return "[no value]";
			if (value.Is_Array () &&
				Array_Viewing_Mode == CSV_ARRAY_VIEWING)
				return value.Description ();
			return value;
			}
		catch (PVL_Exception exception) {return "";}							
	}
return null; 
}

}	//	End of class Parameter_Model.

/*=*****************************************************************************
	Value_Cell_Editor
*/
class Value_Cell_Editor
	extends javax.swing.AbstractCellEditor
	implements TableCellEditor
{
private	Value_Pane			Editor = null;
private JTable				Table;
private int					Row,
							Row_Height;


//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_EDITOR	= 1 << 3,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;


Value_Cell_Editor ()
{
if ((DEBUG & DEBUG_EDITOR) != 0)
	System.out.println
		(">-< Parameter_Pane.Value_Cell_Editor");

}

//	CellEditor (via AbstractCellEditor) interface implementation.
public Object getCellEditorValue ()
{
if ((DEBUG & DEBUG_EDITOR) != 0)
	{
	System.out.print
		(">-< Parameter_Pane.Value_Cell_Editor.getCellEditorValue: ");
	try {Editor.Value ().Write ();}
	catch (Exception e) {}
	}
if (Editor.Value ().Is_Array ())
	Table.setRowHeight (Row, Row_Height);
return Editor.Value ();
}

//	TableCellEditor interface implementation.
public Component getTableCellEditorComponent
	(
	JTable	table,
	Object	value,
	boolean	is_selected,
	int		row,
	int		column
	)
{
if ((DEBUG & DEBUG_EDITOR) != 0)
	System.out.println
		(">>> Parameter_Pane.Value_Cell_Editor.getTableCellEditorComponent:\n"
		+ "    " + row + ',' + column
		+ ": selected = " + is_selected);
Editor = null;
if (Array_Viewing_Mode != SUBTREE_ARRAY_VIEWING)
	Table.setRowHeight (Row, Row_Height);
else if (value instanceof String)
	{
	if ((DEBUG & DEBUG_EDITOR) != 0)
		System.out.println ("    String value - " + value);
	cancelCellEditing ();
	Editor = null;
	}
else
	{
	Value
		a_value = (Value)value;
	if ((DEBUG & DEBUG_EDITOR) != 0)
		{
		System.out.println ("    Value value -");
		try {a_value.Write ();}
		catch (Exception e) {}
		}
	Editor = new Value_Pane (a_value, Parameter_Pane.this);
	Editor.Tree ().setEditable (true);
	Table = table;
	Row = row;
	Row_Height = Table.getRowHeight ();
	if (a_value.Is_Array ())
		{
		Editor.setPreferredSize
			(new Dimension (100, 5 * table.getRowHeight ()));
		Editor.Tree ().expandPath (Last_Value_Path_Expanded);
		Table.setRowHeight (Row, 5 * Row_Height);
		if (! Editor.Tree ().isExpanded (0))
			/*
				If the top level has not been expanded, do so.
				This will occur if the Last_Value_Path_Expanded is null.
			*/
			Editor.Tree ().expandRow (0);
		}
	}
if ((DEBUG & DEBUG_EDITOR) != 0)
	System.out.println
		("<<< Parameter_Pane.Value_Cell_Editor.getTableCellEditorComponent");
return Editor;
}

}	//	End of class Value_Cell_Editor.

/*==============================================================================
	Test stub
*/
public static void main (String[] arguments)
{
System.out.println (ID);

if (arguments.length == 0)
	{
	System.err.println ("Usage: java Parameter_Pane <PVL_filename>");
	System.exit (1);
	}

if (System.getProperty ("os.name").equals ("Mac OS X"))
	{
	try {UIManager.setLookAndFeel
		(UIManager.getCrossPlatformLookAndFeelClassName ());}
	catch (Exception exception) {/* Just leave the existing LAF. */}
	}

Parameter_Pane
	parameter_pane = null;
try
	{
	InputStream
		source_stream;
	if ((source_stream = Streams.Get_Stream (arguments[0])) != null)
		parameter_pane = new Parameter_Pane
				(new Parameter (new Parser (source_stream)));
	else
		{
		System.err.println
			("Unable to access source: " + arguments[0]);
		System.exit (1);
		}
	}
catch (Exception exception)
	{
	System.err.println (exception.getMessage ());
	System.exit (2);
	}

final JFrame
	frame = new JFrame ("Parameter_Pane");
frame.setDefaultCloseOperation (JFrame.EXIT_ON_CLOSE);
frame.setContentPane (parameter_pane);
frame.pack ();
frame.setVisible (true);
}

}	//	End of class Parameter_Pane.
