/*	PostgreSQL_Data_Port

PIRL CVS: $ID$

Copyright (C) 2008  Arizona Board of Regents on behalf of the Planetary
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.PVL.Parameter;
import PIRL.PVL.Value;
import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;

import java.util.Vector;


/**	Provides access to a PostgreSQL database via a <i>JDBC_Data_Port</i>.
<p>
	@see		JDBC_Data_Port

	@author		Bradford Castalia, UA/PIRL
	@version	1.5 
*/
public class PostgreSQL_Data_Port
	extends JDBC_Data_Port
	implements Data_Port
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Database.PostgreSQL_Data_Port (1.19 2006/08/09 00:50:32)";

/**	Data_Port Type name.
*/
public static final String
	Port_Type = "PostgreSQL";

/**	Required Configuration parameters and their default values.
<p>
<dl compact>
	<dt><code>{@link Database#DRIVER}</code>
		<dd>org.postgresql.Driver
	<dt><code>{@link Configuration#HOST}</code>
		<dd>localhost
	<dt><code>{@link Database#DATABASE}</code>
		<dd>public
</dl>
*/
public static final String
	URL_Specs[][] =
	{
	{Database.DRIVER,		"org.postgresql.Driver"},
	{Configuration.HOST,	"localhost"},
	{Database.DATABASE,		"public"}
	};

/**	The Configuration parameter to use when specifying a specific
	port for connection to the database server.
*/
public static final String
	PORT = "port";

/**	Optional Configuration parameters.
<p>
	These parameters are based on the use of the PostgreSQL JDBC driver, 
	version 8.3.
<p>
<h4>Connection Parameters</h4>
<dl>
	<dt><code>user</code>
		<dd>The database user on whose behalf the connection is being made. [none]
	<dt><code>password</code>
		<dd>The database user's password. [none]
	<dt><code>ssl</code>
		<dd>Connect using SSL. The driver must have been compiled with SSL 
			support. This property does not need a value associated with it. 
			The mere presence of it specifies a SSL connection. However, for 
			compatibility with future versions, the value "true" is 
			preferred. [false]
	<dt><code>sslfactory</code>
		<dd>The provided value is a class name to use as the SSLSocketFactory
			when establishing a SSL connection. [none]
	<dt><code>sslfactoryarg</code>
		<dd>This value is an optional argument to the constructor of the 
			sslfactory class provided above. [none]
	<dt><code>compatible</code>
		<dd>Act like an older version of the driver to retain compatibility
			with older applications. At the moment this controls two driver 
			behaviors: the handling of binary data fields, and the handling of
			parameters set via setString().  Older versions of the driver used
			this property to also control the protocol used to connect to the 
			backend. This is now controlled by the protocolVersion property. 
			[none]
	<dt><code>protocolVersion</code>
		<dd>The driver supports both the V2 and V3 frontend/backend protocols.
			The V3 protocol was introduced in 7.4 and the driver will by 
			default try to connect using the V3 protocol, if that fails it will
			fall back to the V2 protocol. If the protocolVersion property is 
			specified, the driver will try only the specified protocol (which 
			should be either "2" or "3"). Setting protocolVersion to "2" may be 
			used to avoid the failed attempt to use the V3 protocol when 
			connecting to a version 7.3 or earlier server, or to force the 
			driver to use the V2 protocol despite connecting to a 7.4 or 
			greater server. [3]
	<dt><code>loglevel</code>
		<dd>Set the amount of logging information printed to the 
			DriverManager's current value for LogStream or LogWriter. It 
			currently supports values of org.postgresql.Driver.DEBUG (2) and 
			org.postgresql.Driver.INFO (1). INFO will log very little 
			information while DEBUG will produce significant detail. This 
			property is only really useful if you are a developer or are having
			problems with the driver. [0]
	<dt><code>charSet</code>
		<dd>The character set to use for data sent to the database or recieved 
			from the database. This property is only relevent for server 
			versions less than or equal to 7.2. The 7.3 release was the first
			with multibyte support compiled by default and the driver uses its
			character set translation facilities instead of trying to do it 
			itself. [none]
	<dt><code>allowEncodingChanges</code>
		<dd>When using the V3 protocol the driver monitors changes in certain 
			server configuration parameters that should not be touched by end 
			users. The client_encoding setting is set by the driver and should
			not be altered. If the driver detects a change it will abort the 
			connection. There is one legitimate exception to this behavior 
			though, using the COPY command on a file residing on the server's
			filesystem. The only means of specifying the encoding of this file
			is by altering the client_encoding setting. The JDBC team considers
			this a failing of the COPY command and hopes to provide an 
			alternate means of specifying the encoding in the future, but for 
			now there is this URL parameter. Enable this only if you need to 
			override the client encoding when doing a copy. [false]
	<dt><code>logUnclosedConnections</code>
		<dd>Clients may leak Connection objects by failing to call its close()
			method. Eventually these objects will be garbage collected and the 
			finalize() method will be called which will close the Connection if
			caller has neglected to do this himself. The usage of a finalizer 
			is just a stopgap solution. To help developers detect and correct 
			the source of these leaks the logUnclosedConnections URL parameter 
			has been added. It captures a stacktrace at each Connection opening
			and if the finalize() method is reached without having been closed 
			the stacktrace is printed to the log. [false]
	<dt><code>prepareThreshold</code>
		<dd>Determine the number of PreparedStatement executions required
			before switching over to use server side prepared statements. The 
			default is five, meaning start using server side prepared 
			statements on the fifth execution of the same PreparedStatement 
			object. [5]
	<dt><code>loginTimeout</code>
		<dd>Specify how long to wait for establishment of a database 
			connection. The timeout is specified in seconds. [Pulled timeout 
			from DriverManager.getLoginTimeout()
	<dt><code>stringtype</code>
		<dd>Specify the type to use when binding PreparedStatement parameters
			set via setString(). If stringtype is set to varchar (the default), 	
			such parameters will be sent to the server as varchar parameters. 
			If stringtype is set to unspecified, parameters will be sent to the
			server as untyped values, and the server will attempt to infer an 
			appropriate type. This is useful if you have an existing 
			application that uses setString() to set parameters that are 
			actually some other type, such as integers, and you are unable to 
			change the application to use an appropriate method such as 
			setInt(). [varchar]
</dl>
*/
public static final String
	Optional_Parameters[] =
	{
	"user",
	"password",
	"ssl",
	"sslfactory",
	"sslfactoryarg",
	"compatible",
	"protocolVersion",
	"loglevel",
	"charSet",
	"allowEncodingChanges",
	"logUnclosedConnections",
	"prepareThreshold",
	"loginTimeout",
	"stringtype"
	};

private static final String
	NL = Database.NL;


//	DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_OPEN		= 1 << 0,
	
	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs the abstract JDBC_Data_Port finalizer class, registers
	the class ID and name of the Data_Port type, and sets the flag to
	treat schema as Database "catalogs".
*/
public PostgreSQL_Data_Port ()
{
ID_Type (ID, Port_Type);
Treat_Schema_As_Catalog = true;
}

/*==============================================================================
	Methods
*/
//	Data_Port Implementation
/**	Implements the Data_Port <code>Parameters</code> method.
<p>
	@see	Data_Port#Parameters()
*/
public Configuration Parameters ()
	throws Database_Exception
{
try
	{
	Parameter
		parameters = new Parameter (Port_Type),
		group = new Parameter ("Required");
	int
		index;
	for (index = 0;
		 index < URL_Specs.length;
		 index++)
		group.Add (new Parameter (URL_Specs[index][0])
			.Value (URL_Specs[index][1]));
	parameters.Add (group);

	group = new Parameter ("Optional");
	group.Add (new Parameter (PORT));
	group.Add (new Parameter (Database.CATALOG));
	for (index = 0;
		 index < Optional_Parameters.length;
		 index++)
		group.Add (new Parameter (Optional_Parameters[index]));
	parameters.Add (group);
	return new Configuration (parameters);
	}
catch (Exception exception)
	{
	//	This shouldn't happen if the coding is correct.
	throw new Database_Exception
		(
		ID + NL
		+"There is a coding flaw in the Parameters method!" + NL
		+ exception.getMessage ()
		);
	}
}

//	Data_Port Implementation
/**	Implements the Data_Port <code>Open</code> method.
<p>
	The Configuration that is supplied is conditionally set
	(existing parameters of the same name are not reset) with
	the required {@link #URL_Specs URL specifications} default
	values. Then the Configuration is provided to the base
	JDBC_Data_Port <code>{@link JDBC_Data_Port#Configure Configure}</code>
	method to load the JDBC driver and save the Configuration.
<p>
	The URL specifying how the connection will be made to the
	database server has the form:
<p>
	<b>jdbc:postgresql://</b>[<i>HOST</i>][<b>:</b><i>PORT</i>]<b>/</b><i>CATALOG</i>[<b>?</b><i>param1</i><b>=</b><i>value1</i>[<b>&</b><i>param2</i><b>=</b><i>value2</i>][...]]
<p>
	All of the parameters will be sought in the Configuration.
	The paramN options are in the {@link #Optional_Parameters
	optional parameters} list. Parameters with these names in the
	Configuration will be included in the URL, otherwise they will
	be omitted.
<p>
	The URL specification is provided to the JDBC_Data_Port <code>{@link
	JDBC_Data_Port#Open_Data_Port Open_Data_Port}</code> method which
	opens a connection to the database server.
<p>
	@param	configuration	The Configuration for this Data_Port.
	@throws	Database_Exception	If no Configuration is provided
		or the JDBC_Data_Port fails to load the driver or make
		the connection to the database server.
*/
public void Open
	(
	Configuration	configuration
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_OPEN) != 0)
	System.out.println (">>> PostgreSQL_Data_Port.Open: Configuration -" + NL
		+ configuration.Description ());
//	Setup the configuration.
if (configuration == null)
	throw new Database_Exception
		(
		ID + NL
		+"An invalid (null) configuration was specified."
		);

//	Don't be case sensitive here.
boolean
	case_sensitive = configuration.Case_Sensitive (false);
try
	{
	//	Set the required parameters.
	if ((DEBUG & DEBUG_OPEN) != 0)
		{
		System.out.println
			("    PostgreSQL_Data_Port.Open: Set_Conditionally (URL_Specs) -");
		for (int element = 0;
				 element < URL_Specs.length;
				 element++)
			System.out.println
				("      " + URL_Specs[element][0]
				+" - " + URL_Specs[element][1]);
		}
	configuration.Set_Conditionally (URL_Specs);

	//	Register the configuration, which loads our driver.
	if ((DEBUG & DEBUG_OPEN) != 0)
		System.out.println
			("    PostgreSQL_Data_Port.Open: Configure");
	Configure (configuration);

	//	Assemble the URL.
	String
		URL = "jdbc:"
			+ Port_Type.toLowerCase ()
			+ "://" + Config_Value (Configuration.HOST);
	String
		URL_part;
	if (! (URL_part = Config_Value (PORT)).equals (""))
		URL += ":" + URL_part;
	URL += "/" + Config_Value (Database.DATABASE);
	if ((DEBUG & DEBUG_OPEN) != 0)
		System.out.println
			("    PostgreSQL_Data_Port.Open: Assembling the URL -" + NL
			+"      Port_Type - " + Port_Type.toLowerCase () + NL
			+"           HOST - " + Config_Value (Configuration.HOST) + NL
			+"           PORT - " + Config_Value (PORT) + NL
			+"       DATABASE - " + Config_Value (Database.DATABASE) + NL
			+"    ->URL - " + URL + NL
			+"    Optional parameters -");

	//	Optional parameters:
	String
		delimiter = "?";
	for (int part = 0;
			 part < Optional_Parameters.length;
			 part++)
		{
		if ((DEBUG & DEBUG_OPEN) != 0)
			System.out.println
				("      " + Optional_Parameters[part]
				+" - " + Config_Value (Optional_Parameters[part]));
		URL_part = Config_Value (Optional_Parameters[part]);
		if (! URL_part.equals (""))
			{
			URL += delimiter + Optional_Parameters[part] + "=" + URL_part;
			if (delimiter.equals ("?"))
				delimiter = "&";
			}
		}

	//	Open the data port connection.
	if ((DEBUG & DEBUG_OPEN) != 0)
		System.out.println
			("    PostgreSQL_Data_Port.Open: Open_Data_Port (" + URL + ")");
	Open_Data_Port (URL);
	}
catch (Configuration_Exception exception)
	{
	configuration.Case_Sensitive (case_sensitive);

	//	Copy the exception with password masked message.
	Configuration_Exception
		configuration_exception = new Configuration_Exception
			(Database_Exception.masked_String (exception.getMessage ()),
			exception.getCause ());
	configuration_exception.setStackTrace (exception.getStackTrace ());
	throw new Database_Exception (configuration_exception);
	}
catch (Database_Exception exception)
	{
	configuration.Case_Sensitive (case_sensitive);
	throw exception;
	}
}

//	Data_Port Implementation
/**	Specialization of the Query method to provide implementation of the
	limit functionality using the server specific capability.
<p>
	@see	JDBC_Data_Port#Query(String, int)
*/
public Vector Query
	(
	String	SQL_query,
	int		limit
	)
	throws Database_Exception
{
if (limit > 0 &&
	SQL_query != null &&
	SQL_query.toUpperCase ().indexOf ("LIMIT") < 0)
	SQL_query += " LIMIT " + limit;

return super.Query (SQL_query, limit);
}

//	Data_Port Implementation
/**	Specialization of the Rename table functionality using server
	specific syntax.
<p>
	@param	table	The name of the table to be affected. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used.
	@param	name	The new name for the table. If this is null,
		nothing is done.
	@throws	Database_Exception	If no catalog or table name is
		available, or the database server rejected the operation.
*/
public void	Rename
	(
	String	table,
	String	name
	)
	throws Database_Exception
{
if (name != null)
	{
	String
		catalog = catalog_name ("Rename table", table);
	table = table_name ("Rename table", table);
	try
		{
		if (Database.Matches
				(Tables (catalog), table, Case_Sensitive_Identifiers))
			Update ("ALTER TABLE "
				+ catalog + Component_Delimiter + table + " RENAME TO " 
				+ name);
		}
	catch (Database_Exception exception)
		{
		throw new Database_Exception
			(
			"Unable to Rename table \"" + table + "\" to \"" + name + "\".",
			exception
			);
		}
	}
}


}
