# -*- coding: utf-8 -*-
#
# Configuration file for the Sphinx documentation builder.
#
# This file does only contain a selection of the most common options. For a
# full list see the documentation:
# http://www.sphinx-doc.org/en/master/config

import glob
import os.path
import subprocess
import sys
from sphinx.errors import SphinxWarning
from docutils import nodes

# -- Helper functions --------------------------------------------------------


# A :issue: role to link to github issues.
# Used like :issue:`2364`
def issue_role(name, rawtext, text, lineno, inliner, options=None, content=None):
    options = options or {}
    try:
        issue_num = int(text.strip())
        if issue_num <= 0:
            raise ValueError
    except ValueError:
        msg = inliner.reporter.error('Invalid issue number: "%s"' % text, line=lineno)
        prb = inliner.problematic(rawtext, rawtext, msg)
        return [prb], [msg]
    template = issue_url + "/{n}"
    ref = template.format(n=issue_num)
    issue_text = "#{issue_no}".format(issue_no=issue_num)
    link = nodes.reference(text=issue_text, refuri=ref, **options)
    return [link], []


# -- Load our extensions -------------------------------------------------
def setup(app):
    # Our own pygments lexers
    from sphinx.highlighting import lexers

    this_dir = os.path.dirname(os.path.realpath(__file__))
    sys.path.insert(0, this_dir)
    from fish_indent_lexer import FishIndentLexer
    from fish_synopsis import FishSynopsisDirective, FishSynopsisLexer

    lexers["fish-docs-samples"] = FishIndentLexer()
    lexers["fish-synopsis"] = FishSynopsisLexer()
    app.add_directive("synopsis", FishSynopsisDirective)

    app.add_config_value("issue_url", default=None, rebuild="html")
    app.add_role("issue", issue_role)


# The default language to assume
highlight_language = "fish-docs-samples"

# -- Project information -----------------------------------------------------

project = "fish-shell"
copyright = "2022, fish-shell developers"
author = "fish-shell developers"
issue_url = "https://github.com/fish-shell/fish-shell/issues"

# Parsing FISH-BUILD-VERSION-FILE is possible but hard to ensure that it is in the right place
# fish_indent is guaranteed to be on PATH for the Pygments highlighter anyway
ret = subprocess.check_output(
    ("fish_indent", "--version"), stderr=subprocess.STDOUT
).decode("utf-8")
# The full version, including alpha/beta/rc tags
release = ret.strip().split(" ")[-1]
# The short X.Y version
version = release.rsplit(".", 1)[0]


# -- General configuration ---------------------------------------------------

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
#
# source_suffix = ['.rst', '.md']
source_suffix = ".rst"

# The master toctree document.
master_doc = "index"

# The languages this uses, also used for content autogenerated by Sphinx.
#
# This is also used if you do content translation via gettext catalogs.
# Usually you set "language" from the command line for these cases.
#
# It also sets the html lang= attribute which would be useful to e.g. screenreaders.
# Currently we only have english.
language = "en"

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This pattern also affects html_static_path and html_extra_path.
exclude_patterns = []

# The name of the Pygments (syntax highlighting) style to use.
pygments_style = None


# -- Options for HTML output -------------------------------------------------

# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
# !!! If you change this you also need to update the @import at the top
# of _static/pygments.css
html_theme_path = ["."]
html_theme = "python_docs_theme"

# Don't add a weird "_sources" directory
html_copy_source = False

# Custom sidebar templates, must be a dictionary that maps document names
# to template names.
#
# The default sidebars (for documents that don't match any pattern) are
# defined by theme itself.  Builtin themes are using these templates by
# default: ``['localtoc.html', 'relations.html', 'sourcelink.html',
# 'searchbox.html']``.
#
html_sidebars = {"**": ["globaltoc.html", "searchbox.html", "localtoc.html"]}


# -- Options for LaTeX output ------------------------------------------------

# The default font is "GNU FreeSans" or something which I've never heard of.
# Make this something that might actually be installed.
latex_elements = {
    'fontpkg': r'''
    \setmainfont{Noto Serif}
    \setsansfont{Noto Sans}
    \setmonofont{Noto Sans Mono}
    ''',
}

# Grouping the document tree into LaTeX files. List of tuples
# (source start file, target name, title,
#  author, documentclass [howto, manual, or own class]).
latex_documents = [
    (
        master_doc,
        "fish-shell.tex",
        "fish-shell Documentation",
        "fish-shell developers",
        "manual",
    ),
]

# The default pdflatex doesn't handle unicode.
# Switch to an engine that does (why pdflatex still exists and is still the default? I don't know)
latex_engine = 'xelatex'

# -- Options for manual page output ------------------------------------------


def get_command_description(path, name):
    """ Return the description for a command, by parsing its synopsis line """
    with open(path) as opened:
        for line in opened:
            if line.startswith(name + ' - '):
                _, desc = line.split(' - ', 1)
                return desc.strip()
            elif line.startswith('``' + name + '`` - '):
                _, desc = line.split('`` - ', 1)
                return desc.strip('`')
    raise SphinxWarning("No description in file %s" % os.path.basename(path))

# Newer sphinxen apparently create another subdirectory which breaks our man lookup.
# Unbreak it (#7996)
man_make_section_directory = False

# One entry per manual page. List of tuples
# (source start file, name, description, authors, manual section).
man_pages = [
    (master_doc, "fish-doc", "", [author], 1),
    ("tutorial", "fish-tutorial", "", [author], 1),
    ("language", "fish-language", "", [author], 1),
    ("interactive", "fish-interactive", "", [author], 1),
    ("relnotes", "fish-releasenotes", "", [author], 1),
    ("completions", "fish-completions", "", [author], 1),
    (
        "fish_for_bash_users",
        "fish-for-bash-users",
        "",
        [author],
        1,
    ),
    ("faq", "fish-faq", "", [author], 1),
]
for path in sorted(glob.glob("cmds/*")):
    docname = os.path.splitext(path)[0]
    cmd = os.path.basename(docname)
    man_pages.append((docname, cmd, get_command_description(path, cmd), "", 1))


# -- Options for Texinfo output ----------------------------------------------

# Grouping the document tree into Texinfo files. List of tuples
# (source start file, target name, title, author,
#  dir menu entry, description, category)
texinfo_documents = [
    (
        master_doc,
        "fish-shell",
        "fish-shell Documentation",
        author,
        "fish-shell",
        "the friendly interactive shell",
        "Miscellaneous",
    )
]


# -- Options for Epub output -------------------------------------------------

# Bibliographic Dublin Core info.
epub_title = project

# The unique identifier of the text. This can be a ISBN number
# or the project homepage.
#
# epub_identifier = ''

# A unique identification for the text.
#
# epub_uid = ''

# A list of files that should not be packed into the epub file.
epub_exclude_files = ["search.html"]

# Enable smart-quotes
# default action is 'qDe': quotes, Dashes, ellipsis. Skip dashes for --options
smartquotes = True
smartquotes_action = 'qe'

linkcheck_ignore = [r'https://github.com/fish-shell/fish-shell/issues/\d+']
