/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <stdio.h>
#include <string.h>
#include <stack.h>
#include "RCX_Disasm.h"
#include "RCX_Constants.h"
#include "RCX_SourceTag.h"

#ifdef DEBUG
//#define CHECK_LENGTHS
#endif

static int ArgsLength(ULong args);
static void SPrintOutputNames(char *argText, const UByte outs);
static void SPrintCondition(char *text, const UByte *code);
static void SPrintValue(char *value, int type, short data);
static int ComputeOffset(UByte b1, UByte b2=0);


#define LOOKUP(i,a)	(((unsigned)(i)<sizeof(a)/sizeof(char*)) ? a[i] : "?")
#define WORD(ptr)	((short)((((ptr)[1]) << 8) + ((ptr)[0])))

static const char *inputTypeNames[] = { "None", "Switch", "Temp", "Light", "Angle" };
static const char *inputModeNames[] = { "Raw", "Boolean", "Edge", "Pulse", "Percent", "Celcius", "Fahrenheit", "Angle" };
static const char *outputDirName[] = { "Rev", "Flip", "Fwd" };
static const char *outputModeName[] = { "Float", "Off", "On" };
static const char *relNames[] = { "<=", ">=", "!=", "==" };
static const char *typeNames[] = {
	"Var", "Timer", "Const", "%3", "Random", "TachoCount", "TachoSpeed", "MotorCurrent",
	"Program", "Input", "%10", "%11", "%12", "%13", "Watch", "Message",
	"AGC", "%17", "ScoutRules", "SensorParam", "TimerLimit", "Counter", "CounterLimit", "ActiveEvents", 
	"EventFeedback", "EventState", "FastTimer", "ClickCounter", "UpperLimit", "LowerLimit", "Hysteresis", "ClickTime",
	"%32", "SerSetting", "BatteryLevel", "FirmwareVersion", "Indirect"};

RCX_Disasm gRCX_Disasm;

enum DataFormat
{
	kAF_None = 0,
	kAF_Skip8,
	kAF_Skip16,
	kAF_Raw8,
	kAF_Raw16,
	kAF_Value8,
	kAF_Value16,
	kAF_Var,
	
	kAF_Outputs,
	kAF_OutputMode,
	kAF_OutputDir,	
	kAF_InputType,
	kAF_InputMode,
	
	kAF_Condition,
	kAF_Jump8,
	kAF_Jump16,
	kAF_Offset8,
	kAF_Offset16
};

static const int argFormatLengths[]={
	0, 1, 2, 1, 2, 2, 3, 1,
	1, 1, 1, 1, 1,
	5, 1, 2, 1, 2
};


#define kArgFormatWidth 5
#define kArgFormatMask	((1 << kArgFormatWidth) - 1)

#define ARG(index, format)	((format) << (index) * kArgFormatWidth)

#define ARGS1(f0)				(ARG(0,f0))
#define ARGS2(f0,f1)			(ARG(0,f0) + ARG(1,f1))
#define ARGS3(f0,f1,f2)			(ARG(0,f0) + ARG(1,f1) + ARG(2,f2))
#define ARGS4(f0,f1,f2,f3)		(ARG(0,f0) + ARG(1,f1) + ARG(2,f2) + ARG(3,f3))
#define ARGS5(f0,f1,f2,f3,f4)	(ARG(0,f0) + ARG(1,f1) + ARG(2,f2) + ARG(3,f3) + ARG(4,f4))



class Instruction
{
public:
	const char*	fName;
	UByte		fOpcode;
	ULong		fArgs;
};


static Instruction sInitData[] = {

{ "out", 0x21, ARGS1(kAF_OutputMode) },
{ "pwr", 0x13, ARGS2(kAF_Outputs, kAF_Value8) },
{ "dir", 0xe1, ARGS1(kAF_OutputDir) },
{ "senm", 0x42, ARGS2(kAF_Raw8, kAF_InputMode) },
{ "sent", 0x32, ARGS2(kAF_Raw8, kAF_InputType) },

{ "wait", 0x43, ARGS1(kAF_Value16) },
{ "playt", 0x23, ARGS2(kAF_Raw16, kAF_Raw8) },
{ "plays", 0x51, ARGS1(kAF_Raw8) },
{ "view", 0x33, ARGS1(kAF_Value16) },

{ "setv", 0x14, ARGS2(kAF_Var, kAF_Value16) },
{ "sumv", 0x24, ARGS2(kAF_Var, kAF_Value16) },
{ "subv", 0x34, ARGS2(kAF_Var, kAF_Value16) },
{ "divv", 0x44, ARGS2(kAF_Var, kAF_Value16) },
{ "mulv", 0x54, ARGS2(kAF_Var, kAF_Value16) },
{ "andv", 0x84, ARGS2(kAF_Var, kAF_Value16) },
{ "orv", 0x94, ARGS2(kAF_Var, kAF_Value16) },
{ "absv", 0x74, ARGS2(kAF_Var, kAF_Value16) },
{ "sgnv", 0x64, ARGS2(kAF_Var, kAF_Value16) },

{ "start", 0x71, ARGS1(kAF_Raw8) },
{ "stop", 0x81, ARGS1(kAF_Raw8) },
{ "stop", 0x50, 0},

{ "msgz", 0x90, 0},
{ "msg", 0xb2, ARGS1(kAF_Value8) },
{ "senz", 0xd1, ARGS1(kAF_Raw8) },
{ "logz", 0x52, ARGS1(kAF_Raw16)},
{ "log", 0x62, ARGS1(kAF_Value8) },
{ "polld", 0xa4, ARGS2(kAF_Raw16, kAF_Raw16)},
{ "setw", 0x22, ARGS2(kAF_Raw8, kAF_Raw8) },
{ "txs", 0x31, ARGS1(kAF_Raw8) },
{ "offp", 0x60, 0},

{ "tmrz", 0xa1, ARGS1(kAF_Raw8) },
{ "calls", 0x17, ARGS1(kAF_Raw8) },

{ "loops", 0x82, ARGS1(kAF_Value8)},
{ "loopc", 0x37, ARGS1(kAF_Offset8) },
{ "loopcl", 0x92, ARGS1(kAF_Offset16) },

{ "jmpl", 0x72,  ARGS1(kAF_Jump16)},
{ "chkl", 0x95, ARGS2(kAF_Condition, kAF_Offset16) },
{ "chk", 0x85, ARGS2(kAF_Condition, kAF_Offset8) },
{ "jmp", 0x27, ARGS1(kAF_Jump8) },

// opcodes that are primarily for Direct mode
{ "ping", 0x10, 0 },
{ "poll", 0x12, ARGS1(kAF_Value8) },
{ "pollp", 0x15, ARGS5(kAF_Raw8, kAF_Raw8, kAF_Raw8, kAF_Raw8, kAF_Raw8) },
{ "memmap", 0x20, 0 },
{ "_task", 0x25, ARGS4(kAF_Skip8, kAF_Raw8, kAF_Skip8, kAF_Raw16) },
{ "_sub", 0x35, ARGS4(kAF_Skip8, kAF_Raw8, kAF_Skip8, kAF_Raw16) },
{ "_firm", 0x75, ARGS3(kAF_Raw16, kAF_Raw16, kAF_Skip8) },
{ "pollb", 0x30, 0},
{ "delt", 0x40, 0},
{ "delt", 0x61, ARGS1(kAF_Raw8) },
{ "dels", 0x70, 0},
{ "dels", 0xc1, ARGS1(kAF_Raw8)},
{ "prgm", 0x91, ARGS1(kAF_Raw8) },
{ "boot", 0xa5, ARGS5(kAF_Raw8, kAF_Raw8, kAF_Raw8, kAF_Raw8, kAF_Raw8) },
{ "tout", 0xb1, ARGS1(kAF_Raw8) }, 



// CyberMaster
{ "Drive", 0x41, ARGS1(kAF_Raw8) },
//{ "OnWait", 0xc2, ARGS2(kAF_Raw8, kAF_Raw8) },  // superceded by uart
{ "OnWaitDifferent", 0x53, ARGS3(kAF_Raw8,  kAF_Raw8, kAF_Raw8) },
{ "ClearTacho", 0x11, ARGS1(kAF_Raw8) },

// Added for Scout
{ "cntd", 0xa7, ARGS1(kAF_Raw8) },
{ "cnti", 0x97, ARGS1(kAF_Raw8) },
{ "cnts", 0xd4, ARGS2(kAF_Raw8, kAF_Value16) },
{ "cntz", 0xb7, ARGS1(kAF_Raw8) },

{ "decvjn", 0xf2, ARGS2(kAF_Var, kAF_Jump8) },
{ "decvjnl", 0xf3, ARGS2(kAF_Var, kAF_Jump16) },

{ "event", 0x03, ARGS1(kAF_Value16) },
{ "mone", 0xb4, ARGS2(kAF_Value16, kAF_Jump8) },
{ "monel", 0xb5, ARGS2(kAF_Value16, kAF_Jump16) },
{ "monex", 0xb0, 0 },

{ "monal", 0x73, ARGS2(kAF_Raw8, kAF_Jump16) },
{ "monax", 0xa0, 0 },

{ "gdir", 0x77, ARGS1(kAF_OutputDir) },
{ "gout", 0x67, ARGS1(kAF_OutputMode) },
{ "gpwr", 0xa3, ARGS2(kAF_Outputs, kAF_Value8) },

{ "light", 0x87, ARGS1(kAF_Raw8) },
{ "lsbt", 0xe3, ARGS1(kAF_Value16) },
{ "lscal", 0xc0, 0 },
{ "lsh", 0xd3, ARGS1(kAF_Value16) },
{ "lslt", 0xc3, ARGS1(kAF_Value16) },
{ "lsut", 0xb3, ARGS1(kAF_Value16) },


{ "playv", 0x02, ARGS2(kAF_Var, kAF_Raw8) },
{ "pollm", 0x63, ARGS2(kAF_Raw16, kAF_Raw8) },
{ "remote", 0xd2, ARGS1(kAF_Raw16) },
{ "rules", 0xd5, ARGS5(kAF_Raw8,kAF_Raw8,kAF_Raw8,kAF_Raw8,kAF_Raw8) },
{ "scout", 0x47, ARGS1(kAF_Raw8) },
{ "setfb", 0x83, ARGS1(kAF_Value16) },
{ "setp", 0xd7, ARGS1(kAF_Raw8) },
{ "sound", 0x57, ARGS1(kAF_Raw8) },
{ "tmrs", 0xc4, ARGS2(kAF_Raw8, kAF_Value16) },
{ "vll", 0xe2, ARGS1(kAF_Value8) },
{ "rts", 0xf6, 0 },

// new for RCX 2.0
{ "disp", 0xe5, ARGS3(kAF_Skip8, kAF_Raw8, kAF_Value16) },
{ "playz", 0x80, 0},
{ "uart", 0xc2, ARGS2(kAF_Raw8, kAF_Raw8) },
{ "set", 0x05, ARGS2(kAF_Value8, kAF_Value16) },
{ "dele", 0x06, 0 },
{ "sete", 0x93, ARGS3(kAF_Raw8, kAF_Raw8, kAF_Raw8) },
{ "mute", 0xd0, 0 },
{ "speak", 0xe0, 0 },
{ "cale", 0x4, ARGS4(kAF_Raw8, kAF_Raw8, kAF_Raw8, kAF_Raw8) },
{ "msgs", 0xf7, ARGS1(kAF_Raw8) },

// new for Spybotics
//{ "find", 0xd5, ARGS4(kAF_Var, kAF_Raw8, kAF_Raw8, kAF_Value8) },
{ "playm", 0xc7, ARGS1(kAF_Raw8) },
{ "playsv", 0xe7, ARGS1(kAF_Var) },
//{ "polle", 0x11, ARGS1(kAF_Raw8) },
{ "pop", 0x01, ARGS1(kAF_Raw8) },
//{ "push", 0xe3, ARGS1(kAF_Value16) },
{ "relz", 0x36, 0},
};

static Instruction *sDispatch[256] = { 0};
static int sInited = 0;

void InitDispatch();

class SrcListState
{
public:
	bool	fEnabled;
	short	fIndex;
	long	fOffset;
};


RCX_Disasm::RCX_Disasm()
{
	InitDispatch();
}


void RCX_Disasm::Print(RCX_Printer *dst, const UByte *code, int length)
{
	Print(dst, code, length, 0, 0, 0);
}


void RCX_Disasm::Print(RCX_Printer *dst, const UByte *code, int length,
		RCX_SourceFiles *sf, const RCX_SourceTag *tags, int tagCount)
{
	UShort pc = 0;
	RCX_Result result;
	stack<SrcListState> stateStack;
	
	// ignore tags if no source files
	if (!sf) tagCount = 0;
		
	while(true)
	{
		// tag processing happens before length check to allow
		// fragment end tags (which have a pc at the end of
		// the fragment) to be processed
		while (tagCount && (tags->fAddress <= pc) )
		{
			switch(tags->fType)
			{
				case RCX_SourceTag::kBegin:
				case RCX_SourceTag::kBeginNoList:
				{
					SrcListState ns;
					ns.fEnabled = (tags->fType == RCX_SourceTag::kBegin);
					ns.fIndex = tags->fSrcIndex;
					ns.fOffset = tags->fSrcOffset;
					stateStack.push(ns);
					break;
				}
				case RCX_SourceTag::kNormal:
				{
					if (stateStack.empty()) break;
					SrcListState &s = stateStack.top();
					if (tags->fSrcIndex != s.fIndex) break;
					if (s.fEnabled)
						s.fOffset = sf->Print(dst, s.fIndex, s.fOffset, tags->fSrcOffset) + 1;
					break;
				}
				case RCX_SourceTag::kEnd:
				{
					if (stateStack.empty()) break;
					SrcListState &s = stateStack.top();
					if (s.fEnabled)
						sf->Print(dst, s.fIndex, s.fOffset, tags->fSrcOffset);
					stateStack.pop();
					break;
				}
			}
			
			tags++;
			tagCount--;
		}
		
		if (length <= 0) break;
	
		result = Print1(dst, code, length, pc);
		pc += result;
		code += result;
		length -= result;
	}
}


RCX_Result RCX_Disasm::Print1(RCX_Printer *dst, const UByte *code, int length, UShort pc)
{
	char text[256];
	char line[256];

	RCX_Result result = SPrint1(text, code, length, pc);
	
	if (result < 1)
	{
		result = 1;
		sprintf(text, "?");
	}

	sprintf(line, "%03d %-32s ", pc, text);
	dst->Print(line);
	
	for(int i=0; i<result; i++)
	{
		sprintf(line, "%02x ", code[i]);
		dst->Print(line);
	}
	sprintf(line, "\n");
	dst->Print(line);
	
	return result;
}	


static void SPrintArg(char *text, ULong format, const UByte *code, UShort pc);

void SPrintArg(char *text, ULong format, const UByte *code, UShort pc)
{
	char buf[32];
	
	switch(format)
	{	
		case kAF_Raw8:
			sprintf(text, "%d", code[0]);
			break;
		case kAF_Raw16:
			sprintf(text, "%d", (short)WORD(code));
			break;
		case kAF_Value8:
			SPrintValue(text, code[0], code[1]);
			break;
		case kAF_Value16:
			SPrintValue(text, code[0], (short)WORD(code+1));
			break;	
		case kAF_Var:
			sprintf(text, "var[%d]", code[0]);
			break;
		case kAF_Outputs:
			SPrintOutputNames(text, code[0] & 7);
			break;
		case kAF_OutputMode:
			SPrintOutputNames(buf, code[0] & 7);
			sprintf(text, "%s, %s", buf, LOOKUP((code[0] >> 6) & 3, outputModeName));
			break;
		case kAF_OutputDir:
			SPrintOutputNames(buf, code[0] & 7);
			sprintf(text, "%s, %s", buf, LOOKUP((code[0] >> 6) & 3, outputDirName));
			break;
		case kAF_InputType:
			sprintf(text,"%s", LOOKUP(code[0], inputTypeNames));
			break;
		case kAF_InputMode:
			sprintf(text,"%s", LOOKUP((code[0] >> 5) & 7, inputModeNames));
			break;
		case kAF_Condition:
			SPrintCondition(text, code);
			break;
		case kAF_Jump8:
			sprintf(text, "%d", pc + ComputeOffset(code[0], 0)); 
			break;
		case kAF_Jump16:
			sprintf(text, "%d", pc + ComputeOffset(code[0], code[1])); 
			break;
		case kAF_Offset8:
			sprintf(text, "%d", pc + (signed char)code[0]);
			break;
		case kAF_Offset16:
			sprintf(text, "%d", pc + (short)WORD(code));
			break;
		case kAF_None:
		case kAF_Skip8:
		case kAF_Skip16:
		default:
			text[0] = 0;
			break;		
	}
}


RCX_Result RCX_Disasm::SPrint1(char *text, const UByte *code, int length, UShort pc)
{
	int iLength;
	Instruction *inst;
	char argText[256];
	UByte op;
	ULong args;	
	bool argPrinted = false;
	const UByte *ptr;
	
	// lookup the instruction
	if (length < 1) return -1;
	op = *code;
	inst = sDispatch[op];
	if (!inst) return -1;
	
	// make sure there's enough code for entire instruction
	iLength = ArgsLength(inst->fArgs) + 1;
	if (length < iLength) return -1;

	// print the instruction name
	sprintf(text,"%-10s ", inst->fName);
	text += strlen(text);
	
	// print the args
	ptr = code+1;
	for(args = inst->fArgs; args; args>>=kArgFormatWidth)
	{
		int af = args & kArgFormatMask;
		
		SPrintArg(argText, af, ptr, pc + (ptr - code));
		if (argText[0])
		{
			sprintf(text, "%s%s", argPrinted ? ", " : "", argText);
			text += strlen(text);
			argPrinted = true;
		}
		
		ptr += argFormatLengths[af];
	}

	return iLength;	
}


void InitDispatch()
{
	unsigned i;
	
	if (sInited) return;
	
	for(i=0; i < sizeof(sInitData) / sizeof(Instruction); i++)
	{
		int op = sInitData[i].fOpcode;
		
		#ifdef DEBUG
		if (sDispatch[op])
			printf("opcode %02x already defined\n", op);
		#endif
		
		sDispatch[op ^ 0x8] = &sInitData[i];
		sDispatch[op] = &sInitData[i];
		
		
		#ifdef CHECK_LENGTHS
		// check length
		int argLen = op & 7;
		if (argLen > 5) argLen -= 6;
		
		int computedLen = ArgsLength(sInitData[i].fArgs);
		
		if (computedLen != argLen)
			printf("opcode %02x: possible wrong length of %d\n", op, computedLen);
			
		#endif
	}
	
	sInited = 1;
}


void SPrintOutputNames(char *argText, const UByte outs)
{
	char *ptr = argText;
	
	if (outs)
	{
		// list outputs
		for(int i=0; i<3; i++)
			if (outs & (1 << i))
				*ptr++ = (char)('A' + i);
	}
	else
		*ptr++ = '0';

	*ptr = 0;
}


int ArgsLength(ULong args)
{
	int length = 0;
	
	for( ; args; args >>= kArgFormatWidth)
	{
		length += argFormatLengths[args & kArgFormatMask];
	}
	
	return length;
}


void SPrintValue(char *text, int type, short data)
{
	switch(type)
	{
		case 0:
			sprintf(text, "var[%d]", data);
			break;
		case 2:
			sprintf(text,"%d", data);
			break;
		default:
			sprintf(text, "%s(%d)", LOOKUP(type, typeNames), data);
			break;
	}
}


void SPrintCondition(char *text, const UByte *code)
{
	char v1Text[16];
	char v2Text[16];
		
	SPrintValue(v1Text, (code[0] & 0x3f), (short)WORD(code+2));
	SPrintValue(v2Text, (code[1] & 0x3f), code[4]); 
	sprintf(text, "%s %s %s", v1Text, LOOKUP((code[0]>>6) & 3, relNames), v2Text);
}


int ComputeOffset(UByte b1, UByte b2)
{
	int x = (b1 & 0x7f) + (b2 << 7);
	
	if (b1 & 0x80) x = -x;
	
	return x;
}


void RCX_Printer::Print(const char *text)
{
	Print(text, strlen(text));
}
