/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * xf_io.h: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: xf_io.h,v 2.13 2003/10/19 12:48:12 blusseau Exp $
 */

#ifndef __XF_IO_H__
#define __XF_IO_H__

#include <netinet/in.h>
#include <time.h>
#include "cnx_if_detect.h"
#include "ged_if.h"
#include "g_string_ref.h"

#include "common_defines.h"

/*********************************************/
/* structure defining the IO of a connection */
/*********************************************/
typedef struct
{
	/* const values */
	int sock_fd;					/* the socket descriptor */
	time_t start_time;			/* creation time of this entry */
	struct in_addr user_ip;		/* user ip */
	CNX_IF_TYPE if_type;       /* type of the network interface this connection uses to go out */
	
	/* variable values */
	GList *incoming_commands;  /* glist of g_string, each one containing a string to send */
										/* g_string stored here are ALWAYS ended by a '|' */
	GString *in_partial;			/* partially received string */
	GList *outgoing_commands;	/* glist of g_string_ref, each one containing a string to send */

	CLOSE_STATUS closed;			/* socket status */
	time_t closed_date;			/* date when the closed variable was changed */

	/* counters */
	guint64 in_data;				/* downloaded bytes */
	guint64 out_data;				/* uploaded bytes */
} XF_IO;

#define XFIO_SET_CLOSE_STATUS(xfio,status)		{(xfio)->closed=(status);(xfio)->closed_date=gl_cur_time;}

/* ------------------------------------------------------------------------------------ */
/* --------------------- XF_IO creation/destruction functions ------------------------- */
/* ------------------------------------------------------------------------------------ */
/*************************************************************************************/
/* create a new XF_IO. The XF_IO entry is automatically registered in the IO handler */
/*************************************************************************************/
/* if remote_addr ==NULL, the function computes it */
/***************************************************/
XF_IO *create_xfio(int sock_fd, struct in_addr *remote_addr);

/*************************************************************************************/
/* delete a XF_IO. The XF_IO entry is automatically unregistered from the IO handler */
/* and the socket is closed.                                                         */
/*************************************************************************************/
void delete_xfio(XF_IO *to_del);

/* ------------------------------------------------------------------------------------ */
/* --------------------------- XF_IO incoming functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */
/************************************************************/
/* same as peek_first_string_of_glist except the GString is */
/* also removed from the GList. You must free it yourself   */
/* when you don't need it anymore.                          */
/************************************************************/
/* output: the pointer or NULL */
/*******************************/
GString *xfio_take_first_string_of_glist_incoming(XF_IO *xfio);

/* ------------------------------------------------------------------------------------ */
/* --------------------------- XF_IO outgoing functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */

/********************************************************/
/* append a copy of the given string to the given glist */
/********************************************************/
void xfio_append_new_cmd_to_glist_outgoing(XF_IO *xfio, GString_ref *string);

#define send_const_str_to_xfio(xfio,const_str)      {\
               GString_ref *__local_str;\
               __local_str=g_string_ref_new(const_str); \
               xfio_append_new_cmd_to_glist_outgoing(xfio,__local_str); \
               g_string_ref_free(__local_str);}

#define send_gstr_to_xfio(xfio,gstr)      {\
               GString_ref *__local_str;\
               __local_str=g_string_ref_new(gstr->str); \
               xfio_append_new_cmd_to_glist_outgoing(xfio,__local_str); \
               g_string_ref_free(__local_str);}

/* for the following function, the gstring reference is stolen, don't free it later */
#define send_stolen_gstr_to_xfio(xfio,gstr)      {\
               GString_ref *__local_str;\
               __local_str=conv_gstring_to_gstring_ref(gstr); \
               xfio_append_new_cmd_to_glist_outgoing(xfio,__local_str); \
               g_string_ref_free(__local_str);}

/* ------------------------------------------------------------------------------------ */
/* ----------------------------- XF_IO misc functions --------------------------------- */
/* ------------------------------------------------------------------------------------ */
/************************/
/* return I/O statistic */
/************************/
void xfio_get_io_stats(guint64 *incoming_bytes, guint64 *outgoing_bytes);

/* ------------------------------------------------------------------------------------ */
/* ----------------------------- XF_IO misc test macros ------------------------------- */
/* ------------------------------------------------------------------------------------ */
#define XFIO_IS_CLOSED(xfio)		(xfio->closed==CNX_CLOSED)
#define XFIO_INPUT_AVAILABLE(xfio)	(xfio->incoming_commands!=NULL)

/* ------------------------------------------------------------------------------------ */
/* ------------------------- GED initialization function ------------------------------ */
/* ------------------------------------------------------------------------------------ */
/************************************************************/
/* function initializing the handler of all the connections */
/************************************************************/
GED_CALLBACKS *xf_io_handler_init(void);

#endif
