/* DChub - a Direct Connect Hub for Linux
 *
 * Copyright (C) 2001 Eric Prevoteau
 *
 * xmltools.h: Copyright (C) Yves BLUSSEAU
 *
 * $Id: users_xml.h,v 2.14 2003/11/22 18:15:47 ericprev Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __USERS_XML_H__
#define __USERS_XML_H__

#ifdef WITH_GCRYPT
#include <gcrypt.h>
#define MD5_DIGEST_LENGTH 16
#define MD5_ALGO(source,len,dest)   gcry_md_hash_buffer(GCRY_MD_MD5,dest,source,len)
#else
#include <openssl/md5.h>
#define MD5_ALGO MD5
#endif

/* Levels use by the hub */
extern const char OPERATOR[];
extern const char KICK[];
extern const char BAN[];
extern const char UNBAN[];
extern const char REDIRECT[];
extern const char MNGUSERS[];
extern const char STARTPRG[];
extern const char MNGDB[];
extern const char MNGCLUSTERS[];
extern const char SIGNKICK[];

#define MAX_NICK_LEN 32 /* MAX LEN FOR NICKNAME */

#define ENC_MD5_PASSWD_LEN (MD5_DIGEST_LENGTH*8/6)+2

#define MASTER_LEVEL 32767 /* Max value for a gint16 */

/*********************************/
/* Encrypt and encode a password */
/***************************************************************************/
/* input:  msg: the password to encrypt                                    */
/*         output: a buffer to receive the encrypt password                */
/*                 The buffer must have space for ENC_MD5_PASSWD_LEN bytes */
/***************************************************************************/
void encrypt_password(const unsigned char *msg, unsigned char *output);

/*************************************/
/* load the users XML file in memory */
/*************************************/
/* return: TRUE=ok, FALSE=error      */
/*************************************/
gboolean loadUsersFile(const char* filename);

/******************************************/
/* retrieve the level for the given right */
/***************************************************************/
/* input:  right: is the right to search                       */
/* output: level if the right is found, MASTER_LEVEL otherwise */
/***************************************************************/
gint16 get_right_level(const char *right_name);

/*********************************************/
/* retrieve the attributes of the given nick */
/*******************************************************************/
/* input: nick: is the nick to search                              */
/*        level: is a pointeur to receive the level of the user    */
/*        (can be NULL if you don't want to retrieve it).          */
/*        *password: is a pointeur to receive the password of      */
/*        the user (can be NULL if you don't want to retrieve it). */
/*        The password must be free()d when no more required.      */
/*        protected: is a pointeur to recaive a boolean that said  */
/*        if the user is kickage or not (can be NULL if you don't  */
/*        want to retrieve it).                                    */
/*        *comment: is a pointeur to receive comment concerning    */
/*        the user (can be NULL if you don't want to retrieve it). */
/*        The comment must be free()d when no more required.       */
/*                                                                 */
/* output: TRUE if the user is found                               */
/*******************************************************************/
gboolean xml_get_user(const char *nick, gint16 *level, char **password, gboolean *protected, char **comment);

/***********************************/
/* change the properties of a user */
/***********************************/
/*******************************************************************/
/* input: nick: is the nick to search                              */
/*        level: is a pointeur to the new level of the user        */
/*        (can be NULL).                                           */
/*        *password: is a pointeur to the new password of the user */
/*        (can be NULL).                                           */
/*        protected: is a pointeur to the new value that said      */
/*        if the user is kickablee or not (can be NULL).           */
/*        *comment: is a pointeur to the new comment concerning    */
/*        the user (can be NULL).                                  */
/*                                                                 */
/* output: TRUE if the user is found                               */
/*******************************************************************/
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_chg_user_properties(const char *nick, gint16 *level, char **password,
								 gboolean *protected, char **comment);

/******************/
/* add a new user */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_add_user(char *nick, gint16 level, const char *password, gboolean protected, const char *comment);

/*****************/
/* delete a user */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_del_user(char *nicktodel);

/*****************/
/* rename a user */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_rename_user(const char *old_nick, const char* new_nick);

void xml_dump_users(const char* filename);

void xml_free_users_resource();

/***************************************************************/
/* create the list of all users having the given minimum level */
/***************************************************************/
/* the returned Gstring must be freed when no more useful */
/**********************************************************/
void full_users_list_with_level(GPtrArray *gpa, gint16 minlevel);

#endif /* __USERS_XML_H__ */
