/* DChub - a Direct Connect Hub for Linux
 *
 * Copyright (C) 2001 Eric Prevoteau
 *
 * users_xml.c: Copyright (C) Yves BLUSSEAU
 *
 * $Id: users_xml.c,v 2.25 2003/11/22 18:15:46 ericprev Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <string.h>
#include <glib.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include "hl_locks.h"
#include "xmltools.h"
#include "users_xml.h"

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>    /* Gray Watson's library */
#define show_alloc() dmalloc_log_unfreed()
#define show_stats() dmalloc_log_stats()
#else
#define show_alloc()    /* nothing */
#define show_stats()    /* nothing */
#endif

/* Levels use by the hub */
const char OPERATOR[]="OPERATOR";
const char KICK[]	 ="KICK";
const char BAN[]	 ="BAN";
const char UNBAN[]	 ="UNBAN";
const char REDIRECT[]="REDIRECT";
const char MNGUSERS[]="MNGUSERS";
const char STARTPRG[]="STARTPRG";
const char MNGDB[]	 ="MNGDB";
const char MNGCLUSTERS[]="MNGCLUSTERS";
const char SIGNKICK[]="SIGNKICK";

/* entities used in XML */
const char ENT_PRIVILEGES[]	="privileges";
const char ENT_RIGHT[]		="right";
const char ENT_USERS[]		="users";
const char ENT_NICK[]		="nick";

/* attribute used in XML */
const char ATTR_LEVEL[]		="level";
const char ATTR_NAME[]		="name";
const char ATTR_PASSWORD[]	="password";
const char ATTR_PROTECTED[]	="protected";

static char *xml_file=NULL;
time_t time_user_file_loaded=0;
HL_MUTEX users_doc_lock=HL_MUTEX_INIT;
xmlDocPtr user_doc=NULL;
HL_MUTEX  levels_table_lock=HL_MUTEX_INIT;
GHashTable *levels_hash_table=NULL;

#define MAX_BUF_CONV_LEN		256	/* MAX LEN FOR STRING IN UTF8 FOR ALL XML DATAS */
#define DEFINE_BUF_CONV(var)	char var[MAX_BUF_CONV_LEN]

#define STR_LAT1ToUTF8(in,out) 						\
   {												\
	   if (in) {									\
	       int outlen=MAX_BUF_CONV_LEN;				\
		   int inlen=strlen(in);					\
           isolat1ToUTF8(out, &outlen, in, &inlen);	\
		   out[outlen]='\0';						\
       } else										\
		   *out='\0';								\
	}												\

#define STR_UTF8ToLAT1(in,out) 						\
   {												\
	   if (in) {									\
	       int outlen=MAX_BUF_CONV_LEN;				\
	       int inlen=strlen(in);					\
	       UTF8Toisolat1(out, &outlen, in, &inlen);	\
	       out[outlen]='\0';						\
       } else										\
	       *out='\0';								\
   }												\

#define MALLOC_STR_UTF8ToLAT1(in,out) 				\
   {												\
	   if (in) {									\
           int inlen=strlen(in);					\
           int outlen=inlen;						\
           out=malloc(outlen);						\
	       UTF8Toisolat1(out, &outlen, in, &inlen);	\
	       *((out)+outlen)='\0';					\
       } else										\
           out=NULL;								\
   }

static void to64(char *s, unsigned long v, int n) {
    static unsigned char itoa64[] =         /* 0 ... 63 => ASCII - 64 */
        "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

    while (--n >= 0) {
        *s++ = itoa64[v&0x3f];
        v >>= 6;
    }
}

void encrypt_password(const unsigned char *msg, unsigned char *output) {	
	unsigned char MD5bin[16];
	unsigned char *final=output;
	
	unsigned long l;

	MD5_ALGO(msg,strlen(msg), MD5bin);
	l = (MD5bin[0]<<16)  | (MD5bin[1]<<8)   | (MD5bin[2]);  to64(final, l, 4); final += 4;
	l = (MD5bin[3]<<16)  | (MD5bin[4]<<8)   | (MD5bin[5]);  to64(final, l, 4); final += 4;
	l = (MD5bin[6]<<16)  | (MD5bin[7]<<8)   | (MD5bin[8]);  to64(final, l, 4); final += 4;
	l = (MD5bin[9]<<16)  | (MD5bin[10]<<10) | (MD5bin[11]); to64(final, l, 4); final += 4;
	l = (MD5bin[12]<<16) | (MD5bin[13]<<10) | (MD5bin[14]); to64(final, l, 4); final += 4;
	l =                    (MD5bin[15]);                    to64(final, l, 2); final += 2;
	*final='\0';
}

void refresh_level_table() {
	xmlXPathObjectPtr search;
	GString* xpath_expr=g_string_new("");
	HL_LOCK_WRITE(levels_table_lock);
	if (levels_hash_table) {
		/* Destroy the old hash */
		g_hash_table_destroy(levels_hash_table);
	}
	levels_hash_table=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);

	g_string_sprintf(xpath_expr,"//%s/%s", ENT_PRIVILEGES, ENT_RIGHT);
	search = getNodeSet (user_doc, xpath_expr->str); /* retrieve all right */
	g_string_free(xpath_expr,TRUE);
	
	if (search) {
		int i;
		xmlNodeSetPtr nodeset = search->nodesetval;
		for (i=0; i < nodeset->nodeNr; i++) {
			xmlNodePtr xml_right_node=nodeset->nodeTab[i];
			xmlChar *xml_name=xmlGetProp(xml_right_node, ATTR_NAME); /* get the right node name */
			if (xml_name) {
				xmlChar *xml_level=xmlGetProp(xml_right_node, ATTR_LEVEL); /* get the level of the node */
				if (xml_level) {
					gint* value=g_new(int,1);
					*value=xmlXPathCastStringToNumber(xml_level);
					// printf("Right Name:[%s] Level:[%d]\n",xml_name,*value);
					if (*value > MASTER_LEVEL) {
						printf ("*** level can't be greater than %d. %s level set to %d\n",
								MASTER_LEVEL, xml_name, MASTER_LEVEL);
						*value=MASTER_LEVEL;
					}
					g_hash_table_insert(levels_hash_table, g_strdup(xml_name), value);
					xmlFree(xml_level);
				}
				xmlFree(xml_name);
			}
		}
		xmlXPathFreeObject(search);
	}
	HL_UNLOCK_WRITE(levels_table_lock);
}

gboolean refresh_user_doc() {
	struct stat user_file_stat;
	if (!stat(xml_file, &user_file_stat)) {
		if (time_user_file_loaded < user_file_stat.st_mtime) {
			HL_LOCK_WRITE(users_doc_lock);
			/* Recheck the time: avoid multiple reload by different processus */
			if (!stat(xml_file, &user_file_stat) &&
				(time_user_file_loaded < user_file_stat.st_atime)) {
				/* load the new file in memory */
				xmlDocPtr new_user_doc=loadXMLFile(xml_file);
				if (new_user_doc) {			
					/* replace old DOM with the new one */
					if (user_doc) {
						xmlFreeDoc(user_doc);
						printf ("Users file %s reloaded.\n",xml_file);
					}
					user_doc=new_user_doc;
					refresh_level_table();
					time_user_file_loaded=time(NULL);
				}
			}
			HL_UNLOCK_WRITE(users_doc_lock);	
		}
	}
	return user_doc ? TRUE : FALSE; /* ok if a DOM already in memory */
}

gboolean loadUsersFile(const char* filename) {
	gboolean result=FALSE;
	if (filename) {
		HL_LOCK_WRITE(users_doc_lock);
		if (xml_file)
			free(xml_file);
		xml_file=strdup(filename);
		HL_UNLOCK_WRITE(users_doc_lock);
		result=refresh_user_doc();
		if (result)
			printf("Users file %s loaded.\n",filename);
	}
	return result;
}

gboolean xml_save_users() {
	gboolean result=FALSE;
	if (xmlSaveFormatFile(xml_file, user_doc,1) == -1) {
		/* an error occured */
		fprintf(stderr,"Error occured saving file %s: %s\n",xml_file, strerror(errno));
	} else {
		time_user_file_loaded=time(NULL);
		result=TRUE;
	}
	return result;
}

gint16 get_right_level(const char *right_name) {
	gint* result;
	HL_LOCK_READ(levels_table_lock);
	result=g_hash_table_lookup(levels_hash_table, right_name);
	HL_UNLOCK_READ(levels_table_lock);
	return result ? *result : MASTER_LEVEL;
}

/*******************************************/
/* Change/Set the level property of a user */
/*******************************************/
/* note: only use internaly (No lock, etc...) */
/**********************************************/
void set_user_level(xmlNodePtr node, int level) {
	xmlChar* xml_level;
	gboolean must_free=FALSE;

	if (level > MASTER_LEVEL) {
		printf ("*** level can't be greater than %d. level set to %d\n",
				MASTER_LEVEL, MASTER_LEVEL);
		level=MASTER_LEVEL;
	}
	
	if (level == MASTER_LEVEL)
		xml_level="MASTER";
	else {
		xml_level=xmlXPathCastNumberToString(level);
		must_free=TRUE;
	}	
		
	if (xmlHasProp(node,ATTR_LEVEL))
		xmlSetProp(node,ATTR_LEVEL, xml_level);
	else
		xmlNewProp(node, ATTR_LEVEL, xml_level);
	if (must_free)
		xmlFree(xml_level);
}

/**********************************************/
/* Change/Set the password property of a user */
/**********************************************/
/* note: only use internaly (No lock, etc...) */
/**********************************************/
void set_user_password(xmlNodePtr node, const char* password) {
	xmlAttrPtr attribute=xmlHasProp(node,ATTR_PASSWORD);
	if (password == NULL || strlen(password) == 0) {
		/* delete the property for an empty password */
		if (attribute)
			xmlRemoveProp(attribute);
	} else {
		unsigned char passMD5ascii[ENC_MD5_PASSWD_LEN];
		encrypt_password(password, passMD5ascii);
		if (attribute)
			xmlSetProp(node,ATTR_PASSWORD,passMD5ascii);
		else
			xmlNewProp(node,ATTR_PASSWORD,passMD5ascii);
	}
}

/****************************/
/* Protect/Unprotect a user */
/**********************************************/
/* note: only use internaly (No lock, etc...) */
/**********************************************/
void set_user_protection(xmlNodePtr node, gboolean protect) {
	xmlAttrPtr attribute=xmlHasProp(node, ATTR_PROTECTED);
	if (protect == FALSE) {
		/* delete the property for an empty password */
		if (attribute)
			xmlRemoveProp(attribute);
	} else {
		xmlChar* xml_protected=xmlXPathCastBooleanToString(protect);
		if (attribute)
			xmlSetProp(node, ATTR_PROTECTED, xml_protected);
		else
			xmlNewProp(node, ATTR_PROTECTED, xml_protected);
		xmlFree(xml_protected);
	}
}

gboolean find_user(const char *nicktofind, gint16 *level, char **password, gboolean *protected, char **comment) {
	xmlXPathObjectPtr search;
	xmlNodePtr nick_node=NULL;
	DEFINE_BUF_CONV(nicktofindUTF8);
	GString* xpath_expr=g_string_new("");
	STR_LAT1ToUTF8(nicktofind, nicktofindUTF8);
	g_string_sprintf(xpath_expr,"//%s/%s[@%s=\"%s\"]", ENT_USERS, ENT_NICK, ATTR_NAME, nicktofindUTF8);
	// printf("XPATH_EXPR:%s\n",xpath_expr->str);
	
	HL_LOCK_READ(users_doc_lock);
	search = getNodeSet (user_doc,xpath_expr->str); /* retrieve all nick */
	if (search) {
		xmlNodeSetPtr nodeset = search->nodesetval;
		if (nodeset->nodeNr != 0)
			nick_node=nodeset->nodeTab[0]; /* Only one nick must exists */
		xmlXPathFreeObject(search);
	}
	if (nick_node) {
		if (level) {
			xmlChar *xml_level=xmlGetProp(nick_node, ATTR_LEVEL);
			if (xml_level) {
				if (strcasecmp(xml_level,"MASTER") == 0)
					*level=MASTER_LEVEL;
				else
					*level=atoi(xml_level);
				xmlFree(xml_level);
			}
		}
		if (password) {
			xmlChar *xml_password=xmlGetProp(nick_node, ATTR_PASSWORD);
			if (xml_password) {
				*password=strdup(xml_password);
			}
		}
		if (protected) {
			xmlChar *xml_protected=xmlGetProp(nick_node, ATTR_PROTECTED);
			if (xml_protected) {
				*protected=(xmlStrcmp(xml_protected, (const xmlChar *) "false") ? TRUE : FALSE);
				xmlFree(xml_protected);
			}
		}
		if (comment) {
			xmlChar *xml_comment=xmlNodeGetContent(nick_node);
			if (xml_comment) {
				MALLOC_STR_UTF8ToLAT1(xml_comment, *comment);
				xmlFree(xml_comment);
			}
		}
	}
	HL_UNLOCK_READ(users_doc_lock);
	g_string_free(xpath_expr,TRUE);
	return nick_node ? TRUE : FALSE;
}

/*********************************************/
/* retrieve the attributes of the given nick */
/*******************************************************************/
/* input: nick: is the nick to search                              */
/*        level: is a pointeur to receive the level of the user    */
/*        (can be NULL if you don't want to retrieve it).          */
/*        *password: is a pointeur to receive the password of      */
/*        the user (can be NULL if you don't want to retrieve it). */
/*        The password must be free()d when no more required.      */
/*        protected: is a pointeur to recaive a boolean that said  */
/*        if the user is kickage or not (can be NULL if you don't  */
/*        want to retrieve it).                                    */
/*        *comment: is a pointeur to receive comment concerning    */
/*        the user (can be NULL if you don't want to retrieve it). */
/*        The comment must be free()d when no more required.       */
/*                                                                 */
/* output: TRUE if the user is found                               */
/*******************************************************************/
gboolean xml_get_user(const char *nick, gint16 *level, char **password, gboolean *protected, char **comment) {
	if (!refresh_user_doc())
		return FALSE;
	return find_user(nick, level, password, protected, comment);
}

/******************/
/* add a new user */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_add_user(char *nick, gint16 level, const char *password, gboolean protected, const char* comment) {
	xmlNodePtr cur;
	xmlNodePtr new_user_node=NULL;
	DEFINE_BUF_CONV(strUTF8);
	
	if (!refresh_user_doc() || nick == NULL)
		return FALSE;
	
	HL_LOCK_WRITE(users_doc_lock);
	cur = xmlDocGetRootElement(user_doc)->children;
	if (cur!=NULL) {
		while (cur != NULL && (xmlStrcmp(cur->name, (const xmlChar *) "users"))) {
			cur=cur->next;
		}	
		if (cur != NULL) {
			/* I'm on the users node */
			if (comment)
				STR_LAT1ToUTF8(comment,strUTF8);

			new_user_node=xmlNewChild(cur, NULL, "nick", comment ? strUTF8 : NULL);

			STR_LAT1ToUTF8(nick,strUTF8);
			xmlNewProp(new_user_node, ATTR_NAME, strUTF8);

			if (level != 0)
				set_user_level(new_user_node, level);

			if (password != NULL)
				set_user_password(new_user_node, password);

			set_user_protection(new_user_node, protected);
			
			if (!xml_save_users()) { /* save the file */
				/* an error occured -> Revert */
				xmlUnlinkNode(new_user_node);
				xmlFreeNode(new_user_node);
				new_user_node=NULL;
			}
		}
	}
	HL_UNLOCK_WRITE(users_doc_lock);
	return new_user_node ? TRUE : FALSE;
}

/*****************/
/* delete a user */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_del_user(char *nicktodel) {
	xmlXPathObjectPtr search;
	xmlNodePtr nick_node=NULL;
	gboolean result=FALSE;
	DEFINE_BUF_CONV(nicktodelUTF8);
	GString* xpath_expr;
	
	if (!refresh_user_doc() || nicktodel == NULL)
		return FALSE;

	xpath_expr=g_string_new("");
	STR_LAT1ToUTF8(nicktodel, nicktodelUTF8);
	g_string_sprintf(xpath_expr,"//%s/%s[@%s=\"%s\"]", ENT_USERS, ENT_NICK, ATTR_NAME, nicktodelUTF8);
	
	HL_LOCK_WRITE(users_doc_lock);
	search = getNodeSet (user_doc,xpath_expr->str); /* retrieve all nick */
	if (search) {
		xmlNodeSetPtr nodeset = search->nodesetval;
		if (nodeset->nodeNr != 0)
			nick_node=nodeset->nodeTab[0]; /* Only one nick must exists */
		xmlXPathFreeObject(search);
	}
	if (nick_node) {
		xmlNodePtr parent=nick_node->parent;
		xmlNodePtr temp=xmlCopyNode(nick_node,1);
		xmlUnlinkNode(nick_node);
		result=TRUE;
		if (!xml_save_users()) { /* save the file */
			/* an error occured -> Revert */
			xmlAddChild(parent, temp); /* recreate the node */
			result=FALSE;
		} else {
			xmlFreeNode(temp);
		}
		xmlFreeNode(nick_node);
	}
	HL_UNLOCK_WRITE(users_doc_lock);
	g_string_free(xpath_expr,TRUE);
	return result;
}

/*****************/
/* rename a user */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_rename_user(const char *old_nick, const char* new_nick) {
	xmlXPathObjectPtr search;
	xmlNodePtr nick_node=NULL;
	gboolean result=FALSE;

	DEFINE_BUF_CONV(old_nickUTF8);
	GString* xpath_expr;

	if (!refresh_user_doc() || old_nick == NULL || new_nick == NULL)
		return FALSE;

	STR_LAT1ToUTF8(old_nick, old_nickUTF8);
	xpath_expr=g_string_new("");
	g_string_sprintf(xpath_expr,"//%s/%s[@%s=\"%s\"]", ENT_USERS, ENT_NICK, ATTR_NAME, old_nickUTF8);
	
	HL_LOCK_WRITE(users_doc_lock);
	search = getNodeSet (user_doc,xpath_expr->str); /* retrieve all nick */
	if (search) {
		xmlNodeSetPtr nodeset = search->nodesetval;
		if (nodeset->nodeNr != 0)
			nick_node=nodeset->nodeTab[0]; /* Only one nick must exists */
		xmlXPathFreeObject(search);
	}
	if (nick_node) {
		DEFINE_BUF_CONV(new_nickUTF8);
		STR_LAT1ToUTF8(new_nick,new_nickUTF8);
		xmlSetProp(nick_node, ATTR_NAME, new_nickUTF8);
		result=TRUE;
		if (!xml_save_users()) { /* save the file */
			/* an error occured -> Revert */
			xmlSetProp(nick_node,ATTR_NAME, old_nickUTF8);
			result=FALSE;
		}
	}
	HL_UNLOCK_WRITE(users_doc_lock);
	g_string_free(xpath_expr,TRUE);
	return result;
}

/***********************************/
/* change the properties of a user */
/***********************************/
/*******************************************************************/
/* input: nick: is the nick to search                              */
/*        level: is a pointeur to the new level of the user        */
/*        (can be NULL).                                           */
/*        *password: is a pointeur to the new password of the user */
/*        (can be NULL).                                           */
/*        protected: is a pointeur to the new value that said      */
/*        if the user is kickablee or not (can be NULL).           */
/*        *comment: is a pointeur to the new comment concerning    */
/*        the user (can be NULL).                                  */
/*                                                                 */
/* output: TRUE if the user is found                               */
/*******************************************************************/
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_chg_user_properties(const char *nick, gint16 *level,
								  char **password, gboolean *protected,
								  char **comment) {
	xmlXPathObjectPtr search;
	xmlNodePtr nick_node=NULL;
	gboolean result=FALSE;
	DEFINE_BUF_CONV(nickUTF8);
	GString* xpath_expr;

	if (!refresh_user_doc() || nick == NULL)
		return FALSE;
	
	STR_LAT1ToUTF8(nick, nickUTF8);

	xpath_expr=g_string_new("");
	g_string_sprintf(xpath_expr,"//%s/%s[@%s=\"%s\"]", ENT_USERS, ENT_NICK, ATTR_NAME, nickUTF8);
	
	HL_LOCK_WRITE(users_doc_lock);
	search = getNodeSet (user_doc,xpath_expr->str); /* retrieve all nick */
	if (search) {
		xmlNodeSetPtr nodeset = search->nodesetval;
		if (nodeset->nodeNr != 0)
			nick_node=nodeset->nodeTab[0]; /* Only one nick must exists */
		xmlXPathFreeObject(search);
	}

	if (nick_node) {
		xmlNodePtr temp=xmlCopyNode(nick_node,1);

		if (level)
			set_user_level(nick_node, *level);

		if (password)
			set_user_password(nick_node, *password);

		if (protected)
			set_user_protection(nick_node, *protected);
		
#warning function to set comment to be written

		if (!xml_save_users()) { /* save the file */
			/* an error occured -> Revert */
			xmlReplaceNode(nick_node,temp);
			xmlFreeNode(nick_node);
			result=FALSE;
		} else {
			xmlFreeNode(temp);
			result=TRUE;
		}
	}
	HL_UNLOCK_WRITE(users_doc_lock);
	return result;
}

/******************************************************************/
/* create the list of all registered users with their information */
/******************************************************************/
/* the returned Gstring must be freed when no more useful */
/**********************************************************/
void full_users_list_with_level(GPtrArray *gpa, gint16 minlevel) {
	xmlXPathObjectPtr search;
	xmlNodePtr nick_node=NULL;
	GString *xpath_expr;

	if (refresh_user_doc()) {
		xpath_expr=g_string_new("");
		g_string_sprintf(xpath_expr,"//%s/%s", ENT_USERS, ENT_NICK);
		HL_LOCK_READ(users_doc_lock);
		search = getNodeSet (user_doc, xpath_expr->str); /* retrieve all nick */
		g_string_free(xpath_expr,TRUE);
		if (search) {
			int i;
			int level;
			DEFINE_BUF_CONV(name);
			char *password;
			char *protected;
			DEFINE_BUF_CONV(comment);
			xmlNodeSetPtr nodeset = search->nodesetval;
			for (i=0; i < nodeset->nodeNr; i++) {
				xmlChar *xml_level;
				nick_node=nodeset->nodeTab[i];
				xml_level=xmlGetProp(nick_node, ATTR_LEVEL);
				if (xml_level) {
					if (strcasecmp(xml_level,"MASTER") == 0)
						level=MASTER_LEVEL;
					else
						level=atoi(xml_level);
					if (level >= minlevel) {
						xmlChar *xml_name;
						xmlChar *xml_password;
						xmlChar *xml_protected;
						xmlChar *xml_comment;
						GString* out=g_string_new("");
			
						xml_name=xmlGetProp(nick_node, ATTR_NAME);
						STR_UTF8ToLAT1(xml_name, name);
						if (xml_name)
							xmlFree(xml_name);
						
						xml_password=xmlGetProp(nick_node, ATTR_PASSWORD);
						if (xml_password) {
							password="*crypt*";
						    xmlFree(xml_password);
						} else
							password="*no pass*";
						
						protected="*";
						xml_protected=xmlGetProp(nick_node, ATTR_PROTECTED);
						if (xml_protected) {
							gboolean is_protected=(xmlStrcmp(xml_protected, (const xmlChar *) "false") ? TRUE : FALSE);
							xmlFree(xml_protected);
							if (is_protected)
								protected="*protected*";
						}
						
						xml_comment=xmlNodeGetContent(nick_node);
						STR_UTF8ToLAT1(xml_comment, comment);
						if (xml_comment)
							xmlFree(xml_comment);
						
						if (level == MASTER_LEVEL)
							g_string_sprintf(out,"%s:MASTER:%s:%s:%s",
											  name,
											  password,
											  protected,
											  comment);
						else if (level == -1)
							g_string_sprintf(out,"%s:*DISABLE*:%s:%s:%s",
											  name,
											  password,
											  protected,
											  comment);
						else
							g_string_sprintf(out,"%s:%d:%s:%s:%s",
											  name,
											  level,
											  password,
											  protected,
											  comment);
						g_ptr_array_add(gpa, out);
					}
					xmlFree(xml_level);
				}
			}
			xmlXPathFreeObject(search);
		}
		HL_UNLOCK_READ(users_doc_lock);
	}
}

void xml_dump_users(const char *filename) {
	xmlSaveFormatFile(filename, user_doc,1);
}

void xml_free_users_resource() {
	xmlFreeDoc(user_doc);
	if (levels_hash_table)
		g_hash_table_destroy(levels_hash_table);
}
