/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * user_cnx_lst.h: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: user_cnx_lst.h,v 2.16 2003/10/25 12:28:45 ericprev Exp $
*/

#ifndef __USER_CNX_LST_H__
#define __USER_CNX_LST_H__

#include <glib.h>
#include <time.h>
#include "hl_locks.h"
#include "g_string_ref.h"
#include "cnx_if_detect.h"
#include "xf_io.h"

/*****************************************/
/* structure of a locally connected user */
/*****************************************/
typedef struct
{
	XF_IO *user_xfio;				/* user xfer information */

	/* DC users information */
	const char *user_cnx_type;
	GString *user_nick;
	guint64 shared_size;
	GString *user_mail;
	GString *user_description;
	guint16 user_flag;			/* user flag */
	guint16 ext_flag;			/* Extended flag */
	gint16 level;						/* user level privilege */
	gboolean IsBot;
	GString *client_version;
   
	/* connection timeout */
	time_t last_time;          /* use for Time Out */

	/* flood detection */
	time_t last_hcall;			/* flood detection system */
	time_t last_msg;				/* flood detection system */
	int msg_count;					/* flood detection system */

	/* statistic information */
	int sr_count;						/* number of search send by the user */
	unsigned int in_cnx_count; 	/* log the number of attempt */
	unsigned int out_cnx_count; 	/* log the number of attempt */

	unsigned int ext_display_flag;    /* extended display flag (* a or'ed value of edf enum) */

} LOCAL_USER_CNX_ENTRY;

/************************************************************/
/* function initializing the handler of all the connections */
/************************************************************/
GED_CALLBACKS *user_cnx_entry_handler_init(void);

/* -------------------------------------------------------------------------- */
/* -------------------- creation/destruction functions ---------------------- */
/* -------------------------------------------------------------------------- */

/**********************/
/* create a new entry */
/******************************************************************************************/
/* input: xfio to used [the pointer is stolen, the caller must ignore it after this call] */
/*        nickname to use (must be !=NULL)                                                */
/*        shared_size: size of the shared data of this user                               */
/*        umail, udesc, ucliver: can be NULL                                              */
/******************************************************************************************/
/* output: address of the newly created and inserted LOCAL_USER_CNX_ENTRY * */
/****************************************************************************/
LOCAL_USER_CNX_ENTRY *uc_create_entry(XF_IO *xfio, char *nickname, guint64 shared_size,
									  char *umail, char *udesc, char *ucliver,
									  guint16 user_flag, guint16 ext_flag,
									  gint16 level, gboolean isbot);

/*******************************************************************/
/* remove an entry from the cnx_entry_list and free its ressources */
/*******************************************************************************/
/* this function can be used as value destruction function in g_hash_table_new */
/*******************************************************************************/
void uc_destroy_entry(LOCAL_USER_CNX_ENTRY *luce);

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------ misc functions ---------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************************************/
/* search for the LOCAL_USER_CNX_ENTRY of the given nickname       */
/* output: LOCAL_USER_CNX_ENTRY * of the user or NULL if not found */
/*******************************************************************/
LOCAL_USER_CNX_ENTRY *user_cnx_entry_get_by_nickname(const char *nickname);

/***************************************************************************/
/* rename the given LOCAL_USER_CNX_ENTRY with a uniq invalid user nickname */
/***************************************************************************/
void rename_luce_for_deletion(LOCAL_USER_CNX_ENTRY *luce);

/*************************************/
/* number of users locally connected */
/*************************************/
int user_cnx_entry_get_nb_users(void);

/**************************************/
/* set and clear ext flag of the user */
/**************************************/
void user_cnx_entry_update_ext_flag_by_nickname(const char *nickname, unsigned int flag_to_set, unsigned int flag_to_clear);

#define send_str_to_luce(luce,gstrref)		xfio_append_new_cmd_to_glist_outgoing((luce)->user_xfio,(gstrref))

#define send_const_str_to_luce(luce,str)		{\
					GString_ref *__local_str;\
					__local_str=g_string_ref_new(str); \
					send_str_to_luce(luce,__local_str);	\
					g_string_ref_free(__local_str);}

#define send_gstr_to_luce(luce,gstr)		{\
					GString_ref *__local_str;\
					__local_str=g_string_ref_new(gstr->str); \
					send_str_to_luce(luce,__local_str);	\
					g_string_ref_free(__local_str);}

/* for the following function, the gstring reference is stolen, don't free it later */
#define send_stolen_gstr_to_luce(luce,gstr)		{\
					GString_ref *__local_str;\
					__local_str=conv_gstring_to_gstring_ref(gstr); \
					send_str_to_luce(luce,__local_str);	\
					g_string_ref_free(__local_str);}

#endif
