/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * process_ucmd.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: process_ucmd.c,v 2.45 2003/11/22 07:01:07 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <ctype.h>
#ifdef WITH_GCRYPT
#include <gcrypt.h>
#define MD4_DIGEST_LENGTH 16
#define MD4_ALGO(source,len,dest)   gcry_md_hash_buffer(GCRY_MD_MD4,dest,source,len)
#else
#include <openssl/md4.h>
#define MD4_ALGO MD4
#endif
#include <glib.h>

#include "process_ucmd.h"
#include "user_cnx_lst.h"
#include "macro.h"
#include "global_user_if.h"
#include "multi_public_chat.h"
#include "gvar.h"
#include "hub_cmd.h"
#include "toolkit.h"
#include "md5.h"
#include "md_db.h"
#include "users_xml.h"
#include "db_xml.h"

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************************************************************************/
/* check if the user has sent the mandatory commands to have a full access */
/***************************************************************************/
/* output: 0=not allowed, !=0=allowed */
/**************************************/
static int client_can_accept_command(LOCAL_USER_CNX_ENTRY *luce)
{
	/* check if the client has sent the $MyINFO tag and if not, check if the missing MyINFO is allowed */
	if((luce->ext_display_flag&HAVE_SENT_MYINFO)==0)
	{
		int no_my_info;

		if (!db_int_get("NO_MY_INFO", &no_my_info))
			no_my_info=0;

		if(!no_my_info) {
			XFIO_SET_CLOSE_STATUS(luce->user_xfio,CNX_WILL_DIE);
			shutdown(luce->user_xfio->sock_fd,SHUT_RD);

			fprintf(stderr,"Kicking '%s' because it has not sent the $MyINFO command.\n",luce->user_nick->str);
			return 0;	/* access forbidden */
		}
	}
	return 1;		/* access granted */
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/************************/
/* process $ConnectToMe */
/************************/
int ucmd_connecttome_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *wanted_nick;
	char *myip;
	char *t;
	GString *org_str;
	int ret=1;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	luce->out_cnx_count++;
	org_str=g_string_new(str->str);
	if (luce->ext_flag & SILENT_CNX)
	{
		goto leave;
	}
	t=str->str+strlen("$ConnectToMe ");
	SKIP_SPACE(t);
	wanted_nick=t;
	t=strchr(wanted_nick,' ');
	if(t==NULL)
		goto leave;
	*t++='\0';
	SKIP_SPACE(t);
	myip=t;
	
	t=strchr(myip,'|');
	if(t!=NULL)
	{
		char *dupl;

		*t='\0';

		dupl=strdup(org_str->str);

		/* send to a user only if its REV_SILENT_CNX is not set */
		glus_send_to_named_user(wanted_nick,REV_SILENT_CNX,0,0,0,org_str);	/* don't free org_str */

		SEND_EVT_TO("connecttome",luce->user_nick->str,2,wanted_nick,dupl);
		free(dupl);

		org_str=NULL;
		ret=0;
	}
	
	leave:
	if(org_str!=NULL)
		g_string_free(org_str,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/******************************************************************/
/* process $GetINFO query and send the $MyINFO of the wanted user */
/*********************************************************************/
/* string format:  $MyINFO $ALL nickname aaa$ $xxxf$bbb$yyy$         */
/*       aaa is user description                                     */
/*       xxx is connection type (ex: Cable)                          */
/*       f is a 1 byte flag. Default value: \x01                     */
/*             if not behind firewall, bit 1 must be set, else clear */
/*       bbb is e-mail (empty string)                                */
/*       yyy is size of shared data in bytes                         */
/*********************************************************************/
static int ucmd_get_info_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *wanted_nick;
	char *mynick;
	char *t;
	int ret=1;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	t=str->str+strlen("$GetINFO ");
	SKIP_SPACE(t);
	wanted_nick=t;
	t=strchr(wanted_nick,' ');
	if(t==NULL)
		goto leave;
	*t++='\0';
	SKIP_SPACE(t);
	mynick=t;
	
	t=strchr(mynick,'|');
	if(t!=NULL)
	{
		GLUS_USER_INFO *gui;
		*t='\0';

		if(strcmp(mynick,luce->user_nick->str))
			goto leave;

		gui=glus_get_user_info(wanted_nick);
		if(gui!=NULL)
		{
			GString *output;

			output=g_string_new("");
#ifdef GLIB_INT_64_BUG
			g_string_sprintf(output,"$MyINFO $ALL %s %s$ $%s%c$%s$%s$|",
					 gui->user_nick->str,
					 gui->user_description->str,
					 gui->user_cnx_type,
					 (char)gui->user_flag,
					 gui->user_mail->str,
					 llunsigned_to_str(gui->shared_size));
#else
			g_string_sprintf(output,"$MyINFO $ALL %s %s$ $%s%c$%s$%Lu$|",
										gui->user_nick->str,
										gui->user_description->str,
										gui->user_cnx_type,
										(char)gui->user_flag,
										gui->user_mail->str,
										gui->shared_size);
#endif
			send_stolen_gstr_to_luce(luce,output);	/* don't free output */
			ret=0;
			glus_free_user_info(gui);
		}
	}
	
	leave:
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/************************/
/* detect flood on chat */
/************************/
static int ucmd_allow_small_event(LOCAL_USER_CNX_ENTRY *ptr, GString *str)
{
	if (ptr->level < get_right_level(OPERATOR)) /* OPERATOR can flood */
	{
		ptr->msg_count++;
		if (str)
			ptr->msg_count += 1 + (str->len / 100);
		else
			ptr->msg_count++;
		if (ptr->last_msg == gl_cur_time)
		{
			if (ptr->msg_count >= gl_silent_limit)
			{
				if (ptr->msg_count > gl_kick_limit)
				{
					unsigned int hip;

					hip=ntohl(ptr->user_xfio->user_ip.s_addr);
					printf("ccflood: <%s> %hhu.%hhu.%hhu.%hhu %lu %Lu/%Lu:%u/%u\n",
									(ptr->user_nick!=NULL)?ptr->user_nick->str:"",
			 						(unsigned char)(hip>>24)&0xff,
			 						(unsigned char)(hip>>16)&0xff,
			 						(unsigned char)(hip>>8)&0xff,
			 						(unsigned char)hip&0xff,
			 						(long unsigned int)(gl_cur_time - ptr->user_xfio->start_time),
			 						ptr->user_xfio->out_data,
			 						ptr->user_xfio->in_data,
			 						ptr->out_cnx_count,
			 						ptr->in_cnx_count
			 						);
					shutdown(ptr->user_xfio->sock_fd, SHUT_RD);
				}
				return 0;
			}
		}
		else
		{
			long int diff;
	
			diff = gl_cur_time - ptr->last_msg;
			if (ptr->msg_count > diff)
				ptr->msg_count -= diff;
			else
				ptr->msg_count = 0;
			ptr->last_msg = gl_cur_time;
		}
	}
	return(1);
}

/*******************/
/* process < query */
/*******************/
static int ucmd_global_chat_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	int ret=1;
	int msg_content;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	if (ucmd_allow_small_event(luce, str) == 0)
		goto global_chat_process_fin;
	if ((gl_moderate_mode) && (luce->level < get_right_level(OPERATOR)))
	{
		send_const_str_to_luce(luce,"<Kernel> the central chat is in moderate mode|");
		goto global_chat_process_fin;
	}
	if (luce->ext_flag & SILENT_CC)
	{
		goto global_chat_process_fin;
	}


	/* be a bot or have your nick between <> */
	if(  ((!strncmp(str->str+1, luce->user_nick->str, luce->user_nick->len)) && (!strncmp(&(str->str[1+luce->user_nick->len]),"> ",2)))
		 || (luce->IsBot)
	  )
	{
		gchar *str_dup;

		str_dup=g_strdup(str->str);

		if (luce->IsBot)
		{
			char *botmsg;
			char *t;
			t=strdup(str->str);
			botmsg=strchr(t, '>');
			if (botmsg!=NULL)
			{
				/* the string starts with "<somewhat> " */
				msg_content=botmsg-t+1;
				if (str->str[msg_content]!='+')
				{
					GLUS_SEND_TO_EVERYONE(str);		/* don't free str */
					str=NULL;
				}
				else
					process_user_hub_command(luce,&str->str[msg_content],NULL);
			}
			else
				str=NULL;
		}
		else
		{
			msg_content=1+luce->user_nick->len+2;  /* if there is a content, its position is here */

			if (str->str[msg_content]!='+')
			{
				GLUS_SEND_TO_EVERYONE(str);		/* don't free str */
				str=NULL;
			}
			else
				process_user_hub_command(luce,&str->str[msg_content],NULL);
		}
		SEND_EVT_TO("globalchat",luce->user_nick->str,1,str_dup);
		g_free(str_dup);
		ret=0;
	}

 	global_chat_process_fin:
	if(str)
		g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/******************************/
/* process $GetNickList query */
/******************************/
static int ucmd_getnicklist_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	GString *ulst;
	GString *oplst;
	GString *lists[2];

	/* a fake user named "Hub-Security" is created instead DC admin cannot send command in chat */
	/* build a $NickList with the list of all users */
	ulst=g_string_new("$NickList ");

	/* and the $OpList with the list of all operators */
	oplst=g_string_new("$OpList ");

	/* only operator and master can see the hub-security "fake" user */
	if(luce->level >= get_right_level(OPERATOR))
	{
		ulst=g_string_append(ulst,"Hub-Security$$");
		oplst=g_string_append(oplst,"Hub-Security$$");
	}
	
#if 1
#warning code disabled (useful ?)
#else
	if ((ptr->is_op & (OP_PRIV | HAVE_OP_KEY)) == OP_PRIV) /* GHOST OP can see them as Op*/
	{
		oplst=g_string_append(oplst,ptr->user_nick->str);
		oplst=g_string_append(oplst,"$$");
	}
#endif
	
	glus_get_nick_lists(&lists[0],&lists[1]);
	g_string_append(ulst,lists[0]->str);
	g_string_append(oplst,lists[1]->str);
	g_string_free(lists[0],TRUE);
	g_string_free(lists[1],TRUE);

	ulst=g_string_append(ulst,"||");					/* user & oplist are always ended by a doubled pipe */
	send_stolen_gstr_to_luce(luce,ulst);		/* don't free ulst */

	oplst=g_string_append(oplst,"||");				/* user & oplist are always ended by a doubled pipe */
	send_stolen_gstr_to_luce(luce,oplst);		/* don't free oplst */

	g_string_free(str,TRUE);

	SEND_EVT_TO("getnicklist",luce->user_nick->str,0);
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/*****************/
/* process $Kick */
/*********************************/
/* string format: $Kick nickname */
/*********************************/
static int ucmd_kick_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *wanted_nick;
	char *t;
	int ret=1;
	int silent_kick;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return (0);
	}

	if (!db_int_get("SILENT_KICK", &silent_kick))
		silent_kick=0;

	t=str->str+strlen("$Kick ");
	SKIP_SPACE(t);
	wanted_nick=t;
	t=strchr(wanted_nick,'|');
	if(t!=NULL)
	{
		GString *query;
		*t='\0';

		/* translate the command into hub command */
		query=g_string_new("");
		if(silent_kick==0)
			g_string_sprintf(query,"<%s> -kick \"%s\" ...|",luce->user_nick->str,wanted_nick);
		else
			g_string_sprintf(query,"<%s> -skick \"%s\" ...|",luce->user_nick->str,wanted_nick);
		
		process_hub_command(luce,query->str);
		g_string_free(query,TRUE);
		ret=0;
	}
	
	g_string_free(str,TRUE);
	return ret;
}
	
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/******************************/
/* process $MultiSearch query */
/*********************************************/
/* MultiSearch is deprecated, just ignore it */
/*********************************************/
static int ucmd_multisearch_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
#if 0
	GString *org_str;
	char *t;
	char *u;
	char *v;
	int ret=1;

	if (luce->ext_flag & SILENT_SR)
	{
		g_string_free(str,TRUE);
		return (1);
	}
	org_str=g_string_new(str->str);

	/* a $Search query has the following format: */
	/* $Search Hub:nick a?b?c?d?eeeee| where */
	/* a is F if size doesn't matter, else T */
	/* b is F if size is "at least", else T (at most) */
	/* c is the size in byte */
	/* d is data type: 1=any,2=audio,3=compressed,4=document,5=exe,6=picture,7=videos,8=folder */
	/* and eeee is the pattern to find */

	t=str->str+strlen("$Search ");
	SKIP_SPACE(t);
	u=strchr(t,':');
	if(u==NULL)
		goto leave;
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if(u>v)
		goto leave;
	t=v;				/* skip the return address (Hub:nick or hostip:port) */
	SKIP_SPACE(t);

	if((*t!='F')&&(*t!='T'))	/* size matter ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	if((*t!='F')&&(*t!='T'))	/* size at least or at most ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted size */
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted type */
	if(t==NULL)
		goto leave;
	t++;

	if(strlen(t)<2)			/* nothing to search  (or just the trailing | ) ? */
		goto leave;

	/* the query is valid, send it */
#if 1
#warning code disabled (cluster)
#else
	send_string_to_all_rhubs(org_str);
#endif
	ret=0;

	SEND_EVT_TO("multisearch",luce->user_nick->str,1,org_str->str);
	leave:
	g_string_free(org_str,TRUE);
	g_string_free(str,TRUE);
	return ret;
#else
	g_string_free(str,TRUE);
	return 0;
#endif
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/*************************/
/* process $MyINFO query */
/*********************************************************************/
/* string format:  $MyINFO $ALL nickname aaa$ $xxxf$bbb$yyy$         */
/*       aaa is user description                                     */
/*       xxx is connection type (ex: Cable)                          */
/*       f is a 1 byte flag. Default value: \x01                     */
/*             if not behind firewall, bit 1 must be set, else clear */
/*       bbb is e-mail (empty string)                                */
/*       yyy is size of shared data in bytes                         */
/*********************************************************************/
static int ucmd_myinfo_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *t;
	char *fnd_nick;
	char *fnd_desc;
	char *fnd_cnx_type;
	unsigned char fnd_flag;
	char *fnd_email;
	char *fnd_size;
	int ret=0;
	int have_changed=0;

	t=str->str;
	if(strncmp(t,"$MyINFO $ALL ",strlen("$MyINFO $ALL ")))
	{
		have_err:
		ret=1;
		goto eofunc;
	}

	t+=strlen("$MyINFO $ALL ");
	SKIP_SPACE(t);

	/* extract the nickname */
	fnd_nick=t;
	t=strchr(fnd_nick,' ');
	if(t==NULL)
		goto have_err;
	*t++='\0';
	SKIP_SPACE(t);

	/* extract the description */
	fnd_desc=t;
	t=strchr(fnd_desc,'$');
	if(t==NULL)
		goto have_err;
	*t++='\0';

	if(strncmp(t," $",2))
		goto have_err;
	t+=2;

	/* extract the connection type */
	fnd_cnx_type=t;
	t=strchr(fnd_cnx_type,'$');
	if(t==NULL)
		goto have_err;
	*t++='\0';

	/* extract the email */
	fnd_email=t;
	t=strchr(fnd_email,'$');
	if(t==NULL)
		goto have_err;
	*t++='\0';

	/* extract the shared size */
	fnd_size=t;
	t=strchr(fnd_size,'$');
	if(t==NULL)
		goto have_err;
	*t++='\0';

	/* well, nearly all fields have been extracted */
	/* only fnd_flag still is merged with fnd_cnx_type */
	if(strlen(fnd_cnx_type)>1)
	{
		fnd_flag=fnd_cnx_type[strlen(fnd_cnx_type)-1];
		fnd_cnx_type[strlen(fnd_cnx_type)-1]='\0';
	}
	else
		goto have_err;

	/* ok, it's time to check parameters */
	if(strcmp(fnd_nick,luce->user_nick->str))
		goto have_err;			/* nickname doesn't match -> abort */

	/* update information only when needed */
	{
		const char *t;
		guint64 new_share_size;

		/* dchub 0.2.5 */
		if (luce->user_cnx_type[0] == '\0')
		{
			luce->ext_flag &= ~(SILENT_PM | SILENT_CC | SILENT_SR | SILENT_CNX | OTIST_CC);
		}
		t=fnd_static_cnx_type(fnd_cnx_type);
		if(luce->user_cnx_type!=t)					/* t and user_cnx_type point on const static string, that's why we can avoid a strcmp */
		{
			luce->user_cnx_type=t;
			have_changed=1;
		}

#ifdef GLIB_INT_64_BUG
		str_to_llunsigned(fnd_size, &(new_share_size));
#else
		sscanf(fnd_size,"%Lu",&(new_share_size));
#endif
		if(new_share_size!=luce->shared_size)
		{
			luce->shared_size=new_share_size;
			have_changed=1;
		}

		if(strcmp(luce->user_mail->str,fnd_email))
		{
			g_string_assign(luce->user_mail,fnd_email);
			have_changed=1;
		}

		if(strcmp(luce->user_description->str,fnd_desc))
		{
			g_string_assign(luce->user_description,fnd_desc);
			have_changed=1;
		}

		if(luce->user_flag!=(((guint16)fnd_flag)&255))
		{
			luce->user_flag=((guint16)fnd_flag)&255;
			have_changed=1;
		}
	}

	if(have_changed)
	{
		luce->ext_display_flag|=HAVE_SENT_MYINFO;

		/* send the myinfo to everyone */
		glus_do_my_info( luce->user_nick->str, luce->user_description->str, luce->user_cnx_type,
		                 luce->user_flag, luce->user_mail->str, luce->shared_size,
		                 luce->ext_flag, luce->level, luce->client_version->str);

		{
			GString *nw;
			nw=g_string_new("");
#ifdef GLIB_INT_64_BUG
			g_string_sprintf(nw,"$MyINFO $ALL %s %s$ $%s%c$%s$%s$|",
#else
			g_string_sprintf(nw,"$MyINFO $ALL %s %s$ $%s%c$%s$%Lu$|",
#endif
									luce->user_nick->str,
									luce->user_description->str,
									luce->user_cnx_type,
									(char)luce->user_flag,
									luce->user_mail->str,
#ifdef GLIB_INT_64_BUG
									llunsigned_to_str(luce->shared_size)
#else
				 					luce->shared_size
#endif
				 					);
		
			SEND_EVT_TO("myinfo",luce->user_nick->str,1,nw->str);
			g_string_free(nw,TRUE);
		}
	}
	eofunc:
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/************************/
/* process $OpForceMove */
/************************************************************************/
/* string format:  $OpForceMove $Who:nick$Where:redirectip$Msg:message| */
/************************************************************************/
/* Added by [Tele2]Shoggot*/
/**************************/
static int ucmd_redir_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *t;
	char *temp;
	int ret=1;
	char nick[21];
	char ip[122];
	char message[255];
	GString *query2;
	query2=g_string_new("");
	t=str->str;

	sscanf(t, "$OpForceMove $Who:%20[^$]$Where:%121[^$]$Msg:%254[^|]|", nick, ip, message);
	
	if(message[0] == '\0')
	{
		printf("Erroneous op_force_move command received\n");
		goto abrt;
	}

	if((temp = strstr(t, "$Msg:")) == NULL)
	{
		printf("Erroneous op_force_move command received\n");
		goto abrt;
	}

	g_string_sprintf(query2,"<%s> -redir \"%s\" \"%s\" \"%s\"|",luce->user_nick->str,nick,ip,message);
	process_hub_command(luce,query2->str);
	ret=0;
	abrt:
	g_string_free(query2,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/***************************/
/* process $RevConnectToMe */
/***************************/
int ucmd_revconnecttome_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *wanted_nick;
	char *mynick;
	char *t;
	GString *org_str=NULL;
	int ret=1;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	luce->out_cnx_count++;
	if (luce->ext_flag & SILENT_CNX)
	{
		goto leave;
	}
	org_str=g_string_new(str->str);
	t=str->str+strlen("$RevConnectToMe ");
	SKIP_SPACE(t);
	mynick=t;
	t=strchr(mynick,' ');
	if(t==NULL)
		goto leave;
	*t++='\0';
	SKIP_SPACE(t);
	wanted_nick=t;
	
	t=strchr(wanted_nick,'|');
	if(t!=NULL)
	{
		gchar *ptr_cpy;

		*t='\0';

		if(strcmp(mynick,luce->user_nick->str))
			goto leave;

		ptr_cpy=g_strdup(org_str->str);
		{
			LOCAL_USER_CNX_ENTRY *ent;
		
			ent=user_cnx_entry_get_by_nickname(wanted_nick);
			if (ent)
			{
#if 1
#warning code disabled (useful ?)
#else
				if (ent->ext_flag & REV_SILENT_CNX)
				{
					if ((ent->tutor != NULL) && (!strcmp(ent->tutor, ptr->user_nick->str)))
						send_str_to_cnx(ent, org_str->str);
				}
				else
#endif
					send_gstr_to_luce(ent, org_str);
				if (ent != luce)
					ent->in_cnx_count++;
				else
					printf("clientbug: %s\n", ent->user_nick->str); /* That hides something ....*/
			}
			else
			{
				/* the user is not locally on the hub */
				GLUS_SEND_TO_A_NICK(wanted_nick,org_str);	/* don't free org_str */
				org_str=NULL;
			}
		}

		SEND_EVT_TO("revconnect",luce->user_nick->str,2,wanted_nick,ptr_cpy);
		g_free(ptr_cpy);
		ret=0;
	}
	
	leave:
	if(org_str)
		g_string_free(org_str,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/***************/
/* process $SR */
/**************************************************************************************************/
/* string format:  $SR nickname filename\005filesize slot/ratio\005hubname (hub addr)\005destnick */
/**************************************************************************************************/
static int ucmd_sr_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *t;
	int ret=1;
	GString *org;
	
	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	/* keep a copy of the incoming string. This string will be sent to the dest nick */
	org=g_string_new(str->str);

	t=str->str+strlen("$SR ");
	SKIP_SPACE(t);

#define SKIP_TO_NEXT(t,sep1,sep2,want)											\
										while((*t)&&(*t!=sep1)&&(*t!=sep2))\
											t++;\
										if(*t!=want)\
											goto abrt;\
											t++;

	/* we are on the source nickname */
	SKIP_TO_NEXT(t,' ',5,' ')

	/* we are on the filename */
	t=strchr(t,5);
	if(t==NULL)
		goto abrt;
	t++;

	/* we are on the filesize */
	SKIP_TO_NEXT(t,' ',5,' ')

	/* we are on the slot ratio */
	SKIP_TO_NEXT(t,' ',5,5)

	/* we are on the hubname */
	t=strchr(t,5);
	if(t==NULL)
		goto abrt;

	/* modify the $SR to create the one sent to the client */
	org=g_string_truncate(org,t-str->str);
	org=g_string_append_c(org,'|');

	str=g_string_erase(str,0,t-str->str+1);	/* keep everything after the separator */
	t=str->str;
	t=strchr(t,'|');
	if(t!=NULL)
	{
		*t='\0';
		GLUS_SEND_TO_A_NICK(str->str,org);		/* don't free org */
		SEND_EVT_TO("sr",luce->user_nick->str,1,org->str);
		org=NULL;
		ret=0;
	}
	/*
	else
	{
		;
		abrt:
		all directory SR are invalid It's a DC hub bug.
		printf("invalid $SR result\n");
	}
	*/
	abrt:
	
	if(org)
		g_string_free(org,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************/
/* process $XSR */
/*******************************************************************************************************/
/* string format:  $XSR CRC nickname filename\005filesize slot/ratio\005hubname (hub addr)\005destnick */
/*******************************************************************************************************/
static int ucmd_xsr_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *t;
	char *v;

	int ret=1;
	GString *org;
	
	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	/* keep a copy of the incoming string. This string will be sent to the dest nick */
	org=g_string_new(str->str);

	t=str->str+strlen("$XSR ");
	SKIP_SPACE(t);

	/* we are on the CRC */
	v=t;
	t=strchr(t,' ');
	if((t==NULL)||((t-v)!=(2*MD4_DIGEST_LENGTH)))
		goto abrt;
	SKIP_SPACE(t);

	/* we are on the source nickname */
	SKIP_TO_NEXT(t,' ',5,' ')

	/* we are on the filename */
	t=strchr(t,5);
	if(t==NULL)
		goto abrt;
	t++;

	/* we are on the filesize */
	SKIP_TO_NEXT(t,' ',5,' ')

	/* we are on the slot ratio */
	SKIP_TO_NEXT(t,' ',5,5)

	/* we are on the hubname */
	t=strchr(t,5);
	if(t==NULL)
		goto abrt;

	/* modify the $SR to create the one sent to the client */
	org=g_string_truncate(org,t-str->str);
	org=g_string_append_c(org,'|');

	str=g_string_erase(str,0,t-str->str+1);	/* keep everything after the separator */
	t=str->str;
	t=strchr(t,'|');
	if(t!=NULL)
	{
		*t='\0';
		GLUS_SEND_TO_A_NICK(str->str,org);		/* don't free org */
		org=NULL;
		ret=0;
	}
	/*
	else
	{
		;
		abrt:
		all directory SR are invalid It's a DC hub bug.
		printf("invalid $SR result\n");
	}
	*/
	abrt:
	
	if(org)
		g_string_free(org,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/*************************/
/* process $Search query */
/*************************/
static int ucmd_search_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	GString *org_str;
	char *t;
	char *u;
	char *v;
	int ret=1;
	char tmp_ret_adr[128];
	int level;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	luce->sr_count++;
	if (((gl_cur_time - luce->last_hcall) < gl_sr_interval) && ((luce->level >= get_right_level(OPERATOR))))
	{
		send_const_str_to_luce(luce,"<Kernel> search dropped. Wait 10sec between 2 searchs|");
		g_string_free(str,TRUE);
		return (0);
	}
	luce->last_hcall = gl_cur_time;
	org_str=g_string_new(str->str);

	/* a $Search query has the following format: */
	/* $Search Hub:nick a?b?c?d?eeeee| where */
	/* a is F if size doesn't matter, else T */
	/* b is F if size is "at least", else T (at most) */
	/* c is the size in byte */
	/* d is data type: 1=any,2=audio,3=compressed,4=document,5=exe,6=picture,7=videos,8=folder */
	/* and eeee is the pattern to find */

	t=str->str+strlen("$Search ");
	SKIP_SPACE(t);
	u=strchr(t,':');
	if(u==NULL)
		goto leave;
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if(u>v)
		goto leave;
	if((v-t)>(sizeof(tmp_ret_adr)-1))	/* a return address longer than the buffer, probably an error */
		goto leave;
	memcpy(tmp_ret_adr,t,v-t);				/* keep a copy of the address */
	tmp_ret_adr[v-t]='\0';
	t=v;				/* skip the return address (Hub:nick or hostip:port) */
	SKIP_SPACE(t);

	if((*t!='F')&&(*t!='T'))	/* size matter ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	if((*t!='F')&&(*t!='T'))	/* size at least or at most ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted size */
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted type */
	if(t==NULL)
		goto leave;
	t++;

	if(strlen(t)<2)			/* nothing to search  (or just the trailing | ) ? */
		goto leave;

	/* perform some checks against the return address */
	/* the return address can have 2 formats: "Hub:nick" or "hostip:port" */
	if(!strncmp(tmp_ret_adr,"Hub:",strlen("Hub:")))
	{
		level=0;
		/* it is a passive search */
		if(strcmp(tmp_ret_adr+strlen("Hub:"),luce->user_nick->str))
			goto leave_on_malformed_ret;	/* the nick name in the return address is not the nick of the current connection */
	}
	else
	{
		int a;
		unsigned int v1,v2,v3,v4;
		unsigned int hip;

		level=1;
		/* it is an active search */
		t=strchr(tmp_ret_adr,':');		/* cannot fail, it was tested earlier in the function */
		*t++='\0';

		a=atoi(t);
		if((a<=0)||(a>65535))
			goto leave_on_malformed_ret;			/* the reply port is not in the range [1:65535] */
		level++;
		if(sscanf(tmp_ret_adr,"%u.%u.%u.%u",&v1,&v2,&v3,&v4)!=4)
			goto leave_on_malformed_ret;			/* the return address does not look like an IP */
		level++;
		if((v1>255)||(v2>255)||(v3>255)||(v4>255))
			goto leave_on_malformed_ret;			/* malformed values */
		level++;
		hip=(v1<<24)|(v2<<16)|(v3<<8)|v4;
		if(hip!=ntohl(luce->user_xfio->user_ip.s_addr))
		{
			static const char *err_msg[]={"Nickname of the connection and of the query are different",
										  "Port not in the range 1-65535",
										  "Not an IP",
										  "Not a valid IP",
										  "IP of the connection and of the query are different"};
			leave_on_malformed_ret:

			if ((luce->level < get_right_level(OPERATOR)))
			{
				/* close the connection of this user */
				XFIO_SET_CLOSE_STATUS(luce->user_xfio,CNX_WILL_DIE);
				shutdown(luce->user_xfio->sock_fd,SHUT_RD);

				fprintf(stderr,"Kicking '%s' because it uses malformed reply path in search query (%s), error: %s\n",luce->user_nick->str,org_str->str,err_msg[level]);
				goto leave;			/* incorrect IP */
			}
		}
	}

	/* the query is valid, send it */
	GLUS_SEND_TO_EVERYONE(org_str);	/* don't free org_str */

	SEND_EVT_TO("search",luce->user_nick->str,1,org_str->str);
	org_str=NULL;
	ret=0;

	leave:
	if(org_str)
		g_string_free(org_str, TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/**************************/
/* process $XSearch query */
/**************************/
static int ucmd_xsearch_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	GString *org_str;
	char *t;
	char *u;
	char *v;
	int ret=1;
	guint8 file_crc[MD4_DIGEST_LENGTH];
	unsigned long file_size;
	GString *return_path=NULL;
	GString *std_search=NULL;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	luce->sr_count++;
	if (((gl_cur_time - luce->last_hcall) < gl_sr_interval) && ((luce->level < get_right_level(OPERATOR))))
	{
		send_const_str_to_luce(luce,"<Kernel> search dropped. Wait 10sec between 2 searchs|");
		g_string_free(str,TRUE);
		return (0);
	}
	luce->last_hcall = gl_cur_time;
	org_str=g_string_new(str->str);

	/* a $XSearch query has the following format: */
	/* $XSearch CRC length Hub:nick a?b?c?d?eeeee| where */
	/* a is F if size doesn't matter, else T */
	/* b is F if size is "at least", else T (at most) */
	/* c is the size in byte */
	/* d is data type: 1=any,2=audio,3=compressed,4=document,5=exe,6=picture,7=videos,8=folder */
	/* and eeee is the pattern to find */

	t=str->str+strlen("$XSearch ");
	SKIP_SPACE(t);

	/* analyze the CRC */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if((v-t)!=(2*MD4_DIGEST_LENGTH))
		goto leave;
	id_ascii_to_bin(t,file_crc);
	t=v;
	SKIP_SPACE(t);
	
	/* analyze the length */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	sscanf(t,"%lu",&file_size);
	t=v;
	SKIP_SPACE(t);

	/* analyze the return path */
	u=strchr(t,':');
	if(u==NULL)
		goto leave;
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if(u>v)
		goto leave;
	return_path=g_string_new(t);
	g_string_truncate(return_path,v-t);
	t=v;				/* skip the return address (Hub:nick or hostip:port) */
	SKIP_SPACE(t);
	/* beginning of the search pattern */
	std_search=g_string_new(t);

	if((*t!='F')&&(*t!='T'))	/* size matter ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	if((*t!='F')&&(*t!='T'))	/* size at least or at most ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted size */
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted type */
	if(t==NULL)
		goto leave;
	t++;

	if(strlen(t)<2)			/* nothing to search  (or just the trailing | ) ? */
		goto leave;

	/* remove the trailing | */
	if(std_search->str[std_search->len-1]=='|')
		g_string_truncate(std_search,std_search->len-1);

	/* the query is valid, send it */
	glus_do_xsearch(return_path->str,file_crc,file_size,std_search->str);

	SEND_EVT_TO("xsearch",luce->user_nick->str,1,org_str->str);
	org_str=NULL;
	ret=0;

	leave:
	if(std_search)
		g_string_free(std_search,TRUE);
	if(return_path)
		g_string_free(return_path,TRUE);
	if(org_str)
		g_string_free(org_str, TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/**************************/
/* process $MD4Get0 query */
/**************************/
static int ucmd_md4get0_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	GString *org_str;
	char *t;
	char *v;
	int ret=1;
	guint8 md_crc[MD4_DIGEST_LENGTH];
	unsigned long file_size;
	MD_INFO *mi;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	luce->sr_count++;
	if (((gl_cur_time - luce->last_hcall) < gl_sr_interval) && ((luce->level < get_right_level(OPERATOR))))
	{
		send_const_str_to_luce(luce,"<Kernel> search dropped. Wait 10sec between 2 searchs|");
		g_string_free(str,TRUE);
		return (0);
	}
	luce->last_hcall = gl_cur_time;
	org_str=g_string_new(str->str);

	/* a $MD4Get0 query has the following format: */
	/* $MD4Get0 CRC length| */
	t=str->str+strlen("$MD4Get0 ");
	SKIP_SPACE(t);

	/* we are on the nickname */
	/* check if the nickname in the query matchs the nickname of the connection */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if((v-t)!=luce->user_nick->len)
		goto leave;
	if(strncmp(t,luce->user_nick->str,luce->user_nick->len))
		goto leave;

	t=v;
	SKIP_SPACE(t);

	/* analyze the CRC */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if((v-t)!=(2*MD4_DIGEST_LENGTH))
		goto leave;
	id_ascii_to_bin(t,md_crc);
	t=v;
	SKIP_SPACE(t);
	
	/* analyze the length */
	v=strchr(t,'|');
	if(v==NULL)
		goto leave;
	sscanf(t,"%lu",&file_size);
	while(t<v)
	{
		if(!isdigit(*t))
			goto leave;
		t++;
	}

	/* try to find a CRC already in the database */
	mi=get_md4(md_crc,file_size);
	if(mi!=NULL)
	{
		/* we have the CRC, reply immediately */
		glus_md4set_received(mi->global_crc,mi->filesize,mi->partial_crc,mi->nb_seg,mi->filename,luce->user_nick->str);
		free_md_info(mi);
	}
	else
	{
		/* the query is valid, send it */
		GLUS_SEND_TO_EVERYONE(org_str);	/* don't free org_str */

		SEND_EVT_TO("md4get0",luce->user_nick->str,1,org_str->str);
		org_str=NULL;
		ret=0;
	}

	leave:
	if(org_str)
		g_string_free(org_str, TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/*************************/
/* process $MD4Set query */
/*************************/
static int ucmd_md4set_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	GString *org_str;
	char *t;
	char *v;
	int i;
	int ret=1;
	guint8 file_crc[MD4_DIGEST_LENGTH];
	guint8 *l0_crc=NULL;
	unsigned char *s_l0;
	unsigned long file_size;
	unsigned nb_seg;
	char *filename;
	char *nickname=NULL;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	luce->sr_count++;
	if (((gl_cur_time - luce->last_hcall) < gl_sr_interval) && ((luce->level < get_right_level(OPERATOR))))
	{
		send_const_str_to_luce(luce,"<Kernel> search dropped. Wait 10sec between 2 searchs|");
		g_string_free(str,TRUE);
		return (0);
	}
	luce->last_hcall = gl_cur_time;
	org_str=g_string_new(str->str);

	/* a $MD4Set query has the following format: */
	/* $MD4Set CRC length L0CRC filename| */
	t=str->str+strlen("$MD4Set ");
	SKIP_SPACE(t);

	/* we are on the nickname */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	nickname=malloc(v-t +1);
	if(nickname==NULL)
		goto leave;
	memcpy(nickname,t,v-t);
	nickname[v-t]='\0';

	t=v;
	SKIP_SPACE(t);

	/* analyze the CRC */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if((v-t)!=(2*MD4_DIGEST_LENGTH))
		goto leave;
	id_ascii_to_bin(t,file_crc);
	t=v;
	SKIP_SPACE(t);
	
	/* analyze the length */
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	sscanf(t,"%lu",&file_size);
	t=v;
	SKIP_SPACE(t);

	/* analyze the l0 CRC */
	nb_seg=(file_size+PARTSIZE-1)/PARTSIZE;
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if((v-t)!=(nb_seg*2*MD4_DIGEST_LENGTH))	/* size of l0 checksum */
		goto leave;
	s_l0=t;
	while(t<v)
	{
		if(!isxdigit(*t))
			goto leave;
		t++;
	}
	l0_crc=malloc(nb_seg*MD4_DIGEST_LENGTH);
	if(l0_crc==NULL)
		goto leave;
	for(i=0;i<nb_seg;i++)
	{
		id_ascii_to_bin(s_l0+2*MD4_DIGEST_LENGTH*i,l0_crc+i*MD4_DIGEST_LENGTH);
	}
	SKIP_SPACE(t);
	
	if((i=strlen(t))<2)			/* nothing filename  (or just the trailing | ) ? */
		goto leave;

	filename=strdup(t);
	if(filename==NULL)
		goto leave;
	filename[i-1]='\0';		/* remove the trailing | */

	/* here, we have: */
	/*	global MD4 CRC: file_crc[MD4_DIGEST_LENGTH] */
	/* partial MD4 CRC: l0_crc (array of nb_seg*MD4_DIGEST_LENGTH) */
	/* the file size: file_size */
	/* the number of segment in the file: nb_seg */
	/* the filename : filename */

	/* the query is valid, send it */
	glus_md4set_received(file_crc,file_size,l0_crc,nb_seg,filename,nickname);
	free(filename);

	SEND_EVT_TO("md4get0",luce->user_nick->str,1,org_str->str);
	ret=0;

	leave:
	if(nickname!=NULL)
		free(nickname);
	if(l0_crc!=NULL)
		free(l0_crc);
	if(org_str)
		g_string_free(org_str, TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */


/*********************/
/* process $To query */
/*********************/
int ucmd_private_chat_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *dest_nick;
	char *source_nick;
	char *t;
	GString *org_cmd;
	int ret=1;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	org_cmd=g_string_new(str->str);
	
	if (luce->level < get_right_level(OPERATOR))
	{
		if (ucmd_allow_small_event(luce, NULL) == 0)
			goto leave;
	}

	/* extract the dest nickname and the source nickname */
	t=str->str+strlen("$To: ");
	SKIP_SPACE(t);
	dest_nick=t;
	t=strchr(dest_nick,' ');
	if(t==NULL)
		goto leave;

	*t++='\0';
	if(strncmp(t,"From: ",strlen("From: ")))
		goto leave;

	t+=strlen("From: ");
	SKIP_SPACE(t);
	source_nick=t;
	t=strchr(source_nick,' ');
	if(t==NULL)
		goto leave;

	*t++='\0';
	if(*t!='$')
		goto leave;

	t++;
	if(strcmp(source_nick,luce->user_nick->str))
		goto leave;			/* the source nickname does not match with connexion nickname */
	
#if 1
#warning code disabled (useful ?)
#else
	if (luce->ext_flag & SILENT_PM)
	{
		if (luce->ext_flag & LOCK_PM_TUTOR)
			goto leave;
		if (ptr->tutor== NULL)
			goto leave;
		if (strcasecmp(dest_nick, ptr->tutor))
			goto leave;
	}
#endif

	if(!strcasecmp(dest_nick,"Hub-Security"))
	{
		ret=process_hub_command(luce,t);
	}
	else
	{
		char *skip_beginning;

		skip_beginning=strchr(t,' ');
		if (skip_beginning == NULL)
			goto leave;
		SKIP_SPACE(skip_beginning);
#if 1
#warning code disabled (useful ?)
#else
		{
			int idx;

			if ((idx = find_pubchat_idx_by_name(dest_nick)) != -1)
			{
				send_to_users_of_multi_pubchat_except2(idx, t, luce);
			}
		}
#endif

		if(skip_beginning[0]!='+')
		{
			GLUS_USER_INFO *ent;

			ent=glus_get_user_info(dest_nick);
			if (ent)
			{
#if 1
#warning code disabled (useful ?)
#else
				if (ent->ext_flag & OTIST_PM)
				{ 
					if ((ent->tutor != NULL) && (!strcmp(ent->tutor, ptr->user_nick->str)))
						send_str_to_cnx(ent, org_cmd->str);
				}
				else
#endif
				{
					GLUS_SEND_TO_A_NICK(dest_nick,g_string_new(org_cmd->str));	/* don't free org_cmd */
					org_cmd=NULL;
				}
				glus_free_user_info(ent);
			}
		}
		else
			process_user_hub_command(luce,skip_beginning,dest_nick);

		SEND_EVT_TO("privchat",luce->user_nick->str,2,dest_nick,t);
		ret=0;
	}

	leave:
	if(org_cmd!=NULL)
		g_string_free(org_cmd,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/****************************/
/* process $UniSearch query */
/****************************/
static int ucmd_unisearch_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	GString *org_str;
	char *t;
	char *u;
	char *v;
	char *destnick;
	int ret=1;
	char *beginning_std_search;

	if(!client_can_accept_command(luce))
	{
		g_string_free(str,TRUE);
		return ret;
	}

	if(luce->ext_flag & SILENT_SR)
	{
		g_string_free(str,TRUE);
		return ret;
	}
	
	org_str=g_string_new(str->str);

	/* a $UniSearch query has the following format: */
	/* $UniSearch destnick ip:port a?b?c?d?eeeee| where */
	/* a is F if size doesn't matter, else T */
	/* b is F if size is "at least", else T (at most) */
	/* c is the size in byte */
	/* d is data type: 1=any,2=audio,3=compressed,4=document,5=exe,6=picture,7=videos,8=folder */
	/* and eeee is the pattern to find */

	t=str->str+strlen("$UniSearch ");
	SKIP_SPACE(t);

	destnick=t;
	t=strchr(t,' ');
	if(t==NULL)
		goto leave;
	*t++='\0';
	SKIP_SPACE(t);

	beginning_std_search=t;

	u=strchr(t,':');
	if(u==NULL)
		goto leave;
	v=strchr(t,' ');
	if(v==NULL)
		goto leave;
	if(u>v)
		goto leave;
	t=v;				/* skip the return address (Hub:nick or hostip:port) */
	SKIP_SPACE(t);

	if((*t!='F')&&(*t!='T'))	/* size matter ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	if((*t!='F')&&(*t!='T'))	/* size at least or at most ? T or F */
		goto leave;
	t=strchr(t,'?');
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted size */
	if(t==NULL)
		goto leave;
	t++;

	t=strchr(t,'?');			/* skip the wanted type */
	if(t==NULL)
		goto leave;
	t++;

	if(strlen(t)<2)			/* nothing to search  (or just the trailing | ) ? */
		goto leave;

	/* the query is valid, send it */
	{
		GString *rebuild_str;
		rebuild_str=g_string_new("");
		g_string_sprintf(rebuild_str,"$Search %s",beginning_std_search);
		GLUS_SEND_TO_A_NICK(destnick,rebuild_str);		/* don't free rebuild_str */
	}
	ret=0;

	SEND_EVT_TO("unisearch",luce->user_nick->str,1,org_str->str);
	leave:
	g_string_free(org_str,TRUE);
	g_string_free(str,TRUE);
	return ret;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/**************************/
/* process $Version query */
/**************************/
int ucmd_version_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	str=g_string_erase(str,0,strlen(cmd));
	if(str->len>0)
	{
		str=g_string_truncate(str,str->len-1);		/* remove trailing | */
	}
	
	g_string_assign(luce->client_version,str->str);
	g_string_free(str,TRUE);
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/****************************/
/* process $ForceSend query */
/****************************/
int ucmd_force_send_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	if(!luce->IsBot) {
		g_string_free(str,TRUE);
		return 1;
	}

	str=g_string_erase(str,0,strlen(cmd));
	GLUS_SEND_TO_EVERYONE(str);		/* don't free str */
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/******************************/
/* process $ForceSendTo query */
/******************************/
int ucmd_force_send_to_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *t;
	char *nickname;

	if(!luce->IsBot) {
		g_string_free(str,TRUE);
		return 1;
	}

	str=g_string_erase(str,0,strlen(cmd));
	if(str->len<3)		/* at least a nick, a space and a pipe */
	{
		g_string_free(str,TRUE);
		return 1;
	}

	nickname=str->str;
	t=strchr(nickname,' ');
	if(t==NULL)
	{
		g_string_free(str,TRUE);
		return 1;
	}

	*t++='\0';

	GLUS_SEND_TO_A_NICK_char_ptr(nickname,t);

	g_string_free(str,TRUE);
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************************/
/* toggle capability flags */
/***************************/
static void capability_to_cnx_entry(LOCAL_USER_CNX_ENTRY *luce, char *cap_name)
{
	if(!strcmp(cap_name,"UTF8"))
	{
		luce->ext_display_flag|=HAVE_UTF8;
		return;
	}

	if(!strcmp(cap_name,"TAG"))
	{
		luce->ext_display_flag|=HAVE_TAG;
		return;
	}

	if(!strcmp(cap_name,"XSearch"))
	{
		luce->ext_display_flag|=HAVE_XSEARCH;
		return;
	}

	if(!strcmp(cap_name,"MD4x"))
	{
		luce->ext_display_flag|=HAVE_MD4x;
		return;
	}

}

/*************************/
/* process $Capabilities */
/*****************************************************/
/* string format: $Capabilities list$of$capabilities */
/*****************************************************/
static int ucmd_capabilities_process(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param)
{
	char *t;
	gchar **cap_array;
	int i;

	t=str->str+strlen("$Capabilities ");
	SKIP_SPACE(t);

	cap_array=g_strsplit(t,"$",0);
	if(cap_array!=NULL)
	{
		i=0;
		while(cap_array[i]!=NULL)
		{
			t=strchr(cap_array[i],'|');
      	if(t!=NULL)
         	*t='\0';

			if(strlen(cap_array[i]))
			{
				capability_to_cnx_entry(luce,cap_array[i]);
			}
			i++;
		}

		g_strfreev(cap_array);
	}
	g_string_free(str,TRUE);
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------- command dispatcher ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
typedef struct
{
	const char *cmd;
	size_t cmd_len;
	int (*fnc)(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString *str, char *xtra_param);
      /* function to call. This function MUST free str when it is not useful anymore */
      /* the function must return 0 if the connection must remain alive (no error or */
      /* recoverable error) and a value !=0 if there is a fatal error) */
   char *xtra_param;
} KNOWN_CMD;

/* all the fuction must free they input */
static KNOWN_CMD known_cmd[]=	{
							{"$Capabilities ",	sizeof("$Capabilities ")-1,	ucmd_capabilities_process,	NULL},		/* ok */
							{"$ConnectToMe ",		sizeof("$ConnectToMe ")-1,		ucmd_connecttome_process,	NULL},		/* ok */
							{"$ForceSend ",		sizeof("$ForceSend ")-1,		ucmd_force_send_process,	NULL},
							{"$ForceSendTo ",		sizeof("$ForceSendTo ")-1,		ucmd_force_send_to_process,NULL},
							{"$GetINFO ",			sizeof("$GetINFO ")-1,			ucmd_get_info_process,		NULL},		/* ok */
							{"$GetNickList",		sizeof("$GetNickList")-1,		ucmd_getnicklist_process,	NULL},		/* ok */
							{"$Kick ",				sizeof("$Kick ")-1,				ucmd_kick_process,			NULL},
							{"$MD4Get0 ",			sizeof("$MD4Get0 ")-1,			ucmd_md4get0_process,		NULL},		/* ok */
							{"$MD4Set ",			sizeof("$MD4Set ")-1,			ucmd_md4set_process,			NULL},		/* ok */
							{"$MultiSearch ",		sizeof("$MultiSearch ")-1,		ucmd_multisearch_process,	NULL},
							{"$MyINFO ",			sizeof("$MyINFO ")-1,			ucmd_myinfo_process,			NULL},		/* ok */
							{"$OpForceMove ",		sizeof("$OpForceMove ")-1,		ucmd_redir_process,			NULL},
							{"$RevConnectToMe ",	sizeof("$RevConnectToMe ")-1,	ucmd_revconnecttome_process,NULL},		/* ok */
							{"$SR ",					sizeof("$SR ")-1,					ucmd_sr_process,				NULL},		/* ok */
							{"$Search ",			sizeof("$Search ")-1,			ucmd_search_process,			NULL},		/* ok */
							{"$To: ",				sizeof("$To: ")-1,				ucmd_private_chat_process,	NULL},		/* ok */
							{"$UniSearch ",		sizeof("$UniSearch ")-1,		ucmd_unisearch_process,		NULL},		/* ok */
							{"$Version ",			sizeof("$Version ")-1,			ucmd_version_process,		NULL},		/* ok */
							{"$XSR ",				sizeof("$XSR ")-1,				ucmd_xsr_process,				NULL},		/* ok */
							{"$XSearch ",			sizeof("$XSearch ")-1,			ucmd_xsearch_process,		NULL},		/* ok */
							{"<",						sizeof("<")-1,						ucmd_global_chat_process,	NULL},		/* ok */
							{NULL,0,NULL,NULL},
};

/************************************************************************/
/* this function takes the first incoming command and choose what to do */
/************************************************************************/
/* return value: <0: fatal error; ==0: ok; >0: non fatal error */
/***************************************************************/
int process_one_command(LOCAL_USER_CNX_ENTRY *luce)
{
	GString *inp;
	int fnd=0;
	int ret=0;

	inp=xfio_take_first_string_of_glist_incoming(luce->user_xfio);
	if(inp==NULL)
		return ret;		/* this case should never appears but who knows ... */

	if (inp->str[0] == '|') /* the | is frequently sent so handle it before */
	{
		g_string_free(inp,TRUE);
		return ret;
	}

#ifdef DEBUG
	printf("%s\n",inp->str);
#endif
	{
		int min = 0;
		int max = ARRAY_SIZE (known_cmd) - 2; /* 1 for le NULL line and 1 because count start at 0 */

		do
		{
			int i = (min + max) / 2;
			int cmp;
      
			cmp = strncmp(inp->str,known_cmd[i].cmd,known_cmd[i].cmd_len);
			if (cmp == 0)
			{
				fnd=1;
				ret=(known_cmd[i].fnc)(known_cmd[i].cmd,luce,inp,known_cmd[i].xtra_param);
				break;
			}
      	else if (cmp < 0)
				max = i - 1;
      	else
				min = i + 1;
		}
		while (min <= max);
	}

	if(!fnd)
	{
		SEND_EVT_TO("unknowncmd",luce->user_nick->str,1,inp->str);
		g_string_free(inp,TRUE);
	}
	return ret;
}

