/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * plugin.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
  $Id: plugin.c,v 2.11 2003/11/28 14:34:56 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_LIBDL

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include <glib.h>

#include "plugin.h"
#include "pluginT.h"
#include "../plugin/dchub_plugin_api.h"
#include "gvar.h"
#include "users_xml.h"
#include "db_xml.h"
#include "global_user_if.h"

static GPtrArray *loaded_plugins=NULL;	/* array of PLUGIN entries */

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************/
/* private functions of the plugin support */
/*******************************************/

/**************************************************************************/
/* return the PLUGIN entry of the plugin named with the given plugin_name */
/**************************************************************************/
/* output: NULL=not found else PLUGIN * */
/****************************************/
static PLUGIN *get_plugin_ptr_by_name(char *plugin_name)
{
	int i;
	PLUGIN *plug;

	if(loaded_plugins==NULL)
		return NULL;

	for(i=0;i<loaded_plugins->len;i++)
	{
		plug=(PLUGIN*)g_ptr_array_index(loaded_plugins,i);

		if(!strcmp(plugin_name,plug->plugin_name))
			return plug;
	}

	return NULL;
}

/******************************************************************************************/
/* check if the plugin named plugin_name is loaded (==exists in the loaded_plugins array) */
/******************************************************************************************/
/* output: 0=no, 1=yes */
/***********************/
static int is_a_loaded_plugin(char *plugin_name)
{
	int i;
	PLUGIN *plug;

	if(loaded_plugins==NULL)
		return 0;

	for(i=0;i<loaded_plugins->len;i++)
	{
		plug=(PLUGIN*)g_ptr_array_index(loaded_plugins,i);

		if(!strcmp(plugin_name,plug->plugin_name))
			return 1;
	}

	return 0;
}

/**************************************************************************************/
/* this function searchs inside all plugin event handler handlers for the given event */
/******************************************************************************************************************/
/* input: the name of the event                                                                                   */
/* output: a GPtrArray containing functions to call (can be empty or NULL)(must be released when becoming unused) */
/******************************************************************************************************************/
static GPtrArray *get_all_function_addresses_for_an_evt(const char *evt_name)
{
	int i,j;
	GPtrArray *gpa;
	PLUGIN *plug;
	PLUGIN_EVT_HANDLER *peh;

	if(loaded_plugins==NULL)
		return NULL;

	gpa=g_ptr_array_new();
	if(gpa==NULL)
		return gpa;

	for(i=0;i<loaded_plugins->len;i++)
	{
		plug=(PLUGIN*)g_ptr_array_index(loaded_plugins,i);

		if(plug->evt_hdl!=NULL)	/* no registered event ? */
		{
			for(j=0;j<plug->evt_hdl->len;j++)
			{
				peh=&(g_array_index(plug->evt_hdl,PLUGIN_EVT_HANDLER,j));
				if(!strcmp(peh->evt_name,evt_name))
				{
					/* we have found a matching entry */
					g_ptr_array_add(gpa,peh->fnc);
				}
			}
		}
	}

	return gpa;
}

/***********************************************************************************/
/* this function creates a new empty plugin entry (only the plugin name is filled) */
/* and add it to the loaded_plugins list. It does not load the plugin itself.      */
/***********************************************************************************/
/* output: PLUGIN* = pointer on the newly allocated entry */
/**********************************************************/
static PLUGIN *create_and_add_plugin_entry(char *plugin_name)
{
	PLUGIN *plug;

	plug=malloc(sizeof(PLUGIN));
	if(plug!=NULL)
	{
		plug->plugin_name=strdup(plugin_name);
		if(plug->plugin_name!=NULL)
		{
			plug->handle=NULL;
			plug->evt_hdl=NULL;

			if(loaded_plugins==NULL)
				loaded_plugins=g_ptr_array_new();
			g_ptr_array_add(loaded_plugins,plug);
		}
		else
		{
			free(plug);
			plug=NULL;
		}
	}
	return plug;
}

/***********************************************************************************/
/* unregister the given (event,fnc) couple from the event list of the given plugin */
/***********************************************************************************/
static void unregister_evt_event_by_pointer(PLUGIN *plug, char *evt_name, PLUGIN_EVENT_FUNCTION fnc)
{
	unsigned int i;

	if(plug->evt_hdl==NULL)	/* no registered event ? */
		return;

	for(i=0;i<plug->evt_hdl->len;i++)
	{
		PLUGIN_EVT_HANDLER *peh;

		peh=&(g_array_index(plug->evt_hdl,PLUGIN_EVT_HANDLER,i));
		if((peh->fnc==fnc)&&(!strcmp(peh->evt_name,evt_name)))
		{
			/* we have found the entry to delete */
			free(peh->evt_name);
			g_array_remove_index_fast(plug->evt_hdl,i);
			return;
		}
	}
}

/*******************************************************************************/
/* register the given (event,fnc) couple to the event list of the given plugin */
/*******************************************************************************/
static void register_evt_event_by_pointer(PLUGIN *plug, char *evt_name, PLUGIN_EVENT_FUNCTION fnc)
{
	int i;
	PLUGIN_EVT_HANDLER nw;

	if(plug->evt_hdl==NULL)	/* no registered event ? */
	{
		plug->evt_hdl=g_array_new(FALSE,FALSE,sizeof(PLUGIN_EVT_HANDLER));
	}
	else
	{
		/* check if the same couple is not already in the event list */
		for(i=0;i<plug->evt_hdl->len;i++)
		{
			PLUGIN_EVT_HANDLER *peh;
	
			peh=&(g_array_index(plug->evt_hdl,PLUGIN_EVT_HANDLER,i));
			if((peh->fnc==fnc)&&(!strcmp(peh->evt_name,evt_name)))
			{
				/* we have found the same entry */
				fprintf(stderr,"register_evt_event_by_pointer: plugin %s tries to register 2 times the same couple (%s:%p). Ignored.\n",
												plug->plugin_name,evt_name,fnc);
				return;
			}
		}
	}

	nw.evt_name=strdup(evt_name);
	if(nw.evt_name==NULL)
	{
		fprintf(stderr,"register_evt_event_by_pointer: out of memory to allocate new event for plugin %s. Ignored.\n", plug->plugin_name);
		return;
	}

	nw.fnc=fnc;

	g_array_append_val(plug->evt_hdl,nw);
}

/*******************************************************************/
/* this function deletes a plugin entry using its PLUGIN pointer   */
/* all plugin registered event are unregistered and its entry is   */
/* removed from loaded_plugins list. This function does not unload */
/* the plugin. It must be called AFTER the plugin was unloaded     */
/*******************************************************************/
static void delete_plugin_entry_by_pointer(PLUGIN *plug)
{
	if(loaded_plugins==NULL)
		return;

	if(plug==NULL)
		return;

	/* remove the pointer from the array */
	if(g_ptr_array_remove_fast(loaded_plugins,plug)==FALSE)
		return;

	if(plug->evt_hdl!=NULL)
	{
		while(plug->evt_hdl->len)
		{
			unregister_evt_event_by_pointer(plug,g_array_index(plug->evt_hdl,PLUGIN_EVT_HANDLER,0).evt_name,g_array_index(plug->evt_hdl,PLUGIN_EVT_HANDLER,0).fnc);
		}
		g_array_free(plug->evt_hdl,TRUE);
		plug->evt_hdl=NULL;
	}

	if(plug->handle)
	{
		fprintf(stderr,"WARNING: in delete_plugin_entry_by_pointer, handle of plugin '%s' is not NULL\n",plug->plugin_name);
	}

	if(plug->plugin_name)
		free(plug->plugin_name);
	free(plug);
}


/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**********************************************************/
/* public functions of the plugin support for the plugins */
/**********************************************************/

/***********************************************************************************/
/* unregister the given (event,fnc) couple from the event list of the given plugin */
/***********************************************************************************/
void unregister_dchub_plugin_event(PLUGIN *plug, char *evt_name, PLUGIN_EVENT_FUNCTION fnc)
{
	if((plug==NULL)||(evt_name==NULL)||(fnc==NULL))
	{
		fprintf(stderr,"ERROR: unregister_dchub_plugin_event called with a NULL parameters: %p %p %p\n",plug,evt_name,fnc);
		return;
	}

	unregister_evt_event_by_pointer(plug,evt_name,fnc);
}

/*******************************************************************************/
/* register the given (event,fnc) couple to the event list of the given plugin */
/*******************************************************************************/
void register_dchub_plugin_event(PLUGIN *plug, char *evt_name, PLUGIN_EVENT_FUNCTION fnc)
{
	if((plug==NULL)||(evt_name==NULL)||(fnc==NULL))
	{
		fprintf(stderr,"ERROR: register_dchub_plugin_event called with a NULL parameters: %p %p %p\n",plug,evt_name,fnc);
		return;
	}

	register_evt_event_by_pointer(plug,evt_name,fnc);
}

/*************************************************************************/
/* this is a set of well known functions provided to scripting languages */
/*************************************************************************/

/*******************************/
/* send a DC command to a user */
/*******************************/
void plugin_send_to_named_user(const char *nickname, char *string)
{
	if((nickname==NULL)||(string==NULL))
	{
		fprintf(stderr,"plugin_send_to_named_user: invalid input parameters\n");
		return;
	}

	GLUS_SEND_TO_A_NICK_char_ptr(nickname,string);
}

/**********************************/
/* send a DC command to all users */
/**********************************/
void plugin_send_to_all_users(char *string)
{
	if(string==NULL)
	{
		fprintf(stderr,"plugin_send_to_all_users: invalid input parameters\n");
		return;
	}

	GLUS_SEND_TO_EVERYONE_char_ptr(string);
}

/*************************************/
/* send a message on the public chat */
/*************************************/
void plugin_send_gchat_msg(const char *sender_nickname, char *string)
{
	gchar *str;

	if((sender_nickname==NULL)||(string==NULL))
	{
		fprintf(stderr,"plugin_send_gchat_msg: invalid input parameters\n");
		return;
	}

	str=g_strconcat("<",sender_nickname,"> ",string,"|",NULL);
	GLUS_SEND_TO_EVERYONE_char_ptr(str);
	g_free(str);
}

/************************************/
/* send a message on a private chat */
/************************************/
void plugin_send_pchat_msg(const char *sender_nickname, const char *dest_nickname, char *string)
{
	gchar *str;

	if((sender_nickname==NULL)||(dest_nickname==NULL)||(string==NULL))
	{
		fprintf(stderr,"plugin_send_pchat_msg: invalid input parameters\n");
		return;
	}

   /* build and send the private chat message */
	str=g_strconcat("$To: ",dest_nickname," From: ",sender_nickname," $<",sender_nickname,"> ",string,"|",NULL);

	GLUS_SEND_TO_A_NICK_char_ptr(dest_nickname,str);
	g_free(str);
}

/*********************/
/* disconnect a user */
/*********************/
void plugin_disconnect(const char *nickname)
{
	if(nickname==NULL)
	{
		fprintf(stderr,"plugin_disconnect: invalid input parameters\n");
		return;
	}

	glus_disconnect_named_user(nickname);
}

/***************/
/* kick a user */
/***************/
void plugin_kick(const char *nickname)
{
	if(nickname==NULL)
	{
		fprintf(stderr,"plugin_kick: invalid input parameters\n");
		return;
	}

	glus_kick_named_user(NULL,nickname);
}

/********************************/
/* nickname connection duration */
/********************************/
/* output: -1=not found */
/************************/
int plugin_nick_duration(const char *nickname)
{
	time_t duration;
	GLUS_USER_INFO *gui;

	if(nickname==NULL)
	{
		fprintf(stderr,"plugin_nick_duration: invalid input parameters\n");
		return -1;
	}

	gui=glus_get_user_info(nickname);
	if(gui==NULL)
		return -1;

	duration=gl_cur_time-gui->cnx_start_time;
	glus_free_user_info(gui);

	return (int)duration;
}

/******************************************/
/* get the type of account used by a user */
/******************************************/
gint16 plugin_nick_type(const char *nickname)
{
	gint16 result;

	if (nickname == NULL) {
		fprintf(stderr,"plugin_nick_type: invalid input parameters\n");
		return 0;
	}
	if (!xml_get_user(nickname, &result, NULL, NULL, NULL))
		result=0; /* if user is not found return level 0 */
	return result;
}

/**********************************/
/* get hub shared size and #users */
/**********************************/
void plugin_get_hub_user_start(guint64 *shared_size, unsigned int *nb_users)
{
	glus_hub_users_stat(shared_size,nb_users);
	return;
}

/*************************************************/
/* return max number of users allowed on the hub */
/*************************************************/
int get_max_users(void)
{
	int max_nb;

	if (!db_int_get("MAXUSER", &max_nb))
		max_nb=100;

	return max_nb;
}


/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/******************************************************/
/* public functions of the plugin support for the hub */
/******************************************************/

/*************************************************************************************/
/* load the plugin having the given name. The plugin must be in the plugin directory */
/**********************************************************************************************/
/* input: plugin_name= the name of the plugin                                                 */
/*        output= an allocated GString. Information and error messages will be appended to it */
/**********************************************************************************************/
void plugin_start(char *plugin_name, GString *output)
{
	PLUGIN *plug;
	GString *plug_fullpath;
	int (*fnc)(PLUGIN *);

	if(plugin_directory==NULL)
	{
		g_string_sprintfa(output,"plugin support is disabled (no plugin directory defined)");
		return;
	}

	if(is_a_loaded_plugin(plugin_name))
	{
		g_string_sprintfa(output,"plugin is already loaded");
		return;
	}

	plug=create_and_add_plugin_entry(plugin_name);
	if(plug==NULL)
	{
		g_string_sprintfa(output,"not enough memory to create entry for plugin '%s'",plugin_name);
		return;
	}

	/* time to load the plugin really */
	plug_fullpath=g_string_new(plugin_directory);
	g_string_sprintfa(plug_fullpath,"/%s",plugin_name);
	plug->handle=dlopen(plug_fullpath->str,RTLD_LAZY|RTLD_GLOBAL);
	if(plug->handle==NULL)
	{
		g_string_sprintfa(output,"load error: %s",dlerror());
		delete_plugin_entry_by_pointer(plug);
		return;
	}

	/* find the function 'plugin_beginning' in the plugin */
	fnc=dlsym(plug->handle,"plugin_beginning");
	if(fnc==NULL)
	{
		g_string_sprintfa(output,"no function 'plugin_beginning' defined by the plugin");
		dlclose(plug->handle);
		plug->handle=NULL;
		delete_plugin_entry_by_pointer(plug);
		return;
	}

	/* and call 'plugin_beginning'. If output is !=0, it is an error */
	if((*fnc)(plug)!=0)
	{
		g_string_sprintfa(output,"'plugin_beginning' function fails.");
		dlclose(plug->handle);
		plug->handle=NULL;
		delete_plugin_entry_by_pointer(plug);
		return;
	}

	g_string_sprintfa(output,"Ok");
	return;
}

/********************************************/
/* unload the plugin having the given name. */
/**********************************************************************************************/
/* input: plugin_name= the name of the plugin                                                 */
/*        output= an allocated GString. Information and error messages will be appended to it */
/**********************************************************************************************/
void plugin_stop(char *plugin_name, GString *output)
{
	void (*fnc)(PLUGIN *);
	PLUGIN *plug;

	if(plugin_directory==NULL)
	{
		g_string_sprintfa(output,"plugin support is disabled (no plugin directory defined)");
		return;
	}

	plug=get_plugin_ptr_by_name(plugin_name);
	if(plug==NULL)
	{
		g_string_sprintfa(output,"plugin '%s' is not loaded",plugin_name);
		return;
	}

	/* find the function 'plugin_end' in the plugin */
	fnc=dlsym(plug->handle,"plugin_end");
	if(fnc==NULL)
	{
		g_string_sprintfa(output,"WARNING: no function 'plugin_end' defined by the plugin");
	}

	/* and call 'plugin_end'. If output is !=0, it is an error */
	(*fnc)(plug);

	/* unload the plugin */
	dlclose(plug->handle);
	plug->handle=NULL;

	/* and free its entry */
	delete_plugin_entry_by_pointer(plug);
	g_string_sprintfa(output,"Ok");
	return;
}

/**********************************************************************************************/
/* if the plugin directory is defined and contains a file named AUTOSTART, all plugins having */
/* their name in this file are loaded.                                                        */
/**********************************************************************************************/
void load_all_autostart_plugin(void)
{
	FILE *f;
	GString *str;

	if((plugin_directory==NULL)||(strlen(plugin_directory)==0))
	{
		fprintf(stderr,"No plugin directory defined, plugin support is disabled.\n");
	}

	str=g_string_new(plugin_directory);
	str=g_string_append(str,"/AUTOSTART");
	f=fopen(str->str,"rb");
	if(f==NULL)
	{
		fprintf(stderr,"No plugin to start automatically.\n");
	}
	else
	{
		char buf[2048];
		GString *output;

		output=g_string_new("");
		while(fgets(buf,sizeof(buf),f)!=NULL)
		{
			char *r;

			r=strpbrk(buf,"\r\n");
			if(r!=NULL)
				*r='\0';

			fprintf(stderr,"Loading '%s' plugin:",buf);
			fflush(stderr);
			g_string_truncate(output,0);
			plugin_start(buf,output);
			fprintf(stderr,"%s\n",output->str);
		}

		fclose(f);

		g_string_free(output,TRUE);
	}
	g_string_free(str,TRUE);
}

/***************************************************************/
/* send event to plugin (same prototype as perl event handler) */
/***************************************************************/
void send_evt_to_plugin(const char *evt_name, const char *evt_emitter, int nb_args, ...)
{
	GPtrArray *gpa;
	int i;
	va_list ap;

	gpa=get_all_function_addresses_for_an_evt(evt_name);
	if(gpa==NULL)
		return;

	if(gpa->len!=0)
	{
		GStringChunk *gsc;
		GArray *param;
		PLUGIN_PARAM pp;
		char buf[512];

		gsc=g_string_chunk_new(32);
		param=g_array_new(FALSE,FALSE,sizeof(PLUGIN_PARAM));

		/* build parameter array */
		/* name: "event", value: evt_name */
		pp.varname=g_string_chunk_insert(gsc,"event");
		pp.varvalue=g_string_chunk_insert(gsc,evt_name);
		param=g_array_append_val(param,pp);

		/* name: "nickname", value: evt_emitter */
		pp.varname=g_string_chunk_insert(gsc,"nickname");
		pp.varvalue=g_string_chunk_insert(gsc,evt_emitter);
		param=g_array_append_val(param,pp);

		/* name: "argc", value: nb_args */
		pp.varname=g_string_chunk_insert(gsc,"argc");
		sprintf(buf,"%u",nb_args);
		pp.varvalue=g_string_chunk_insert(gsc,buf);
		param=g_array_append_val(param,pp);

		/* name: "0"->"...", value: first argument -> ... argument */
		va_start(ap,nb_args);
		for(i=0;i<nb_args;i++)
		{
			char *arg;
			char tmp_num[10];

			arg=va_arg(ap,char*);

			sprintf(tmp_num,"%u",i);

			pp.varname=g_string_chunk_insert(gsc,tmp_num);
			pp.varvalue=g_string_chunk_insert(gsc,arg);
			param=g_array_append_val(param,pp);
   	}
 
		for(i=0;i<gpa->len;i++)
		{
			((PLUGIN_EVENT_FUNCTION)(g_ptr_array_index(gpa,i)))(param);
		}

		g_array_free(param,TRUE);
		g_string_chunk_free(gsc);
		va_end(ap);
	}

	g_ptr_array_free(gpa,TRUE);
}

/* HAVE_LIBDL */
#endif
