#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <errno.h>
#include "mmap_tools.h"
#include "xmltools.h"
#include "users_xml.h"

#ifdef HAVE_GETOPT_LONG
	#include <getopt.h>
#endif

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>    /* Gray Watson's library */
#define show_alloc() dmalloc_log_unfreed()
#define show_stats() dmalloc_log_stats()
#else
#define show_alloc()    /* nothing */
#define show_stats()    /* nothing */
#endif

/************************************************************/
/* each entry of the password file has the following format */
/*
ABBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBCCCCCCCCCCCCCD
where A='F' if the entry is empty and 'U' if it is used
B...B is a 32 bytes C-string (always ended by a '\0') containing a nickname. 
  => the longest nickname takes 31 chars 
C...C is a 16 butes C-string (always ended by a '\0') containing a password. 
  => the longest password takes 15 chars               
D is a 1 byte value: D='O' if this nick is an operator, 
                     D='N' if this nick is a normal user,
                     D='M' if this nick is the master of the hub
                        (==operator + special function) 
    only the master of the hub can modify configuration and user file remotely.
************************************************************/
#define PASSWD_ENTRY (1+32+16+1)

#define NICK_PASS_POS 1
#define NICK_PASS_LEN 32
#define PASS_PASS_POS (NICK_PASS_POS+NICK_PASS_LEN)
#define PASS_PASS_LEN 16
#define TYPE_PASS_POS (PASS_PASS_POS+PASS_PASS_LEN)

const char* db_user_file;
const char* xml_user_file;
char* dchub_templatedir=NULL;

static void	display_usage(char *prog) {
	fprintf(stderr,"Usage: %s [options] binary_user_file xml_user_file\n"
			"Options:\n"
			"\t-h, --help                   Display this help\n"
			"\t-T, --templatedir=DIRECTORY  name of the directory containing the XML template\n"
			"\t                             files [DEFAULT: %s]\n"
#ifndef HAVE_GETOPT_LONG
			"\n  WARNING this build do not support long arguments\n"
#endif	
			"\n", prog, DCHUB_TEMPLATEDIR);
}

static void parse_command_line_options(int argc,char **argv) {
#ifdef HAVE_GETOPT_LONG
	static struct option optab[]= {
		{"help",no_argument,NULL,'h'},           /* get help */
		{"templatedir",required_argument,NULL,'T'},	/* the template dir containing some XML files */
		{NULL,0,NULL,'\0'}               /* last option */
	};
#endif
	static const char *short_opt="hT:";
	int ch;
	opterr = 0;
	
#ifdef HAVE_GETOPT_LONG
	while((ch=getopt_long(argc,argv,short_opt,optab,NULL))!=EOF) {
#else
	while((ch=getopt(argc,argv,short_opt))!=EOF) {		
#endif
		switch(ch) {
			case 'h':
				display_usage(argv[0]);
				exit(0);
			case 'T':
				dchub_templatedir=strdup(optarg);
				break;
			default:
				fprintf(stderr,"Unknown option: %c\n",ch);
				exit(1);
		}
	}

	if ((argc-optind) != 2) {
		display_usage(argv[0]);
		exit(0);
	}

	db_user_file=argv[optind];
	xml_user_file=argv[optind+1];
}
	
int main(int argc, char **argv)  {
	FILE* fdb_user_file;
	char buf[PASSWD_ENTRY];
	struct stat st;
	gboolean new_xml_file=FALSE;

	int operator_level;
	
	parse_command_line_options(argc,argv);

	initXMLTools(dchub_templatedir);

	if (dchub_templatedir==NULL)
		dchub_templatedir=strdup(DCHUB_TEMPLATEDIR);

	fdb_user_file=fopen(db_user_file,"rb");
	if (fdb_user_file == NULL) {
		printf("Can't load user file %s: %s\n",db_user_file,strerror(errno));
		exit(1);
	}
	
	if (stat(xml_user_file, &st) != 0) {
		FILE* fxml_user_file;
		GString *template_user_file;
		size_t flen;
		unsigned char *fadr;
		
		template_user_file=g_string_new("");
		g_string_printf(template_user_file,"%s/%s",dchub_templatedir,"users.xml.in");
	
		if (stat(template_user_file->str, &st) != 0) {
			printf("Can't load template file %s: %s\n", template_user_file->str, strerror(errno));
			g_string_free(template_user_file, TRUE);
			exit (1);
		}
		
		printf("Creating a new users file %s\n",xml_user_file);

		fxml_user_file=fopen(xml_user_file,"wb");
		if (fxml_user_file == NULL) {
			printf("Can't create user file %s: %s\n",xml_user_file,strerror(errno));
			exit(1);
		}
			
        /* copy the template */
		fadr=map_file(template_user_file->str,&flen);
		fwrite(fadr,1,flen,fxml_user_file);
		unmap_file(fadr,flen);
		fclose(fxml_user_file);
		g_string_free(template_user_file,TRUE);

		new_xml_file=TRUE;
	}
	
	if (!loadUsersFile(xml_user_file)) {
		fprintf(stderr,"Error can't load %s.\n",xml_user_file);
		exit(1);
	}

	if (new_xml_file) {
		/* we remove the MASTER account */
		xml_del_user("MASTER");
	}
	
	operator_level=get_right_level(OPERATOR);
	
	while (fread(buf,1,PASSWD_ENTRY,fdb_user_file)==PASSWD_ENTRY) {
		if(buf[0]=='U') {
			int level_to_use;
			char *nick=buf+NICK_PASS_POS;
			char *password=buf+PASS_PASS_POS;
			switch (((int)buf[TYPE_PASS_POS])&255) {
				case 'O':
					level_to_use=operator_level;
					break;
				case 'M':
					level_to_use=MASTER_LEVEL;
					break;
				default:
					level_to_use=0;
					break;
			}
			
			if (xml_get_user(nick,NULL,NULL,NULL,NULL)) {
				/* nick already exist */
				printf("User %s already exists: not imported\n",nick);
			} else {
				xml_add_user(nick, level_to_use, password, FALSE, NULL);
			}
		}
	}

	fclose(fdb_user_file);
			
	/* Free all XML resources for users file */
	xml_free_users_resource();

	/* Free XML Tools */
	freeXMLTools();

	if (dchub_templatedir)
		free(dchub_templatedir);

	return 0;
}
