/* DChub - a Direct Connect Hub for Linux
 *
 * Copyright (C) 2001 Eric Prevoteau
 *
 * migrate_dchub_configuration.c: Copyright (C) Yves BLUSSEAU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
 $Id: migrate_dchub_configuration.c,v 1.2 2003/11/22 07:39:55 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <errno.h>
#include "mmap_tools.h"
#include "xmltools.h"
#include "db_xml.h"

#ifdef HAVE_GETOPT_LONG
	#include <getopt.h>
#endif

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>    /* Gray Watson's library */
#define show_alloc() dmalloc_log_unfreed()
#define show_stats() dmalloc_log_stats()
#else
#define show_alloc()    /* nothing */
#define show_stats()    /* nothing */
#endif

const char* db_conf_file;
const char* xml_conf_file;
char* dchub_templatedir=NULL;

typedef struct s_db_sym {
   char			*name;
   char			*str_value;
   int			int_value;
   int			float_value;
   int			type;
} t_db_sym;

enum { EDB_TYPE_INT, EDB_TYPE_FLOAT, EDB_TYPE_STR };

static void	display_usage(char *prog) {
	fprintf(stderr,"Usage: %s [options] binary_conf_file xml_conf_file\n"
			"Options:\n"
			"\t-h, --help                   Display this help\n"
			"\t-T, --templatedir=DIRECTORY  name of the directory containing the XML template\n"
			"\t                             files [DEFAULT: %s]\n"
#ifndef HAVE_GETOPT_LONG
			"\n  WARNING this build do not support long arguments\n"
#endif	
			"\n", prog, DCHUB_TEMPLATEDIR);
}

static void parse_command_line_options(int argc,char **argv) {
#ifdef HAVE_GETOPT_LONG
	static struct option optab[]= {
		{"help",no_argument,NULL,'h'},           /* get help */
		{"templatedir",required_argument,NULL,'T'},	/* the template dir containing some XML files */
		{NULL,0,NULL,'\0'}               /* last option */
	};
#endif
	static const char *short_opt="hT:";
	int ch;
	opterr = 0;
	
#ifdef HAVE_GETOPT_LONG
	while((ch=getopt_long(argc,argv,short_opt,optab,NULL))!=EOF) {
#else
	while((ch=getopt(argc,argv,short_opt))!=EOF) {		
#endif
		switch(ch) {
			case 'h':
				display_usage(argv[0]);
				exit(0);
			case 'T':
				dchub_templatedir=strdup(optarg);
				break;
			default:
				fprintf(stderr,"Unknown option: %c\n",ch);
				exit(1);
		}
	}

	if ((argc-optind) != 2) {
		display_usage(argv[0]);
		exit(0);
	}

	db_conf_file=argv[optind];
	xml_conf_file=argv[optind+1];
}
	
static char	*fgets_line(FILE *file, int end) {
	char	*out;
	unsigned int	s;
	int	i;
	int	c;
	 
	s = 48;
	out=malloc(s);
	out[0] = '\0';
	i = 0;
	while (1) {
		c = fgetc(file);
		if (c == EOF) {
			free (out);
			return (NULL);
		}
		if (i >= s) {
			s *= 2;
			out = realloc(out, s);
		}
		if (c == end) {
			out[i] = 0;
			return (out);
		}
		out[i++] = c;
	}
}

gboolean migrate_db_table (const char* filename, FILE* file) {
	t_db_sym	*tmp;
	int	entries_expected;
	//t_db_sym **table;
	int	entries = 0;
	
	if (fscanf(file, "%d\n", &entries_expected) != 1)
		return FALSE;

	//printf("%d entries expected\n",entries_expected);

	while (1) {
		tmp = malloc(sizeof(t_db_sym));
		if ((fread(tmp, 1, sizeof(t_db_sym), file) != (sizeof(t_db_sym)))) /* read one entry */
			break;
		entries++;
		tmp->name = NULL;
		tmp->str_value = NULL;
		if ((tmp->name = fgets_line(file, '\0')) == 0) /* read the name for the entry */
			goto load_error;
		if ((tmp->str_value = fgets_line(file, '\0')) == 0) {
			free(tmp->name);
			
		  load_error:
			printf("%s file is corrupted %d entry read\n", filename, entries);
			free (tmp);
			return FALSE;
		}
		// printf("Name:[%s] Type:%d Int:%d Float:[%f] Str:[%s]\n",tmp->name, tmp->type,
		//	   tmp->int_value, (float)tmp->float_value, tmp->str_value);

		/* look if the key already exist */
		if (db_type_get(tmp->name) != -1) {
			printf ("Key %s already exist. Key not imported\n",tmp->name);
		} else {
			switch (tmp->type) {
				case EDB_TYPE_STR:
					db_str_set(tmp->name,tmp->str_value);
					break;
				case EDB_TYPE_INT:
					db_int_set(tmp->name,tmp->int_value);
					break;
				case EDB_TYPE_FLOAT:
					db_float_set(tmp->name,tmp->float_value);
					break;
			}
		}
	
		free(tmp->str_value);
		free(tmp->name);
		free(tmp);
	}
	free(tmp);
	
	if (entries != entries_expected) {
		fprintf(stderr, "WARNING database is corrupted.");
		return FALSE;
	}
	
	printf("%d Key%s migrate from %s.\n", entries, (entries>1)?"s":"",filename);
	return TRUE;
}

int main(int argc, char **argv)  {
	FILE* fdb_conf_file;
	struct stat st;
	gboolean new_xml_file=FALSE;
	int result=1;
	
	parse_command_line_options(argc,argv);

	initXMLTools(dchub_templatedir);

	if (dchub_templatedir==NULL)
		dchub_templatedir=strdup(DCHUB_TEMPLATEDIR);

	fdb_conf_file=fopen(db_conf_file,"rb");
	if (fdb_conf_file == NULL) {
		printf("Can't load configuration file %s: %s\n", db_conf_file, strerror(errno));
		exit(1);
	}
	
	if (stat(xml_conf_file, &st) != 0) {
		FILE* fxml_conf_file;
		GString *template_conf_file;
		size_t flen;
		unsigned char *fadr;
		
		template_conf_file=g_string_new("");
		g_string_sprintf(template_conf_file,"%s/%s",dchub_templatedir,"conf.xml.in");
	
		if (stat(template_conf_file->str, &st) != 0) {
			printf("Can't load template file %s: %s\n", template_conf_file->str, strerror(errno));
			g_string_free(template_conf_file, TRUE);
			exit (1);
		}
		
		printf("Creating a new configuration file %s\n",xml_conf_file);

		fxml_conf_file=fopen(xml_conf_file,"wb");
		if (fxml_conf_file == NULL) {
			printf("Can't create configuration file %s: %s\n",xml_conf_file,strerror(errno));
			exit(1);
		}
			
        /* copy the template */
		fadr=map_file(template_conf_file->str,&flen);
		fwrite(fadr,1,flen,fxml_conf_file);
		unmap_file(fadr,flen);
		fclose(fxml_conf_file);
		g_string_free(template_conf_file,TRUE);

		new_xml_file=TRUE;
	}
	
	if (!loadDBFile(xml_conf_file)) {
		fprintf(stderr,"Error can't load %s.\n",xml_conf_file);
		exit(1);
	}

	if (migrate_db_table(db_conf_file, fdb_conf_file))
		result=0;
	
	fclose(fdb_conf_file);
			
	/* Free all XML resources for users file */
	xml_free_db_resource();

	/* Free XML Tools */
	freeXMLTools();

	if (dchub_templatedir)
		free(dchub_templatedir);

	return result;
}
