/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * main.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: main.c,v 2.56 2003/11/22 13:47:08 blusseau Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#ifdef HAVE_SIGNAL_H
	#include <signal.h>
	#define HAVE_SIGNAL 1
#else
	#undef HAVE_SIGNAL
#endif

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_POLL_H
#include <sys/poll.h>
#else
#include "my_poll.h"
#endif
#include <sys/wait.h>

#include <glib.h>
#include <regex.h>

#ifdef WIN32
	#include <windows.h>
	#include <winsock2.h>
	#include <getopt.h>
#else
	#ifdef HAVE_SYS_TIME_H
		#include <sys/time.h>
	#endif
 	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#ifdef HAVE_UNISTD_H
		#include <unistd.h>
	#endif  /* HAVE_UNISTD_H */
#endif

#ifndef WIN32
	#include <pwd.h>
#endif

#ifdef HAVE_ERRNO_H
	#include <errno.h>
#else
	extern int errno;
	#ifndef EINPROGRESS
		#define EINPROGRESS	115
	#endif
	#ifndef EOPNOTSUPP
		#define EOPNOTSUPP	95
	#endif
	#ifndef EINTR
		#define EINTR		4
	#endif
#endif

#ifdef HAVE_GETOPT_LONG
	#include <getopt.h>
#endif

#include "config.h"
#include "main.h"
#include "network.h"
#include "do_hub_reg.h"

#ifdef WITH_PERL
	#include "emb_perl.h"
#endif
#include "timed_out_string.h"
#include "tos_key.h"
#include "color_debug.h"
#include "gvar.h"
#include "plugin.h"
#include "ged.h"
#include "main_cnx.h"
#include "hub_cnx.h"
#include "toolkit.h"
#include "xf_io.h"
#include "bin_xf_io.h"
#include "user_cnx_lst.h"
#include "hub_cnx_lst.h"
#include "main_cnx_handshake.h"
#include "hub_cnx_handshake.h"
#include "hub_cnx_connect.h"
#include "global_user_if.h"
#include "multi_public_chat.h"
#include "hub_passwd.h"
#include "lmp.h"
#include "md_db.h"
#include "xmltools.h"
#include "users_xml.h"
#include "db_xml.h"
#include "mmap_tools.h"

#ifdef sun
	#define HAVE_NO_THREAD 1
#endif
#ifndef HAVE_INET_ATON
	#include "inet_aton.c"
#endif

#ifdef WIN32
 WSADATA gl_ws_info;
#endif

#ifdef __CYGWIN__
#define GLIB_INT_64_BUG
#endif

#ifdef WIN32
#define GLIB_INT_64_BUG
#endif

#ifdef GLIB_INT_64_BUG
char	*llunsigned_to_str(guint64 a);
#endif

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>    /* Gray Watson's library */
#define show_alloc() dmalloc_log_unfreed()
#define show_stats() dmalloc_log_stats()
#else
#define show_alloc()    /* nothing */
#define show_stats()    /* nothing */
#endif

time_t gl_cur_time;
unsigned short hub_port=411;
unsigned short reg_hub_port=0;

unsigned short cluster_port=410;		/* default cluster port */

GString	*hubname=NULL;
char	*tos_filename=NULL;
char	*dchub_templatedir=NULL;
int	gl_sr_interval;
int	gl_moderate_mode;
int	gl_silent_limit;
int	gl_kick_limit;
int	gl_buf_size;

#if 0
int	gl_reg_sock = -1;
int	gl_resolver_fd = -1;
int	gl_resolv_id = 0;
#endif

int	gl_peek_user = 0;

int	daemonize = 0;
char* pidfile = NULL;
char* logfile = NULL;
int fd_logfile = -1;

char	*gl_bind_addr;

/***************************/
/* configuration file vars */
/***************************/
char *conf_filename=NULL;		/* name of the configuration file */

/******************/
/* user file vars */
/******************/
char *user_filename=NULL;		/* name of the user file */
char *user_uid=NULL;			/* name of the user */
struct passwd *user_pw=NULL;	/* passwd entry containing pw_uid of the user to use */

/*********************/
/* LOCK_PATTERN vars */
/*********************/
GPtrArray *gl_lock_patern = NULL;
G_LOCK_DEFINE(gl_lock_patern);

/***************************************/
/* some vars used for Perl interpreter */
/***************************************/
char *perl_script_dir=NULL;
char *perl_script_main=NULL;

/*****************************************/
/* somes vars used for external programs */
/*****************************************/
char *ext_prog_dir=NULL;

/********************************/
/* directory containing plugins */
/********************************/
char *plugin_directory=NULL;

/****************************************************************************/
/* it is not possible to start an external program from user thread because */
/* starting a program adds a user and this requires write access on cnx_lst */
/* Unfortunately, when a user thread manages its connection, it sets a read */
/* lock on the structure.                                                   */
/****************************************************************************/
GPtrArray *prog_to_start=NULL;

/*******************************************************/
/* add an external program to start by the main thread */
/*******************************************************/
void add_prog_to_start(char *progname)
{
	if(prog_to_start==NULL)
		prog_to_start=g_ptr_array_new();

	g_ptr_array_add(prog_to_start,strdup(progname));
}

#ifndef HAVE_SOCKETPAIR
int socketpair(int d, int type, int protocol, int sv[2])
{
	return(-1);
}
#endif

#ifndef HAVE_BZERO
void bzero (void *s, size_t n)
{
	memset(s, 0, n);
}
#endif


#ifndef HAVE_INET_NETWORK
unsigned long int inet_network(const char *str)
{
	return(inet_addr(str));
}
#endif

/**************************/
/* an extanded socketpair */
/* return 0 if it's Ok    */
/*        1 if fail       */
/**************************/

int	dchub_socketpair(int d, int type, int protocol, int sockfd[2])
{
	/* if(socketpair(AF_INET,SOCK_STREAM,0,sockfd)==-1) */
	if(socketpair(d, type, protocol, sockfd)==-1)
	{
#ifndef EOPNOSUPPORT
#define EOPNOSUPPORT EOPNOTSUPP
#endif
		if(errno!=EOPNOSUPPORT)				/* according to the man, it should be this */
		{											/* but the value is not defined in errno.h */
			perror("dchub_socketpair:socketpair");
			return 1;
		}
		else
		{
			int temp_sock;
			struct sockaddr_in lcl;
			unsigned int len_lcl=sizeof(lcl);

			perror("dchub_socketpair:socketpair (trying manually)");

			/* linux doesn't seems to allow socketpair in INET domain, we will do this manually */
			/* but it is a bit long */

			/* create 2 sockets */
			sockfd[0]=_x_tcp(0);
			if(sockfd[0]==-1)
			{
				fail:
				return 1;
			}

			temp_sock=_x_tcp(0);
			if(temp_sock==-1)
			{
				close(sockfd[0]); // ok
				goto fail;
			}

			if(getsockname(temp_sock,(void*)&lcl,&len_lcl)!=0)
			{
				perror("dchub_socketpair:getsockname");
				close(temp_sock); // ok
				close(sockfd[0]); // ok
				goto fail;
			}
			listen(temp_sock,1);
			
			/* and connect the first on the second. The accept returns the true second */
			/* don't make a blocking connect else everything hangs */
			set_non_bloquant_sock(sockfd[0]);

			if(connect(sockfd[0],(void*)&lcl,len_lcl)!=0)
			{
				if(errno!=EINPROGRESS)
				{
					perror("start_new_ext_prog:connect");
					close(temp_sock); // ok
					close(sockfd[0]); // ok
					goto fail;
				}
			}
			
			sockfd[1]=accept(temp_sock,NULL,NULL);
			if(sockfd[1]==-1)
			{
				perror("start_new_ext_prog:accept");
				close(temp_sock); // ok
				close(sockfd[0]); // ok
				goto fail;
			}

			set_bloquant_sock(sockfd[0]);

			shutdown(temp_sock,SHUT_RDWR);
			close(temp_sock); // ok

			/* that's all folks :) easy isn't it ? */
		}
	}
return 0;
}

#if 0
/* will be use in dchub 0.3.0 for non treadable systems */
static void create_resolver(void)
{
	int sockfd[2];			/* 0=hub side, 1=program side */
	int num;

	/* allocate a socketpair (one side for the hub, the other for the dns resolver) */
	if (dchub_socketpair(AF_INET,SOCK_STREAM,0,sockfd))
		{
			failure:
			fprintf(stderr, "unable to stard resolver process\n");
			exit (1);
		}
	if ((num=fork()) == -1);
	{
		/* bad news */
		printf("fork fails in resolver init\n");
		shutdown(sockfd[1],SHUT_RDWR);
		close(sockfd[1]); // ok
		goto failure;

	}
	if (num == 0); /* the soon, dns resolver*/
	{
		char	dns[256];
		char	out[300];
		struct in_addr	addr;
		int	i;
		char	*p1;
		char	*p2;
		int	offset = 0;

		/* close all unnecessary descriptor */ // ok
		{
			int i;
			int mx;

#ifdef WIN32
			mx=gl_ws_info.iMaxSockets;
#else
			mx=sysconf(_SC_OPEN_MAX);
#endif
			for(i=3;i<mx;i++)
				if(i!=sockfd[1])
					close(i); // ok
		}
		while (1)
		{					/* send number:foo.bar.com\n */ /* replay number:XXX.XXX.XXX.XXX\n */
			if (strchr(dns, '\n') == NULL)
			{
				i = read(sockfd[0], dns + offset, 255 - offset);
				offset += i;
			}
			if (i <= -1)
				exit (1);
			dns[i] = '\0';
			
			p1 =  strchr(dns, '\n');
			if (p1)
				*p1 = '\0';
				
				
			p2 = strchr(dns, ':');
			if (p2)
			{
				if (str_to_inaddr(p2 + 1, &addr))
					sprintf(out, "%d:F\n", atoi(dns)); /* resolve fail */
				else
					sprintf(out, "%d:%s\n", atoi(dns), inet_ntoa(addr));
				}
			}
			p1++; /* seek end of first requet */
			if (dns + offset == p1)
				offset = 0;
			else
			{
				memmove(dns, p2, offset - (p2 - dns));
				offset -= (p2 - dns);
			}
			dns[offset] = '\0';
		}	
	/* set the global variable */
	shutdown(sockfd[1],SHUT_RDWR);
	close(sockfd[1]); // ok
	gl_resolver_fd = sockfd[0];
}

static int	dns_used = 0;
static char	local_resolv_buf[128 + 1];
static int	local_resolv_buf_size = 0;
static int	local_resolv_id = 0;

int		get_resolved_id()
{
	if (dns_used == 0)
	{
		local_resolv_buf_size += read(gl_resolver_fd, local_resolv_buf + local_resolv_buf_size, 128 - local_resolv_buf_size);
		/* add gertion error */
		local_resolv_buf[local_resolv_buf_size] = '\0';
		local_resolv_id = atoi(local_resolv_buf);
	}
	return(local_resolv_id);
}

int	get_resolved_dns(struct in_addr *addr)
{
	char	*p1;
	char	*p2;
	int	ret = 0;
	
	p1 = strchr(local_resolv_buf, ':');
	p2 = strchr(p1, '\n');
	
	if (dns_used == 0)
		return (-1);
	
	if (p1[1] == 'F') /* if the resolv fail  RCV : id:F\n*/
		ret = 1;
	else		/* resolf sucess */
	{
		p2 = '\0'; /* remplace the '\n' by a '\0' */
		if (str_to_inaddr(p1 + 1, addr))
			ret = 1; /* sould never append */
		*p2 = '\n'; /* restor the '\n' */
	}
	
	p2++; /* seek the end of the sting */
	if ((p2 - local_resolv_buf) == local_resolv_buf_size)
	{
		dns_used = 0;
	}
	else
	{
		memmove(local_resolv_buf, p2, local_resolv_buf_size - (p2 - local_resolv_buf));
		local_resolv_buf_size -= p2 - local_resolv_buf;
		local_resolv_buf[local_resolv_buf_size] = '\0';
		if (strchr(local_resolv_buf, '\n') == NULL)
			dns_used = 0; /* no more resolved dns in buffer */
	}
	return (ret);
}

int	start_resolve(char *dns)
{
	char		buf[256];
	
	local_resolv_id++;
	
	if (local_resolv_id> 1500000)
		local_resolv_id = 1;
		
	snprintf(buf, 256, "%d:%s\n", local_resolv_id, dns);
	write(gl_resolver_fd, buf, strlen(buf));
	return (local_resolv_id);
}

#endif

struct my_args
{
	int	*fds;
	char	*filename;
};

static void fork_child(struct my_args *param)
{
	GPtrArray *arg_exec;
	char buf[64];
	GString *str_hub_ip;
	int	*sockfd;
	char	*progname;

	set_user_id();

	progname = param->filename;
	sockfd = param->fds;
	/* close all unnecessary descriptor */ // ok
	{
		int i;
		int mx;

#ifdef WIN32
		mx=gl_ws_info.iMaxSockets;
#else
		mx=sysconf(_SC_OPEN_MAX);
#endif
		for(i=3;i<mx;i++)
		if(i!=sockfd[1])
		close(i); // ok
	}

	sprintf(buf,"%u",sockfd[1]);

	str_hub_ip=get_default_host_ip(gl_bind_addr);
	/* create the arguments of the program to start */
	arg_exec=g_ptr_array_new();
	g_ptr_array_add(arg_exec,g_strconcat(ext_prog_dir,"/",progname,NULL));
	g_ptr_array_add(arg_exec,progname);
	g_ptr_array_add(arg_exec,buf);
	g_ptr_array_add(arg_exec,str_hub_ip->str);
	g_ptr_array_add(arg_exec,NULL);
						
	execv(g_ptr_array_index(arg_exec,0),(void*)(arg_exec->pdata));
	perror("execv");
	printf("execv fails: %s %s %s\n",(char*)g_ptr_array_index(arg_exec,0),progname,buf);
#ifdef WIN32	
	_endthread();
#endif
	exit(1);
}

/**************************************************************************/
/* start a new external program and attach it to the hub as a normal user */
/******************************************************************************/
/* the external program is always seen as an operator (not master of the hub) */
/******************************************************************************/
/* progname: name of the program (contains only valid DC char) */
/***************************************************************/
static void start_new_ext_prog(char *progname)
{
	int sockfd[2];			/* 0=hub side, 1=program side */
	LOCAL_USER_CNX_ENTRY *luce;
	int num;
	gint16 level;
	struct my_args param;


	if(ext_prog_dir==NULL)
	{
		fprintf(stderr,"No external program directory defined.\n");
		return;
	}

	/* allocate a socketpair (one side for the hub, the other for the program) */
	if (dchub_socketpair(AF_INET,SOCK_STREAM,0,sockfd))
		return;

	/* create I/O structure */
	if (!xml_get_user(progname,&level,NULL,NULL,NULL))	/* look if the bot is register     */
		level=get_right_level(OPERATOR);				/* and take is level, if not found */
	                                                    /* use the OPERATOR level          */
	luce=uc_create_entry(create_xfio(sockfd[0],NULL),		/* create and register an I/O structure */
	                     progname,		/* nickname */
	                     0,				/* shared size */
	                     NULL,			/* mail */
	                     NULL,			/* description */
	                     NULL,			/* client version */
	                     1,				/* user flag */
	                     0,				/* ext flag */
						 level, 		/* default level for a Bot */
						 TRUE);			/* it's a BOT */
	                     
	param.filename = progname;
	param.fds = sockfd;

#ifdef WIN32
	num = _beginthread(fork_child, 0, &param);
	/* CreateProcess((null), cmd, ...) */
#else
	if ((num=fork()) == -1)
	{
		/* bad news */
		printf("fork fails: %s\n",progname);
		shutdown(sockfd[1],SHUT_RDWR);
		close(sockfd[1]);
		/* let the main loop handles structure deletion */
	}
	/* double fork to avoid zombie */
	else if (num == 0) /* main forks intermediate */
	{
		if ((num=fork())==-1) { /* intermediate fork the prog  */
			/* bad news */
			printf("fork fails: %s\n",progname);
			exit(1);
		} else if (num == 0) {
		 	fork_child(&param); /* do the job :-) */
		}
		exit(0); /* intermediate quits */
			 /* new prog will quit here too */
	}
	else
#endif
	{	/* notify to all users the incoming of this program */

		close(sockfd[1]);			/* close the remote side */ 
		waitpid(num, NULL, 0 );			/* clean up intermediate */

		/* send the hello string to everyone */
		glus_do_hello(luce->user_nick->str);

		/* send the myinfo to everyone */
		glus_do_my_info(luce->user_nick->str, luce->user_description->str, luce->user_cnx_type,
						luce->user_flag, luce->user_mail->str, luce->shared_size, 
						luce->ext_flag, luce->level, luce->client_version->str);

		luce->ext_display_flag|=HAVE_SENT_MYINFO;

		 /* and to conclude, update oplist of all users */
		{
			GString *lists[2];
			glus_get_nick_lists(&lists[0],&lists[1]);

			g_string_prepend(lists[1],"$OpList ");
			g_string_append(lists[1],"||");
			GLUS_SEND_TO_EVERYONE(lists[1]);		/* don't free lists[1] */

			g_string_free(lists[0],TRUE);			/* but don't forget to free lists[0] */
		}
	}
	printf("extproc ID: %d\n",num);
}

/**************************************************************/
/* check periodically if any external program must be started */
/**************************************************************/
static void check_prog_to_start(void)
{
	if(prog_to_start==NULL)
		return;

	while(prog_to_start->len!=0)
	{
		char *progname;

		progname=g_ptr_array_remove_index_fast(prog_to_start,0);
		if(progname!=NULL)
		{
			start_new_ext_prog(progname);
			free(progname);
		}
	}
}

int user_wants_to_quit=0;

#ifdef HAVE_SIGNAL
static void reopenlog (void)
{
	if(fd_logfile!=-1)
	{
		close(fd_logfile);
	}

	/* create the log file if necessary */
	if (logfile!=NULL)
	{
		fd_logfile = open(logfile, O_WRONLY | O_CREAT | O_APPEND, 0666);
		if (fd_logfile < 0)
		{
			fprintf(stderr,"Unable to create %s\n",logfile);
		}
		else
		{
			if(dup2(fd_logfile,1)<0)
			{
				fprintf(stderr,"Unable to make the dup2\n");
			}
			if(dup2(fd_logfile,2)<0)
			{
				fprintf(stdout,"Unable to make the dup2\n");
			}
		}
	}
}

static void catch_sig(int sig)
{
	switch(sig)
	{
		case SIGTERM:
		case SIGQUIT:
		case SIGINT:
						user_wants_to_quit=1;
						break;

		case SIGHUP:
						reopenlog();
						break;
	}
}

#endif

/************************************/
/* we don't want to receive SIGPIPE */
/************************************/
static void set_sig(void)
{
#ifdef HAVE_SIGNAL
	struct sigaction act;
	sigset_t set;

	/* ignore SIGPIPE */
	sigemptyset(&set);
	sigaddset(&set,SIGCHLD);
	sigaddset(&set,SIGPIPE);
	act.sa_handler=SIG_IGN;
	act.sa_mask=set;
	act.sa_flags=SA_RESTART;

	sigaction(SIGCHLD,&act,NULL);
	sigaction(SIGPIPE,&act,NULL);
	sigprocmask(SIG_UNBLOCK,&set,NULL);

	/* and catch some other sig */
	sigemptyset(&set);
	sigaddset(&set,SIGINT);
	sigaddset(&set,SIGQUIT);
	sigaddset(&set,SIGTERM);
	sigaddset(&set,SIGHUP);
	act.sa_handler=catch_sig;
	act.sa_mask=set;
	act.sa_flags=SA_RESTART;

	sigaction(SIGINT,&act,NULL);
	sigaction(SIGQUIT,&act,NULL);
	sigaction(SIGTERM,&act,NULL);
	sigaction(SIGHUP,&act,NULL);
	sigprocmask(SIG_UNBLOCK,&set,NULL);
#endif
}

static void create_default_conffile(void) {
	FILE *fxml_conf_file;
	GString *template_conf_file;
	size_t flen;
	unsigned char *fadr;
	struct stat st;

	template_conf_file=g_string_new("");
	g_string_sprintf(template_conf_file,"%s/%s",
					(dchub_templatedir ? dchub_templatedir : DCHUB_TEMPLATEDIR),
					"conf.xml.in");

	if (stat(template_conf_file->str, &st) != 0) {
		printf ("Error: template file %s doesn't exist.\n", template_conf_file->str);
		g_string_free(template_conf_file, TRUE);
		exit(1);
	}
			
	fxml_conf_file=fopen(conf_filename,"wb");
	if (fxml_conf_file == NULL) {
		printf("Can't create configuration file %s: %s\n", conf_filename, strerror(errno));
		exit(1);
	}
				
	/* copy the template */
	fadr=map_file(template_conf_file->str, &flen);
	fwrite(fadr, 1, flen, fxml_conf_file);
	unmap_file(fadr, flen);
	fclose(fxml_conf_file);
	g_string_free(template_conf_file, TRUE);
	
	if (!loadDBFile(conf_filename)) {
		fprintf(stderr,"%s is not a valid configuration file\n",conf_filename);
		exit(1);
	}

	/* init the default variable */
	db_str_set("HUBNAME","noname");		/* hubname */
	db_str_set("HUBADDR","");			/* address of this server: (for DC Hublist) */
	db_str_set("HUBDESC","");			/* description of this server (for DC Hublist) */

	db_int_set("REG_HUB",0);			/* the server should be registered on DC Hublist server ? (0=No) */
	db_str_set("REG_SERV","vandel405.dynip.com");/* address of the DC Hublist server */
#if 0
	db_str_set("REG_SERV","dreamland.gotdns.org");/* address of a testing DC Hublist server */
#endif
	db_int_set("MAXUSER",100);			/* maximum number of users allowed on this hub */
	db_int_set("REDIR_FULL",0);			/* when the hub is full, redirect users to another hub ? (0=No) */
	db_str_set("REDIR_ADDR","");		/* address of the hub to redirect user to */
#if 0
	db_str_set("LOCK_PATTERN","^.*");	/* default denied nicknames: all nicknames */
#endif

	db_int_set("ONLY_REG",0);			/* this is a public hub */
			      						
	db_int_set("PUBCHAT",1);			/* public chat deactivated */

	db_int_set("MULTICHAT",1);			/* private chat deactivated */

	db_int_set("MIN_SR_INTERVAL",10);	/* interval deetewn 2 search */

	db_int_set("SPAM_SILENT_LIMIT",10);	/* nbr message needed befor silence the user */

	db_int_set("SPAM_KICK_LIMIT",10);	/* nbr message needed befor diconnect the user */

	db_int_set("BUFFER_SCALE",8);	    /* buffer size in Ko */

	db_int_set("ALL_REDIR", 0);			/* redirect all user */
}

static void create_default_userfile(void) {
	FILE *fxml_user_file;
	GString *template_user_file;
	size_t flen;
	unsigned char *fadr;
	struct stat st;

	template_user_file=g_string_new("");
	g_string_sprintf(template_user_file,"%s/%s",
					(dchub_templatedir ? dchub_templatedir : DCHUB_TEMPLATEDIR),
					"users.xml.in");

	if (stat(template_user_file->str, &st) != 0) {
		printf ("Error: template file %s doesn't exist.\n", template_user_file->str);
		g_string_free(template_user_file, TRUE);
		exit(1);
	}
			
	fxml_user_file=fopen(user_filename,"wb");
	if (fxml_user_file == NULL) {
		printf("Can't create user file %s: %s\n", user_filename, strerror(errno));
		exit(1);
	}
				
	/* copy the template */
	fadr=map_file(template_user_file->str, &flen);
	fwrite(fadr, 1, flen, fxml_user_file);
	unmap_file(fadr, flen);
	fclose(fxml_user_file);
	g_string_free(template_user_file, TRUE);
}
	
static void	display_usage(char *prog)
{
	fprintf(stderr,"Usage: %s [options]\n"
			"Options:\n"
			" -h, --help                      Display this help\n"
			" -c, --conf=FILENAME             use the given filename as configuration file\n"
			" -u, --user=FILENAME             use the given filename as user/passwd file\n"
			" -P, --hubpasswd=FILENAME        use the given filename as hub password file\n"
			" -r, --crc=FILENAME              use the given filename as database for file CRC\n"
			" -U, --UID=USERNAME              ask DcHub to change UID after having bind the\n"
			"                                 main port.\n"
			" -i, --init                      create the default configuration and user file\n"
			"                                 content. You must use this option the first\n"
			"                                 time you start the hub. Note: This function\n"
			"                                 deletes existing configuration and user file.\n"
			" -m, --missing                   create the default configuration and user file\n"
			"                                 content if the files are missing.\n"
			" -d, --pscriptdir=DIRECTORY      name of the directory containing perl script.\n"
			" -s, --pscriptinit=FILENAME      name of the filename loaded when perl starts.\n"
			" -T, --templatedir=DIRECTORY     name of the directory containing the XML template\n"
			"                                 files [DEFAULT: %s]\n"
			" -n, --newport=NUMBER            change listening port of the hub. After using\n"
			"                                 this option, the new port is stored in the\n"
			"                                 database and the option is no more required.\n"
			" -C, --clusterport=NUMBER        change cluster listening port of the hub. After\n"
			"                                 using this option, the new port is stored in\n"
			"                                 the database and the option is no more required\n"
			" -f, --forceport=NUMBER          force DCHUB to register a specific Port\n"
			" -e, --exprogdir=DIRECTORY       name of the directory containing external\n"
			"                                 programs.\n"
			"                                 (NOTE: if a file named 'AUTOSTART' exists in\n"
			"                                 this directory, this file contains the name of\n"
			"                                 all external programs to start (1 per line)\n"
			" -b, --bind=IP                   if the machin have more than one external IP\n"
			" -l, --linkdir=DIRECTORY         name of the directory containing plugins.\n"
			"                                 (NOTE: if a file named 'AUTOSTART' exists in\n"
			"                                 this directory, this file contains the name of\n"
			"                                 all plugins to load on start (1 per line)\n"
			" -w, --pidfile                   sets the file to which DcHub records the daemon\n"
			"                                 process id.\n"
			" -L, --logfile                   sets the file to which DcHub will write is log.\n"
			" -z, --daemonize                 daemonize DcHub.\n"
			
#ifndef HAVE_GETOPT_LONG
			"\n  WARNING this build do not support long arguments\n"
#endif	
			"\n",prog,DCHUB_TEMPLATEDIR);
}

static void parse_command_line_options(int argc,char **argv)
{
#ifdef HAVE_GETOPT_LONG
	static struct option optab[]= {
		{"help",no_argument,NULL,'h'},           /* get help */
		{"conf",required_argument,NULL,'c'},	/* define the configuration file */
		{"user",required_argument,NULL,'u'},	/* define the user/passwd file */
		{"hubpasswd",required_argument,NULL,'P'},	/* define the hub passwd file */
		{"UID",required_argument,NULL,'U'},	/* change UID after running */
		{"init",no_argument,NULL,'i'},			/* initialize configuration & user/passwd files to their default value */
		{"pidfile", required_argument,NULL,'w'},
		{"logfile", required_argument,NULL,'L'},
		{"daemonize", no_argument,NULL,'z'},
		{"missing",no_argument,NULL,'m'},		/* add missing variables to the configuration & user/passwd files */
		{"pscriptdir",required_argument,NULL,'d'},	/* the Perl script dir */
		{"pscriptinit",required_argument,NULL,'s'},	/* the Perl script containing the main */
		{"templatedir",required_argument,NULL,'T'},	/* the template dir containing some XML files */
		{"newport",required_argument,NULL,'n'},	/* the new hub port to use */
		{"clusterport",required_argument,NULL,'C'},	/* the new hub port to use */
		{"forceport",required_argument,NULL,'f'},	/* the new hub port to reg */
		{"exprogdir",required_argument,NULL,'e'},	/* the dir containing external program */
		{"bind",required_argument,NULL,'b'},	/*  */ 
		{"linkdir",required_argument,NULL,'l'},	/* the dir containing plugins */
		{"crc",required_argument,NULL,'r'},	/* define the MD file */
		{NULL,0,NULL,'\0'}               /* last option */
	};
#endif
	static const char *short_opt="hc:u:igzw:L:md:s:T:n:C:e:U:f:"
	                             "b:l:P:r:";
	int do_init=0;
	int add_missing=0;
	int ch;
	struct stat st;
	int new_hub_port=-1;
	int new_cluster_port=-1;
	int tmp_reg_hub_port=-1;
	char *ed2kdb_file=NULL;

#if 0
	if(argc==1)
	{
		display_usage(argv[0]);
		exit(0);
	}
#endif
	opterr = 0;
#ifdef HAVE_GETOPT_LONG
	while((ch=getopt_long(argc,argv,short_opt,optab,NULL))!=EOF)
#else
	while((ch=getopt(argc,argv,short_opt))!=EOF)
#endif
	{
		switch(ch)
		{
			case 'h':
							display_usage(argv[0]);
							exit(0);
	
			case 'c':
							if(conf_filename!=NULL)
								free(conf_filename);
							conf_filename=strdup(optarg);
							break;

			case 'w':
							if (pidfile!=NULL)
								free(pidfile);
							pidfile=strdup(optarg);
							break;

			case 'L':
							if (logfile!=NULL)
								free(logfile);
							logfile=strdup(optarg);
							break;

			case 'z':
							daemonize = 1;
							break;

			case 'u':
							if(user_filename!=NULL)
								free(user_filename);
							user_filename=strdup(optarg);
							break;

			case 'P':
							if(hub_passwd_file!=NULL)
								free(hub_passwd_file);
							hub_passwd_file=strdup(optarg);
							break;

			case 'r':
							if(ed2kdb_file!=NULL)
								free(ed2kdb_file);
							ed2kdb_file=strdup(optarg);
							break;

			case 'U':
							if(user_uid!=NULL)
								free(user_uid);
							user_uid=strdup(optarg);
							break;
			case 'b':
							if(gl_bind_addr!=NULL)
								free(gl_bind_addr);
							if (is_ip(optarg))
								gl_bind_addr=strdup(optarg);
							else
							{
								gl_bind_addr=NULL;
								printf("Error invalid bind param\n");
							}
							break;
	
			case 'i':
							do_init=1;
							break;
	
			case 'm':
							add_missing=1;
							break;

			case 'd':
							if(perl_script_dir!=NULL)
								free(perl_script_dir);
							perl_script_dir=strdup(optarg);
							break;
	
			case 's':
							if(perl_script_main!=NULL)
								free(perl_script_main);
							perl_script_main=strdup(optarg);
							break;

			case 'T':
							if(dchub_templatedir != NULL)
								free(dchub_templatedir);
							dchub_templatedir=strdup(optarg);
							break;

			case 'l':
							if ( plugin_directory != NULL )
								free( plugin_directory );
							plugin_directory = strdup( optarg );
							break;

			case 'n':
							new_hub_port=strtoul(optarg,NULL,10);
							if((new_hub_port<1)||(new_hub_port>65535))
							{
								fprintf(stderr,"invalid -n parameter\n");
								exit(1);
							}
							break;

			case 'C':
							new_cluster_port=strtoul(optarg,NULL,10);
							if((new_cluster_port<1)||(new_cluster_port>65535))
							{
								fprintf(stderr,"invalid -C parameter\n");
								exit(1);
							}
							break;

			case 'f':
							tmp_reg_hub_port=strtoul(optarg,NULL,10);
							if((tmp_reg_hub_port<1)||(tmp_reg_hub_port>65535))
							{
								fprintf(stderr,"invalid -f parameter\n");
								exit(1);
							}
							reg_hub_port = tmp_reg_hub_port;
							break;


			case 'e':
							if(ext_prog_dir!=NULL)
								free(ext_prog_dir);
							ext_prog_dir=strdup(optarg);
							break;

			default:
							fprintf(stderr,"Unknown option: %c\n",ch);
							exit(1);
		}
	}

	/* init XML Tools */
	if (!initXMLTools(dchub_templatedir)) {
		exit(1);
	}

	/* check if required variables are filled ... and filled with correct values */

	/* ------------------------------- */
	/* 1) check the configuration file */
	/* ------------------------------- */
	if(conf_filename==NULL) {
		GString* default_conf_filename=g_string_new("");
		g_string_sprintf(default_conf_filename,"%s/%s",DCHUB_DBDIR,"conf.xml");
		conf_filename=strdup(default_conf_filename->str);
		g_string_free(default_conf_filename, TRUE);
	}

	if(stat(conf_filename,&st)==0) 
	{
		if(!S_ISREG(st.st_mode))
		{
			fprintf(stderr,"The configuration filename you gave is not a file.\n");
			exit(1);
		}
	}
	else
	{
		if((do_init|add_missing)==0)
		{
			fprintf(stderr,"The configuration filename you gave (%s) does not exist,\nyou must use --init or --missing to create it.\n",conf_filename);
			exit(1);
		}
	}

	{
		GString *tm;
		char *p2;

		tm=g_string_new(conf_filename);

		p2=strrchr(tm->str,'/');
		if(p2==NULL)
			tm=g_string_assign(tm,"tos_tbl");
		else
		{
			tm=g_string_truncate(tm,p2+1-tm->str);
			tm=g_string_append(tm,"tos_tbl");
		}
			
		tos_filename=tm->str;
		g_string_free(tm,FALSE);
	}

	/* ------------------------------- */
	/* 2) check the user file          */
	/* ------------------------------- */
	if(user_filename==NULL)
	{
		GString* default_user_filename=g_string_new("");
		g_string_sprintf(default_user_filename,"%s/%s",DCHUB_DBDIR,"users.xml");
		user_filename=strdup(default_user_filename->str);
		g_string_free(default_user_filename, TRUE);
	}

	if(stat(user_filename,&st)==0) 
	{
		if(!S_ISREG(st.st_mode))
		{
			fprintf(stderr,"The user filename you gave is not a file.\n");
			exit(1);
		}
	}
	else
	{
		if((do_init|add_missing)==0)
		{
			fprintf(stderr,"The user filename you gave does not exist,\nyou must use --init or --missing to create it.\n");
			exit(1);
		}
	}

	/* ------------------------------- */
	/* 2a) check the hub passwd file   */
	/* ------------------------------- */
	if(hub_passwd_file==NULL)
	{
		GString* default_passwd_filename=g_string_new("");
		g_string_sprintf(default_passwd_filename,"%s/%s",DCHUB_DBDIR,"dchub.hubpasswd");
		hub_passwd_file=strdup(default_passwd_filename->str);
		g_string_free(default_passwd_filename, TRUE);
	}

	if(stat(hub_passwd_file,&st)==0) 
	{
		if(!S_ISREG(st.st_mode))
		{
			fprintf(stderr,"The hub password filename you gave is not a file.\n");
			exit(1);
		}
	}

	hp_lmp=lmp_new(hub_passwd_file,sizeof(HUB_PASS_ROW),-1);
	if(hp_lmp==NULL)
	{
		fprintf(stderr,"Unable to set up the LMP entry on the given hub password file.\n");
		exit(1);
	}

	/* -------------------------- */
	/* 2b) check the MD_db file   */
	/* -------------------------- */
	if(ed2kdb_file==NULL)
	{
		GString* default_ed2kdb_filename=g_string_new("");
		g_string_sprintf(default_ed2kdb_filename,"%s/%s",DCHUB_DBDIR,"dchub.crc");
		ed2kdb_file=strdup(default_ed2kdb_filename->str);
		g_string_free(default_ed2kdb_filename, TRUE);
	}

	if(stat(ed2kdb_file,&st)==0) 
	{
		if(!S_ISREG(st.st_mode))
		{
			fprintf(stderr,"The CRC filename you gave is not a file.\n");
			exit(1);
		}
	}

	if(init_md_db(ed2kdb_file)!=0)
	{
		fprintf(stderr,"Unable to set up the CRC database.\n");
		exit(1);
	}

	/* ------------------------------- */
	/* 3) check perl vars              */
	/* ------------------------------- */
	if( ((perl_script_dir!=NULL)&&(perl_script_main==NULL)) ||
		 ((perl_script_dir==NULL)&&(perl_script_main!=NULL)) )
	{
		fprintf(stderr,"To use perl script, you must use -d and -s at the same time (either both of them or none of them).\n");
		exit(1);
	}

	/* initialize file ? */
	if(do_init==1)
	{
		/* first: erase previous file */
		unlink(conf_filename);	
		unlink(user_filename);

		set_user_id();
		
	    /* 2nd: create the default configuration file */
		create_default_conffile();
		
		/* 3rd: create the default user/passwd file */
		create_default_userfile();
		
		fprintf(stderr,"Init done. Check the configuration file if you want to modify\n"
							"it. NEVER edit user file. After, you can start the hub without\n"
							"the --init option.\n"
							"Use the account 'MASTER' (see below) to configure the hub\n"
							"directly from any DC client and the 'Hub-Security' private chat\n");
		fprintf(stderr,"This is the default configuration:\n"
							"HUBNAME => 'noname'\n"
							"HUBADDR => ''\n"
							"HUBDESC => ''\n"
							"REG_HUB => 0\n"
							"REG_SERV => 'vandel405.dynip.com'\n"
							"MAXUSER => 100\n"
							"REDIR_FULL => 0\n"
							"REDIR_ADDR => ''\n"
							"ONLY_REG => '0'\n"
							"PUBCHAT => '1'\n"
							"MULTICHAT => '1'\n"
							"MIN_SR_INTERVAL => '10'\n"
							"SPAM_SILENT_LIMIT => '10'\n"
							"SPAM_KICK_LIMIT => '10'\n"
							"BUFFER_SCALE => '8'\n"
#if 0
							"LOCK_PATTERN -> '^#*' /* remplace | by  in this string\n" */
#endif
							"\n"
							"In the user file, there is only 1 user who is the master of\n"
							"the hub. The nickname is 'MASTER' and its password is 'MASTER'\n"
							"\n");
		fprintf(stderr,"The easiest way to configure the hub is to start it and enter\n"
							"inside with the login/password 'MASTER'. Then, open a private\n"
							"chat with \"Hub-Security\" and use some commands of manual\n"
							"(see Documentation/commands.txt)\n"
							"\n");
		exit(0);
	}

	if (add_missing == 1 ) {

		set_user_id();

		if (stat(conf_filename, &st) != 0) {
			/* create a default conf file */
			create_default_conffile();
		}
		if (stat(user_filename, &st) != 0) {
			/* create a default users file */
			create_default_userfile();
		}
	}

	/* we are ready to start the hub */
	if (daemonize) {
		int child;
		switch(child = fork()) {
		    case -1: /* failed */
				fprintf(stderr,"fork() failed\n");
				exit(1);
		    case 0: { /* child */
				waitpid( getppid(), NULL, 0 );
				setsid();
				chdir("/");
				break;
		    }
		    default:
				exit(0);
		}
	}

	/* write the pid file */
	if (pidfile!=NULL) {
		int pid=getpid();
		FILE *pf = fopen(pidfile, "w");
		if(pf==NULL) {
			fprintf(stderr,"Unable to create %s\n",pidfile);
			exit(1);
		}
		fprintf(pf, "%d\n", pid);
		fclose(pf);
	}

	/* create the log file if necessary */
	if (logfile!=NULL)
	{
		fd_logfile = open(logfile, O_WRONLY | O_CREAT | O_APPEND, 0666);
		if (fd_logfile < 0)
		{
			fprintf(stderr,"Unable to create %s\n",logfile);
			exit(1);
		}
			
		if(dup2(fd_logfile,1)<0)
		{
			fprintf(stderr,"Unable to make the dup2\n");
			exit(1);
		}

		if(dup2(fd_logfile,2)<0)
		{
			fprintf(stdout,"Unable to make the dup2\n");
			exit(1);
		}
	}
	
	/* open the configuration file */
	if (!loadDBFile(conf_filename)) {
		fprintf(stderr,"%s is not a valid configuration file\n",conf_filename);
		exit(1);
	}

	if(new_hub_port!=-1)
	{
		/* if a port is provided, save the port number into the database */
		printf ("update HUBPORT to %d\n", new_hub_port);
		db_int_set("HUBPORT",new_hub_port);
	}

	if(new_cluster_port!=-1)
	{
		printf ("update CLUSTERPORT to %d\n", new_cluster_port);
		db_int_set("CLUSTERPORT",new_cluster_port);
	}
			
	/* initialize hub_port param to the saved value or to the default 411 value */
	if(!db_int_get("HUBPORT", &new_hub_port))
		new_hub_port=411;

	if(!db_int_get("CLUSTERPORT",&new_cluster_port))
		new_cluster_port=410;
	hub_port=new_hub_port;
	cluster_port=new_cluster_port;

	/* open the user file */
	if (!loadUsersFile(user_filename)) {
		fprintf(stderr,"%s is not a valid users file\n",user_filename);
		exit(1);
	}

	/* automatically starts some external programs */
	if(ext_prog_dir!=NULL)
	{
		GString *str;
		FILE *f;

		str=g_string_new("");
		g_string_sprintf(str,"%s/AUTOSTART",ext_prog_dir);
		f=fopen(str->str,"rt");
		if(f!=NULL)
		{
			char buf[2048];

			while(fgets(buf,sizeof(buf),f)!=NULL)
			{
				char *r;

				r=strpbrk(buf,"\r\n");
				if(r!=NULL)
					*r='\0';

				add_prog_to_start(buf);
			}
			
			fclose(f); // ok
		}

		g_string_free(str,TRUE);
	}
}

/* #define DEBUG_MEMORY */

#ifdef DEBUG_MEMORY
#include <mcheck.h>
static void
enable (int sig)
{
	mtrace ();
	signal (SIGUSR1, enable);
}

static void
disable (int sig)
{
	muntrace ();
	signal (SIGUSR2, disable);
}
#endif /* DEBUG_MEMORY */

int load_dyn_conf(void)
{
	char	*hub_name;
	char	*pattern_list;

	hub_name=db_str_get("HUBNAME");

	if (!db_int_get("REG_HUB",&do_hub_registration))
		do_hub_registration=0;

	if (!db_int_get("MIN_SR_INTERVAL" , &gl_sr_interval))
	{
		gl_sr_interval = 10;
	}
	if (!db_int_get("SPAM_SILENT_LIMIT" , &gl_silent_limit))
	{
		gl_silent_limit = 10;
	}
	if (!db_int_get("SPAM_KICK_LIMIT" , &gl_kick_limit))
	{
		gl_kick_limit = 10;
	}
	if (!db_int_get("BUFFER_SCALE" , &gl_buf_size))
	{
		gl_buf_size = 8;
	}
	if (gl_buf_size < 2)
		gl_buf_size = 2;
	gl_buf_size *= 1024;
	
	if ((pattern_list = db_str_get("LOCK_PATTERN")) != NULL)
	{
		char *tmp;
		G_LOCK(gl_lock_patern);
		if(gl_lock_patern!=NULL)
		{
			int i;
			for(i=0;i<gl_lock_patern->len;i++)
				tmp=g_ptr_array_index(gl_lock_patern,i);
			g_ptr_array_free(gl_lock_patern,TRUE);
		}

		if (*pattern_list != '\0')
		{
			gl_lock_patern=g_ptr_array_new();
			printf("pattern1: %s\n",pattern_list);
			while ((tmp=strchr(pattern_list, '')) != NULL)
			{
				*tmp++='\0';
				g_ptr_array_add(gl_lock_patern,(void*)pattern_list);
				pattern_list=tmp;
			}
			if(pattern_list!=NULL)
				g_ptr_array_add(gl_lock_patern,(void*)pattern_list);
			printf("pattern3: %s\n",pattern_list);
		}
		G_UNLOCK(gl_lock_patern);
	}

	if (hubname != NULL)
		g_string_free(hubname, TRUE);	
	hubname=g_string_new("$HubName ");

	if (hub_name != NULL)
	{
		hubname=g_string_append(hubname,hub_name);
		free(hub_name);
	}
	hubname=g_string_append(hubname,"|");
	
	return (0);
}

/***************************************************************/
/* change the user ID of the process if a different one is set */
/***************************************************************/
void set_user_id(void)
{
#ifndef WIN32
	if(user_pw!=NULL)
	{
		if (setuid(user_pw->pw_uid))
			perror("setuid");
		return;
	}

	if (user_uid!=NULL)
	{
		struct passwd* pw;

		user_pw = getpwnam(user_uid);
		if (user_pw!=NULL)
		{
			if (setuid(user_pw->pw_uid))
				perror("setuid");
		}
		pw = getpwuid(getuid());
		if (pw)
			printf("dchub is now running as %s\n", pw->pw_name);
	}
#endif
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------------- main loop GED ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static void main_loop_tasks_always1(const struct ged_callbacks *ged)
{
	/* removed in DCHub 0.2.3 and back in 0.2.5*/
	check_hub_registration();
#if 0
#ifdef HAVE_NO_THREAD
	if (gl_reg_sock != -1)
	{
		if(FD_ISSET(gl_reg_sock,&rd)) /* gl_resolver_fd  dchub 0.2.5 */
		{
			check_hub_registration();
			last_reg_time = gl_cur_time; /* that the interval beetwen 2 reg */
		}
	}
	else
	{
		if (do_hub_registration)
		{
			if((last_reg_time + (DELAY_BETWEEN_REGISTRATION-60)) < gl_cur_time) {
				char	*reg_serv = NULL;

				reg_serv=db_str_get("REG_SERV");
				if (reg_serv) {
					try_new_registration(reg_serv); /* start a registration */
					last_reg_time = gl_cur_time;
					free(reg_serv);
				}
			}
		}
	}
#endif /* ! HAVE_NO_THREAD */
#endif /* 0 */

}

/**********************/
/* low priority tasks */
/**********************/
static void main_loop_tasks_periodic(const GED_CALLBACKS *ged)
{
	timeout_tos();
	check_prog_to_start();

	/* for even lower priority tasks, only 6 times slower (1 every minute) */
	{
		static int tempo=0;

		tempo++;
		if(tempo==6)
		{
			tempo=0;

			update_dynamic_if_ip();
		}
	}

	g_mem_profile();
}

static GED_CALLBACKS main_loop_ged={
                                   "main loop GED",
                                   NULL,
                                   NULL,
                                   NULL,
                                   main_loop_tasks_periodic,
                                   main_loop_tasks_always1,
                                   NULL,
                                   NULL
                                   };


GED_CALLBACKS *manage_main_loop_tasks_init(void)
{
	return &main_loop_ged;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------------------- main () ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
int main(int argc,char **argv)
{
#if 0
#ifdef HAVE_NO_THREAD
	time_t last_reg_time = 0;
#endif
#endif

#ifdef WIN32
{
	int wsarc;

	if ((wsarc=WSAStartup(MAKEWORD( 2, 0 ), &gl_ws_info))<0)
	{
		fprintf(stderr, "Error initializing Winsock (rc:%d / Err: %d)\n", wsarc, WSAGetLastError());
		exit(1);
	}
}
  /* ws_changetitle (char *, int); */
#endif


#ifdef DEBUG_MEMORY
	mcheck(0);
#endif /* DEBUG_MEMORY */

	printf("DChub v" VERSION "\n");
	parse_command_line_options(argc,argv);
	
#ifndef HAVE_NO_PTHREAD
	g_thread_init(NULL);
#endif
	/* ignore some signals */
	set_sig();
#ifdef DEBUG_MEMORY
	signal (SIGUSR1, enable);
	signal (SIGUSR2, disable);
#endif /* DEBUG_MEMORY */

	load_dyn_conf();
	load_tos_data(tos_filename);

	/* create the dns resolver process */
	/* create_resolver(); */ /* dchub 0.2.5 */

	/* create the hub socket */
	ged_register(main_sck_init, 0);				/* accepting connection from user client */
	ged_register(hub_sck_init, 0);				/* accepting connection from the cluster */
	ged_register(hub_cnx_connect_init, 0);		/* initiating connection to the cluster */

	/* handler for basic I/O functions (should have a very high priority to improve speed */
	ged_register(xf_io_handler_init, 1);
	ged_register(bin_xf_io_handler_init, 1);

	/* handler for incoming connections handshake */
	ged_register(cnx_handshake_init,3);	/* its priority must be lowest than XF_IO to be faster */
	ged_register(hub_handshake_init,3);	/* its priority must be lowest than BIN_XF_IO to be faster */

	/* handler for the running connections */
	ged_register(user_cnx_entry_handler_init,3);	/* its priority must be lowest than XF_IO to be faster */
	ged_register(hub_cnx_entry_handler_init,3);	/* its priority must be lowest than XF_IO to be faster */

	/* register some virtual user functions */
	multi_public_chat_init();

	/* handler for the tasks of the main loop */
	ged_register(manage_main_loop_tasks_init,999);		/* lowest priority */

	set_user_id();

#ifdef WITH_PERL
	init_perl();
#endif

#ifdef HAVE_LIBDL
	load_all_autostart_plugin();
#endif

	SEND_EVT_TO("hubinit","hubinit",0);

	/* remove in dchub 0.2.5 */
#if 0
#ifndef HAVE_NO_THREAD
	strat_reg_thread(); /* will be remove in the next soon */
#endif
#endif

	/* begin of the main loop */
	while(!user_wants_to_quit)
	{
#if 0
		static int omega1=0,omega2=0,omega3=0;
		if ((omega1 != cnx_lst->wait_len) || (omega2 != cnx_lst->len) || (omega3 != route_table->server_count))
		{
			printf("there is " COLOMAJ1 "%d" COLONONE " user in queue, " COLOMAJ1 "%d" COLONONE" user login, "COLOMAJ1 "%d" COLONONE" hub in cluster\n",
						cnx_lst->wait_len, cnx_lst->len, route_table->server_count);
			omega1 = cnx_lst->wait_len;
			omega2 = cnx_lst->len;
			omega3 = route_table->server_count;
		}
#endif

		/* FD_SET(gl_resolver_fd,&rd); */	/* resolver fd  dchub 0.2.5 */
		/* n=max(n,gl_resolver_fd); */

		/********************************************/
		/********************************************/
		/* Generic Event Decoder support */
		{
			GArray *struct_pollfd;
			int poll_ret;

			struct_pollfd=g_array_new(FALSE,FALSE,sizeof(struct pollfd));

			ged_pre_poll(struct_pollfd);
			poll_ret=poll((struct pollfd*)(struct_pollfd->data),struct_pollfd->len,1000);

			/* update current time */
			gl_cur_time=time(NULL);
#if 0
			check_timed_action();
#endif

			if(poll_ret>0)
			{
				/* an event occurs */
				ged_post_poll(struct_pollfd);
			}
			else if(poll_ret==-1)
			{
				if(errno!=EINTR)
				{
					perror("main_loop: select");
					printf(COLOYEL1"error in file desc, user_wants_to_quit = %d"COLONONE"\n", user_wants_to_quit);
					user_wants_to_quit = 1;
				}
			}

			g_array_free(struct_pollfd,TRUE);

			ged_periodic();
			ged_always();
		}
	} /* end of the main loop */

	printf("ok\n");
	
	/***************************************************/
	/*************** END close soft ********************/ 
	/***************************************************/

	/* Free all XML resources for users file */
	xml_free_users_resource();

	/* Free all XML resources for configuration file */
	xml_free_db_resource();

	/* Free XML Tools */
	freeXMLTools();
		
	save_tos_data(tos_filename);
#ifdef WITH_PERL
	exit_perl();
#endif

	ged_exit();

	if(fd_logfile!=-1)
	{
		close(fd_logfile);
	}
	
	if(pidfile!=NULL) {
		unlink(pidfile);	
	}

	exit(0);
}

