/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * lmp.h: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: lmp.h,v 2.3 2003/02/27 17:39:39 ericprev Exp $
*/

#ifndef __LMP_H__
#define __LMP_H__

typedef struct
{
	/* values to provide */
	int record_size;						/* size of a record in the base */

	/* values filled by lmp_* */
	int fd;									/* file descriptor */
	int is_locked;							/* 0=fd is not locked, 1= fd is locked */
	void *mapped_addr;					/* address where the file is mapped into memory */
												/* NOTE: the first record is reserved for LMP */
	unsigned long mapped_size;			/* number of bytes mapped into memory */
	int nb_records;						/* number of records (== (int)(mapped_size/record_size) ) */
} LMP_ENTRY;

/******************************************************/
/* create a new LMP entry (and the file if not exist) */
/********************************************************************/
/* input: prealloc_records= -1, no prealloc else, try to immediatly */
/*        allocate space on disc (it is not in bytes but in record) */
/*        The lack of free disc space is an error                   */
/* output: a newly allocated LMP entry or NULL                      */
/********************************************************************/
LMP_ENTRY *lmp_new(char *filename, int record_size, long int prealloc_records);

/***********************************/
/* lock and map a file into memory */
/***********************************/
/* output: 0=ok, 1=error */
/*************************/
int lmp_lock_and_map(LMP_ENTRY *);

/*********************************************/
/* unmap file from memory, it remains locked */
/*********************************************/
void lmp_unmap(LMP_ENTRY *);

/**************************************************************/
/* unlock a file. If file is not yet unmapped, it is unmapped */
/**************************************************************/
void lmp_unmap_and_unlock(LMP_ENTRY *);

/***************************************************************/
/* close file and destroy LMP entry (the file remains present) */
/***************************************************************/
void lmp_close(LMP_ENTRY *);

/********************************************************************/
/* append the given record to the given LMP. The LMP must be locked */
/* After the call, the LMP remains locked but becomes unmapped      */
/********************************************************************/
/* output: 0=ok, 1=error */
/*************************/
int lmp_append_record(LMP_ENTRY *, void *value);

typedef struct
{
	/* values filled by lmp_* */
	int fd;									/* file descriptor */
	int is_locked;							/* 0=fd is not locked, 1= fd is locked */
	void *mapped_addr;					/* address where the file is mapped into memory */
												/* NOTE: the first record is reserved for LMP */
	unsigned long mapped_size;			/* number of bytes mapped into memory */
} LMP0_ENTRY;

/******************************************************/
/* create a new LMP0 entry (and the file if not exist) */
/********************************************************************/
/* input: prealloc_records= -1, no prealloc else, try to immediatly */
/*        allocate space on disc (it is not in bytes but in record) */
/*        The lack of free disc space is an error                   */
/* output: a newly allocated LMP0 entry or NULL                     */
/********************************************************************/
LMP0_ENTRY *lmp0_new(char *filename, long int prealloc_records);

/***********************************/
/* lock and map a file into memory */
/***********************************/
/* output: 0=ok, 1=error */
/*************************/
int lmp0_lock_and_map(LMP0_ENTRY *);

/*********************************************/
/* unmap file from memory, it remains locked */
/*********************************************/
void lmp0_unmap(LMP0_ENTRY *);

/***************************************************/
/* map a file into memory, the file must be locked */
/***************************************************/
int lmp0_map_locked(LMP0_ENTRY *);

/**************************************************************/
/* unlock a file. If file is not yet unmapped, it is unmapped */
/**************************************************************/
void lmp0_unmap_and_unlock(LMP0_ENTRY *);

/***************************************************************/
/* close file and destroy LMP entry (the file remains present) */
/***************************************************************/
void lmp0_close(LMP0_ENTRY *);

/********************************************************************/
/* append the given record to the given LMP. The LMP must be locked */
/* After the call, the LMP remains locked but becomes unmapped      */
/********************************************************************/
/* output: -1=error, else position of the row in the LMP file */
/**************************************************************/
off_t lmp0_append_record(LMP0_ENTRY *, guint32 data_len, const guint8 *data);

#endif
