/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hub_passwd.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hub_passwd.c,v 2.8 2003/04/17 15:41:29 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <glib.h>

#include "hub_passwd.h"
#include "chunk_codes.h"
#include "toolkit.h"
#include "lmp.h"
#include "gvar.h"

/*********************/
/* hub password file */
/*********************/
char *hub_passwd_file=NULL;		/* file containing hub_id:cluster_id of all hubs allowed to connect to this one */

/**************************/
/* hub password LMP entry */
/**************************/
LMP_ENTRY *hp_lmp=NULL;

/*****************************************************/
/* search and return the row having the given hub_id */
/*****************************************************/
static HUB_PASS_ROW *get_hub_passwd_row_from_hub_id(guint8 *hub_id)
{
	HUB_PASS_ROW *ret=NULL;
	int i;

	if(lmp_lock_and_map(hp_lmp))
	{
		fprintf(stderr,"get_hub_passwd_row_from_hub_id: unable to map file.\n");
		return ret;
	}

	for(i=1;i<hp_lmp->nb_records;i++)	/* we don't use the first record, it is a header */
	{
		if( (((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags!=HPR_EF_EMPTY)&&
			 (!memcmp(hub_id,((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].hub_id,HUB_ID_LEN)) )
		{
			if( (((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].special_flags) & INVALID_HPR_SF)
				break;

			ret=malloc(sizeof(HUB_PASS_ROW));
			if(ret)
			{
				memcpy(ret,&(((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i]),sizeof(HUB_PASS_ROW));
			}
			break;
		}
	}

	lmp_unmap_and_unlock(hp_lmp);
	return ret;
}

/***********************************************************************/
/* search in the passwd for a hub_id matching the given one and return */
/* its cluster_id in the appropriate array.                            */
/***********************************************************************/
/* output: FALSE = hub_id not found               */
/*         TRUE = hub_id found, cluster_id loaded */
/**************************************************/
gboolean hub_passwd_get_clus_id_from_hub_id(guint8 *hub_id, guint8 *cluster_id)
{
	HUB_PASS_ROW *passwd_row;

	passwd_row=get_hub_passwd_row_from_hub_id(hub_id);
	if(passwd_row==NULL)
		return FALSE;		/* not found */

	memcpy(cluster_id,passwd_row->cluster_id,MD_BLOC_SIZE);

	free(passwd_row);
	return TRUE;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- misc functions ----------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************************************************/
/* update the last_connect_attempt field of the row with the given hub_address */
/*******************************************************************************/
void hub_passwd_touch_last_attempt_from_hub_address(const char *hub_address)
{
	int i;

	if(lmp_lock_and_map(hp_lmp))
	{
		fprintf(stderr,"hub_passwd_touch_last_attempt_from_hub_address: unable to map file.\n");
		return;
	}

	for(i=1;i<hp_lmp->nb_records;i++)	/* we don't use the first record, it is a header */
	{
		if( (((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags!=HPR_EF_EMPTY)&&
			 (!strcmp(hub_address,((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].hub_address)) )
		{
			((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].last_connection_attempt=gl_cur_time;
			break;
		}
	}

	lmp_unmap_and_unlock(hp_lmp);
	return;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------------ hub commands ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**********************/
/* set a password row */
/**********************/
void ucmd_hp_set(HUB_PASS_ROW *hpr)
{
	int i;
	int empty=-1;

	if(lmp_lock_and_map(hp_lmp))
	{
		fprintf(stderr,"ucmd_hp_set: unable to map file.\n");
		return;
	}

	/* first, we try to replace an existing entry with the same HUB_ID */
	for(i=1;i<hp_lmp->nb_records;i++)	/* we don't use the first record, it is a header */
	{
		if((((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags!=HPR_EF_EMPTY))
		{
			if(!memcmp(hpr->hub_id,((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].hub_id,HUB_ID_LEN))
			{
				memcpy(&(((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i]),hpr,sizeof(HUB_PASS_ROW));
				lmp_unmap_and_unlock(hp_lmp);
				return;
			}
		}

		if((empty==-1)&&(((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags==HPR_EF_EMPTY))
			empty=i;
	}

	if(empty!=-1)
	{
		memcpy(&(((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[empty]),hpr,sizeof(HUB_PASS_ROW));
		lmp_unmap_and_unlock(hp_lmp);
		return;
	}

	lmp_append_record(hp_lmp,hpr);
	lmp_unmap_and_unlock(hp_lmp);
}

/**********************/
/* get a password row */
/**********************/
HUB_PASS_ROW *ucmd_hp_get(guint8 *hub_id)
{
	return get_hub_passwd_row_from_hub_id(hub_id);
}

/*************************/
/* get all password rows */
/**********************************************************************************/
/* NOTE: some rows can have entry_flags set to HPR_EF_EMPTY and should be ignored */
/**********************************************************************************/
HUB_PASS_ROW *ucmd_hp_get_rows(int *nb_rows)
{
	HUB_PASS_ROW *ret=NULL;

	if(lmp_lock_and_map(hp_lmp))
	{
		fprintf(stderr,"get_hub_passwd_row_from_hub_id: unable to map file.\n");
		return ret;
	}

	*nb_rows=(hp_lmp->nb_records-1);
	if((*nb_rows)!=0)
	{
		ret=malloc(sizeof(HUB_PASS_ROW)*(*nb_rows));
		if(ret!=NULL)
		{
			memcpy(ret,&(((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[1]),*nb_rows*sizeof(HUB_PASS_ROW));
		}
	}

	lmp_unmap_and_unlock(hp_lmp);
	return ret;
}

/*************************/
/* delete a password row */
/*******************************/
/* output: TRUE: delete ok     */
/*         FALSE: delete error */
/*******************************/
gboolean ucmd_hp_del(guint8 *hub_id)
{
	int i;

	if(lmp_lock_and_map(hp_lmp))
	{
		fprintf(stderr,"ucmd_hp_del: unable to map file.\n");
		return FALSE;
	}

	/* first, we try to find an existing entry with the same HUB_ID */
	for(i=1;i<hp_lmp->nb_records;i++)	/* we don't use the first record, it is a header */
	{
		if((((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags!=HPR_EF_EMPTY))
		{
			if(!memcmp(hub_id,((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].hub_id,HUB_ID_LEN))
			{
				/* we have found one, just delete it */
				((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags=HPR_EF_EMPTY;
				lmp_unmap_and_unlock(hp_lmp);
				return TRUE;
			}
		}
	}

	lmp_unmap_and_unlock(hp_lmp);
	return FALSE;
}

/**************************************/
/* clear connection attempt timestamp */
/**************************************/
/* output: TRUE: delete ok     */
/*         FALSE: delete error */
/*******************************/
gboolean ucmd_hp_reset_last_cnx_attempt(guint8 *hub_id)
{
	int i;

	if(lmp_lock_and_map(hp_lmp))
	{
		fprintf(stderr,"ucmd_hp_reset_last_cnx_attempt: unable to map file.\n");
		return FALSE;
	}

	/* first, we try to find an existing entry with the same HUB_ID */
	for(i=1;i<hp_lmp->nb_records;i++)	/* we don't use the first record, it is a header */
	{
		if((((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].entry_flags!=HPR_EF_EMPTY))
		{
			if(!memcmp(hub_id,((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].hub_id,HUB_ID_LEN))
			{
				/* we have found one, clear its timestamp */
				((HUB_PASS_ROW*)(hp_lmp->mapped_addr))[i].last_connection_attempt=0;
				lmp_unmap_and_unlock(hp_lmp);
				return TRUE;
			}
		}
	}

	lmp_unmap_and_unlock(hp_lmp);
	return FALSE;
}


