/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hub_cnx_lst.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hub_cnx_lst.c,v 2.34 2003/11/22 13:47:08 blusseau Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
	#include <unistd.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef WITH_GCRYPT
#include <gcrypt.h>
#define MD4_DIGEST_LENGTH 16
#define MD4_ALGO(source,len,dest)   gcry_md_hash_buffer(GCRY_MD_MD4,dest,source,len)
#else
#include <openssl/md4.h>
#define MD4_ALGO MD4
#endif
#include <glib.h>

#include "gvar.h"
#include "ged_if.h"
#include "bin_xf_io.h"
#include "global_user_if.h"
#include "hub_cnx_lst.h"
#include "toolkit.h"
#include "md5.h"
#include "md_db.h"
#include "users_xml.h"
#include "db_xml.h"

/* list of clusters */
/* the value is a CLUSTER_ENTRY * */
static GList *cluster_entry_list=NULL;

#ifndef HAVE_NO_MUTEX
HL_MUTEX cluster_entry_lock = HL_MUTEX_INIT;
#endif

/***********************************************************************************/
/* all HUB_CNX_ENTRY listed below are ignored when a send to all hubs is performed */
/***********************************************************************************/
static gboolean exclude_all_clusters=FALSE;		/* if set to TRUE, not relay is done on the clusters */
static GList *excluded_hub_entry=NULL;

static CLUSTER_ENTRY *get_cluster_entry_from_id(guint8 *clus_id);
static void send_chunk_to_all_hubs(GByteArray *gba);
#if 0
static void call_fnc_for_each_hub(void (*fnc)(HUB_CNX_ENTRY *hce,void *param),void *param);
#endif
static HUB_CNX_ENTRY *find_hub_cnx_entry_with_a_user(const char *nickname);
/* -------------------------------------------------------------------------- */
/* -------------------- creation/destruction functions ---------------------- */
/* -------------------------------------------------------------------------- */

/**********************/
/* create a new entry */
/******************************************************************************************/
/* input: bxfio to used [the pointer is stolen, the caller must ignore it after this call]*/
/******************************************************************************************/
/* output: address of the newly created and inserted HUB_CNX_ENTRY * */
/*********************************************************************/
HUB_CNX_ENTRY *hc_create_entry(BIN_XF_IO *bxfio, char *hubname, HUB_CNX_TYPE hc_type,
               guint32 hop_cost, guint8 *remote_hub_id, guint8 *clus_id)
{
	CLUSTER_ENTRY *ce;
	HUB_CNX_ENTRY hce;

	ce=get_cluster_entry_from_id(clus_id);
	if(ce==NULL)
	{
		/* the cluster does not exist, we must create it */
		ce=malloc(sizeof(CLUSTER_ENTRY));
		if(ce==NULL)
		{
			fprintf(stderr,"hc_create_entry: out of memory.\n");
			delete_bin_xfio(bxfio);	/* because this function steals bxfio reference, we must destroy it */
			return NULL;
		}

		memcpy(ce->cluster_id,clus_id,MD_BLOC_SIZE);
		ce->hc_array=g_array_new(FALSE,FALSE,sizeof(HUB_CNX_ENTRY));
		HL_LOCK_WRITE(cluster_entry_lock);
		cluster_entry_list=g_list_append(cluster_entry_list,ce);
		HL_UNLOCK_WRITE(cluster_entry_lock);
	}

	/* create the new hub entry */
	hce.hub_bxfio=bxfio;
	memcpy(hce.remote_hub_id,remote_hub_id,HUB_ID_LEN);
	hce.remote_hub_name=strdup(hubname);
	hce.hop_cost=hop_cost;
	hce.last_time=gl_cur_time;

	hce.hub_users=g_hash_table_new(g_str_hash,g_str_equal);

	HL_LOCK_WRITE(cluster_entry_lock);
	g_array_append_val(ce->hc_array,hce);
	HL_UNLOCK_WRITE(cluster_entry_lock);
	return &(g_array_index(ce->hc_array,HUB_CNX_ENTRY,ce->hc_array->len-1));
}

/**************************************************************************/
/* this function handles all incoming requests from all hubs of a cluster */
/**************************************************************************/
/* entry: key= user_nick (the value is inside the next parameter)  */
/*        value= HUB_USER_CNX_ENTRY * to process                   */
/*        user_data= NULL (unused, just required for the GHfunc)   */
/*******************************************************************/
/* NOTE: the array has LOCK_READ set */
/****************************************************/
/* output: always TRUE, we want to destroy the user */
/****************************************************/
static gboolean delete_one_hub_user_entry(gpointer key, gpointer value, gpointer user_data)
{
	HUB_USER_ENTRY *hue=value;
	/* notify to everyone this user has left the cluster */
	/* we have no special case to handle because the hub entry was already removed */
	glus_do_quit(hue->user_nick->str);

	/* and free memory */
	g_string_free(hue->user_nick,TRUE);	/* only this value is mandatory */

	if(hue->user_mail)
		g_string_free(hue->user_mail,TRUE);
	if(hue->user_description)
		g_string_free(hue->user_description,TRUE);
	if(hue->client_version)
		g_string_free(hue->client_version,TRUE);

	free(hue);
	return TRUE;
}

/*******************************************************************/
/* remove an entry from the cnx_entry_list and free its ressources */
/*******************************************************************************/
/* this function can be used as value destruction function in g_hash_table_new */
/*******************************************************************************/
static void hc_destroy_hub_entry(CLUSTER_ENTRY *ce,HUB_CNX_ENTRY *hce, guint hce_idx)
{
	HUB_CNX_ENTRY tmp_val;

	memcpy(&tmp_val,hce,sizeof(HUB_CNX_ENTRY));

	/* hce is not an allocated structure, it is a value of a Garray */
	g_array_remove_index_fast(ce->hc_array,hce_idx);

	/* now, destroy the hce content */
	delete_bin_xfio(tmp_val.hub_bxfio);

	free(tmp_val.remote_hub_name);

	/* notify to everyone the users of this hub has left the cluster */
	/* we have no special case to handle because the entry of this hub was already removed */
	g_hash_table_foreach_remove(tmp_val.hub_users,delete_one_hub_user_entry,NULL);
	
	/* and free the hash table */
	g_hash_table_destroy(tmp_val.hub_users);
}

/**********************************************************************************************/
/* search for a user in the given hub cnx entry with the given nickname (not null terminated) */
/* if the user exists, its HUB_USER_ENTRY is returned and *is_created is set to FALSE         */
/* if it does not exist, it is created with some default values and *is_created is set to TRUE*/
/*                       but only if with_creation was set to TRUE                            */
/**********************************************************************************************/
static HUB_USER_ENTRY *find_user_named_for_this_hub(HUB_CNX_ENTRY *hce, const unsigned char *nickname, unsigned char nicklen, gboolean *is_created, gboolean with_creation)
{
	char *t;
	HUB_USER_ENTRY *hue;

	t=malloc(nicklen+1);
	if(t==NULL)
		return NULL;		/* fatal error */

	memcpy(t,nickname,nicklen);
	t[nicklen]='\0';

	/* search for this user */
	hue=g_hash_table_lookup(hce->hub_users,t);
	if(hue!=NULL)
	{
		*is_created=FALSE;
		free(t);
		return hue;		/* user found */
	}

	/* create missing user ? */
	if(with_creation==FALSE)
	{
		free(t);
		return hue;
	}

	hue=malloc(sizeof(HUB_USER_ENTRY));
	if(hue==NULL)
	{
		free(t);
		return hue;	/* fatal error */
	}

	hue->user_cnx_type="???";
	hue->user_nick=g_string_new(t);
	free(t);
	hue->shared_size=0;
	hue->user_mail=g_string_new("");
	hue->user_description=g_string_new("");
	hue->user_flag=1;		/* LSB always set */
	hue->ext_flag=0;
	hue->level=0;
	hue->client_version=g_string_new("");

  	g_hash_table_insert(hce->hub_users,hue->user_nick->str,hue);
	*is_created=TRUE;
	return hue;		/* user created */
}


/* -------------------------------------------------------------------------- */
/* ------------------------- processing functions --------------------------- */
/* -------------------------------------------------------------------------- */
#if 0
/***********************************************************************/
/* terminate a user connection. After this call, the luce is destroyed */
/*******************************************************************************/
/* input: with_remove== TRUE, the luce is removed from the user_cnx_entry_list */
/*                   == FALSE, the pointer of the destroyed luce is kept in the*/
/*                user_cnx_entry_list (useful for g_hash_table_foreach_remove) */
/*******************************************************************************/
static void end_user_connection(LOCAL_USER_CNX_ENTRY *luce, gboolean with_remove)
{
	GString *msg;

	/* send a $Quit to everyone */
	glus_do_quit(luce->user_nick->str);

	SEND_EVT_TO("quit",luce->user_nick->str,0);

	/* printf("cnx over1.\n"); */
	{
		unsigned int hip;

		hip=ntohl(luce->user_xfio->user_ip.s_addr);
		printf("cnxover1: <%s> \t%hhu.%hhu.%hhu.%hhu %lu %d %Lu/%Lu:%u/%u\n",
						luce->user_nick->str,
	 					(unsigned char)(hip>>24)&0xff,
		 				(unsigned char)(hip>>16)&0xff,
		 				(unsigned char)(hip>>8)&0xff,
		 				(unsigned char)hip&0xff,
		 				(long unsigned int)(gl_cur_time - luce->user_xfio->start_time),
						luce->sr_count,
		 				luce->user_xfio->out_data,
		 				luce->user_xfio->in_data,
		 				luce->out_cnx_count,
		 				luce->in_cnx_count
						);
	}
	if(with_remove)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		HL_LOCK_WRITE(user_cnx_entry_lock);
		g_hash_table_remove(user_cnx_entry_list,luce->user_nick->str);
		HL_UNLOCK_WRITE(user_cnx_entry_lock);
		HL_LOCK_READ(user_cnx_entry_lock);
	}
	uc_destroy_entry(luce);
	return;
}
#endif

/*************************************************************************************/
/* add or update the user in the chunk. The change must be relay to everyone but hce */
/*************************************************************************************/
/* NOTE: to cluster, we resend the chunk as is (on add or update, not if unchanged */
/***********************************************************************************/
/* Require: SC_USER_NICK. Optional: SC_USER_SHARE,SC_USER_MAIL,SC_USER_DESC */
/* SC_USER_FLAG,SC_USER_EFLAG,SC_USER_LEVEL,SC_USER_CNX_TYPE,SC_USER_CLIVER */
/****************************************************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_add_or_update(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	HUB_USER_ENTRY *hue;
	gboolean is_created=FALSE;
	gboolean is_updated=FALSE;

	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	/* find or create a user with this nick */
	hue=find_user_named_for_this_hub(hce,sc->chunk_content,sc->chunk_size,&is_created,TRUE);	/* allow creation of missing user */
	if(hue==NULL)
		return;	/* something goes really wrong */

	CHUNK_GET_AND_SET_ULVAL_ND(hue->shared_size,cc,SC_USER_SHARE,is_updated);
	CHUNK_GET_AND_SET_NNSTR_in_gstring_ND(hue->user_mail,cc,SC_USER_MAIL,is_updated);
	CHUNK_GET_AND_SET_NNSTR_in_gstring_ND(hue->user_description,cc,SC_USER_DESC,is_updated);
	CHUNK_GET_AND_SET_UVAL_ND(hue->user_flag,cc,SC_USER_FLAG,is_updated);
	CHUNK_GET_AND_SET_UVAL_ND(hue->ext_flag,cc,SC_USER_EFLAG,is_updated);
	CHUNK_GET_AND_SET_SVAL_ND(hue->level,cc,SC_USER_LEVEL,is_updated);

	/* USER_CNX_TYPE uses a static string, not an allocated one */
	sc=chunk_get_subcore(cc,SC_USER_CNX_TYPE);
	if(sc!=NULL)
	{
		const char *cnx_type;
		cnx_type=fnd_static_cnx_type_from_str_len(sc->chunk_content,sc->chunk_size);
		if(strcmp(cnx_type,hue->user_cnx_type))
		{
			hue->user_cnx_type=cnx_type;
			is_updated=TRUE;
		}
	}

	CHUNK_GET_AND_SET_NNSTR_in_gstring_ND(hue->client_version,cc,SC_USER_CLIVER,is_updated);

	/* on creation, say hello to all local users */
	if(is_created)
	{
		exclude_all_clusters=TRUE;
		glus_do_hello(hue->user_nick->str);
		exclude_all_clusters=FALSE;
	}
	
	/* on creation or update, send the myinfo to everyone */
	/* if(is_updated||is_created) */
	/* changed to 'is_updated' only by hhgoth */
	if(is_updated)
	{
		/* to avoid loop, we don't send a message on the link where it came from */
		excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
		glus_do_my_info(hue->user_nick->str,hue->user_description->str, hue->user_cnx_type,
		                hue->user_flag,hue->user_mail->str,hue->shared_size,
		                hue->ext_flag,hue->level,hue->client_version->str);
		excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);
	}
}

/******************************************************************************/
/* remove the user in the chunk. The change must be relay to everyone but hce */
/******************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_USER_NICK */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_remove(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	HUB_USER_ENTRY *hue;
	gboolean is_created;

	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	/* search for a user on this hub connection */
	hue=find_user_named_for_this_hub(hce,sc->chunk_content,sc->chunk_size,&is_created,FALSE);	/* no creation of missing user */
	if(hue==NULL)
		return;	/* the user does not exist, technically, a user has left but has never come in, that's weird :) */
		       	/* we don't propagate such strange message */

	/* remove this user from the hub */
	g_hash_table_remove(hce->hub_users,hue->user_nick->str);

	/* send a quit to all local users and to all hubs except the one where the message came from */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	//glus_do_quit(hue->user_nick->str);	
	/* and don't forget to destroy the memory used by this user */
	delete_one_hub_user_entry(hue->user_nick->str,hue,NULL);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

}

/*********************************************************/
/* add the info of the given glus_user to the chunk_core */
/*********************************************************/
static void add_user_to_user_lst_for_each_hub(GLUS_USER_INFO *gui, CHUNK_CORE *cc)
{
	guint64 share=GUINT64_TO_BE(gui->shared_size);
	guint16 uflag=GUINT16_TO_BE(gui->user_flag);
	guint16 eflag=GUINT16_TO_BE(gui->ext_flag);
	gint16 lvl=GINT16_TO_BE(gui->level);

	chunk_core_append_subchunk(cc,SC_USER_NICK,gui->user_nick->len,gui->user_nick->str);
	chunk_core_append_subchunk(cc,SC_USER_SHARE,sizeof(share),&share);
	chunk_core_append_subchunk(cc,SC_USER_MAIL,gui->user_mail->len,gui->user_mail->str);
	chunk_core_append_subchunk(cc,SC_USER_DESC,gui->user_description->len,gui->user_description->str);
	chunk_core_append_subchunk(cc,SC_USER_FLAG,sizeof(uflag),&uflag);
	chunk_core_append_subchunk(cc,SC_USER_EFLAG,sizeof(eflag),&eflag);
	chunk_core_append_subchunk(cc,SC_USER_LEVEL,sizeof(lvl),&lvl);
	chunk_core_append_subchunk(cc,SC_USER_CNX_TYPE,strlen(gui->user_cnx_type),gui->user_cnx_type);
	chunk_core_append_subchunk(cc,SC_USER_CLIVER,gui->client_version->len,gui->client_version->str);
}

/**********************************/
/* ask for a remote hub user list */
/**********************************/
/* Require: nothing */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_lst_req(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	CHUNK_CORE *cc_lst;
	GPtrArray *gpa;
	int i;

	cc_lst=chunk_core_new(CC_USER_LST);
	if(cc_lst==NULL)
		return;

	/* retrieve the list of all known users except those of hce */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	gpa=glus_get_users_info();
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* convert the array of GLUS_USER_INFO into a chunk array */
	for(i=0;i<gpa->len;i++)
	{
		add_user_to_user_lst_for_each_hub(g_ptr_array_index(gpa,i),cc_lst);
	}

	/* send the chunk to hce */
	bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,
												chunk_convert_chunk_to_mem(cc_lst),
												FALSE);	/* dont copy yhe chunk, don't free gba */
	/* free the chunk */
	chunk_free(cc_lst);

	glus_free_user_info_ptrarray(gpa);
}

/************************/
/* remote hub user list */
/*******************************************************************************************************************/
/* Require: reply of CC_USER_LST_REQ. Require: SC_HUB_USERS and same as CC_USER_IN. A SC_USER_NICK is the start of */
/* values of a new nickname. technically, the reply is an array with a variable number of element per usernothing  */
/*******************************************************************************************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_lst(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	HUB_USER_ENTRY *hue=NULL;
	SUBCHUNK_CORE *sc=NULL;
	gboolean is_created=FALSE;
	gboolean is_updated=FALSE;
	int sc_num=0;
	guint64 __old_val_u64;
	guint16 __old_val_u16;
	gint16 __old_val_s16;

	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);

	while(sc_num<cc->nb_subchunk)
	{
		sc=cc->subchunk+sc_num;

		switch(sc->chunk_type)
		{
			case SC_USER_NICK:	/* we have encountered a new nick */
										if(hue!=NULL)
										{
											/* on creation, say hello to all local users */
											if(is_created)
											{
												exclude_all_clusters=TRUE;
												glus_do_hello(hue->user_nick->str);
												exclude_all_clusters=FALSE;
											}
											
											/* on creation or update, send the myinfo to everyone */
											if(is_updated||is_created)
											{
												/* to avoid loop, we don't send a message on the link where it came from */
												glus_do_my_info(hue->user_nick->str,hue->user_description->str, hue->user_cnx_type,
		                										hue->user_flag,hue->user_mail->str,hue->shared_size,
		                										hue->ext_flag,hue->level,hue->client_version->str);
											}
											hue=NULL;
										}

										is_created=FALSE;
										is_updated=FALSE;

										/* find or create a user with this nick */
										hue=find_user_named_for_this_hub(hce,sc->chunk_content,sc->chunk_size,&is_created,TRUE);	/* allow creation of missing user */
										if(hue==NULL)
											goto abrt;	/* something goes really wrong */
										break;
									
			case SC_USER_SHARE:	
										__old_val_u64=hue->shared_size;
										hue->shared_size=subchunk_get_guint64(sc);
										if(hue->shared_size!=__old_val_u64)
											is_updated=TRUE;
										break;

			case SC_USER_MAIL:
										is_updated|=subchunk_assign_gstring(hue->user_mail,sc);
										break;

			case SC_USER_DESC:
										is_updated|=subchunk_assign_gstring(hue->user_description,sc);
										break;

			case SC_USER_FLAG:
										__old_val_u16=hue->user_flag;
										hue->user_flag=subchunk_get_guint(sc);
										if(hue->user_flag!=__old_val_u16)
											is_updated=TRUE;
										break;

			case SC_USER_EFLAG:
										__old_val_u16=hue->ext_flag;
										hue->ext_flag=subchunk_get_guint(sc);
										if(hue->ext_flag!=__old_val_u16)
											is_updated=TRUE;
										break;

			case SC_USER_LEVEL:
										__old_val_s16=hue->level;
										hue->level=subchunk_get_gint(sc);
										if(hue->level!=__old_val_s16)
											is_updated=TRUE;
										break;

			case SC_USER_CLIVER:
										is_updated|=subchunk_assign_gstring(hue->client_version,sc);
										break;

			case SC_USER_CNX_TYPE:
										{
											const char *cnx_type;
											cnx_type=fnd_static_cnx_type_from_str_len(sc->chunk_content,sc->chunk_size);
											if(strcmp(cnx_type,hue->user_cnx_type))
											{
												hue->user_cnx_type=cnx_type;
												is_updated=TRUE;
											}
										}
										break;

			default:
										fprintf(stderr,"hub_cc_user_lst: unknown subchunk core: %d\n",sc->chunk_type);
										break;
		}
		sc_num++;
	}

	/* don't forget the last nick */
	if(hue!=NULL)
	{
		/* on creation, say hello to all local users */
		if(is_created)
		{
			exclude_all_clusters=TRUE;
			glus_do_hello(hue->user_nick->str);
			exclude_all_clusters=FALSE;
		}
											
		/* on creation or update, send the myinfo to everyone */
		if(is_updated||is_created)
		{
			/* to avoid loop, we don't send a message on the link where it came from */
			glus_do_my_info(hue->user_nick->str,hue->user_description->str, hue->user_cnx_type,
         										hue->user_flag,hue->user_mail->str,hue->shared_size,
         										hue->ext_flag,hue->level,hue->client_version->str);
		}
	}

	abrt:
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);
}

/******************************/
/* ask for a user information */
/******************************************************/
/* Require: SC_USER_NICK (reply will be a CC_USER_IN) */
/******************************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_info_req(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc_nick=NULL;
	GString *nick;
	GLUS_USER_INFO *gui;
	
	sc_nick=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc_nick==NULL)
		return;	/* invalid CC_USER_INFO_REQ chunk */

	nick=g_string_new("");
	subchunk_assign_gstring(nick,sc_nick);

	gui=glus_get_user_info(nick->str);
	if(gui!=NULL)
	{
		GByteArray *gba;
		guint64 share=GUINT64_TO_BE(gui->shared_size);
		guint16 uflag=GUINT16_TO_BE(gui->user_flag);
		guint16 eflag=GUINT16_TO_BE(gui->ext_flag);
		gint16 lvl=GINT16_TO_BE(gui->level);

		gba=chunk_convert_param_to_mem(CC_USER_IN,9,
													SC_PARAM_PTR(SC_USER_NICK,gui->user_nick->len,gui->user_nick->str),
													SC_PARAM_GUINT64(SC_USER_SHARE,share),
													SC_PARAM_PTR(SC_USER_MAIL,gui->user_mail->len,gui->user_mail->str),
													SC_PARAM_PTR(SC_USER_DESC,gui->user_description->len,gui->user_description->str),
													SC_PARAM_GUINT16(SC_USER_FLAG,uflag),
													SC_PARAM_GUINT16(SC_USER_EFLAG,eflag),
													SC_PARAM_GINT16(SC_USER_LEVEL,lvl),
													SC_PARAM_PTR(SC_USER_CNX_TYPE,strlen(gui->user_cnx_type),gui->user_cnx_type),
													SC_PARAM_PTR(SC_USER_CLIVER,gui->client_version->len,gui->client_version->str)
												);
		bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,gba,FALSE);	/* don't copy the chunk, don't free gba */
		glus_free_user_info(gui);
	}

	g_string_free(nick,TRUE);
}

/*****************************************/
/* send a broadcast message to all users */
/***********************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_MSG_CONTENT */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_bcast(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc_content=NULL;
	GString *content;
	
	sc_content=chunk_get_subcore(cc,SC_MSG_CONTENT);
	if(sc_content==NULL)
		return;	/* invalid CC_USER_BCAST chunk */

	content=g_string_new("");
	subchunk_assign_gstring(content,sc_content);

	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	GLUS_SEND_TO_EVERYONE(content);	/* don't free sc_content */
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);
}

/*************************************/
/* send an unicast message to a user */
/***********************************************************************************************/
/* NOTE: if the user is not on this node, we resend the chunk to the destination cluster as is */
/***********************************************************************************************/
/* Require: SC_MSG_CONTENT, SC_USER_NICK */
/*****************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_ucast(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc_nick=NULL;
	SUBCHUNK_CORE *sc_content=NULL;
	GString *nick;
	GString *content;
	
	sc_nick=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc_nick==NULL)
		return;	/* invalid CC_USER_UCAST chunk */

	sc_content=chunk_get_subcore(cc,SC_MSG_CONTENT);
	if(sc_content==NULL)
		return;	/* invalid CC_USER_UCAST chunk */

	nick=g_string_new("");
	content=g_string_new("");
	subchunk_assign_gstring(nick,sc_nick);
	subchunk_assign_gstring(content,sc_content);

	GLUS_SEND_TO_A_NICK(nick->str,content);	/* don't free content */
	g_string_free(nick,TRUE);
}

/****************************************************************************/
/* kick the user in the chunk. The change must be relay to everyone but hce */
/****************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_USER_NICK */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_kick(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	GString *kicked_nick;
	GString *kicker_nick;

	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	kicked_nick=g_string_new("");
	subchunk_assign_gstring(kicked_nick,sc);
	
	sc=chunk_get_subcore(cc,SC_KICK_KICKER);
	if(sc==NULL)
		kicker_nick=NULL;
	else
	{
		kicker_nick=g_string_new("");
		subchunk_assign_gstring(kicker_nick,sc);
	}

	/* the glus_kick function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_kick_named_user((kicker_nick!=NULL)?kicker_nick->str:NULL,
	                     kicked_nick->str);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	g_string_free(kicked_nick,TRUE);
	if(kicker_nick)
		g_string_free(kicker_nick,TRUE);
}

/***************************************************************************/
/* ban the user in the chunk. The change must be relay to everyone but hce */
/***************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_USER_NICK, SC_KICK_DURATION */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_ban(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	GString *kicked_nick;
	GString *kicker_nick;
	unsigned long duration=24*3600;	/* default duration is 24 hours */
	gboolean have_duration=FALSE;

	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	kicked_nick=g_string_new("");
	subchunk_assign_gstring(kicked_nick,sc);
	
	sc=chunk_get_subcore(cc,SC_KICK_KICKER);
	if(sc==NULL)
		kicker_nick=NULL;
	else
	{
		kicker_nick=g_string_new("");
		subchunk_assign_gstring(kicker_nick,sc);
	}

	CHUNK_GET_AND_SET_UVAL_ND(duration,cc,SC_KICK_DURATION,have_duration)
	
	/* the glus_kick function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_ban_named_user((kicker_nick!=NULL)?kicker_nick->str:NULL,
	                     kicked_nick->str, duration);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	g_string_free(kicked_nick,TRUE);
	if(kicker_nick)
		g_string_free(kicker_nick,TRUE);
}

/*************************************************************************/
/* ban the IP in the chunk. The change must be relay to everyone but hce */
/*************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_USER_IP, SC_KICK_DURATION */
/*****************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_banip(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	GString *kicker_nick;
	unsigned long duration=24*3600;	/* default duration is 24 hours */
	struct in_addr ip;
	gboolean have_duration=FALSE;
	gboolean have_ip=FALSE;

	sc=chunk_get_subcore(cc,SC_USER_IP);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	CHUNK_GET_AND_SET_UVAL_ND(ip.s_addr,cc,SC_USER_IP,have_ip);
	if(have_ip==FALSE)
	{
		fprintf(stderr,"hub<=>hub: error, hub_cc_user_banip: no SC_USER_IP chunk\n");
		return;
	}
	
	sc=chunk_get_subcore(cc,SC_KICK_KICKER);
	if(sc==NULL)
		kicker_nick=NULL;
	else
	{
		kicker_nick=g_string_new("");
		subchunk_assign_gstring(kicker_nick,sc);
	}

	CHUNK_GET_AND_SET_UVAL_ND(duration,cc,SC_KICK_DURATION,have_duration)
	
	/* the glus_ban_ip function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_ban_ip((kicker_nick!=NULL)?kicker_nick->str:NULL,
					ip,duration);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	if(kicker_nick)
		g_string_free(kicker_nick,TRUE);
}

/*****************************************************************************/
/* unban the user in the chunk. The change must be relay to everyone but hce */
/*****************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_USER_NICK */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_uban(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	GString *kicked_nick;

	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	kicked_nick=g_string_new("");
	subchunk_assign_gstring(kicked_nick,sc);
	
	/* the glus_kick function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_uban_named_user(kicked_nick->str);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	g_string_free(kicked_nick,TRUE);
}

/**********************************************************************************/
/* disconnect the user in the chunk. The change must be relay to everyone but hce */
/**********************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/***********************************************/
/* Require: SC_USER_NICK */
/*****************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_user_disc(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	SUBCHUNK_CORE *sc;
	GString *kicked_nick;

	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		return;	/* invalid CC_USER_IN chunk */

	kicked_nick=g_string_new("");
	subchunk_assign_gstring(kicked_nick,sc);
	
	/* the glus_kick function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_disconnect_named_user(kicked_nick->str);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	g_string_free(kicked_nick,TRUE);
}

/****************************************************************************/
/* process an extended search. The search must be relay to everyone but hce */
/****************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/********************************************************************/
/* Require: SC_RETURN_PATH, SC_FILE_CRC, SC_FILE_LEN, SC_STD_SEARCH */
/********************************************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_xsearch(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	guint8 *fcrc;
	guint32 file_length;
	GString *return_path;
	GString *std_search;
	SUBCHUNK_CORE *sc;

	return_path=g_string_new("");
	std_search=g_string_new("");
	
	sc=chunk_get_subcore(cc,SC_RETURN_PATH);
	if(sc==NULL)
		goto abrt;
	subchunk_assign_gstring(return_path,sc);
	
	sc=chunk_get_subcore(cc,SC_FILE_CRC);
	if((sc==NULL)||(sc->chunk_size!=MD4_DIGEST_LENGTH))
		goto abrt;
	fcrc=sc->chunk_content;

	sc=chunk_get_subcore(cc,SC_FILE_LEN);
	if(sc==NULL)
		goto abrt;
	file_length=subchunk_get_guint(sc);

	sc=chunk_get_subcore(cc,SC_STD_SEARCH);
	if(sc==NULL)
		goto abrt;
	subchunk_assign_gstring(std_search,sc);
	
	/* the glus_do_xsearch function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_do_xsearch(return_path->str, fcrc, file_length, std_search->str);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	abrt:
	g_string_free(return_path,TRUE);
	g_string_free(std_search,TRUE);
}

/**********************************************************************************/
/* process an incoming md4set. The message must be relay to every cluster but hce */
/**********************************************************************************/
/* NOTE: to cluster, we resend the chunk as is */
/*******************************************************************************/
/* Require: SC_FILE_CRC, SC_FILE_LEN, SC_FILE_L0CRC, SC_FILENAME, SC_USER_NICK */
/*******************************************************************************/
/* don't free the CHUNK_CORE */
/*****************************/
static void hub_cc_md4set(HUB_CNX_ENTRY *hce,CHUNK_CORE *cc)
{
	guint8 *fcrc;
	guint8 *l_crc;
	guint32 file_length;
	GString *filename;
	GString *user_nick;
	int nb_seg;
	SUBCHUNK_CORE *sc;

	filename=g_string_new("");
	user_nick=g_string_new("");

	/* file global CRC */
	sc=chunk_get_subcore(cc,SC_FILE_CRC);
	if((sc==NULL)||(sc->chunk_size!=MD4_DIGEST_LENGTH))
		goto abrt;
	fcrc=sc->chunk_content;

	/* file size */
	sc=chunk_get_subcore(cc,SC_FILE_LEN);
	if(sc==NULL)
		goto abrt;
	file_length=subchunk_get_guint(sc);

	/* file l0 CRC */
	sc=chunk_get_subcore(cc,SC_FILE_L0CRC);
	if(sc==NULL)
		goto abrt;
	nb_seg=(file_length+PARTSIZE-1)/PARTSIZE;
	if(sc->chunk_size!=(nb_seg*MD4_DIGEST_LENGTH))
		goto abrt;
	l_crc=sc->chunk_content;

	/* filename */
	sc=chunk_get_subcore(cc,SC_FILENAME);
	if(sc==NULL)
		goto abrt;
	subchunk_assign_gstring(filename,sc);

	/* usernick */
	sc=chunk_get_subcore(cc,SC_USER_NICK);
	if(sc==NULL)
		goto abrt;
	subchunk_assign_gstring(user_nick,sc);

	/* the glus_md4set_receive function does everything, we just add hce to the excluded hub list to prevent the message to go back */
	excluded_hub_entry=g_list_prepend(excluded_hub_entry,hce);
	glus_md4set_received(fcrc,file_length,l_crc,nb_seg,filename->str,user_nick->str);
	excluded_hub_entry=g_list_remove(excluded_hub_entry,hce);

	/* free memory */
	abrt:
	g_string_free(filename,TRUE);
	g_string_free(user_nick,TRUE);
}

/******************************************************/
/* process incoming chunks on the given hub connexion */
/***************************************************************************/
/* if the connection must be deleted, just set its BIN_XF_IO to CNX_CLOSED */
/***************************************************************************/
static void process_hub_incoming_chunks(CLUSTER_ENTRY *ce,HUB_CNX_ENTRY *hce)
{
	CHUNK_CORE *cc;

	/* process all available chunks */
	while((cc=bin_xfio_take_first_chunk_of_glist_incoming(hce->hub_bxfio))!=NULL)
	{
		switch(cc->chunk_type)
		{
			case CC_USER_IN:
							hub_cc_user_add_or_update(hce,cc);
							break;

			case CC_USER_OUT:
							hub_cc_user_remove(hce,cc);
							break;

			case CC_USER_LST_REQ:
							hub_cc_user_lst_req(hce,cc);
							break;

			case CC_USER_LST:
							hub_cc_user_lst(hce,cc);
							break;

			case CC_USER_INFO_REQ:
							hub_cc_user_info_req(hce,cc);
							break;

			case CC_BCAST:
							hub_cc_bcast(hce,cc);
							break;

			case CC_UCAST:
							hub_cc_ucast(hce,cc);
							break;

			case CC_USER_KICK:
							hub_cc_user_kick(hce,cc);
							break;

			case CC_USER_BAN:
							hub_cc_user_ban(hce,cc);
							break;

			case CC_USER_BANIP:
							hub_cc_user_banip(hce,cc);
							break;

			case CC_USER_UBAN:
							hub_cc_user_uban(hce,cc);
							break;

			case CC_USER_DISC:
							hub_cc_user_disc(hce,cc);
							break;

			case CC_XSEARCH:
							hub_cc_xsearch(hce,cc);
							break;

			case CC_MD4SET:
							hub_cc_md4set(hce,cc);
							break;

			default:		fprintf(stderr,"Unknown chunk code: %d\n",cc->chunk_type);
							break;
		}

		/* free the extracted chunk */
		chunk_free(cc);

		/* abort the loop if the connection is closed */
		if(BIN_XFIO_IS_CLOSED(hce->hub_bxfio))
			break;
	}
}

/*************************************/
/* main handler of the cluster entry */
/**************************************************************************/
/* this function handles all incoming requests from all hubs of a cluster */
/**************************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                        */
/*        user_data= NULL (unused, just required for the GHfunc)   */
/*******************************************************************/
/* NOTE: the array has LOCK_READ set */
/**********************************************************/
/* output: TRUE= the entry is removed from the hash table */
/*         FALSE= the entry remains in the hash table     */
/**********************************************************/
static gboolean process_cluster_entry(gpointer value, gpointer user_data)
{
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;
	int i;

	/* process each entry of hc_array of ce */
	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));
		if(BIN_XFIO_IS_CLOSED(hce->hub_bxfio))
			continue;

		if(BIN_XFIO_INPUT_AVAILABLE(hce->hub_bxfio))
		{
			process_hub_incoming_chunks(ce,hce);
		}
	}	

	/* discard closed hub connection of ce */
	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));
		if(!BIN_XFIO_IS_CLOSED(hce->hub_bxfio))
			continue;

		hc_destroy_hub_entry(ce,hce,i);
	}

	if(ce->hc_array->len==0)
	{
		g_array_free(ce->hc_array,TRUE);
		ce->hc_array=NULL;
		return TRUE;	/* discard this cluster */
	}
	return FALSE;		/* don't discard the entry */
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GLUS handlers ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/*****************************************************/
/* function used to say a "$Hello" on the connection */
/*****************************************************/
static void hc_user_do_hello(const char *nickname)
{
	GByteArray *gba;

	gba=chunk_convert_param_to_mem(CC_USER_IN,1,
												SC_PARAM_PTR(SC_USER_NICK,strlen(nickname),nickname)
											);
	send_chunk_to_all_hubs(gba);	/* don't free gba */
}

/* -------------------------------------------------------------------------- */
/****************************************************/
/* function used to say a "$Quit" on the connection */
/****************************************************/
static void hc_user_do_quit(const char *nickname)
{
	GByteArray *gba;

	gba=chunk_convert_param_to_mem(CC_USER_OUT,1,
												SC_PARAM_PTR(SC_USER_NICK,strlen(nickname),nickname)
											);
	send_chunk_to_all_hubs(gba);	/* don't free gba */
}

/* -------------------------------------------------------------------------- */
/******************************************************/
/* function used to say a "$MyINFO" on the connection */
/******************************************************/
static void hc_user_do_myinfo(const char *nickname, const char *description, const char *user_cnx_type, const guint16 user_flag, const char *user_mail, const guint64 shared_size, const guint16 ext_flag, const gint16 level, const char *client_version)
{
	GByteArray *gba;
	guint64 share=GUINT64_TO_BE(shared_size);
	guint16 uflag=GUINT16_TO_BE(user_flag);
	guint16 eflag=GUINT16_TO_BE(ext_flag);
	gint16 lvl=GINT16_TO_BE(level);

	gba=chunk_convert_param_to_mem(CC_USER_IN,9,
												SC_PARAM_PTR(SC_USER_NICK,strlen(nickname),nickname),
												SC_PARAM_GUINT64(SC_USER_SHARE,share),
												SC_PARAM_PTR(SC_USER_MAIL,strlen(user_mail),user_mail),
												SC_PARAM_PTR(SC_USER_DESC,strlen(description),description),
												SC_PARAM_GUINT16(SC_USER_FLAG,uflag),
												SC_PARAM_GUINT16(SC_USER_EFLAG,eflag),
												SC_PARAM_GINT16(SC_USER_LEVEL,lvl),
												SC_PARAM_PTR(SC_USER_CNX_TYPE,strlen(user_cnx_type),user_cnx_type),
												SC_PARAM_PTR(SC_USER_CLIVER,strlen(client_version),client_version)
											);
	send_chunk_to_all_hubs(gba);	/* don't free gba */
}


/* -------------------------------------------------------------------------- */
/**************************************************************************************/
/* send a message to a user having the given name and required user_flag and ext_flag */
/**************************************************************************************/
/* output: 0=ok else user not found */
/************************************/
static int hc_send_to_named_user(GLUS_PARAM *glus_param)
{
	HUB_CNX_ENTRY *hce;
	GByteArray *gba;

	HL_LOCK_READ(cluster_entry_lock);
	hce=find_hub_cnx_entry_with_a_user(glus_param->nickname);
	if(hce==NULL)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return 1;	/* not found */
	}

	gba=chunk_convert_param_to_mem(CC_UCAST,2,
												SC_PARAM_PTR(SC_USER_NICK,strlen(glus_param->nickname),glus_param->nickname),
												SC_PARAM_PTR(SC_MSG_CONTENT,glus_param->msg->data->len,glus_param->msg->data->str)
											);
	bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,gba,FALSE);	/* don't copy the chunk, don't free gba */
	HL_UNLOCK_READ(cluster_entry_lock);
	return 0;
}

/* -------------------------------------------------------------------------- */
/**********************************************************************/
/* send a message to all users having required user_flag and ext_flag */
/**********************************************************************/
static void hc_send_to_all_users(GLUS_PARAM *glus_param)
{
	GByteArray *gba;
#warning user_flag and ext_flag not honored 

	gba=chunk_convert_param_to_mem(CC_BCAST,1,
								   SC_PARAM_PTR(SC_MSG_CONTENT,glus_param->msg->data->len,glus_param->msg->data->str));

	send_chunk_to_all_hubs(gba);	/* don't free gba */
}

/* -------------------------------------------------------------------------- */
/****************************************************/
/* process a $XSearch and send it on the connection */
/****************************************************/
static void hc_user_do_xsearch(const char *return_path, const guint8 *file_crc, const guint32 file_length, const char *std_search)
{
	GByteArray *gba;
	guint32 size=GUINT32_TO_BE(file_length);

	gba=chunk_convert_param_to_mem(CC_XSEARCH,4,
												SC_PARAM_PTR(SC_RETURN_PATH,strlen(return_path),return_path),
												SC_PARAM_PTR(SC_FILE_CRC,MD4_DIGEST_LENGTH,file_crc),
												SC_PARAM_GUINT32(SC_FILE_LEN,size),
												SC_PARAM_PTR(SC_STD_SEARCH,strlen(std_search),std_search)
											);
	send_chunk_to_all_hubs(gba);	/* don't free gba */
}

/* -------------------------------------------------------------------------- */
static GLUS_USER_INFO *hc_glus_get_user_info_from_hue(HUB_USER_ENTRY *hue)
{
	GLUS_USER_INFO *gui;
	
	gui=malloc(sizeof(GLUS_USER_INFO));
	if(gui==NULL)
	{
		fprintf(stderr,"hc_glus_get_user_info: out of memory\n");
		return NULL;
	}

	gui->cnx_start_time=0;		/* it is the hub connection start time */
	gui->user_cnx_type=hue->user_cnx_type;
	gui->user_nick=g_string_new(hue->user_nick->str);
	gui->shared_size=hue->shared_size;
	if(hue->user_description)
		gui->user_description=g_string_new(hue->user_description->str);
	else
		gui->user_description=g_string_new("");
	if(hue->user_mail)
		gui->user_mail=g_string_new(hue->user_mail->str);
	else
		gui->user_mail=g_string_new("");
	gui->user_flag=hue->user_flag;
	gui->ext_flag=hue->ext_flag;
	gui->level=hue->level;
	if(hue->client_version)
		gui->client_version=g_string_new(hue->client_version->str);
	else
		gui->client_version=g_string_new("");

	memset(&(gui->user_ip),0,sizeof(gui->user_ip));

	return gui;
}

/*********************************************/
/* retrieve user information from a nickname */
/*********************************************/
/* output: NULL= not found */
/***************************/
static GLUS_USER_INFO *hc_glus_get_user_info(const char *nickname)
{
	HUB_USER_ENTRY *hue;
	HUB_CNX_ENTRY *hce;
	GLUS_USER_INFO *gui;
	gboolean is_created;

	HL_LOCK_READ(cluster_entry_lock);
	hce=find_hub_cnx_entry_with_a_user(nickname);
	if(hce==NULL)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return NULL;		/* user not found */
	}

	hue=find_user_named_for_this_hub(hce,nickname,strlen(nickname),&is_created,FALSE);	/* no creation of missing user */
	if(hue==NULL)
	{
		fprintf(stderr,"hc_glus_get_user_info: internal error, a user has disappeared while the lock is set.\n");
		HL_UNLOCK_READ(cluster_entry_lock);
		return NULL;		/* user not found */
	}

	gui=hc_glus_get_user_info_from_hue(hue);
	HL_UNLOCK_READ(cluster_entry_lock);
	return gui;
}

/* -------------------------------------------------------------------------- */
/*************************************************/
/* add the glus user info of a user to the array */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= HUB_USER_ENTRY * to process                                 */
/*        user_data= GPtrArray of GLUS_USER_INFO                             */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void hc_get_user_info_two(gpointer key, gpointer value, gpointer user_data)
{
	GPtrArray *array_of_gui_ptr=user_data;
	HUB_USER_ENTRY *hue=value;

	g_ptr_array_add(array_of_gui_ptr,hc_glus_get_user_info_from_hue(hue));
}

/*************************************************/
/* add the glus user info of a user to the array */
/*****************************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                                  */
/*        user_data= GPtrArray of GLUS_USER_INFO                             */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void hc_get_user_info_one(gpointer value, gpointer user_data)
{
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;
	int i;

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));
		g_hash_table_foreach(hce->hub_users,hc_get_user_info_two,user_data);
	}	
}

/******************************************************/
/* get all users. (add GLUS_USER_INFO * to the array) */
/******************************************************/
void hc_glus_get_users_info(GPtrArray *array_of_gui_ptr)
{
	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach(cluster_entry_list,hc_get_user_info_one,array_of_gui_ptr);
	HL_UNLOCK_READ(cluster_entry_lock);
}

/************************************/
/* add a user to user (and op) list */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= HUB_USER_ENTRY * to process                                 */
/*        user_data= GString **  ([0]=user nick list, [1]=op nick list)      */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void hc_get_nick_lists_two(gpointer key, gpointer value, gpointer user_data)
{
	GString **lists=user_data;
	HUB_USER_ENTRY *hue=value;

	g_string_append(lists[0],hue->user_nick->str);
	g_string_append(lists[0],"$$");

	if(hue->level >= get_right_level(OPERATOR))
	{
		g_string_append(lists[1],hue->user_nick->str);
		g_string_append(lists[1],"$$");
	}
}

/**********************************/
/* add the nicks of the given hub */
/*****************************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                                  */
/*        user_data= GPtrArray of GLUS_USER_INFO                             */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void hc_get_nick_lists_one(gpointer value, gpointer user_data)
{
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;
	int i;

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));
		g_hash_table_foreach(hce->hub_users,hc_get_nick_lists_two,user_data);
	}	
}

/************************************************************************/
/* add the nicks to the provided strings (each nick is '$$' terminated) */
/************************************************************************/
/* list[0]= user nick list */
/* list[1]= op nick list   */
/***************************/
static void hc_glus_get_nick_lists(GString **lists)
{
	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach(cluster_entry_list,hc_get_nick_lists_one,lists);
	HL_UNLOCK_READ(cluster_entry_lock);
}


/*************************************************/
/* add number of users and share size of the hub */
/*************************************************/
static void hc_add_hub_users_stat(guint64 *shared_size, unsigned int *nb_users)
{
	/* nothing to do */
}

/*****************************************/
/* kick a user having the given nickname */
/*****************************************/
/* output: 0=ok else user not found */
/************************************/
static int hc_kick_named_user(const char *kicking_nickname, const char *nickname)
{
	HUB_CNX_ENTRY *hce;
	HUB_USER_ENTRY *hue;
	gboolean is_created=FALSE;

	CHUNK_CORE *cc;

	HL_LOCK_READ(cluster_entry_lock);
	hce=find_hub_cnx_entry_with_a_user(nickname);
	if(hce==NULL)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return 1;	/* not found */
	}

	hue=find_user_named_for_this_hub(hce,nickname,strlen(nickname),&is_created,FALSE);	/* no creation of missing user */
	if(hue==NULL)
	{
		fprintf(stderr,"hc_kick_named_user: internal error, a user has disappeared while the lock is set.\n");
		HL_UNLOCK_READ(cluster_entry_lock);
		return 1;		/* user not found */
	}

	if (hue->ext_flag & SHILD_MODE)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return 0;	/* user found, nothing to do */
	}

	cc=chunk_core_new(CC_USER_KICK);
	chunk_core_append_subchunk(cc,SC_USER_NICK,strlen(nickname),nickname);
	
	if (kicking_nickname!=NULL)
		chunk_core_append_subchunk(cc,SC_KICK_KICKER,strlen(kicking_nickname),kicking_nickname);

	/* send the chunk to hce */
	bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,
											  chunk_convert_chunk_to_mem(cc),
											  FALSE);	/* dont copy the chunk, don't free gba */
	/* free the chunk */
	chunk_free(cc);

	HL_UNLOCK_READ(cluster_entry_lock);
	return 0;
}

/****************************************/
/* ban a user having the given nickname */
/****************************************/
/* output: 0=ok else user not found */
/************************************/
static int hc_ban_named_user(const char *kicking_nickname, const char *nickname, unsigned long duration)
{
	HUB_CNX_ENTRY *hce;
	HUB_USER_ENTRY *hue;
	gboolean is_created=FALSE;

	CHUNK_CORE *cc;

	HL_LOCK_READ(cluster_entry_lock);
	hce=find_hub_cnx_entry_with_a_user(nickname);
	if(hce==NULL)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return 1;	/* not found */
	}

	hue=find_user_named_for_this_hub(hce,nickname,strlen(nickname),&is_created,FALSE);	/* no creation of missing user */
	if(hue==NULL)
	{
		fprintf(stderr,"hc_ban_named_user: internal error, a user has disappeared while the lock is set.\n");
		HL_UNLOCK_READ(cluster_entry_lock);
		return 1;		/* user not found */
	}

	if (hue->ext_flag & SHILD_MODE)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return 0;	/* user found, nothing to do */
	}

	cc=chunk_core_new(CC_USER_BAN);
	chunk_core_append_subchunk(cc,SC_USER_NICK,strlen(nickname),nickname);
	{
		guint32 ntemp_ban;
		ntemp_ban=GUINT32_TO_BE(duration);
		chunk_core_append_subchunk(cc,SC_KICK_DURATION,sizeof(ntemp_ban),&ntemp_ban);
	}
	
	if(kicking_nickname!=NULL)
	{
		chunk_core_append_subchunk(cc,SC_KICK_KICKER,strlen(kicking_nickname),kicking_nickname);
	}

	/* send the chunk to hce */
	bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,
												chunk_convert_chunk_to_mem(cc),
												FALSE);	/* dont copy the chunk, don't free gba */
	/* free the chunk */
	chunk_free(cc);

	HL_UNLOCK_READ(cluster_entry_lock);
	return 0;
}

/**************************************************************/
/* ban a user having the given IP                             */
/* NOTE: this ban is communicated to all nodes of the cluster */
/**************************************************************/
/* output: 0=ok else user not found */
/************************************/
static int hc_ban_ip(const char *kicking_nickname, struct in_addr ip, unsigned long duration)
{
	CHUNK_CORE *cc;
	GByteArray *gba;
	CLUSTER_ENTRY *ce;
	int num_cluster,num_hub;

	HL_LOCK_READ(cluster_entry_lock);
	cc=chunk_core_new(CC_USER_BANIP);
	chunk_core_append_subchunk(cc,SC_USER_IP,sizeof(ip.s_addr),&(ip.s_addr));
	{
		guint32 ntemp_ban;
		ntemp_ban=GUINT32_TO_BE(duration);
		chunk_core_append_subchunk(cc,SC_KICK_DURATION,sizeof(ntemp_ban),&ntemp_ban);
	}
	
	if(kicking_nickname!=NULL)
	{
		chunk_core_append_subchunk(cc,SC_KICK_KICKER,strlen(kicking_nickname),kicking_nickname);
	}

	/* convert the chunk into array */
	gba=chunk_convert_chunk_to_mem(cc);

	/* now, we must send the packets to all other nodes */
	num_cluster=0;
	while((ce=g_list_nth_data(cluster_entry_list,num_cluster++))!=NULL)
	{
		num_hub=0;
		while(num_hub<ce->hc_array->len)
		{
			/* send the chunk to hce */
			bin_xfio_append_new_gba_to_glist_outgoing(g_array_index(ce->hc_array,HUB_CNX_ENTRY,num_hub).hub_bxfio,
												gba,
												TRUE);	/* copy the chunk */
		
			num_hub++;
		}
	}

	/* free the array */
	g_byte_array_free(gba,TRUE);

	/* free the chunk */
	chunk_free(cc);

	HL_UNLOCK_READ(cluster_entry_lock);
	return 0;
}

/******************************************/
/* unban a user having the given nickname */
/******************************************/
/* output: 0=ok else user not found */
/************************************/
static int hc_uban_named_user(const char *nickname)
{
	GByteArray *gba;

	CHUNK_CORE *cc;

	cc=chunk_core_new(CC_USER_UBAN);
	chunk_core_append_subchunk(cc,SC_USER_NICK,strlen(nickname),nickname);

	/* send the chunk to hce */
	gba=chunk_convert_chunk_to_mem(cc);
	send_chunk_to_all_hubs(gba);	/* don't free gba */

	/* free the chunk */
	chunk_free(cc);

	return 0;
}

/**************************************************************/
/* function called after a user was disconnected from the hub */
/* the function is not here to destroy a user itself, it is   */
/* here to notify other I/O function this user has left       */
/**************************************************************/
static int hc_disconnect_named_user(const char *nickname)
{
	HUB_CNX_ENTRY *hce;
	GByteArray *gba;

	HL_LOCK_READ(cluster_entry_lock);
	hce=find_hub_cnx_entry_with_a_user(nickname);
	if(hce==NULL)
	{
		HL_UNLOCK_READ(cluster_entry_lock);
		return 1;	/* not found */
	}

	gba=chunk_convert_param_to_mem(CC_USER_DISC,1,
												SC_PARAM_PTR(SC_USER_NICK,strlen(nickname),nickname)
											);
	bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,gba,FALSE);	/* don't copy the chunk, don't free gba */
	HL_UNLOCK_READ(cluster_entry_lock);
	return 0;
}

/**************************************************************/
/* function called after a user was disconnected from the hub */
/* the function is not here to destroy a user itself, it is   */
/* here to notify other I/O function this user has left       */
/**************************************************************/
static void hc_user_is_disconnected(const char *nickname)
{
}
/**************************************************************************/
/* get all users having the given IP. (add GLUS_USER_INFO * to the array) */
/**************************************************************************/
void hc_glus_get_users_info_by_ip (GPtrArray *array_of_gui_ptr, struct in_addr ip)
{
	/* nothing to do, cluster users has no IP */
}

/**********************************************/
/* function called when a $MD4Set is received */
/***************************************************************/
/* g_crc is the file global CRC (MD_BLOC_SIZE bytes)           */
/* l_crc is the set of partial CRC (nb_seg*MD_BLOC_SIZE bytes) */
/* nickname is the return path nickname                        */
/***************************************************************/
void hc_md4set_received(const guint8 *g_crc, const guint32 file_length, const guint8 *l_crc, const guint32 nb_seg, const char *filename, const char *nickname)
{
	int ret;
	GByteArray *gba;
	guint32 rfl;

	/* add the value to the database */
	ret=md4_add(g_crc,file_length,l_crc,nb_seg,filename);
	if(ret<0)	/* error when adding CRC */
		return;

	/* and relay it to all hubs */
	rfl=GUINT32_TO_BE(file_length);
	gba=chunk_convert_param_to_mem(CC_MD4SET, 5,
	                                 SC_PARAM_PTR(SC_FILE_CRC,MD_BLOC_SIZE,g_crc),
	                                 SC_PARAM_GUINT32(SC_FILE_LEN,rfl),
	                                 SC_PARAM_PTR(SC_FILE_L0CRC,nb_seg*MD_BLOC_SIZE,l_crc),
	                                 SC_PARAM_PTR(SC_FILENAME,strlen(filename),filename),
	                                 SC_PARAM_PTR(SC_USER_NICK,strlen(nickname),nickname)
	                              );
	send_chunk_to_all_hubs(gba);	/* don't free gba */
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GED handlers ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************************/
/* function called on exit */
/***************************/
static int hc_ged_exit(const struct ged_callbacks *ged)
{
	return 0;
}

static void g_list_foreach_remove(GList **list, gboolean (*func)(gpointer value, gpointer user_data),
                                  gpointer user_data)
{
	GList *cur=g_list_first(*list);

	while(cur!=NULL)
	{
		GList *nxt;
		gpointer data;

		nxt=g_list_next(cur);
		data=cur->data;

		if((func)(data,user_data)==TRUE)
		{
			*list=g_list_remove(*list,data);
		}
		cur=nxt;
	}
}

/********************************/
/* function called at each loop */
/********************************/
static void hc_ged_always1(const struct ged_callbacks *ged)
{
	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach_remove(&cluster_entry_list,process_cluster_entry,NULL);
	HL_UNLOCK_READ(cluster_entry_lock);
}

static GED_CALLBACKS hub_cnx_entry_ged=
					{
						"hub cnx ged",
						hc_ged_exit,
						NULL,					/* we have no descriptor to add */
						NULL,					/* and none to scan */
						NULL,
						hc_ged_always1,
						NULL,
						NULL,
					};

static GLOBAL_USER_CALLBACKS hub_cnx_entry_glus=
					{
						"hub cnx glus",
						hc_send_to_named_user,              /* -> CC_UCAST */
						hc_send_to_all_users,               /* -> CC_BCAST */
						hc_glus_get_user_info,              /* internal */
						hc_glus_get_users_info,             /* internal */
						hc_glus_get_users_info_by_ip,       /* internal */
						hc_glus_get_nick_lists,             /* internal */
						hc_add_hub_users_stat,              /* internal */
						hc_kick_named_user,                 /* -> CC_USER_KICK */
						hc_ban_named_user,                  /* -> CC_USER_BAN */
						hc_ban_ip,                  				/* -> for ban by ip (to be code) */
						hc_uban_named_user,                 /* -> CC_USER_UBAN */
						hc_disconnect_named_user,           /* -> CC_USER_DISC */
						hc_user_is_disconnected,            /* nothing to do */
						hc_user_do_hello,                   /* -> CC_USER_IN */
						hc_user_do_quit,                    /* -> CC_USER_OUT */
						hc_user_do_myinfo,                  /* -> CC_USER_IN */
						hc_user_do_xsearch,                 /* -> CC_XSEARCH */
						hc_md4set_received,						/* -> CC_MD4SET */
					};

/************************************************************/
/* function initializing the handler of all the connections */
/************************************************************/
GED_CALLBACKS *hub_cnx_entry_handler_init(void)
{
   cluster_entry_list=NULL;

	global_user_register(&hub_cnx_entry_glus);
	
   return &hub_cnx_entry_ged;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------ misc functions ---------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static gint compare_clus_entry_with_clus_id(gconstpointer a, gconstpointer b)
{
	const CLUSTER_ENTRY *p=a;
	const guint8 *clus_id=b;

	return memcmp(p->cluster_id,clus_id,MD_BLOC_SIZE);
}

/*************************************/
/* find a cluster entry from its ID  */
/* output: the CLUSTER_ENTRY or NULL */
/*************************************/
static CLUSTER_ENTRY *get_cluster_entry_from_id(guint8 *clus_id)
{
	GList *out;

	out=g_list_find_custom(cluster_entry_list,clus_id,compare_clus_entry_with_clus_id);
	if(out==NULL)
		return NULL;
	return out->data;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
typedef struct
{
	guint8 *hub_id;
	gboolean found;
} FIND_HUB_ID_AND_BOOL;

/*****************************************************/
/* search inside a cluster a hub having the given id */
/*******************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                        */
/*        user_data= FIND_HUB_ID_AND_BOOL pointer                  */
/*******************************************************************/
/* NOTE: the array has LOCK_READ set */
/*************************************/
static void find_an_hub_id(gpointer value, gpointer user_data)
{
	int i;
	FIND_HUB_ID_AND_BOOL *fhi=user_data;
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;

	if(fhi->found==TRUE)	/* already found ? */
		return;

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));
		if(!memcmp(fhi->hub_id,hce->remote_hub_id,HUB_ID_LEN))
		{
			fhi->found=TRUE;
			break;
		}
	}
}

/************************************************************/
/* search in all clusters if a hub with the given id exists */
/************************************************************/
/* output: TRUE= the hub is here      */
/*         FALSE= the hub is not here */
/**************************************/
gboolean hub_connected(guint8 *hub_id)
{
	FIND_HUB_ID_AND_BOOL fhi;

	fhi.hub_id=hub_id;
	fhi.found=FALSE;

	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach(cluster_entry_list,find_an_hub_id,&fhi);
	HL_UNLOCK_READ(cluster_entry_lock);

	return fhi.found;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
typedef struct
{
	const char *nickname;
	HUB_CNX_ENTRY *hce;
} FIND_USER_AND_HUB;

/*****************************************************/
/* search inside a cluster a hub having the given id */
/*******************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                        */
/*        user_data= FIND_USER_AND_HUB pointer                     */
/*******************************************************************/
/* NOTE: the array has LOCK_READ set */
/*************************************/
static void find_a_hub_with_a_user(gpointer value, gpointer user_data)
{
	int i;
	FIND_USER_AND_HUB *fuah=user_data;
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;

	if(fuah->hce!=NULL)	/* already found ? */
		return;

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));
		if(g_hash_table_lookup(hce->hub_users,fuah->nickname)!=NULL)
		{
			fuah->hce=hce;
			break;
		}
	}
}

/**********************************************************/
/* search in all clusters a hub having the given nickname */
/**********************************************************/
/* output: the HUB_CNX_ENTRY or NULL */
/*************************************/
static HUB_CNX_ENTRY *find_hub_cnx_entry_with_a_user(const char *nickname)
{
	FIND_USER_AND_HUB fuah;

	fuah.nickname=nickname;
	fuah.hce=NULL;

	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach(cluster_entry_list,find_a_hub_with_a_user,&fuah);
	HL_UNLOCK_READ(cluster_entry_lock);

	return fuah.hce;
}


/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************************************/
/* send a prebuild chunk to all hubs of a cluster */
/*******************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                        */
/*        user_data= GByteArray * pointer                          */
/*******************************************************************/
/* NOTE: the array has LOCK_READ set */
/*************************************/
static void send_chunk_to_all_hubs_hub(gpointer value, gpointer user_data)
{
	int i;
	GByteArray *gba=user_data;
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));

		if(g_list_find(excluded_hub_entry,hce)!=NULL)	/* excluded hub */
			break;

		bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,gba,TRUE);	/* copy the chunk */
	}
}

/*****************************************************************/
/* send a prebuilt chunk to all known hubs (except excluded_one) */
/*****************************************************************/
/* Note: the given GByteArray reference is stolen, don't free it */
/*****************************************************************/
static void send_chunk_to_all_hubs(GByteArray *gba)
{
	if(exclude_all_clusters==FALSE)
	{
		HL_LOCK_READ(cluster_entry_lock);
		g_list_foreach(cluster_entry_list,send_chunk_to_all_hubs_hub,gba);
		HL_UNLOCK_READ(cluster_entry_lock);
	}
	g_byte_array_free(gba,TRUE);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************************************************/
/* append a cluster statistic to the given gstring */
/***************************************************/
/* entry: value= CLUSTER_ENTRY * to process */
/*        user_data= GString *              */
/********************************************/
/* NOTE: the array has LOCK_READ set */
/*************************************/
static void hub_cnx_append_cluster_stat_one(gpointer value, gpointer user_data)
{
	int i;
	GString *dest=user_data;
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;

	g_string_append(dest,"Cluster ID: ");
	append_MD_to_str(dest,ce->cluster_id,HUB_ID_LEN);
	g_string_append(dest,"\r\n");

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));

		g_string_append(dest,"  HubID: ");
		append_MD_to_str(dest,hce->remote_hub_id,HUB_ID_LEN);
		g_string_append(dest,"\r\n  Hubname: ");
		g_string_append(dest,hce->remote_hub_name);
		g_string_sprintfa(dest,"\r\n  #users: %d\r\n",g_hash_table_size(hce->hub_users));
	}
}

/*******************************************************************/
/* append cluster and cluster's hub statistic to the given gstring */
/*******************************************************************/
void hub_cnx_append_cluster_stats(GString *output)
{
	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach(cluster_entry_list,hub_cnx_append_cluster_stat_one,output);
	HL_UNLOCK_READ(cluster_entry_lock);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
#if 0
typedef struct
{
	void (*fnc)(HUB_CNX_ENTRY *hce,void *param);
	void *param;
} CALL_EACH;

/*********************************************/
/* call a function for each not excluded hub */
/*******************************************************************/
/* entry: value= CLUSTER_ENTRY * to process                        */
/*        user_data= CALL_EACH *                                   */
/*******************************************************************/
/* NOTE: the array has LOCK_READ set */
/*************************************/
static void call_fnc_for_each_hub_hub(gpointer value, gpointer user_data)
{
	int i;
	CALL_EACH *calle=user_data;
	CLUSTER_ENTRY *ce=value;
	HUB_CNX_ENTRY *hce;

	for(i=0;i<ce->hc_array->len;i++)
	{
		hce=&(g_array_index(ce->hc_array,HUB_CNX_ENTRY,i));

		if(g_list_find(excluded_hub_entry,hce)!=NULL)	/* excluded hub */
			break;

		(calle->fnc)(hce,(calle->param));
	}
}

/********************************************************/
/* call the given function for each hub of each cluster */
/* excluded hubs are ignored. No cluster is ignored.    */
/********************************************************/
static void call_fnc_for_each_hub(void (*fnc)(HUB_CNX_ENTRY *hce,void *param),void *param)
{
	CALL_EACH ce;

	ce.fnc=fnc;
	ce.param=param;

	HL_LOCK_READ(cluster_entry_lock);
	g_list_foreach(cluster_entry_list,call_fnc_for_each_hub_hub,&ce);
	HL_UNLOCK_READ(cluster_entry_lock);
}
#endif
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
