/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hub_cnx_connect.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hub_cnx_connect.c,v 2.5 2003/09/21 09:41:02 blusseau Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_SYS_POLL_H
#include <sys/poll.h>
#else
#include "my_poll.h"
#endif

#ifdef WIN32
	#include <windows.h>
	#include <winsock2.h>
#else
	#include <sys/types.h>
	#include <sys/socket.h>
#endif

#include <glib.h>

#include "config.h"
#include "main.h"

#include "ged_if.h"
#include "network.h"
#include "gvar.h"
#include "hub_cnx_connect.h"
#include "hub_passwd.h"
#include "hub_cnx_handshake.h"
#include "hub_cnx_lst.h"

typedef struct
{
	char *hub_address;
	int sock_fd;
	time_t attempt_time;
} HUB_ATTEMPT;

static GArray *hub_cnx_attempt=NULL;		/* array of HUB_ATTEMPT */

#if 0
/* ************************************************************************** */
/* ************************************************************************** */
/* ************************************************************************** */
/* ******************* connection acceptation handler *********************** */
/* ************************************************************************** */
/* ************************************************************************** */
/* ************************************************************************** */

/*****************************************************************/
/* this socket fd is the one used to accept incoming connections */
/*****************************************************************/
static int hub_sck;

static void	new_incoming_connection(void)
{
	int sock_fd;
	struct sockaddr_in remote;
	int x=sizeof(remote);
	char *dummy1;
	int dummy_len;

	sock_fd=accept(hub_sck,(struct sockaddr *)&remote,&x);
	if(sock_fd==-1)
		return;

	/* we first check if someone doesn't try to flood the hub with dummy connection */
	if(get_tos_entry(CNX_TOSKEY,(char *)&(remote.sin_addr),sizeof(remote.sin_addr),0,&dummy1,&dummy_len)==1)
	{
		/*printf(stderr,"Connection attempt from the same ip in less than 10 seconds.\n"); */
		shutdown(sock_fd,SHUT_RDWR);
		close(sock_fd);
		return;
	}

	if(get_tos_entry(BAN_TOSKEY,(char *)&(remote.sin_addr),sizeof(remote.sin_addr),0,&dummy1,&dummy_len)==1)
	{
		/* fprintf(stderr,"Connection attempt from a banned ip.\n"); */
		shutdown(sock_fd,SHUT_RDWR);
		close(sock_fd);
		return;
	}

	/* put the address of the remote host in the CNX tos list */
	add_tos_entry(CNX_TOSKEY,10,(char *)&(remote.sin_addr),sizeof(remote.sin_addr),NULL,0);

	hub_handshake_register_new_fd(sock_fd,&(remote.sin_addr));
}
#endif

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------------- main functions ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***********************************************/
/* search a running attempt by its hub_address */
/***************************************************/
/* output: -1=not found                            */
/*         else index in the hub_cnx_attempt array */
/***************************************************/
static int attempt_idx_by_hub_address(const char *hub_address)
{
	int i;

	for(i=0;i<hub_cnx_attempt->len;i++)
	{
		if(!strcmp(g_array_index(hub_cnx_attempt,HUB_ATTEMPT,i).hub_address, hub_address) )
			return i;
	}
	return -1;
}

static int comp_hub_cnx_attempt(const void *a, const void *b)
{
	return ((HUB_ATTEMPT*)a)->sock_fd - ((HUB_ATTEMPT*)b)->sock_fd;
}

/*************************************************************************/
/* check if the given hub row is valid, autostart and not yet in attempt */
/*************************************************************************/
static void try_hub_cnx_row(HUB_PASS_ROW *hpr)
{
	HUB_ATTEMPT ha;
	struct sockaddr_in addr;
	int ret;

	if(hpr->entry_flags!=HPR_EF_BUSY)	/* populated entry ? */
		return;
	if((hpr->special_flags&HPR_SF_AUTO_START)==0)	/* in autostart ? */
		return;
	if((hpr->special_flags&INVALID_HPR_SF)!=0)	/* in error ? */
		return;
	if((hpr->last_connection_attempt+MIN_DELAY_BETWEEN_2_ATTEMPTS)>gl_cur_time)	/* already recently attempted */
		return;
	if(hub_connected(hpr->hub_id)==TRUE)	/* hub already connected ? */
		return;
	if(attempt_idx_by_hub_address(hpr->hub_address)!=-1)	/* attempt already in progress ? */
		return;


	/* well, it's time to try this row */
	hub_passwd_touch_last_attempt_from_hub_address(hpr->hub_address);

	/* only this part of the code can hang the hub */
	{
		char *dup_addr;
		char *t;
		addr.sin_family=AF_INET;
		dup_addr=strdup(hpr->hub_address);
		t=strchr(dup_addr,':');
		if(t!=NULL)
		{
			*t++='\0';
			addr.sin_port=htons(atoi(t));
		}
		else
			addr.sin_port=htons(410);		/* standard port number -1 */

		if(str_to_inaddr(dup_addr,&(addr.sin_addr)))
		{
			fprintf(stderr,"try_hub_cnx_row: unable to resolv %s. Will retry later\n",dup_addr);
			free(dup_addr);
			return;
		}
		free(dup_addr);
	}

	/* create a socket and try to bind it */
	ha.hub_address=strdup(hpr->hub_address);
	ha.attempt_time=gl_cur_time;
	ha.sock_fd=socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if(ha.sock_fd==-1)
	{
		perror("try_hub_cnx_row:socket:");
		return;
	}

	set_non_bloquant_sock(ha.sock_fd);
	ret=connect(ha.sock_fd,(struct sockaddr *)&addr,sizeof(addr));
	if((ret==-1)&&(errno!=EINPROGRESS))
	{
		fprintf(stderr,"try_hub_cnx_row: connect %s -> %s\n",hpr->hub_address,strerror(errno));
		close(ha.sock_fd);
		return;
	}
	
	/* add the new entry to the array, we just have to wait for a WRITE ready now */
	g_array_append_val(hub_cnx_attempt,ha);

	/* sort to speed up future scan */
	qsort(hub_cnx_attempt->data,hub_cnx_attempt->len,sizeof(HUB_ATTEMPT),comp_hub_cnx_attempt);
}

/***************************************************/
/* handle an event occuring on a hub_attempt entry */
/**************************************************************/
/* output: on succes or error, do what you want with sock_fd  */
/*         and then, set it to -1 to force destruction of the */
/*         entry                                              */
/**************************************************************/
static void hub_cnx_attempt_done(HUB_ATTEMPT *ha, int fd_events)
{
	if(fd_events&POLLOUT)
	{
		hub_handshake_register_new_fd(ha->sock_fd,NULL);
		ha->sock_fd=-1;		/* everything is done */
		return;
	}

	close(ha->sock_fd);
	ha->sock_fd=-1;		/* on error, we discard the entry */
	return;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GED handlers ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************/
/* function called at end */
/**************************/
static int hub_cnx_connect_exit(const GED_CALLBACKS *ged)
{
	return 0;
}

/****************************/
/* add FD event to wait for */
/****************************/
static void hub_cnx_connect_add_fd(const GED_CALLBACKS *ged, GArray *struct_pollfd_array)
{
	int i;
	struct pollfd nw;

	for(i=0;i<hub_cnx_attempt->len;i++)
	{
		nw.fd=g_array_index(hub_cnx_attempt,HUB_ATTEMPT,i).sock_fd;
		nw.events=POLLOUT;
		nw.revents=0;

		g_array_append_val(struct_pollfd_array,nw);
	}
}

/*******************************/
/* check for the arrived event */
/*******************************/
static void hub_cnx_connect_scan_fd(const GED_CALLBACKS *ged, GArray *struct_pollfd_array, int low_range, int high_range)
{
	struct pollfd *nw;
	int i;
	int minus=0;
	HUB_ATTEMPT *ha;

	i=low_range;
	while(i<high_range)
	{
		int idx;

		nw=&(g_array_index(struct_pollfd_array,struct pollfd,i));

		/* now, we must find the xfio with the same sock_fd */
		/* it is pretty easy, its index (idx) is >=xfio_minus, <xf_io_array->len and sock_fd[idx] <=nw->sock_fd */
		idx=minus;
		while(idx<hub_cnx_attempt->len)
		{
			ha=&(g_array_index(hub_cnx_attempt,HUB_ATTEMPT, idx));
			if(ha->sock_fd==nw->fd)
			{
				/* we have found it */
				minus=idx+1;	 /* the next time, start after this idx */
				if(nw->revents&(POLLOUT|POLLERR|POLLHUP))
				{
					/* process outgoing data */
					hub_cnx_attempt_done(ha, nw->revents);
				}
				break;
			}

			if(ha->sock_fd>nw->fd)
			{  /* the fd does not exists, we are already above what we search for */
				minus=i;	  /* the next time, start here */
				break;
			}
			idx++;
		}
		if(idx>=hub_cnx_attempt->len)  /* end of array reached */
			break;						/* all other fd will be unavailable */

		i++;
	}

	/* now, purge deleted entry of the hub_cnx_attempt array */
	i=0;
	while(i<hub_cnx_attempt->len)
	{
		if(g_array_index(hub_cnx_attempt,HUB_ATTEMPT, i).sock_fd==-1)
		{
			free(g_array_index(hub_cnx_attempt,HUB_ATTEMPT, i).hub_address);
			g_array_remove_index(hub_cnx_attempt,i);		/* keep the array sorted */
		}
		else
			i++;
	}
}

/********************************/
/* function called periodically */
/********************************/
static void hub_cnx_connect_periodic(const struct ged_callbacks *ged)
{
	static int one_of_six=0;

	one_of_six++;
	if(one_of_six==6)
	{
		HUB_PASS_ROW *hpr;
		int nb_rows;
		int i;

		/* this function is called only 1 of 6 consecutives calls of the periodic function */
		one_of_six=0;
		
		hpr=ucmd_hp_get_rows(&nb_rows);
		if(hpr!=NULL)
		{
			for(i=0;i<nb_rows;i++)
			{
				try_hub_cnx_row(&hpr[i]);
			}
			free(hpr);
		}
	}
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------- initialization of the handler --------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static GED_CALLBACKS hub_cnx_connect_ged=
				{
					"hub_cnx_connect GED", /* ged_name For debug */
					hub_cnx_connect_exit, /* destructor */
					hub_cnx_connect_add_fd, 
					hub_cnx_connect_scan_fd,
               hub_cnx_connect_periodic,
               NULL,
               NULL,
					NULL
				};

/*********************************************/
/* function initializing the hub_cnx_connect */
/*********************************************/
GED_CALLBACKS *hub_cnx_connect_init(void)
{
	hub_cnx_attempt=g_array_new(FALSE,FALSE,sizeof(HUB_ATTEMPT));
	return &hub_cnx_connect_ged;
}

