/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hub_cmd.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
  $Id: hub_cmd.c,v 2.59 2003/11/22 16:30:55 blusseau Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
	#include <windows.h>
	#include <winsock2.h>
	#include <getopt.h>
#else
	#ifdef HAVE_SYS_TIME_H
		#include <sys/time.h>
	#endif
 	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#ifdef HAVE_UNISTD_H
		#include <unistd.h>
	#endif  /* HAVE_UNISTD_H */
#endif

#ifdef HAVE_ERRNO_H
	#include <errno.h>
#else
	extern int errno;
#endif

#include <glib.h>
#include "hub_cmd.h"
#include "gvar.h"
#include "macro.h"
#include "users_xml.h"
#include "db_xml.h"
#include "toolkit.h"
#include "timed_out_string.h"
#include "tos_key.h"
#include "main.h"
#include "multi_public_chat_cmd.h"
#include "emb_perl.h"
#include "user_cnx_lst.h"
#include "global_user_if.h"
#include "hub_passwd.h"
#include "md5.h"
#include "hub_cnx_lst.h"
#include "user_cnx_lst.h"

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>    /* Gray Watson's library */
#define show_alloc() dmalloc_log_unfreed()
#define show_stats() dmalloc_log_stats()
#else
#define show_alloc()    /* nothing */
#define show_stats()    /* nothing */
#endif

#ifndef INADDR_ANY
#define INADDR_ANY      0x00000000
#endif
#ifndef INADDR_NONE
#define INADDR_NONE     0xFFFFFFFF
#endif

#ifdef __CYGWIN__
#define GLIB_INT_64_BUG
#endif

#ifdef WIN32
#define GLIB_INT_64_BUG
#endif

#ifdef GLIB_INT_64_BUG
char	*llunsigned_to_str(guint64 a);
#endif


static int do_reload(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	load_dyn_conf();
	g_string_sprintfa((*dest_base),"Reload done|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return (0);
}

#ifdef DEBUG

static int do_dump(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{

	if(splitted_str->len<2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		int i;
		const char *user;
		CNX_ENTRY *ptr2;

		for(i=1; i<splitted_str->len; i++)
		{
			user=g_ptr_array_index(splitted_str,i);
			if(user!=NULL)
			{
				GString *tmp;

				ptr2 = find_cnx_by_nickname(user); 
				tmp = g_string_new((*dest_base)->str);
				
				if (ptr2 == NULL)
					g_string_sprintfa(tmp, "nick : %s not found in strucure|", user);
				else
				{
#ifdef GLIB_INT_64_BUG
					g_string_sprintfa(tmp, "user_nick=%s, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, share=%s, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",
#else
					g_string_sprintfa(tmp, "user_nick=%s, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, share=%Lu, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",
#endif					
									ptr2->user_nick->str,
									ptr2->sock_fd,
									ptr2->user_cnx_type,
									ptr2->is_op,
									inet_ntoa(ptr2->ip),
#ifdef GLIB_INT_64_BUG
									llunsigned_to_str(ptr2->shared_size),
#else
									ptr2->shared_size,
#endif
									ptr2->incoming_commands,
									ptr2->outgoing_commands,
									ptr2->server_id,
									ptr2,
									ptr2->next);
				}
				send_const_str_to_luce(luce,tmp->str);
				g_string_free(tmp, TRUE);
			}
		}
	}
	return (0);
}

static int do_dump_queue(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	CNX_ENTRY	 *ptr2;
	GString	*tmp;

	for (ptr2=cnx_lst->wait; ptr2; ptr2 = ptr2->next)
	{
		tmp = g_string_new((*dest_base)->str);
		if (ptr2->handshake_done==1)
		{
			g_string_sprintfa(tmp,"user_nick=%s, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",
						ptr2->user_nick->str,
						ptr2->sock_fd,
						ptr2->user_cnx_type,
						ptr2->is_op,
						inet_ntoa(ptr2->ip),
						ptr2->incoming_commands,
						ptr2->outgoing_commands,
						ptr2->server_id,
						ptr2,
						ptr2->next);
		}
		else
		{
			g_string_sprintfa(tmp,"%duser_nick=?????, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",ptr2->connected,
						ptr2->sock_fd,
						ptr2->user_cnx_type,
						ptr2->is_op,
						inet_ntoa(ptr2->ip),
						ptr2->incoming_commands,
						ptr2->outgoing_commands,
						ptr2->server_id,
						ptr2,
						ptr2->next);
		}
		send_const_str_to_luce(luce,tmp->str);
		g_string_free(tmp, TRUE);
	}
	return(0);
}

#endif

static int do_set(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s. Usage : set var value\r\n|",cmd);
	else if (!strcmp(g_ptr_array_index(splitted_str,1), "MODERATE"))
	{
		printf("(%s)", (char *)g_ptr_array_index(splitted_str,2));
		sscanf(g_ptr_array_index(splitted_str,2), "%d", &gl_moderate_mode);
		if (gl_moderate_mode)
			g_string_sprintfa((*dest_base)," moderate mode enabled\r\n|");
		else
			g_string_sprintfa((*dest_base)," moderate mode disabled\r\n|");
	}
	else
	{
		g_string_sprintfa((*dest_base),"arg 1 in MODERATE\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return(0);
}

/****************************/
/* get ip from a named user */
/****************************/
static int do_get_ip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s user_name\r\n|",cmd);
	}
	else
	{
		const char *user;

		user=g_ptr_array_index(splitted_str,1);
		if(user!=NULL)
		{
			GLUS_USER_INFO *gui;

			gui=glus_get_user_info(user);
			if (gui)
			{
				g_string_sprintfa((*dest_base),"ip:%s %s\r\n|", gui->user_nick->str, inet_ntoa(gui->user_ip));
				glus_free_user_info(gui);
			}
			else
			{
				g_string_sprintfa((*dest_base),"user %s not found.\r\n|", user);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************************/
/* give the username having the same IP */
/**********************************-*****/
static int do_rev_ip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s user_name\r\n|",cmd);
	}
	else
	{
		const char *str_ip;
		str_ip=g_ptr_array_index(splitted_str,1);

		if(str_ip!=NULL)
		{
			int i;
			struct in_addr ip;

			if (inet_aton(str_ip,&ip) == 0)	/* address invalid ? */
				g_string_sprintfa((*dest_base),"unable to parse %s\r\n|", str_ip);
			else
			{
				GPtrArray *gui_array;

				gui_array=glus_get_users_info_by_ip(ip);
				for(i=0;i<gui_array->len;i++)
				{
					GLUS_USER_INFO *gui;

					gui=g_ptr_array_index(gui_array,i);
					g_string_sprintfa((*dest_base),"rev:%s %s\r\n",inet_ntoa(gui->user_ip), gui->user_nick->str);
				}

				glus_free_user_info_ptrarray(gui_array);
				g_string_append((*dest_base),"end of list\r\n|");
			}
		}
		else
		{
			g_string_sprintfa((*dest_base),"internal error in %s\r\n|", __FILE__);
		}
	} 
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************/
/* set a new hub password entry */
/********************************/
static int do_hpset(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 5)		/* at least 4 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id cluster_id hub_addr flags\r\n|",cmd);
	}
	else
	{
		HUB_PASS_ROW nw_hpr;
		gchar **fields;

		if(strlen(g_ptr_array_index(splitted_str,1))!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
			goto abrt;
		}

		if(strlen(g_ptr_array_index(splitted_str,2))!=(2*MD_BLOC_SIZE))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  cluster_id is a %d characters string|",cmd,MD_BLOC_SIZE*2);
			goto abrt;
		}

		nw_hpr.entry_flags=HPR_EF_BUSY;
		id_ascii_to_bin(g_ptr_array_index(splitted_str,1),nw_hpr.hub_id);
		id_ascii_to_bin(g_ptr_array_index(splitted_str,2),nw_hpr.cluster_id);
		strncpy_max(nw_hpr.hub_address, g_ptr_array_index(splitted_str,3), sizeof(nw_hpr.hub_address));
		memset(nw_hpr.reserved,0,sizeof(nw_hpr.reserved));
		nw_hpr.special_flags=0;

		fields=g_strsplit(g_ptr_array_index(splitted_str,4),".",0);
		if(fields)
		{
			char **flag;
			char *option;

			flag=fields;
			while((option=*flag++)!=NULL)
			{
				if(!strcmp(option,"None"))
					continue;
				if(!strcmp(option,"Autostart"))
				{
					nw_hpr.special_flags|=HPR_SF_AUTO_START;
					continue;
				}
				if(!strcmp(option,"Rejected"))
				{
					nw_hpr.special_flags|=HPR_SF_REJECTED;
					continue;
				}

				g_string_sprintfa((*dest_base),"Unkown special flag '%s' (ignored)\r\n",option);
			}
			g_strfreev(fields);
		}
		
		ucmd_hp_set(&nw_hpr);
		g_string_sprintfa((*dest_base),"%s done\r\n|",cmd);
	}
	abrt:
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**************************************************************************************************/
/* convert a HUB_PASS_ROW structure into a human readable form and append it to the given gstring */
/**************************************************************************************************/
static void append_decoded_hpr(GString *dest_base, HUB_PASS_ROW *hpr)
{
	if(hpr->entry_flags==HPR_EF_EMPTY)	/* the entry is empty */
		return;

	g_string_append(dest_base,"HubID: ");
	append_MD_to_str(dest_base,hpr->hub_id,HUB_ID_LEN);
	g_string_append(dest_base,"\r\n   ClusterID: ");
	append_MD_to_str(dest_base,hpr->cluster_id,HUB_ID_LEN);
	g_string_append(dest_base,"\r\n   HubAddr: ");
	g_string_sprintfa(dest_base,"%s\r\n   Flags:",hpr->hub_address);
	
	if(hpr->special_flags==0)
		g_string_append(dest_base," None\r\n");
	else
	{
		if(hpr->special_flags&HPR_SF_AUTO_START)
			g_string_append(dest_base," AutoStart");
		if(hpr->special_flags&HPR_SF_REJECTED)
			g_string_append(dest_base," Rejected(invalid_cluster_id)");
		g_string_append(dest_base,"\r\n");
	}
}

/****************************/
/* get a hub password entry */
/****************************/
static int do_hpget(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		char *str_id=g_ptr_array_index(splitted_str,1);

		if(strlen(str_id)!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
		}
		else
		{
			HUB_PASS_ROW *hpr;
			guint8 hub_id[HUB_ID_LEN];

			/* convert the ascii ID to its binary version */
			id_ascii_to_bin(str_id,hub_id);

			hpr=ucmd_hp_get(hub_id);
			if(hpr==NULL)
			{
				g_string_sprintfa((*dest_base),"No known hub with the ID %s\r\n|",str_id);
			}
			else
			{
				append_decoded_hpr((*dest_base),hpr);
				g_string_append_c((*dest_base),'|');

				free(hpr);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*******************************/
/* delete a hub password entry */
/*******************************/
static int do_hpdel(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		char *str_id=g_ptr_array_index(splitted_str,1);

		if(strlen(str_id)!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
		}
		else
		{
			gboolean erased;
			guint8 hub_id[HUB_ID_LEN];

			/* convert the ascii ID to its binary version */
			id_ascii_to_bin(str_id,hub_id);

			erased=ucmd_hp_del(hub_id);
			if(erased==FALSE)
			{
				g_string_sprintfa((*dest_base),"Fail to delete the hub with the ID %s\r\n|",str_id);
			}
			else
			{
				g_string_sprintfa((*dest_base),"hub with the ID %s was deleted\r\n|",str_id);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************************/
/* schedule a hub entry for connection */
/***************************************/
static int do_hpcnx(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		char *str_id=g_ptr_array_index(splitted_str,1);

		if(strlen(str_id)!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
		}
		else
		{
			gboolean erased;
			guint8 hub_id[HUB_ID_LEN];

			/* convert the ascii ID to its binary version */
			id_ascii_to_bin(str_id,hub_id);

			erased=ucmd_hp_reset_last_cnx_attempt(hub_id);
			if(erased==FALSE)
			{
				g_string_sprintfa((*dest_base),"The hub with the ID %s was not found\r\n|",str_id);
			}
			else
			{
				g_string_sprintfa((*dest_base),"hub with the ID %s was scheduled for connection attempt in the next 60 seconds\r\n|",str_id);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************/
/* get all hub password entries */
/********************************/
static int do_hpgetall(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 1)		/* at least the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s\r\n|",cmd);
	}
	else
	{
		HUB_PASS_ROW *hpr;
		int nb_rows;
		int i;

		hpr=ucmd_hp_get_rows(&nb_rows);
		if(nb_rows==0)
		{
			g_string_sprintfa((*dest_base),"No known hub ID\r\n|");
		}
		else
		{
			for(i=0;i<nb_rows;i++)
			{
				append_decoded_hpr((*dest_base),&(hpr[i]));
			}
			g_string_append_c((*dest_base),'|');
			free(hpr);
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**************************************/
/* get all running cluster connection */
/**************************************/
static int do_hpstat(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 1)		/* at least the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s\r\n|",cmd);
	}
	else
	{
		g_string_sprintfa((*dest_base),"Cluster connections:\r\n");
		
		hub_cnx_append_cluster_stats(*dest_base);

		g_string_sprintfa((*dest_base),"End of list.\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************************/
/* compute the MD5 hash of the given string */
/********************************************/
static int do_md5cmd(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		guint8 md[MD_BLOC_SIZE];
		char *str=g_ptr_array_index(splitted_str,1);

		do_md5(str,strlen(str),md);

		g_string_sprintfa((*dest_base),"MD5hash of '%s' is ",str);
		append_MD_to_str((*dest_base),md,MD_BLOC_SIZE);

		g_string_sprintfa((*dest_base),"\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/************************************/
/* display the list of all commands */
/************************************/
static int do_help(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	g_string_sprintfa((*dest_base),"** Operator command list:\r\n"
					  "\r\n"
					  "-help                         display this help.\r\n"
					  "+help                         normal user help.\r\n");
	
	if (luce->level >= get_right_level(KICK))
        g_string_sprintfa((*dest_base), "\r\n"
                          "-kick {nick OR ip} [reason]   kick the given user out of the hub.\r\n"
                          "-skick {nick OR ip} [reason]  silent kick (Without printing anything on the chat).\r\n");

	if (luce->level >= get_right_level(REDIRECT))
        g_string_sprintfa((*dest_base), "\r\n"
                          "-redir nick address reason    redirect the given user to this address.\r\n");
	
	if (luce->level >= get_right_level(OPERATOR))
        g_string_sprintfa((*dest_base), "\r\n"
						  "-bcast msg                    send the given message to everyone.\r\n"
                          "-version nick                 the client-version used by the given user.\r\n"
                          "\r\n"
                          "-getip nick                   Get IP of a user.\r\n"
                          "-revip ip                     Reveal's all the users having the given IP.\r\n");
	
	if (luce->level >= get_right_level(BAN))
		g_string_sprintfa((*dest_base), "\r\n"
						  "-ban duration {nick OR ip} [reason]\r\n"
						  "                              ban the given user for the given time.\r\n"
						  "                              The duration can be suffixed by 'd' for day or\r\n"
						  "                              'h' for hour. By default, it is in minutes.\r\n");
	if (luce->level >= get_right_level(UNBAN))
		g_string_sprintfa((*dest_base), "-unban {nick OR ip} [{nick OR ip}...]\r\n");
	
	if (luce->level >= get_right_level(BAN) || luce->level >= get_right_level(UNBAN))
		g_string_sprintfa((*dest_base), "-banlist                      print the list of banned ip(s).\r\n");

	if (luce->level >= get_right_level(UNBAN))
		g_string_sprintfa((*dest_base),	"-clearban                     clear the banlist.\r\n");
	
	if (luce->level == MASTER_LEVEL)
		g_string_sprintfa((*dest_base), "\r\n"
						  "-perlclear                    clear all perl autoloaded handlers.\r\n"
						  "-perlreset                    restart perl interpreter.\r\n"
						  "-pluginstart plugin_name      start named plugin.\r\n"
						  "-pluginstop plugin_name       stop named plugin.\r\n");
	
	if (luce->level >= get_right_level(MNGUSERS))
		g_string_sprintfa((*dest_base), "\r\n"
						  "-ulist                        display all existing accounts.\r\n"
						  "-add nick passwd {level OR master}\r\n"
						  "                              create a new registered user.\r\n"
						  "-del nick                     delete a registered user.\r\n"
						  "-rename oldnick newnick       rename a registered user.\r\n"
						  "-passwd nick newpasswd        change the password of a registered user.\r\n"
						  "-access nick [level]          show or set the level of a user account.\r\n"
						  "-protect nick                 declare the user as a protected user.\r\n"
						  "-unprotect nick               declare the user as an unprotected user.\r\n");
	
	if (luce->level >= get_right_level(MNGDB))
		g_string_sprintfa((*dest_base), "\r\n"
						  "-dblist                       display database.\r\n"
						  "-dbshow key                   display one database key.\r\n"
						  "-dbadd key type value         create a new key in the database.\r\n"
						  "-dbset key value              change the value of a key in the database.\r\n"
						  "-dbdel key                    delete a key of the database.\r\n"
						  "-reload                       update some variables in the hub.\r\n"
						  "\r\n"
						  "-set VARIABLE VALUE           custom function (new).\r\n");

	if (luce->level == MASTER_LEVEL)
		g_string_sprintfa((*dest_base), "\r\n"	
						  "-startprg name                start the given external program (-e required)\r\n"
						  "                              during hub start and name must be a valid name\r\n"
						  "                              in this directory.\r\n");
	if (luce->level >= get_right_level(REDIRECT))
		g_string_sprintfa((*dest_base), "\r\n"	
						  "-redirall address             redirect all users to this address.\r\n"
						  "-rediralllocal address        redirect all local users to this address.\r\n");
	
#ifdef DEBUG
	g_string_sprintfa((*dest_base), "\r\n"
					  "-dump user [users...]         dump data from the internal structure (use with care).\r\n"
					  "-dumpqueue                    dump login queue list (use with care).\r\n");
#endif
					  
	if (luce->level >= get_right_level(MNGCLUSTERS))
		g_string_sprintfa((*dest_base), "\r\n"
						  "*** cluster command list:\r\n"
						  "-hpset hub_id cluster_id hub_address flags\r\n"
						  "                              set an entry in the hub password file for the given hub.\r\n"
						  "                              hub_address can be either a FQDN or an IP. hub_id is the HUB_ID\r\n"
						  "                              key of the remote hub. cluster_id is MD5 of the cluster name.\r\n"
						  "                              flags value contains at least one of the following keywords.\r\n"
						  "                              keywords are separated by a dot (.) and contains no space.\r\n"
						  "                              Keywords: None, Autostart, Rejected.\r\n"
						  "-hpdel hub_id                 remove an entry in the hub password file for the given hub.\r\n"
						  "-hpcnx hub_id                 for a connection attempt for the given hub.\r\n"
						  "                              Note: the attempt will occur in the next 60 seconds.\r\n"
						  "-hpget hub_id                 retrieve the hub password file entry having the given hub_id.\r\n"
						  "-hpgetall                     retrieve all the hub password file entries.\r\n"
						  "-hpstat                       list the established cluster connections.\r\n"
						  "-md5 string                   compute the MD5 of the given string. Use this command to compute\r\n"
						  "                              the cluster ID from the cluster name for example.\r\n");
					
	g_string_sprintfa((*dest_base), "\r\n"
					  "\r\n"
					  "Note: If a parameter contains spaces, put the parameter between quotes.\r\n"
					  "Note2: A quoted parameter cannot contains quote itself.\r\n"
					  "\r\n|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/******************************/
/* function use to sort ulist */
/******************************/
gint compare_ulist (gconstpointer a, gconstpointer b) {
	return strcasecmp((*(GString**)a)->str, (*(GString**)b)->str);
}

/***************************************************************************************/
/* display the list of all existing account with their account type and their password */
/***************************************************************************************/
static int do_ulist(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str) {
	GString *out;
	int minlevel=(int)xtra_param;
	GPtrArray *gpa;
	int i;

	gpa=g_ptr_array_new();
	
	full_users_list_with_level(gpa, minlevel);

	out=g_string_new("");
	out=g_string_assign(out,(*dest_base)->str);
	g_string_sprintfa(out,"Nickname:Level:Password:Protected:Comment|");
	send_const_str_to_luce(luce,out->str);
	
	/* sort the array */
	g_ptr_array_sort(gpa, compare_ulist);
	
	/* the user list content is sent in multiple small messages */
	/* to avoid buffer overflow in DC */
	for (i=0; i<gpa->len; i++) {
		GString* gstr=(GString*)g_ptr_array_index(gpa,i);
		out=g_string_assign(out,(*dest_base)->str);
		g_string_sprintfa(out,"%s|", gstr->str);
		g_string_free(gstr,TRUE);
		send_const_str_to_luce(luce,out->str);
	}
	g_string_free(out,TRUE);
	g_ptr_array_free(gpa,TRUE);
	return 0;
}

/*******************************************************/
/* add new registered user to the registered user list */
/*******************************************************/
static int do_add(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=4)		/* 3 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s nick passwd {level OR master}\r\n|",cmd);
	}
	else
	{
		char *new_nick;
		char *password;
		int level;
		char *t;

		/* check login */
		new_nick=g_ptr_array_index(splitted_str,1);
		if((strlen(new_nick)+1)>MAX_NICK_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Nickname too long (max: %d).\r\n|",cmd,MAX_NICK_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if (xml_get_user(new_nick, NULL, NULL, NULL, NULL))
		{
			g_string_sprintfa((*dest_base),"%s: an account with this nickname still exists.\r\n|",cmd);
			goto leave;
		}

		/* check password */
		password=g_ptr_array_index(splitted_str,2);

		/* check level */
		{
			char *level_string;
			level_string=g_ptr_array_index(splitted_str,3);
			if (strcasecmp(level_string,"MASTER") == 0)
				level=MASTER_LEVEL;
			else {
				level=strtol(level_string, &t, 10);
				if(t[0] != '\0') {
					g_string_sprintfa((*dest_base),"%s: bad level parameter:%s must be an integer or MASTER\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,3));
					goto leave;
				}
				if (level > MASTER_LEVEL) {
					g_string_sprintfa((*dest_base),"%s: level can't be greater than %d (MASTER level).\r\nUser %s not created.\r\n|",cmd, MASTER_LEVEL, new_nick);
					goto leave;
				}
			}
		}	

		if (!xml_add_user(new_nick, level, password, FALSE, NULL))
			g_string_sprintfa((*dest_base),"%s: Error while creating nickname %s.\r\n|",cmd, new_nick);
		else
			g_string_sprintfa((*dest_base),"%s: nickname %s created.\r\n|",cmd, new_nick);
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**********************************************************/
/* delete a registered user from the registered user list */
/**********************************************************/
static int do_del(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameter + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *nick_to_del;

		/* check login */
		nick_to_del=g_ptr_array_index(splitted_str,1);
		if((strlen(nick_to_del)+1)>MAX_NICK_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Nickname too long (max: %d).\r\n|",cmd,MAX_NICK_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if (!xml_get_user(nick_to_del, NULL, NULL, NULL, NULL))
		{
			g_string_sprintfa((*dest_base),"%s: No account with this nickname exists.\r\n|",cmd);
			goto leave;
		}

		if (!xml_del_user(nick_to_del))
			g_string_sprintfa((*dest_base),"%s: Error while deleting nickname %s.\r\n|",cmd, nick_to_del);
		else
			g_string_sprintfa((*dest_base),"%s: nickname %s deleted.\r\n|",cmd, nick_to_del);
	}
  leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/******************************************************************/
/* rename an existing registered user of the registered user list */
/******************************************************************/
static int do_rename(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)		/* 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *old_nick;
		char *new_nick;

		/* check old login */
		old_nick=g_ptr_array_index(splitted_str,1);
		if((strlen(old_nick)+1)>MAX_NICK_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Old Nickname too long (max: %d).\r\n|",cmd,MAX_NICK_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if (!xml_get_user(old_nick, NULL, NULL, NULL, NULL))
		{
			g_string_sprintfa((*dest_base),"%s: No account with this nickname (%s) exists.\r\n|",cmd, old_nick);
			goto leave;
		}

		/* check new login */
		new_nick=g_ptr_array_index(splitted_str,2);
		if((strlen(new_nick)+1)>MAX_NICK_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: New Nickname too long (max: %d).\r\n|",cmd,MAX_NICK_LEN-1);
			goto leave;
		}
		if (strlen(new_nick) < 1) {
			g_string_sprintfa((*dest_base),"%s: New Nickname too short.\r\n|",cmd);
			goto leave;
		}			

		/* account already exists ? */
		if (xml_get_user(new_nick, NULL, NULL, NULL, NULL))
		{
			g_string_sprintfa((*dest_base),"%s: An account having the new name (%s) still exists.\r\n|",cmd,new_nick);
			goto leave;
		}

		if(!xml_rename_user(old_nick, new_nick))
			g_string_sprintfa((*dest_base),"%s: Error while updating nickname %s.\r\n|",cmd, old_nick);
		else
			g_string_sprintfa((*dest_base),"%s: nickname %s updated to %s.\r\n|",cmd, old_nick, new_nick);
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/******************************************************/
/* change the password of an existing registered user */
/******************************************************/
static int do_passwd(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len != 3)		/* 2 parameters + the command */
			g_string_sprintfa((*dest_base),"usage: %s username password\r\n|",cmd);
	else {
		char *nick;
		char *password;

		/* check nick */
		nick=g_ptr_array_index(splitted_str,1);

		/* account already exists ? */
		if (!xml_get_user(nick, NULL, NULL, NULL, NULL)) {
			g_string_sprintfa((*dest_base),"%s: No account with this nickname (%s) exists.\r\n|",cmd, nick);
			goto leave;
		}

		/* check password */
		password=g_ptr_array_index(splitted_str,2);

		if (!xml_chg_user_properties(nick, NULL, &password, NULL, NULL))
			g_string_sprintfa((*dest_base),"%s: Error while updating the password of user %s.\r\n|",cmd, nick);
		else
			g_string_sprintfa((*dest_base),"%s: password of user %s updated.\r\n|",cmd, nick);
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************/
/* protect/unprotect a user */
/****************************/
static int do_protection(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len != 2)		/* 1 parameters + the command */
			g_string_sprintfa((*dest_base),"usage: %s username\r\n|",cmd);
	else {
		char *nick;
		gboolean protect=(gboolean)xtra_param;

		/* check nick */
		nick=g_ptr_array_index(splitted_str,1);

		/* account already exists ? */
		if (!xml_get_user(nick, NULL, NULL, NULL, NULL)) {
			g_string_sprintfa((*dest_base),"%s: No account with this nickname (%s) exists.\r\n|",cmd, nick);
			goto leave;
		}

		/* set or unset the protection */
		if (!xml_chg_user_properties(nick, NULL, NULL, &protect, NULL))
			g_string_sprintfa((*dest_base),"%s: Error while updating the protection of user %s.\r\n|",cmd, nick);
		else
			if (protect)
				g_string_sprintfa((*dest_base),"%s: user %s is now protected.\r\n|",cmd, nick);
			else
				g_string_sprintfa((*dest_base),"%s: user %s is now unprotected.\r\n|",cmd, nick);
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*************************************/
/* change the access level of a user */
/*************************************/
static int do_access(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len < 2 || splitted_str->len > 3) {	/* 2 or 3 parameters + the command */
		g_string_sprintfa((*dest_base),"usage: %s username [new_level]\r\n|",cmd);
	} else {
		char *nick;
		gint16 level;
		
		/* check old login */
		nick=g_ptr_array_index(splitted_str,1);

		/* account already exists ? (get the level if exists) */
		if (!xml_get_user(nick, &level, NULL, NULL, NULL)) {
			g_string_sprintfa((*dest_base),"%s: No account with this nickname (%s) exists.\r\n|", cmd, nick);
			goto leave;
		}

		if (splitted_str->len == 3) {
			/* set the level */
			{
				/* check level */
				char *level_string;
				char *t;
				level_string=g_ptr_array_index(splitted_str,2);
				if (strcasecmp(level_string,"MASTER") == 0)
					level=MASTER_LEVEL;
				else {
					int level_in_int=strtol(level_string, &t, 10);
					if(t[0] != '\0') {
						g_string_sprintfa((*dest_base),"%s: bad level parameter:%s must be an integer or MASTER\r\n|",cmd,level_string);
						goto leave;
					}
					if (level_in_int > MASTER_LEVEL) {
						g_string_sprintfa((*dest_base),"%s: level can't be greater than %d (MASTER level).\r\n|",cmd, MASTER_LEVEL);
						goto leave;
					}
					if (level_in_int < -1) {
						g_string_sprintfa((*dest_base),"%s: level can't be lower than -1 (account disable).\r\n|",cmd);
						goto leave;
					}
					level=level_in_int; /* conversion int -> gint16 ok */
				}
				
				/* level ok */
				if (!xml_chg_user_properties(nick, &level, NULL, NULL, NULL))
					g_string_sprintfa((*dest_base),"%s: Error while updating the level of user %s.\r\n|",cmd, nick);
				else
					g_string_sprintfa((*dest_base),"%s: level of user %s updated to %s.\r\n|",cmd, nick, level_string);
			}
		} else {
			/* report the level */
			if (level == MASTER_LEVEL)
				g_string_sprintfa((*dest_base),"%s is at level MASTER.\r\n|", nick);
			else
				g_string_sprintfa((*dest_base),"%s is at level %d.\r\n|", nick, level);
		}
	}
  leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}


/*******************************/
/* function use to sort db key */
/*******************************/
gint compare_db_key (gconstpointer a, gconstpointer b, gpointer user_data) {
	return strcasecmp((char*)g_ptr_array_index(((GPtrArray *)user_data),(int)a),
					  (char*)g_ptr_array_index(((GPtrArray *)user_data),(int)b));
}

/*************************/
/* dump current database */
/*************************/
static int do_db(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=1)		/* 0 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GSList *sortIndex=NULL;
		GStringChunk *gsc=NULL;
		GPtrArray *gpa_name=NULL;
		GPtrArray *gpa_type=NULL;
		GPtrArray *gpa_val=NULL;

		if (db_get_all_keys(&gpa_name, &gpa_type, &gpa_val, &gsc)) {
			int i;
			GString *out;
			
			for( i=0; i<gpa_name->len; i++) {
				sortIndex=g_slist_append(sortIndex,(gpointer)i);
			}
		
			/* sort the array by keyname*/
			sortIndex=g_slist_sort_with_data(sortIndex, compare_db_key, gpa_name);

			out=g_string_new("");

			/* the database content is sent in multiple small messages */
			/* to avoid buffer overflow in DC */
			for(i=0;i<gpa_name->len;i++) {
				out=g_string_assign(out,(*dest_base)->str);
				g_string_sprintfa(out,"'%s'\t'%s'\t'%s'|",
								  (char*)g_ptr_array_index(gpa_type,(int)g_slist_nth_data(sortIndex, i)),
								  (char*)g_ptr_array_index(gpa_name,(int)g_slist_nth_data(sortIndex, i)),
								  (char*)g_ptr_array_index(gpa_val, (int)g_slist_nth_data(sortIndex, i)));
				send_const_str_to_luce(luce,out->str);
			}
			g_string_free(out,TRUE);
			
		}
		g_string_sprintfa((*dest_base),"End of database dump|");

		if(gsc!=NULL)
			g_string_chunk_free(gsc);
		if(gpa_name!=NULL)
			g_ptr_array_free(gpa_name,TRUE);
		if(gpa_type!=NULL)
			g_ptr_array_free(gpa_type,TRUE);
		if(gpa_val!=NULL)
			g_ptr_array_free(gpa_val,TRUE);
		g_slist_free(sortIndex);
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**********************/
/* dump the key value */
/**********************/
static int do_dbshow(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len != 2)	{	/* 1 parameters + the command */
		g_string_sprintfa((*dest_base),"usage: %s key \r\n|", cmd);
	} else {
		char *kn;
		int tp;

		kn=g_ptr_array_index(splitted_str,1);

		tp=db_type_get(kn);
		if(tp == -1) {
			g_string_sprintfa((*dest_base),"no key named '%s'\r\n|",kn);
		} else {
			if(tp == TYPE_INT) {
				/* value is an int */
				int vl;
				if(!db_int_get(kn,&vl))
					vl=0;
				g_string_sprintfa((*dest_base),"'%s'\t'%s'\t'%d'|",
								  DB_TYPE[tp],kn,vl);
			} else if(tp == TYPE_FLOAT) {
				/* value is a float */
				float vl;
				if(!db_float_get(kn,&vl))
					vl=0;
				g_string_sprintfa((*dest_base),"'%s'\t'%s'\t'%f'|",
								  DB_TYPE[tp],kn,vl);
			} else {
				/* value is a string */
				char *vl;
				vl=db_str_get(kn);
				if (vl == NULL) {
					g_string_sprintfa((*dest_base),"'%s'\t'%s'\t''|", DB_TYPE[tp],kn);
				} else {
					g_string_sprintfa((*dest_base),"'%s'\t'%s'\t'%s'|",
									  DB_TYPE[tp],kn,vl);
					free(vl);
				}
			}
		}
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*********************/
/* add the key value */
/*********************/
static int do_dbadd(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len != 4) {	/* 3 parameters + the command */
		g_string_sprintfa((*dest_base),"usage: %s key type value\r\n\t\ttype allowed: 'int', 'str' or 'float'\r\n|", cmd);
	} else {
		char *kn;
		int tp;
		
		kn=g_ptr_array_index(splitted_str,1);

		tp=db_type_get(kn);
		if(tp != -1) {
			g_string_sprintfa((*dest_base),"a key with this name still exists. Error\r\n|");
		} else {
			/* check type before adding */
			int wanted_type;
			char *type=g_ptr_array_index(splitted_str,2);
			for (wanted_type=TYPE_STR; wanted_type<NO_MORE_TYPE; wanted_type++) {
				if (strcmp(type, DB_TYPE[wanted_type]) == 0)
					break;
			}
			
			if (wanted_type == NO_MORE_TYPE) {
				g_string_sprintfa((*dest_base),"Unknown type. Only 'int', 'str' and 'float' are allowed. Error\r\n|");
			} else {
				gboolean result=FALSE;
				if (wanted_type == TYPE_INT) {
					result=db_int_set(kn,atoi(g_ptr_array_index(splitted_str,3)));
				} else if (wanted_type == TYPE_STR) {
					result=db_str_set(kn,g_ptr_array_index(splitted_str,3));
				} else {
					result=db_float_set(kn,atof(g_ptr_array_index(splitted_str,3)));
				}
				if (result == TRUE)
					g_string_sprintfa((*dest_base),"Key %s added.\r\n|",kn);
				else
					g_string_sprintfa((*dest_base),"%s: Error creating %s key. Key wasn't created.\r\n|",cmd,kn);
			}
		}
		load_dyn_conf();
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/************************/
/* update the key value */
/************************/
static int do_dbset(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len < 3) { /* at least 2 parameters + the command */
		g_string_sprintfa((*dest_base),"usage: %s key value\r\n|", cmd);
	} else {
		char *kn;
		int tp;
		
		kn=g_ptr_array_index(splitted_str,1);

		tp=db_type_get(kn);
		if (tp == -1) {
			g_string_sprintfa((*dest_base),"No key with this name still exists. Error\r\n|");
		} else {
			gboolean result=FALSE;
			if (tp == TYPE_INT)
				result=db_int_set(kn,atoi(g_ptr_array_index(splitted_str,2)));
			else if (tp == TYPE_STR)
				result=db_str_set(kn,g_ptr_array_index(splitted_str,2));
			else
				result=db_float_set(kn,atof(g_ptr_array_index(splitted_str,2)));

			if (result)
				g_string_sprintfa((*dest_base),"Key %s updated.\r\n|",kn);
			else
				g_string_sprintfa((*dest_base),"%s: Error updating %s key value. Key wasn't updated.\r\n|", cmd, kn);
		}
		load_dyn_conf();
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************************/
/* test if the given key is a protected key */
/********************************************/
/* out: 0=not protected, 1=yes */
static int is_a_protected_key(char *kn)
{
	static const char *tbl_prot[]={
												"HUBNAME",
												"HUBADDR",
												"HUBDESC",
												"REG_HUB",
												"REG_SERV",
												"MAXUSER",
												"REDIR_FULL",
												"REDIR_ADDR",
												"ONLY_REG",
												NULL
											};
	int i;
	
	i=0;
	while(tbl_prot[i]!=NULL)
	{
		if(!strcmp(kn,tbl_prot[i]))
			return 1;
		i++;
	}

	return 0;
}

/*********************/
/* del the key value */
/*********************/
static int do_dbdel(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (splitted_str->len != 2) {		/* 1 parameters + the command */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	} else {
		char *kn;

		kn=g_ptr_array_index(splitted_str,1);
		if (!is_a_protected_key(kn)) {
			if (db_del(kn))
				g_string_sprintfa((*dest_base),"Key %s deleted.\r\n|",kn);
			else
				g_string_sprintfa((*dest_base),"%s: Error while deleting %s key. Key wasn't deleted.\r\n|",cmd,kn);
		}
		else
		{
			g_string_sprintfa((*dest_base),"Key %s is a protected key, you cannot delete it.\r\n|",kn);
		}
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*****************************/
/* start an external program */
/*****************************/
static int do_startprg(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	char *progname;
	gchar *p;
	struct stat st;

	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		goto abrt;
	}

	if(ext_prog_dir==NULL)
	{
		g_string_sprintfa((*dest_base),"Hub has been started without external program directory.\r\n|");
		goto abrt;
	}

	progname=g_ptr_array_index(splitted_str,1);

	if(strchr(progname,'/')!=NULL)
	{
		g_string_sprintfa((*dest_base),"You cannot start an external program containing / in its name.\r\n|");
		goto abrt;
	}

	p=g_strconcat(ext_prog_dir,"/",progname,NULL);
	if(stat(p,&st)==-1)
	{
		g_string_sprintfa((*dest_base),"Error on %s: %s\r\n|",progname,strerror(errno));
	}
	else
	{
		/*
		start_new_ext_prog(progname)
		*/
		if (glus_user_is_connected(progname)==TRUE)
			g_string_sprintfa((*dest_base),"%s is already take by an other user\r\n|",progname);
		else
		{
			add_prog_to_start(progname);
			g_string_sprintfa((*dest_base),"%s started\r\n|",progname);
		}
	}
		
	g_free(p);

	abrt:
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***********************************************/
/* now lets disconnect given user from do_kick */
/***********************************************/
typedef enum
{
	KT_DISCONNECT,
	KT_KICK_AND_SHORT_BAN,
} KICK_TYPE;

void proceed_user_kick(char *nick, LOCAL_USER_CNX_ENTRY *luce, GString *preason, GString **dest_base, KICK_TYPE ban_type, gboolean no_message)
{
	gint16 level_user_to_kick;
	gboolean is_user_to_kick_protected;
	gboolean is_user_to_kick_reg=xml_get_user(nick, &level_user_to_kick, NULL, &is_user_to_kick_protected, NULL);
	gboolean can_kick=FALSE;
	if (!is_user_to_kick_reg || ((level_user_to_kick < luce->level) && !is_user_to_kick_protected))
		can_kick=TRUE;
	
	if (!can_kick) {
		/* explain why we can't kick */
		if (is_user_to_kick_protected) {
			g_string_sprintfa((*dest_base),"You can't kick %s because it's a protected user.\r\n|", nick);
		} else {
			g_string_sprintfa((*dest_base),"You can't kick a user with a level greater than yours.\r\n|");
		}
		send_const_str_to_luce(luce,(*dest_base)->str);
	} else {
		/* try to kick the user */
		if ( (no_message == FALSE) || ((no_message == TRUE) && (preason != NULL))) {		
			int bot_kick;
			GString *out;
			char *kickmsg = NULL;
			out=g_string_new("");

			switch(ban_type)
				{
					case KT_DISCONNECT:
						kickmsg="is disconnecting";
						break;

					case KT_KICK_AND_SHORT_BAN:
						kickmsg="is kicking";
						break;
				}

			if (preason==NULL) {
				g_string_sprintf(out,"<%s> %s %s %s.|",
								 luce->user_nick->str,
								 luce->user_nick->str,
								 kickmsg,
								 nick);
			} else {
				g_string_sprintf(out,"<%s> %s %s %s because: %s|",
								 luce->user_nick->str,
								 luce->user_nick->str,
								 kickmsg,
								 nick,
								 preason->str);
			}
										
			if (!db_int_get("DISPLAY_BOT_KICK",&bot_kick))
				bot_kick=1;

			if ((bot_kick==0 && (luce->level >= get_right_level(SIGNKICK))) || (no_message==TRUE)) {
				GLUS_SEND_TO_A_NICK(nick,out);	/* don't free out */
			} else {
				GLUS_SEND_TO_EVERYONE(out);     /* don't free out */
			}
		}

		{	/* send "kick" event */
			GLUS_USER_INFO *gui;

			gui=glus_get_user_info(nick);
			if (gui)
				{
					GString *ipadd;
					ipadd = g_string_new("");
					g_string_sprintf(ipadd,"%s",inet_ntoa(gui->user_ip));
					switch(ban_type)
						{
							case KT_DISCONNECT:
							case KT_KICK_AND_SHORT_BAN:
								SEND_EVT_TO("kick",luce->user_nick->str,2,gui->user_nick->str,ipadd->str);
								break;
						}

					glus_free_user_info(gui);
					g_string_free(ipadd,TRUE);
				}
		}

		/* kick the user */
		switch(ban_type)
			{
				case KT_DISCONNECT:
					glus_disconnect_named_user(nick);
					break;

				case KT_KICK_AND_SHORT_BAN:
					glus_kick_named_user(luce->user_nick->str, nick);
					break;
			}
	}
}

/*******************************************/
/* now lets ban given nick/ip from do_ban  */
/*******************************************/
void proceed_user_ban(char *user, LOCAL_USER_CNX_ENTRY *luce, GString *preason, GString **dest_base, unsigned long duration_in_second) {
	gint16 level_user_to_ban;
	gboolean is_user_to_ban_protected;
	gboolean is_user_to_ban_reg=xml_get_user(user, &level_user_to_ban, NULL, &is_user_to_ban_protected, NULL);
	gboolean can_ban=FALSE;
	printf("Try a ban %s -> %s\n", luce->user_nick->str, user);
	if (!is_user_to_ban_reg || ((level_user_to_ban < luce->level) && !is_user_to_ban_protected))
		can_ban=TRUE;
	
	if (!can_ban) {
		/* explain why we can't ban */
		if (is_user_to_ban_protected) {
			g_string_sprintfa((*dest_base),"You can't ban %s because it's a protected user.\r\n|",user);
		} else {
			g_string_sprintfa((*dest_base),"You can't ban a user with a level greater than yours.\r\n|");
		}
		send_const_str_to_luce(luce,(*dest_base)->str);
	} else {
		/* try to kick the user */
		struct sockaddr_in user_addr_in;
		int user_is_ip=inet_aton(user,&(user_addr_in.sin_addr));

		/* Display the banning message in the global chat ? */
		if (preason != NULL) {
			int bot_kick;
			GString *out;

			out=g_string_new("");
			if (user_is_ip) {
				g_string_sprintf(out,"<%s> %s is banning IP %s because: %s|",
								 luce->user_nick->str,
								 luce->user_nick->str,
								 user,
								 preason->str);
			} else {
				g_string_sprintf(out,"<%s> %s is banning %s because: %s|",
								 luce->user_nick->str,
								 luce->user_nick->str,
								 user,
								 preason->str);
			}
			if (!db_int_get("DISPLAY_BOT_KICK",&bot_kick))
				bot_kick=1;

			if(bot_kick==0 && (luce->level >= get_right_level(SIGNKICK)) && !user_is_ip) {
				GLUS_SEND_TO_A_NICK(user,out);	/* don't free out */
			} else
				GLUS_SEND_TO_EVERYONE(out);     /* don't free out */
		}

		/* send "ban" event */
		SEND_EVT_TO("ban",luce->user_nick->str,1,user);

		/* ban the user */
		if (user_is_ip) {
			/* ban by ip */
			glus_ban_ip(luce->user_nick->str, user_addr_in.sin_addr, duration_in_second);
		} else {
			/* ban by nick */
			glus_ban_named_user(luce->user_nick->str, user, duration_in_second);
		}
	}
}


/******************************************/
/* kick the given nick                    */
/* check if nick exists as ip or nickname */
/* and decide what to do                  */
/******************************************/
static int do_flood(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	/* 2 parameters (-flood nick) + reason */
	if (splitted_str->len < 3)
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
		return 0;
	}
	else
	{
		const char *nick;
		GLUS_USER_INFO *gui_nick;
		GString *preason;
		char *reason;
		int flood_count;

		if (!db_int_get("FLOOD_COUNT",&flood_count))
			flood_count=1;

		reason=g_ptr_array_index(splitted_str,2);
		preason=g_string_new(reason);
		if (splitted_str->len > 3)
		{
			int i;
			for (i=3;i<splitted_str->len;i++)
			{
				if(g_ptr_array_index(splitted_str,i)!=NULL)
					g_string_sprintfa(preason," %s",(char*)g_ptr_array_index(splitted_str,i));
			}
		}
		nick=g_ptr_array_index(splitted_str,1);

		gui_nick = glus_get_user_info((char *)nick);
		
		if(gui_nick)
		{
			int i;
			GString *mymsg;
			const unsigned int user_flag=1;

			mymsg=g_string_new("");

			g_string_sprintf(mymsg,"$ $DSL%c$$0$",(char)user_flag);
			for(i=0;i<flood_count;i++)
			{
				GString *floodmsg;
				floodmsg=g_string_new("");
				g_string_sprintf(floodmsg,"$Hello flood%i|$MyINFO $ALL flood%i %s|$To: %s From: flood%i $<flood%i> %s|",i,i,mymsg->str,gui_nick->user_nick->str,i,i,preason->str);
				glus_send_to_named_user(gui_nick->user_nick->str,0,0,0,0,floodmsg);
			}
			g_string_free(mymsg,TRUE);

			proceed_user_kick(gui_nick->user_nick->str, luce, preason, dest_base, KT_DISCONNECT,FALSE);
			glus_free_user_info(gui_nick);
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s not found|",nick);
			send_const_str_to_luce(luce,(*dest_base)->str);
		}
		g_string_free(preason,TRUE);
	}
	printf("flood finished\n");
	return 0;
}

static int do_kick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	/* 2 parameters (-kick nick) + reason */
	if ((splitted_str->len < 2) || 
		(splitted_str->len < 3  && !(luce->level == MASTER_LEVEL)))
	{ /* only a hub-master should be allowed to kick without a reason !!! */
			g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
			send_const_str_to_luce(luce,(*dest_base)->str);
			return 0;
	}
	else
	{
		const char *nick;
		char *reason=NULL;
		struct sockaddr_in addr_in;
		GLUS_USER_INFO *gui_nick;
		GString *preason=NULL;
		gboolean no_message=(gboolean)xtra_param;
		
		if (splitted_str->len > 2)
		{
			/* preparing kicking reason */
			reason=g_ptr_array_index(splitted_str,2);		
			preason=g_string_new(reason);
			if (splitted_str->len > 3)
			{
				int i;
				for (i=3;i<splitted_str->len;i++)
				{
					if(g_ptr_array_index(splitted_str,i)!=NULL)
						g_string_sprintfa(preason," %s",(char*)g_ptr_array_index(splitted_str,i));
				}
			}
		}

		nick=g_ptr_array_index(splitted_str,1);

		/* check if we've got a nick using this name */
		gui_nick = glus_get_user_info((char *)nick);

		/* lets see if we can use the given nick as an IP */
		if (inet_aton(nick,&(addr_in.sin_addr)) != 0)
		{	/* take the string as an IP */
			
			GPtrArray *gui_array;
			/********************************************************************/
			/* got an IP, but maybe where's more than one user with the same ip */
			/* so collect them into an array first and kick them all            */
			/* the nick should be unique                                        */
			/********************************************************************/

			gui_array=glus_get_users_info_by_ip(addr_in.sin_addr);

			/* are we able to use both nick and ip ? what should I do now ? */
			/* first check if we find both nick & ip */
			if(gui_nick && gui_array->len>0)
			{
				int i;
				int use_both=FALSE;

				for(i=0;i<gui_array->len;i++)
				{
					GLUS_USER_INFO *gui_ip;
					gui_ip=g_ptr_array_index(gui_array,i);

					/* the user is using his IP as nick as well ? */
					if(!strcmp(gui_ip->user_nick->str,gui_nick->user_nick->str))
						use_both=TRUE;
				}

				if(use_both==TRUE)
				{	/* found user using ip as nick as well -> kick it !!! */
					GLUS_USER_INFO *gui_ip;

					for(i=0;i<gui_array->len;i++)
					{
						if((gui_ip=g_ptr_array_index(gui_array,i))!=NULL)
						{
							proceed_user_kick(gui_ip->user_nick->str, luce, preason, dest_base, KT_KICK_AND_SHORT_BAN,no_message);
						}
					}
				}
				else
				{	/* different users found -> reply data to decide */
					GLUS_USER_INFO *gui_ip;

					g_string_sprintfa((*dest_base),"Attention found more than one user to kick:\r\n");
					g_string_sprintfa((*dest_base),"IP: %s	Nick: %s\r\n",inet_ntoa(gui_nick->user_ip), gui_nick->user_nick->str);
					for(i=0;i<gui_array->len;i++)
					{
						if((gui_ip=g_ptr_array_index(gui_array,i))!=NULL)
							g_string_sprintfa((*dest_base),"IP: %s	Nick: %s\r\n",inet_ntoa(gui_ip->user_ip), gui_ip->user_nick->str);
					}
					g_string_sprintfa((*dest_base),"Please try the -kick command with the Parameter(ip/nick) you've not used|");
					send_const_str_to_luce(luce,(*dest_base)->str);
				}
			}
			else if(gui_nick && gui_array->len==0)
			{	/* found nick only -> kick it */
				proceed_user_kick(gui_nick->user_nick->str, luce, preason, dest_base, KT_KICK_AND_SHORT_BAN,no_message);
			}
			else if(gui_array->len>0)
			{
				int i;
				GLUS_USER_INFO *gui_ip;

				for (i=0;i<gui_array->len;i++)
				{
					if((gui_ip=g_ptr_array_index(gui_array,i))!=NULL)
					{
						proceed_user_kick(gui_ip->user_nick->str, luce, preason, dest_base, KT_KICK_AND_SHORT_BAN,no_message);
					}
				}
			}
			else
			{
				g_string_sprintfa((*dest_base),"%s not found|",nick);
				send_const_str_to_luce(luce,(*dest_base)->str);
			}

			if(gui_nick)
				glus_free_user_info(gui_nick);
			glus_free_user_info_ptrarray(gui_array);
		}
		else if (gui_nick)
		{
			proceed_user_kick(gui_nick->user_nick->str, luce, preason, dest_base, KT_KICK_AND_SHORT_BAN,no_message);
			glus_free_user_info(gui_nick);
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s not found|",nick);
			send_const_str_to_luce(luce,(*dest_base)->str);
		}
		if(preason!=NULL)
			g_string_free(preason,TRUE);
	}
	/* no need for success-message - we have one in Main-chat */
	return 0;
}
static int do_ban(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<3)		/* at least 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s duration {nick OR ip} [reason]\r\n|", cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		char	*nick;
		char	*reason;
		char 	*duration;
		unsigned long duration_time;
		struct sockaddr_in addr_in;
		GString *preason=NULL;
		char time_unit;
		int a;

		duration=g_ptr_array_index(splitted_str,1);
		nick=g_ptr_array_index(splitted_str,2);		/* a nick or an IP */
		if(splitted_str->len==4)							/* and the optional reason */
			reason=g_ptr_array_index(splitted_str,3);
		else
			reason=NULL;

		a=sscanf(duration,"%lu%c",&duration_time,&time_unit);

		if(a==0)
		{
			g_string_sprintfa((*dest_base),"no valid duration for %s\r\n|",cmd);
			send_const_str_to_luce(luce,(*dest_base)->str);
			goto abrt;
		}
		if(a==1)
			duration_time*=60;		/* duration in minute */
		else {
			if((time_unit!='h')&&(time_unit!='d'))
			{
				g_string_sprintfa((*dest_base),"no valid time unit for %s. Only 'h' and 'd' allowed.\r\n|",cmd);
				send_const_str_to_luce(luce,(*dest_base)->str);
				goto abrt;
			}

			if(time_unit=='h')
				duration_time*=3600;
			if(time_unit=='d')
				duration_time*=(24*3600);
		}

		if(reason!=NULL)
			preason=g_string_new(reason);

		/* lets see if we can use the given nick as an IP */
		if (inet_aton(nick,&(addr_in.sin_addr)) != 0) {
			/* take the string as an IP */
			GPtrArray *gui_array;
			/********************************************************************/
			/* got an IP, but maybe where's more than one user with the same ip */
			/* so collect them into an array first and ban them all             */
			/********************************************************************/

			gui_array=glus_get_users_info_by_ip(addr_in.sin_addr);

			if (gui_array->len > 0) {
				int i;

				for (i=0; i<gui_array->len; i++) {
					GLUS_USER_INFO *gui_ip;
					gui_ip=g_ptr_array_index(gui_array,i);
					/* ban by name */
					proceed_user_ban(gui_ip->user_nick->str, luce, preason, dest_base, duration_time);
				}
			} else {
				/* ban by ip (nick is an IP) */
				proceed_user_ban(nick, luce, preason, dest_base, duration_time);
			}
			glus_free_user_info_ptrarray(gui_array);
		} else {
			/* check if we've got a nick using this name */
			GLUS_USER_INFO *gui_nick= glus_get_user_info((char *)nick);
			if (gui_nick) {
				proceed_user_ban(gui_nick->user_nick->str, luce, preason, dest_base, duration_time);
				glus_free_user_info(gui_nick); 
			} else {
				g_string_sprintfa((*dest_base),"%s not found|",nick);
				send_const_str_to_luce(luce,(*dest_base)->str);
			}
		}
		
		if(preason!=NULL)
			g_string_free(preason,TRUE);
	}
	abrt:
	/* no success-message needed in cause of having one it in the main-chat */
	return 0;
}

static int do_silence(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len < 3)		/* 2 parameters + the command or more*/
	{
		g_string_sprintfa((*dest_base),"usage:%s TYPE(in pPcCsSdDvVlLrRaAoO) username [username ...]\r\n"
		"P->PM\r\n"
		"C->Chat\r\n"
		"S->Search\r\n"
		"D->Download\r\n"
		"V->Vip(Shild)\r\n"
		"L->LOCK_PM_TUTOR\r\n"
		"R->REV_SILENT_CNX\r\n"
		"A->OTIST CC (N/A)\r\n"
		"O->OTIST PM\r\n"	
		"|",cmd);
	}
	else
	{
		char		*type;
		int			i;
		unsigned int	flag = 0;
		unsigned int	unflag = 0;

		type = g_ptr_array_index(splitted_str,1);

		if (strchr(type, 'P'))
			flag |= SILENT_PM;
		if (strchr(type, 'p'))
			unflag |= SILENT_PM;
			
		if (strchr(type, 'C'))
			flag |= SILENT_CC;
		if (strchr(type, 'c'))
			unflag |= SILENT_CC;
			
		if (strchr(type, 'S'))
			flag |= SILENT_SR;
		if (strchr(type, 's'))
			unflag |= SILENT_SR;
			
		if (strchr(type, 'D'))
			flag |= SILENT_CNX;
		if (strchr(type, 'd'))
			unflag |= SILENT_CNX;
			
		if (strchr(type, 'V'))
			flag |= SHILD_MODE;
		if (strchr(type, 'v'))
			unflag |= SHILD_MODE;
			
		if (strchr(type, 'L')) 
			flag |= LOCK_PM_TUTOR;
		if (strchr(type, 'l'))
			unflag |= LOCK_PM_TUTOR;
			
		if (strchr(type, 'R'))
			flag |= REV_SILENT_CNX;
		if (strchr(type, 'r'))
			unflag |= REV_SILENT_CNX;
			
		if (strchr(type, 'A'))
			flag |= OTIST_CC;
		if (strchr(type, 'a'))
			unflag |= OTIST_CC;

		if (strchr(type, 'O'))
			flag |= OTIST_PM;
		if (strchr(type, 'o'))
			unflag |= OTIST_PM;


		for (i = 2; i < splitted_str->len; i++)
		{
			user_cnx_entry_update_ext_flag_by_nickname(g_ptr_array_index(splitted_str,i),flag,unflag);
		}
		g_string_sprintfa((*dest_base)," done|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************/
/* redirect the given nick */
/***************************/
static int do_redir(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=4)		/* 3 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GString *rd_msg;
		char *rd;
		char *nm;
		char *msg;
		rd_msg=g_string_new("");

		nm=g_ptr_array_index(splitted_str,1);
		printf("redirect by: %s\n",nm);
		rd=g_ptr_array_index(splitted_str,2);
		printf("redirect to: %s\n",rd);
		msg=g_ptr_array_index(splitted_str,3);
		printf("redirect message: %s\n",msg);

		/* first the message to display */
		if((strncmp(msg,"You are being re-directed to",28))!=0)
			g_string_sprintf(rd_msg,"$To: %s From: %s $<%s> You are being re-directed to %s because: %s|",nm,luce->user_nick->str,luce->user_nick->str,rd,msg);
		else
			g_string_sprintf(rd_msg,"$To: %s From: %s $<%s> %s|",nm,luce->user_nick->str,luce->user_nick->str,msg);

		/* and then the redirection command */
		g_string_sprintfa(rd_msg,"$ForceMove %s|",rd);

		GLUS_SEND_TO_A_NICK(nm,rd_msg);		/* don't free rd_msg */

		/* g_string_sprintfa((*dest_base),"%s redirected\r\n|",nm); */
	}

	/* send_const_str_to_luce(luce,(*dest_base)->str); */
	return 0;
}

/**********************/
/* redirect all users */
/**********************/
static int do_redirall(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2) {	/* 1 parameter + the command */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GString *rd_msg;
		char *rd;
		rd_msg=g_string_new("");

		rd=g_ptr_array_index(splitted_str,1);

		/* first the message to display */
		g_string_sprintf(rd_msg,"<Hub-Security> You are being redirected to %s.|",rd);

		/* and then the redirection command */
		g_string_sprintfa(rd_msg,"$ForceMove %s|",rd);

#warning full cluster or local user only ?
		GLUS_SEND_TO_EVERYONE(rd_msg);		/* don't free rd_msg */
		
		g_string_sprintfa((*dest_base),"All users has been redirected\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************/
/* send a broadcast message */
/****************************/
static int do_bcast(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		GString *out;
		out=g_string_new("");
		g_string_sprintf(out,"$To: you From: Hub-Mass-Message $<Hub-Mass-Message> %s|", (char*)g_ptr_array_index(splitted_str,1));
		GLUS_SEND_TO_EVERYONE(out);		/* don't free out */
		/* g_string_sprintfa((*dest_base),"Broadcast done\r\n|"); */
	}
	/* send_const_str_to_luce(luce,(*dest_base)->str); */
	return 0;
}

static int do_version(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		GLUS_USER_INFO *gui;
		char *nick;
		nick=g_ptr_array_index(splitted_str,1);

		gui=glus_get_user_info(nick);

		if(gui!=NULL)
		{
			g_string_sprintfa((*dest_base),"the client-version of '%s' is: %s\r\n|",
			                 nick,gui->client_version->str);

			send_const_str_to_luce(luce,(*dest_base)->str);

			glus_free_user_info(gui);
		}
	}
	return 0;
}

#if 0
/************************************/
/* delete a  ban on the given ip(s) */
/************************************/
static int do_unbanip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	unsigned long key=(unsigned long)xtra_param;

	if(splitted_str->len<2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		int i;
		const char *ip;

		for(i=1;i<splitted_str->len;i++)
		{
			ip=g_ptr_array_index(splitted_str,i);
			if(ip!=NULL)
			{
				struct in_addr inp;

				if(inet_aton(ip,&inp)==0)
				{
					g_string_sprintfa((*dest_base),"Fail to unban IP '%s'. Invalid address\r\n",ip);
				}
				else
				{
					delete_this_tos_entry(key,(void*)&inp,sizeof(inp),0);
					g_string_sprintfa((*dest_base),"unban IP %s\r\n",ip);
				}
			}
		}

		g_string_sprintfa((*dest_base),"Unban done\r\n|");
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***********************************/
/* delete a ban on the given ip(s) */
/***********************************/
static int do_unbannick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<2)	 /* 1 parameter + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		int i;
		unsigned long key=(unsigned long)xtra_param;
		const char *nick;

		for (i=1;i<splitted_str->len;i++)
		{
			nick=g_ptr_array_index(splitted_str,i);
			if(nick!=NULL)
			{
				delete_this_tos_entry(key,(void*)nick,strlen(nick),1);
				g_string_sprintfa((*dest_base),"unban Nick %s\r\n",nick);
			}
		}
		g_string_sprintfa((*dest_base),"Unban done\r\n|");
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}
#endif
/*******************************************/
/* delete a ban on the given nick(s)/ip(s) */
/*******************************************/
static int do_unban(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<2)	 /* 1 parameter + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		int i;
		const char *nick;

		for (i=1;i<splitted_str->len;i++)
		{
			nick=g_ptr_array_index(splitted_str,i);
			if(nick!=NULL)
			{
				glus_uban_named_user(nick);
			}
		}
		g_string_sprintfa((*dest_base),"Unban done\r\n|");
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}
static int append_ban_entry(const time_t timeout, const char *ptr1, const int len1, const char *ptr2, const int len2, void *xtra)
{
	/* warning: ptr1 is always a pointer on a valid struct in_addr (and len1 is its length) */
	/*          ptr2 is either a nickname (and len2 is its length) but it can also be NULL */
	GString **dest_base=xtra;

	if((ptr1!=NULL)&&(len1==sizeof(struct in_addr)))
	{
		struct in_addr *inp=(void*)ptr1;
		char timestamp[512];
#ifdef WIN32
		strftime(timestamp,sizeof(timestamp),"%c",localtime(timeout));
#else
		struct tm tm;
		localtime_r(&timeout,&tm);
		strftime(timestamp,sizeof(timestamp),"%c",&tm);
#endif


		if((ptr2==NULL)||(len2==0))
		{
			/* inet_ntoa is not thread safe, it works with a static buffer, we will do the conversion manually */
			g_string_sprintfa((*dest_base),"IP: %u.%u.%u.%u - ban until %s\r\n",
									(unsigned)((ntohl(inp->s_addr)>>24)&255),
									(unsigned)((ntohl(inp->s_addr)>>16)&255),
									(unsigned)((ntohl(inp->s_addr)>>8)&255),
									(unsigned)((ntohl(inp->s_addr))&255),
									timestamp
									);
		}
		else
		{
			char nick[256];

			strncpy_max(nick,ptr2,min(sizeof(nick),(len2+1)));
			/* inet_ntoa is not thread safe, it works with a static buffer, we will do the conversion manually */
			g_string_sprintfa((*dest_base),"IP: %u.%u.%u.%u	Nick: %s - ban until %s\r\n",
									(unsigned)((ntohl(inp->s_addr)>>24)&255),
									(unsigned)((ntohl(inp->s_addr)>>16)&255),
									(unsigned)((ntohl(inp->s_addr)>>8)&255),
									(unsigned)((ntohl(inp->s_addr))&255),
									nick,
									timestamp
									);
		}
	}
	else if(ptr2!=NULL)
	{
		struct tm tm;
		char nick[256];
		char timestamp[512];
		localtime_r(&timeout,&tm);
		strftime(timestamp,sizeof(timestamp),"%c",&tm);

		strncpy_max(nick,ptr2,min(sizeof(nick),(len2+1)));
		g_string_sprintfa((*dest_base),"Nick: %s - ban until %s\r\n",
									nick,
									timestamp
									);
   }
	return 0;	/* continue */
}

/**********************/
/* print the ban list */
/**********************/
static int do_banlist(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if (number_of_tos_entry(BAN_TOSKEY) == 0) {
		g_string_sprintfa((*dest_base),"Ban list is empty.|");
	} else {
		g_string_sprintfa((*dest_base),"Ban list:\r\n");
		scan_all_tos_entry(BAN_TOSKEY,append_ban_entry,dest_base);
		g_string_sprintfa((*dest_base),"|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************/
/* clearing wanted banlist */
/***************************/
static int do_clearban(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
   if(splitted_str->len!=1)      /* the command only */
   {
      g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
   }
   else
   {
      delete_all_same_key(BAN_TOSKEY, dest_base, luce->user_nick->str);
		g_string_sprintfa((*dest_base),"Clearing banlist done\r\n|");
   }

   send_const_str_to_luce(luce,(*dest_base)->str);
   return 0;
}

/******************************/
/* reset the perl interpreter */
/******************************/
static int do_perlreset(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=1)		/* the command only */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
#ifdef WITH_PERL
		restart_perl();
#endif
		g_string_sprintfa((*dest_base),"Perl interpreter restarted\r\n|");
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	return 0;
}
/**************************************/ 
/* clear all autoloaded perl handlers */ 
/**************************************/ 
static int do_clear_autoloaded_perl_handlers(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)                  
{                                
	if (splitted_str->len!=1)
	{                /* the command only */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
#ifdef WITH_PERL
		clear_autoloaded_perl_handlers();
#endif
		g_string_sprintfa((*dest_base),"All autoloaded perl handlers cleared\r\n|");
		send_const_str_to_luce(luce,(*dest_base)->str);
	}                        
	return 0;                
}                                

static int do_pluginstart(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
   if (splitted_str->len!=2)
   {
      g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
   }
   else
   {
      char *plugin_name;
      plugin_name=g_ptr_array_index(splitted_str,1);
      plugin_start(plugin_name,(*dest_base));
      g_string_sprintfa((*dest_base),"|");
   }
   send_const_str_to_luce(luce,(*dest_base)->str); 
   return 0;
}

static int do_pluginstop(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
   if (splitted_str->len!=2)
   {
      g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
   }
   else
   {
      char *plugin_name;
      plugin_name=g_ptr_array_index(splitted_str,1);
      plugin_stop(plugin_name,(*dest_base));
      g_string_sprintfa((*dest_base),"|");
   }
   send_const_str_to_luce(luce,(*dest_base)->str);
   return 0;
}

typedef struct
{
	const char *cmd;
	int (*fnc)(const char *cmd, LOCAL_USER_CNX_ENTRY *ptr, GString **dest_base, 
			   const GString *str, char *xtra_param, const GPtrArray *splitted_str);
	/* function to call. (ERROR: the following sentence is FALSE => This function MUST free str when it is not useful anymore) */
	/* the function must return >=0 if the connection must remain alive (no error (0) or */
	/* recoverable error (>0)) and a value <0 if there is a fatal error) */
	/* *dest_base is a the beginning of the reply to send. To create the full reply, */
	/* add the message you want to send and ends the string with a |. Use send_const_str_to(ptr,(*dest_base)->str) */
	/* to send the reply */
	char *xtra_param;
	const char *level_needed;
} KNOWN_CMD;

static KNOWN_CMD hub_cmd[]={ /* Command MUST BE SORT by name */
        {"-access",     do_access,      NULL,       MNGUSERS},
        {"-add",        do_add,         NULL,       MNGUSERS},  /* ok v0.4.0 */
        {"-ban",        do_ban,         NULL,       BAN},       /* new version */
        {"-banlist",    do_banlist,     NULL,       UNBAN},     /* ok v0.4.0 new version*/
        {"-bcast",      do_bcast,       NULL,       OPERATOR},  /* ok v0.4.0 */
        {"-clearban",   do_clearban,    NULL,       UNBAN},

        /* database management */
        {"-dbadd",      do_dbadd,       NULL,       MNGDB},     /* ok v0.4.0 */
        {"-dbdel",      do_dbdel,       NULL,       MNGDB},     /* ok v0.4.0 */
        {"-dblist",     do_db,          NULL,       MNGDB},     /* ok v0.4.0 */
        {"-dbset",      do_dbset,       NULL,       MNGDB},     /* ok v0.4.0 */
        {"-dbshow",     do_dbshow,      NULL,       MNGDB},     /* ok v0.4.0 */
        {"-del",        do_del,         NULL,       MNGUSERS},  /* ok v0.4.0 */
#ifdef DEBUG                                        
        {"-dump",       do_dump,        NULL,       NULL},      /* disabled */
        {"-dumpqueue",  do_dump_queue,  NULL,       NULL},      /* disabled */
#endif
        {"-flood",      do_flood,       NULL,       OPERATOR},
        {"-getip",      do_get_ip,      NULL,       OPERATOR},      /* ok v0.4.0 */
        {"-help",       do_help,        NULL,       OPERATOR},      /* ok v0.4.0 */
        {"-hpcnx",      do_hpcnx,       NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-hpdel",      do_hpdel,       NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-hpget",      do_hpget,       NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-hpgetall",   do_hpgetall,    NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-hpset",      do_hpset,       NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-hpstat",     do_hpstat,      NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-kick",       do_kick,        (void*)FALSE, KICK},        /* ok v0.4.0 new version */
        {"-md5",        do_md5cmd,      NULL,       MNGCLUSTERS},   /* ok v0.4.0 */
        {"-passwd",     do_passwd,      NULL,       MNGUSERS},      /* ok v0.4.0 */
        {"-perlclear",  do_clear_autoloaded_perl_handlers, NULL, STARTPRG}, /* ok v0.4.0 */
        {"-perlreset",  do_perlreset,   NULL,       STARTPRG},      /* ok v0.4.0 */
        {"-pluginstart ", do_pluginstart, NULL,     STARTPRG},
        {"-pluginstop ", do_pluginstop,	NULL,     STARTPRG},	
        {"-protect",    do_protection,  (void*)TRUE, MNGUSERS},
        {"-redir",      do_redir,       NULL,       REDIRECT},      /* ok v0.4.0 */
        {"-redirall",   do_redirall,    NULL,       REDIRECT},      /* ok v0.4.0 */
#if 0
        {"-rediralllocal", do_local_redirall,NULL,  REDIRECT},      /* disabled */
#endif
        {"-reload",     do_reload,      NULL,       MNGDB},         /* ok v0.4.0 */
        {"-rename",     do_rename,      NULL,       MNGUSERS},      /* ok v0.4.0 */
        {"-revip",      do_rev_ip,      NULL,       OPERATOR},      /* ok v0.4.0 */
        {"-set",        do_set,         NULL,       MNGDB},         /* ok v0.4.0 */
        {"-silence",    do_silence,     NULL,       KICK},          
        {"-skick",      do_kick,        (void*)TRUE, KICK},         /* new version */
        {"-startprg",   do_startprg,    NULL,       STARTPRG},
        {"-ulist",      do_ulist,       (void*)-1, 	MNGUSERS},		/* ok v0.4.0 */
        {"-unban",      do_unban,       NULL,       UNBAN},         /* ok v0.4.0 */
        {"-unprotect",  do_protection,  (void*)FALSE, MNGUSERS},
        {"-version",    do_version,     NULL,       OPERATOR},      /* ok v0.4.0 */
    {NULL,NULL,NULL,NULL},
};

/*********************************************************************************/
/* run the given command. Incoming string is splitted, access rights are checked */
/*********************************************************************************/
static int run_hub_cmd(KNOWN_CMD *hcmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest, const GString *inp)
{
	if ((hcmd->level_needed == NULL) || (luce->level >= get_right_level(hcmd->level_needed)))
	{
		GStringChunk *chunk=NULL;
		GPtrArray *tbl=NULL;
		int ret=0;

		split_string_into_array(&tbl,&chunk,inp->str," ");
		/* now, we can process the command */
		if((chunk!=NULL)&&(tbl!=NULL))
		{
			ret=(hcmd->fnc)(hcmd->cmd,luce,dest,inp,hcmd->xtra_param,tbl);
		}

		/* free allocated data */
		if(tbl!=NULL)
			g_ptr_array_free(tbl,TRUE);
		if(chunk!=NULL)
			g_string_chunk_free(chunk);
		return ret;
	}
	else
	{
		/* not enough rights */
		g_string_sprintfa((*dest),"You are not allowed to use this command.|");
		send_const_str_to_luce(luce,(*dest)->str);
		return 0;
	}
}

/*************************************************************************/
/* the given string has been sent to "Hub-Security", it may be a command */
/* This function processes the string and performs the wanted action	 */
/*************************************************************************/
/* output: =0: ok								 */
/*		   <0: fatal error (=> abort connection) */
/*		   >0: recoverable error (continue)		 */
/*************************************************/
/* the last char of string is a '|' */
int process_hub_command(LOCAL_USER_CNX_ENTRY *luce,char *string)
{
	int ret=0;
	GString *inp=NULL;
	GString *dest=NULL;
	char *t;
	int fnd=0;

	printf("%s\n",string);

	t=strchr(string,' ');
	if (t == NULL)
		goto eoc;
	SKIP_SPACE(t);

	if(*t=='\0')
	{
		ret=1;
		goto eoc;		/* non fatal error: nothing entered */
	}

	inp=g_string_new(t);	/* remove trash before the command */
	if(inp->len>1)
		inp=g_string_truncate(inp,inp->len-1);		/* remove trailing | */


	/* build the beginning of the query reply */
	dest=g_string_new("");
	g_string_sprintf(dest,"$To: %s From: Hub-Security $",luce->user_nick->str);
	{
		int min = 0, max = ARRAY_SIZE (hub_cmd) - 2;
		gchar **split_cmd;

		split_cmd=g_strsplit(inp->str," ",2); /* split the string */
		
		do
		{
			int i = (min + max) / 2;
			int cmp = strcmp(*split_cmd, hub_cmd[i].cmd);
			if (cmp == 0)
			{
				fnd=1;
				ret=run_hub_cmd(&hub_cmd[i],luce,&dest,inp);
				break;
			}
			else if (cmp < 0)
				max = i - 1;
			else
				min = i + 1;
		}
		while (min <= max);

		g_strfreev(split_cmd);
	}

	if(!fnd)
	{
		g_string_sprintfa(dest,"Unknown command: %s|",inp->str);
		send_const_str_to_luce(luce,dest->str);
	}
	eoc:
	if(dest!=NULL)
		g_string_free(dest,TRUE);
	if (inp!=NULL)
		g_string_free(inp,TRUE);
	return 0;
}

/*****************************************/
/* display the list of all user commands */
/*****************************************/
static int do_user_help(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	g_string_sprintfa((*dest_base),	"** User command list:\r\n"
											 "\r\n"
											 "+help\r\n"
											 "               display this help\r\n"
											 "+me text\r\n"
											 "               irc like /me\r\n"
											 "+new_chat chatname [password]\r\n"
											 "               create a new multi-users chat named \"chatname\" with an optional\r\n"
											 "               password. The multi-users chat (mchat in short) works like the\r\n"
											 "               normal private chat except you can be more than 2 users in this chat.\r\n"
											 "               Note: it is not possible to create two mchats with the same name.\r\n"
											 "+list chatname\r\n"
											 "               list the users in the chat named \"chatname\".\r\n"
											 "+join chatname [password]\r\n"
											 "               connect to a multi-users chat named \"chatname\".\r\n"
											 "+leave chatname\r\n"
											 "               leave to a multi-users chat named \"chatname\".\r\n"
			

											 "\r\n|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************************/
/*     a +me function like in irc       */
/****************************************/
static int do_user_me(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	GString *out2;
	const char	*rcv;

	if (dest_base == NULL)
		return 0;
	
	if (luce->ext_flag & SILENT_CC)
	{
		return 0;
	}
	rcv = str->str + (sizeof("me ") - 1);
	if (*rcv == '\0')
		return 0;

	out2=g_string_new("");
	if ((*dest_base)->str[0] == '<')
	{
		g_string_sprintfa(out2,"<*%s%s|", luce->user_nick->str, rcv);
		
		GLUS_SEND_TO_EVERYONE(out2);			/* don't free out2 */
	}
	else
	{
		char	*ptr_nick;
		char	dest_nick[64];
		int	i;

		ptr_nick = (*dest_base)->str + strlen("$To: ");
		ptr_nick = strchr(ptr_nick, ' '); /* This strchr can not retur NULL cause the string is create by the hub */
		ptr_nick += strlen(" From: ");
		for (i = 0; (i < 63) && ptr_nick[i] && ptr_nick[i] != ' '; i++)
			dest_nick[i] = ptr_nick[i];
		dest_nick[i] = '\0';
		g_string_sprintfa(out2,"$To: %s From: %s $<*%s*%s|", dest_nick, luce->user_nick->str, luce->user_nick->str, rcv);
		GLUS_SEND_TO_A_NICK(dest_nick,out2);		/* don't free out2 */
	}
	return 0;
}

#define EVERYONE_MSK 0
static KNOWN_CMD user_hub_cmd[]={
	{"+help",       do_user_help,           NULL,	EVERYONE_MSK},		/* ok v0.4.0 */
	{"+join",       cmd_join_public_chat,   NULL,	EVERYONE_MSK},		/* ok v0.4.0 */
	{"+leave",      cmd_leave_public_chat,  NULL,	EVERYONE_MSK},		/* ok v0.4.0 */
	{"+list",       cmd_list_public_chat,   NULL,	EVERYONE_MSK},		/* ok v0.4.0 */
	{"+me",         do_user_me,             NULL,	EVERYONE_MSK},		/* to rewrite */
	{"+new_chat",   cmd_create_public_chat, NULL,	EVERYONE_MSK},		/* ok v0.4.0 */
	{NULL,NULL,NULL},
};

/*************************************************************************/
/* the given string has been sent to any user but always starts with a + */
/* This function processes the string and performs the wanted action     */
/*************************************************************************/
/* output: =0: ok                                */
/*         <0: fatal error (=> abort connection) */
/*         >0: recoverable error (continue)      */
/*************************************************/
int process_user_hub_command(LOCAL_USER_CNX_ENTRY *luce,char *string, char *remote_name)
{
	int ret=0;
	GString *inp;
	GString *dest=NULL;
	char *t;
#if 0
	int fnd=0;
#endif

	inp=g_string_new(string);

	t=inp->str;
	SKIP_SPACE(t);

	if(*t=='\0')
	{
		ret=1;
		goto eoc;		/* non fatal error: nothing entered */
	}

	/* remote trash before the command */
	inp=g_string_erase(inp,0,t-inp->str);
	if(inp->len>1)
		inp=g_string_truncate(inp,inp->len-1);		/* remove trailing | */
	{
		int min = 0, max = ARRAY_SIZE (user_hub_cmd) - 2;
		gchar **split_cmd;

		split_cmd=g_strsplit(inp->str," ",2); /* split the string */

		do
		{
			int i = (min + max) / 2;
			int cmp = strcmp(*split_cmd, user_hub_cmd[i].cmd);
			
			if (cmp == 0)
			{
#if 0
				fnd=1;
#endif
				/* build the beginning of the query reply */
				dest=g_string_new("");
				if((remote_name==NULL)||(strlen(remote_name)==0))
					g_string_sprintf(dest,"<Kernel> ");
				else
					g_string_sprintf(dest,"$To: %s From: %s $",luce->user_nick->str, remote_name);
				ret=run_hub_cmd(&user_hub_cmd[i],luce,&dest,inp);
				break;
			}
			else if (cmp < 0)
				max = i - 1;
			else
				min = i + 1;
		}
		while (min <= max);

		g_strfreev(split_cmd);
	}

#if 0
	/* if the command is not found, we don't return any error message because */
	/* the command can be handled by a script */
	if(!fnd)
	{
		g_string_sprintfa(dest,"Unknown command: %s|",inp->str);
		send_str_to(ptr,dest->str);
	}
#endif
	eoc:
	if(dest!=NULL)
		g_string_free(dest,TRUE);
	g_string_free(inp,TRUE);

	return 0;
}

typedef struct
{
	const char *cmd;
	void (*fnc)(const char *evt_name, const char *evt_emitter, int nb_args, va_list ap, char *xtra_param);
	/* NOTE: all parameters are read-only because you may have more than 1 event handler for each event */
	char *xtra_param;
} KNOWN_EVENT;

static KNOWN_EVENT hub_event_hdl[]={
	{NULL,NULL,NULL},
};

/**************************************************************************************************/
/* in the same way as it exists for PERL scripts, there is an event handler for embedded commands */
/**************************************************************************************************/
/* evt_name and evt_emitter are the same as PERL scripts one */
/* ap is a va_list of nb_args (char *)                       */
/*************************************************************/
void hub_event_command(const char *evt_name, const char *evt_emitter, int nb_args, va_list ap)
{
	int i;
	i=0;
	while(hub_event_hdl[i].cmd!=NULL)
	{
		if(!strcmp(evt_name,hub_event_hdl[i].cmd))
		{
			(hub_event_hdl[i].fnc)(evt_name,evt_emitter,nb_args,ap,hub_event_hdl[i].xtra_param);
		}
		i++;
	}
}


