/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * global_user_if.h: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: global_user_if.h,v 2.18 2003/10/25 12:28:44 ericprev Exp $
*/

#ifndef __GLOBAL_USER_IF_H__
#define __GLOBAL_USER_IF_H__

#include <netinet/in.h>

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------- user info functions --------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************/
/* generic user info structure */
/*******************************/
typedef struct
{
	time_t cnx_start_time;
	const char *user_cnx_type;
	GString *user_nick;
	guint64 shared_size;
	GString *user_description;
	GString *user_mail;
	guint16 user_flag;
	guint16 ext_flag;
	gint16  level;
	GString *client_version;
	struct in_addr user_ip;
} GLUS_USER_INFO;

#include "global_user.h"

/*********************************************/
/* retrieve user information from a nickname */
/*********************************************/
/* output: NULL= not found */
/***************************/
GLUS_USER_INFO *glus_get_user_info(const char *nickname);

/*************************************************************/
/* free a GLUS_USER_INFO obtained with the previous function */
/*************************************************************/
void glus_free_user_info(GLUS_USER_INFO *glus_info);

/*****************/
/* get all users */
/*************************************************/
/* output: a GPtrArray of GLUS_USER_INFO pointer */
/*************************************************/
GPtrArray *glus_get_users_info(void);

/**************************************/
/* get all users having the given IP. */
/*************************************************/
/* output: a GPtrArray of GLUS_USER_INFO pointer */
/*************************************************/
GPtrArray *glus_get_users_info_by_ip(struct in_addr ip);

/****************************************************************/
/* free a GLUS_USER_INFO obtained with the 2 previous functions */
/****************************************************************/
void glus_free_user_info_ptrarray(GPtrArray *glus_info_ptr_array);

/***********************************************/
/* allocate and populate user and op nick list */
/* the lists must be free at the end           */
/***********************************************/
/* NOTE: (each nick is '$$' terminated) */
/****************************************/
void glus_get_nick_lists(GString **user_list, GString **op_list);

/*****************************************************/
/* get the number of users and share size of the hub */
/*****************************************************/
void glus_hub_users_stat(guint64 *shared_size, unsigned int *nb_users);

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------- send functions -------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************************************************************************/
/* send a message to a user having the given name and required user_flag and ext_flag */
/* zero *_msks force GLUS to ignore these flags                                       */
/* output: 0=ok else user not found                                                   */
/* NOTE: the GString reference is stolen, you must not free it.                       */
/**************************************************************************************/
void glus_send_to_named_user(const char *nickname, guint16 uflag_msk, guint16 wanted_uflag,
                             guint16 eflag_msk, guint16 wanted_eflag, GString *msg);

#define GLUS_SEND_TO_A_NICK(nickname,msg)  glus_send_to_named_user(nickname,0,0,0,0,msg)

#define GLUS_SEND_TO_A_NICK_char_ptr(nickname,msg)  glus_send_to_named_user(nickname,0,0,0,0,g_string_new(msg))

/**************************************************************/
/* send a message to a user defined by a GLUS_PARAM structure */
/**************************************************************/
/* the structure is left untouched */
/***********************************/
void glus_send_to_by_glus_param(GLUS_PARAM *param);

/***************************************************************************************/
/* send a message to all users having required user_flag and ext_flag                  */
/* zero *_msks force GLUS to ignore these flags                                        */
/* if nickname is not NULL, the given user is ignored and will not receive the message */
/* NOTE: the GString reference is stolen, you must not free it.                        */
/***************************************************************************************/
void glus_send_to_all_users(const char *nickname, unsigned int uflag_msk, unsigned int wanted_uflag,
                            unsigned int eflag_msk, unsigned int wanted_eflag, GString *msg);

#define GLUS_SEND_TO_EVERYONE(msg)  glus_send_to_all_users(NULL,0,0,0,0,msg)
#define GLUS_SEND_TO_EVERYONE_char_ptr(msg)  glus_send_to_all_users(NULL,0,0,0,0,g_string_new(msg))

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------- user action functions -------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************/
/* kick the user having the given nickname */
/*******************************************/
/* kicking_nickname can be NULL */
/********************************/
void glus_kick_named_user(const char *kicking_nickname, const char *nickname);

/******************************************/
/* ban the user having the given nickname */
/******************************************/
/* kicking_nickname can be NULL */
/********************************/
void glus_ban_named_user(const char *kicking_nickname, const char *nickname, unsigned long duration);

/******************************************/
/* ban the user having the given nickname */
/******************************************/
/* kicking_nickname can be NULL */
/********************************/
void glus_ban_ip(const char *kicking_nickname, const struct in_addr ip, unsigned long duration);

/**************************************************/
/* unban the user having the given nickname or IP */
/**************************************************/
void glus_uban_named_user(const char *nickname);

/*************************************************/
/* disconnect the user having the given nickname */
/*************************************************/
void glus_disconnect_named_user(const char *nickname);

/******************************************************/
/* notify everyone a user is disconneted from the hub */
/******************************************************/
void glus_user_is_disconnected(const char *nickname);

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------- user communication functions ---------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*****************************************************/
/* function used to say a "$Hello" on the connection */
/*****************************************************/
void glus_do_hello(const char *nickname);

/****************************************************/
/* function used to say a "$Quit" on the connection */
/****************************************************/
void glus_do_quit(const char *nickname);

/******************************************************/
/* function used to say a "$MyINFO" on the connection */
/******************************************************/
void glus_do_my_info(const char *nickname, const char *description, const char *user_cnx_type, const guint16 user_flag, const char *user_mail, const guint64 shared_size, const guint16 ext_flag, const gint16 level, const char *client_version);

/*********************************************************/
/* function used to process a $Xsearch on the connection */
/*********************************************************/
void glus_do_xsearch(const char *return_path, const guint8 *file_crc, const guint32 file_length, const char *std_search);

/**********************************************/
/* function called when a $MD4Set is received */
/***************************************************************/
/* g_crc is the file global CRC (MD_BLOC_SIZE bytes)           */
/* l_crc is the set of partial CRC (nb_seg*MD_BLOC_SIZE bytes) */
/* nickname is the return path nickname                        */
/***************************************************************/
void glus_md4set_received(const guint8 *g_crc, const guint32 file_length, const guint8 *l_crc, const guint32 nb_seg, const char *filename, const char *nickname);

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------- misc user info functions ------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************************************************/
/* get the user information for the given nickname */
/***************************************************/
/* output: NULL if user not found */
/**********************************/
GString *txt_glus_get_user_info(const char *nick);

/*******************************************/
/* get the list of ops/users '$' separated */
/*******************************************/
GString *txt_glus_get_oplist(void);
GString *txt_glus_get_userlist(void);

/********************************************************/
/* check if a user with the given nickname is connected */
/********************************************************/
/* output: TRUE=yes, FALSE=no */
/******************************/
gboolean glus_user_is_connected(const char *nickname);

#endif
