/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * global_user.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: global_user.c,v 2.19 2003/11/05 18:16:18 ericprev Exp $
*/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include "global_user_if.h"
#include "g_string_ref.h"

static GPtrArray *glus_array=NULL;

/**************************************/
/* register a new GLOBAL_USER handler */
/**************************************/
void global_user_register(GLOBAL_USER_CALLBACKS *guc)
{
	if(glus_array==NULL)
		glus_array=g_ptr_array_new();

	g_ptr_array_add(glus_array,guc);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------- user info functions --------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*********************************************/
/* retrieve user information from a nickname */
/*********************************************/
/* output: NULL= not found */
/***************************/
GLUS_USER_INFO *glus_get_user_info(const char *nickname)
{
	GLUS_USER_INFO *gui=NULL;
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			gui=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->get_user_info)(nickname);
			if(gui!=NULL)		/* user found */
				break;
		}
	}

	return gui;
}

/*************************************************************/
/* free a GLUS_USER_INFO obtained with the previous function */
/*************************************************************/
void glus_free_user_info(GLUS_USER_INFO *glus_info)
{
	g_string_free(glus_info->user_nick,TRUE);
	g_string_free(glus_info->user_description,TRUE);
	g_string_free(glus_info->user_mail,TRUE);
	g_string_free(glus_info->client_version,TRUE);
	free(glus_info);
}

/*****************/
/* get all users */
/*************************************************/
/* output: a GPtrArray of GLUS_USER_INFO pointer */
/*************************************************/
GPtrArray *glus_get_users_info(void)
{
	GPtrArray *nw;
	int i;

	nw=g_ptr_array_new();
	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->get_users_info)(nw);
		}
	}

	return nw;
}

/**************************************/
/* get all users having the given IP. */
/*************************************************/
/* output: a GPtrArray of GLUS_USER_INFO pointer */
/*************************************************/
GPtrArray *glus_get_users_info_by_ip(struct in_addr ip)
{
	GPtrArray *nw;
	int i;

	nw=g_ptr_array_new();
	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->get_users_info_by_ip)(nw,ip);
		}
	}

	return nw;
}

/*************************************************************/
/* free a GLUS_USER_INFO obtained with the previous function */
/*************************************************************/
void glus_free_user_info_ptrarray(GPtrArray *glus_info_ptr_array)
{
	int i;

	if(glus_info_ptr_array==NULL)
		return;

	for(i=0;i<glus_info_ptr_array->len;i++)
	{
		glus_free_user_info((GLUS_USER_INFO *)(g_ptr_array_index(glus_info_ptr_array,i)));
	}

	g_ptr_array_free(glus_info_ptr_array,TRUE);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------- send functions -------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************************************************************************/
/* send a message to a user having the given name and required user_flag and ext_flag */ 
/* zero *_msks force GLUS to ignore these flags                                       */
/* output: 0=ok else user not found                                                   */
/* NOTE: the GString reference is stolen, you must not free it.                       */
/**************************************************************************************/
void glus_send_to_named_user(const char *nickname, guint16 uflag_msk, guint16 wanted_uflag,
                             guint16 eflag_msk, guint16 wanted_eflag, GString *msg)
{
	GLUS_PARAM nw;
	int i;
	int ret;

	nw.nickname=nickname;
	nw.uflag_msk=uflag_msk;
	nw.wanted_uflag=wanted_uflag;
	nw.eflag_msk=eflag_msk;
	nw.wanted_eflag=wanted_eflag;
	nw.msg=conv_gstring_to_gstring_ref(msg);

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			ret=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->send_to_named_user)(&nw);
			if(ret==0)		/* user found */
				break;
		}
	}

	g_string_ref_free(nw.msg);
}

/**************************************************************/
/* send a message to a user defined by a GLUS_PARAM structure */
/**************************************************************/
/* the structure is left untouched */
/***********************************/
void glus_send_to_by_glus_param(GLUS_PARAM *param)
{
	int i;
	int ret;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			ret=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->send_to_named_user)(param);
			if(ret==0)		/* user found */
				break;
		}
	}
}

/***************************************************************************************/
/* send a message to all users having required user_flag and ext_flag                  */
/* zero *_msks force GLUS to ignore these flags                                        */
/* if nickname is not NULL, the given user is ignored and will not receive the message */
/* NOTE: the GString reference is stolen, you must not free it.                        */
/***************************************************************************************/
void glus_send_to_all_users(const char *nickname, unsigned int uflag_msk, unsigned int wanted_uflag,
                            unsigned int eflag_msk, unsigned int wanted_eflag, GString *msg)
{
	GLUS_PARAM nw;
	int i;

	nw.nickname=nickname;
	nw.uflag_msk=uflag_msk;
	nw.wanted_uflag=wanted_uflag;
	nw.eflag_msk=eflag_msk;
	nw.wanted_eflag=wanted_eflag;
	nw.msg=conv_gstring_to_gstring_ref(msg);

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->send_to_all_users)(&nw);
		}
	}

	g_string_ref_free(nw.msg);
}

/***********************************************/
/* allocate and populate user and op nick list */
/* the lists must be free at the end           */
/***********************************************/
void glus_get_nick_lists(GString **user_list, GString **op_list)
{
	GString *lists[2];
	int i;

	lists[0]=*user_list=g_string_new("");
	lists[1]=*op_list=g_string_new("");

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->get_nick_lists)(lists);
		}
	}
}

/*****************************************************/
/* get the number of users and share size of the hub */
/*****************************************************/
void glus_hub_users_stat(guint64 *shared_size, unsigned int *nb_users)
{
	int i;

	*shared_size=0;
	*nb_users=0;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if((((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->add_hub_users_stat)!=NULL)
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->add_hub_users_stat)(shared_size,nb_users);
		}
	}
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------- user action functions -------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************/
/* kick the user having the given nickname */
/*******************************************/
void glus_kick_named_user(const char *kicking_nickname,const char *nickname)
{
	int i;
	int ret;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			ret=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->kick_named_user)(kicking_nickname,nickname);
			if(ret==0)		/* user found */
				break;
		}
	}
}

/******************************************/
/* ban the user having the given nickname */
/******************************************/
void glus_ban_named_user(const char *kicking_nickname,const char *nickname, unsigned long duration)
{
	int i;
	int ret;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->ban_named_user!=NULL)
			{
				ret=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->ban_named_user)(kicking_nickname,nickname, duration);
				if(ret==0)		/* user found */
					break;
			}
		}
	}
}

/************************************/
/* ban the user having the given ip */
/************************************/
void glus_ban_ip(const char *kicking_nickname, const struct in_addr ip, unsigned long duration)
{
	int i;
	int ret;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->ban_ip!=NULL)
			{
				ret=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->ban_ip)(kicking_nickname, ip, duration);
				if(ret==0)		/* user found */
					break;
			}
		}
	}
}

/**************************************************/
/* unban the user having the given nickname or IP */
/**************************************************/
void glus_uban_named_user(const char *nickname)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->uban_named_user!=NULL)
			{
				/* we relay the unban to all entries */
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->uban_named_user)(nickname);
			}
		}
	}
}

/*************************************************/
/* disconnect the user having the given nickname */
/*************************************************/
void glus_disconnect_named_user(const char *nickname)
{
	int i;
	int ret;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			ret=(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->disconnect_named_user)(nickname);
			if(ret==0)		/* user found */
				break;
		}
	}
}

/*******************************************************/
/* notify everyone a user is disconnected from the hub */
/*******************************************************/
void glus_user_is_disconnected(const char *nickname)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_is_disconnected)(nickname);
		}
	}
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------- user communication functions ---------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*****************************************************/
/* function used to say a "$Hello" on the connection */
/*****************************************************/
void glus_do_hello(const char *nickname)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if((((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_hello)!=NULL)
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_hello)(nickname);
		}
	}
}

/****************************************************/
/* function used to say a "$Quit" on the connection */
/****************************************************/
void glus_do_quit(const char *nickname)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if((((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_quit)!=NULL)
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_quit)(nickname);
		}
	}
}

/******************************************************/
/* function used to say a "$MyINFO" on the connection */
/******************************************************/
void glus_do_my_info(const char *nickname, const char *description, const char *user_cnx_type, const guint16 user_flag, const char *user_mail, const guint64 shared_size, const guint16 ext_flag, const gint16 level, const char *client_version)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if((((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_my_info)!=NULL)
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_my_info)(nickname,description,user_cnx_type,user_flag,user_mail,shared_size,ext_flag,level, client_version);
		}
	}
}

/*********************************************************/
/* function used to process a $Xsearch on the connection */
/*********************************************************/
void glus_do_xsearch(const char *return_path, const guint8 *file_crc, const guint32 file_length, const char *std_search)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if((((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_xsearch)!=NULL)
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->user_do_xsearch)(return_path,file_crc,file_length,std_search);
		}
	}
}

/**********************************************/
/* function called when a $MD4Set is received */ 
/***************************************************************/
/* g_crc is the file global CRC (MD_BLOC_SIZE bytes)           */
/* l_crc is the set of partial CRC (nb_seg*MD_BLOC_SIZE bytes) */ 
/* nickname is the return path nickname                        */
/***************************************************************/
void glus_md4set_received(const guint8 *g_crc, const guint32 file_length, const guint8 *l_crc, const guint32 nb_seg, const char *filename, const char *nickname)
{
	int i;

	if(glus_array)
	{
		for(i=0;i<glus_array->len;i++)
		{
			if((((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->md4set_received)!=NULL)
				(((GLOBAL_USER_CALLBACKS*)(g_ptr_array_index(glus_array,i)))->md4set_received)(g_crc,file_length,l_crc,nb_seg,filename,nickname);
		}
	}
}


/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------- misc user info functions ------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/***************************************************/
/* get the user information for the given nickname */
/***************************************************/
/* output: NULL if user not found */
/**********************************/
GString *txt_glus_get_user_info(const char *nick)
{
	GLUS_USER_INFO *gui=NULL;
	GString *str=NULL;

	gui=glus_get_user_info(nick);
	if(gui!=NULL)
	{
		str=g_string_new("");
#ifdef GLIB_INT_64_BUG
		g_string_sprintf(str,"%s$%s$%s$%s$%u$%d$",
					((gui->user_cnx_type!=NULL)?gui->user_cnx_type:""),
					llunsigned_to_str(gui->shared_size),
					((gui->user_mail->str!=NULL)?gui->user_mail->str:""),
					((gui->user_description->str!=NULL)?gui->user_description->str:""),
					gui->user_flag,
					gui->level);
#else
		g_string_sprintf(str,"%s$%Lu$%s$%s$%u$%d$",
					((gui->user_cnx_type!=NULL)?gui->user_cnx_type:""),
					gui->shared_size,
					((gui->user_mail->str!=NULL)?gui->user_mail->str:""),
					((gui->user_description->str!=NULL)?gui->user_description->str:""),
					gui->user_flag,
					gui->level);
#endif
		glus_free_user_info(gui);
	}
	return str;
}

/*******************************************/
/* get the list of ops/users '$' separated */
/*******************************************/
GString *txt_glus_get_oplist(void)
{
	GString *str,*dummy;
	int i;
	
	glus_get_nick_lists(&dummy,&str);

	g_string_free(dummy,TRUE); /* we don't want the userlist */

	/* replace the "$$" by a "$" */
	for(i=0;i<str->len;i++)
	{
		if((str->str[i]=='$') && (str->str[i+1]=='$'))
		{
			g_string_erase(str,i,1);
		}
	}
	return str;
}

GString *txt_glus_get_userlist(void)
{
	GString *str,*dummy;
	int i;
	
	glus_get_nick_lists(&str,&dummy);

	g_string_free(dummy,TRUE); /* we don't want the oplist */

	/* replace the "$$" by a "$" */
	for(i=0;i<str->len;i++)
	{
		if((str->str[i]=='$') && (str->str[i+1]=='$'))
		{
			g_string_erase(str,i,1);
		}
	}
	return str;
}

/********************************************************/
/* check if a user with the given nickname is connected */
/********************************************************/
/* output: TRUE=yes, FALSE=no */
/******************************/
gboolean glus_user_is_connected(const char *nickname)
{
	gboolean ret=FALSE;
	GLUS_USER_INFO *gui;

	gui=glus_get_user_info(nickname);
	if(gui!=NULL)
	{
		glus_free_user_info(gui);
		ret=TRUE;
	}
	return ret;
}

