/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * emb_perl.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: emb_perl.c,v 2.5 2003/10/25 13:07:18 blusseau Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef WITH_PERL

#include <EXTERN.h>
#include <perl.h>
#include <glib.h>

#include "emb_perl.h"
#include "xs_init.h"
#include "gvar.h"
#include "hub_cmd.h"

/* 1 = clean out filename's symbol table after each request, 0 = don't */
#ifndef DO_CLEAN
#define DO_CLEAN 0
#endif

static PerlInterpreter *my_perl = NULL;
G_LOCK_DEFINE(my_perl);

/* variables utiles: perl_script_dir et perl_script_main */
static void init_perl_interpreter(void)
{
	dSP;									 /* initialize stack pointer		*/
	ENTER;								  /* everything created after here */
	SAVETMPS;							  /* ...is a temporary variable.	*/
	PUSHMARK(SP);						 /* remember the stack pointer	 */
	XPUSHs(sv_2mortal(newSVpv(perl_script_dir,0))); /* push the table onto the stack  */
	PUTBACK;							 /* make local stack pointer global */
	call_pv("dchub_perl_init", G_SCALAR|G_DISCARD);		/* call the function				 */
	PUTBACK;
	FREETMPS;							  /* free that return value		  */
	LEAVE;							  /* ...and the XPUSHed "mortal" args.*/
}


/******************************************/
/* initialize persistent perl interpreter */
/******************************************/
void init_perl(void)
{
	char *embedding[] = { "", NULL};
#if 0
	char *args[] = { "", DO_CLEAN, NULL };
#endif
	int exitstatus=0;
	gchar *emb_main=NULL;

	if((perl_script_dir==NULL)||(perl_script_main==NULL))
	{
		fprintf(stderr,"Perl scripting disabled.\n");
		return;
	}

	if((my_perl = perl_alloc()) == NULL)
	{
		fprintf(stderr, "init_perl: no memory!");
		exit(1);
	}

	/* build the filename of the main .pl */
	emb_main=g_strconcat(perl_script_dir,"/",perl_script_main,NULL);

	/* and load it when perl starts */
	embedding[1]=emb_main;
	perl_construct(my_perl);
	exitstatus=perl_parse(my_perl,xs_init,2,embedding,NULL);
	if(exitstatus)
	{
		fprintf(stderr,"init_perl: perl_parse error\n");
		exit(1);
	}

	g_free(emb_main);
	exitstatus=perl_run(my_perl);
	printf("Perl Interpreter loaded.\n");

	init_perl_interpreter();
}

/************************************/
/* exit persistent perl interpreter */
/************************************/
void exit_perl(void)
{
	if(my_perl!=NULL)
	{
		PL_perl_destruct_level = 0;
		perl_destruct(my_perl);
		perl_free(my_perl);
		my_perl=NULL;
	}
}

/****************************/
/* restart perl interpreter */
/****************************/
void restart_perl(void)
{
	G_LOCK(my_perl);
	exit_perl();
	init_perl();
	G_UNLOCK(my_perl);
}

/****************************************************************/
/* send an event (with its arguments) to the perl event handler */
/****************************************************************/
void send_evt_to_script(const char *evt_name, const char *evt_emitter, int nb_args, ...)
{
	HV *perl_tbl;
	int i;
	va_list ap;
	va_list set;

	/* before sending the event to PERL scripts, send it to hub_cmd event handler */
	va_start(set,nb_args);
	hub_event_command(evt_name,evt_emitter,nb_args,set);
	va_end(set);

	/* no perl interpreter ? */
	if(my_perl==NULL)
		return;

	/* create the perl tbl which will contain the parameters of the called perl function */
	perl_tbl=newHV();
	/* first, the mandatory entries */

	hv_store(perl_tbl,"event",5,newSVpv(evt_name,0),0);
	hv_store(perl_tbl,"nickname",8,newSVpv(evt_emitter,0),0);
	hv_store(perl_tbl,"argc",4,newSViv(nb_args),0);

    /* and then, the optionnal arguments */
	va_start(ap,nb_args);
	for(i=0;i<nb_args;i++)
	{
		char *arg;
		char tmp_num[10];

		arg=va_arg(ap,char*);

		sprintf(tmp_num,"%u",i);
	
		hv_store(perl_tbl,tmp_num,strlen(tmp_num),newSVpv(arg,0),0);

	}

	{
		dSP;									 /* initialize stack pointer		*/
		ENTER;								  /* everything created after here */
		SAVETMPS;							  /* ...is a temporary variable.	*/
		PUSHMARK(SP);						 /* remember the stack pointer	 */
		XPUSHs(sv_2mortal(newRV_noinc((SV*)perl_tbl))); /* push the table onto the stack  */
		PUTBACK;							 /* make local stack pointer global */
		call_pv("dchub_perl_handler", G_SCALAR|G_DISCARD);		/* call the function				 */
		SPAGAIN;							/* refresh stack pointer */
		PUTBACK;
		FREETMPS;							  /* free that return value		  */
		LEAVE;							  /* ...and the XPUSHed "mortal" args.*/
	}
	/* end of the game */
#if 0
	/* not required due to sv_2mortal on perl_tbl */
	hv_undef(perl_tbl);
#endif
	va_end(ap);
}


/*************************************/
/* clear all autoloaded perl handler */
/*************************************/
void clear_autoloaded_perl_handlers(void)
{
	/* no perl interpreter ? */
	if(my_perl==NULL)
		return;

	G_LOCK(my_perl);

	{
		dSP;        							     						/* initialize stack pointer         */
		ENTER;                                          		     		/* everything created after here    */
		SAVETMPS;                                       		     		/* ...is a temporary variable.      */
		PUSHMARK(SP);                                                      	/* remember the stack pointer       */
		XPUSHs(sv_2mortal(newSVpv(perl_script_dir,0)));                     /* push the table onto the stack    */
		PUTBACK;                                                            /* make local stack pointer global  */
		call_pv("dchub_perl_clear_autoloaded_handlers", G_SCALAR|G_DISCARD);/* call the function                */
		PUTBACK;
		FREETMPS;                                                           /* free that return value           */
		LEAVE;                                                              /* ...and the XPUSHed "mortal" args.*/
	}

	G_UNLOCK(my_perl);
}
#else
	void send_evt_to_script(const char *evt_name, const char *evt_emitter, int nb_args, ...)
	{
	}
#endif /* WITH_PERL */
