/* DChub - a Direct Connect Hub for Linux
 *
 * Copyright (C) 2001 Eric Prevoteau
 *
 * db_xml.c: Copyright (C) Yves BLUSSEAU
 *
 * $Id: db_xml.c,v 1.5 2003/11/22 07:39:55 ericprev Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <string.h>
#include <glib.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include "hl_locks.h"
#include "xmltools.h"
#include "db_xml.h"

#ifdef HAVE_LIBDMALLOC
#include <dmalloc.h>    /* Gray Watson's library */
#define show_alloc() dmalloc_log_unfreed()
#define show_stats() dmalloc_log_stats()
#else
#define show_alloc()    /* nothing */
#define show_stats()    /* nothing */
#endif

/* entities used in XML */
const char ENT_DB_DATABASE[]="database";
const char ENT_DB_KEY[]		="key";

/* attribute used in XML */
const char ATTR_DB_NAME[]	="name";
const char ATTR_DB_TYPE[]	="type";
const char ATTR_DB_VALUE[]	="value";

/* type for keys */
const char* DB_TYPE[]={"str","int","float"};

typedef struct s_db_sym {
   char			*name;
   int			type;
   union {
	   char		*str_value;
	   int		int_value;
	   float	float_value;
   };
} t_db_sym;

typedef struct {
	GPtrArray *name;
	GPtrArray *type;
	GPtrArray *value;
	GStringChunk *chunks;
} dump_db_datas;

static char *db_xml_file=NULL;
time_t time_db_file_loaded=0;
HL_MUTEX db_doc_lock=HL_MUTEX_INIT;
xmlDocPtr db_doc=NULL;
HL_MUTEX  db_table_lock=HL_MUTEX_INIT;
GHashTable *db_hash_table=NULL;

static void destroy_t_db_sym (gpointer data) {
	t_db_sym* key=data;
	//printf("Destroying key:%s\n",key->name);
	free(key->name);
	if (key->type==TYPE_STR)
		free(key->str_value);
	free(key);
}

void refresh_db_hash_table() {
	xmlXPathObjectPtr search;
	GString* xpath_expr=g_string_new("");
	HL_LOCK_WRITE(db_table_lock);
	if (db_hash_table) {
		/* Destroy the old hash */
		g_hash_table_destroy(db_hash_table);
	}
	db_hash_table=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, destroy_t_db_sym);

	g_string_sprintf(xpath_expr,"//%s/%s", ENT_DB_DATABASE, ENT_DB_KEY);
	search = getNodeSet (db_doc, xpath_expr->str); /* retrieve all key */
	g_string_free(xpath_expr,TRUE);
	
	if (search) {
		int i;
		DEFINE_BUF_CONV(name);
		xmlNodeSetPtr nodeset = search->nodesetval;
		for (i=0; i < nodeset->nodeNr; i++) {
			xmlNodePtr xml_key_node=nodeset->nodeTab[i];
			xmlChar *xml_name=xmlGetProp(xml_key_node, ATTR_DB_NAME); /* get the key node name */
			if (xml_name) {
				if (strlen(xml_name) > 0) {
					STR_UTF8ToLAT1(xml_name,name);
					xmlChar *xml_type=xmlGetProp(xml_key_node, ATTR_DB_TYPE); /* get the key type */
					if (xml_type) {
						int type;
						for (type=TYPE_STR; type<NO_MORE_TYPE; type++) {
							if (xmlStrcmp(xml_type, DB_TYPE[type]) == 0) {
								t_db_sym *new_db_sym;
								//xmlChar *xml_value=xmlNodeGetContent(xml_key_node);
								xmlChar *xml_value=xmlGetProp(xml_key_node, ATTR_DB_VALUE); /* get the key value */
								DEFINE_BUF_CONV(value);
								new_db_sym=malloc(sizeof(t_db_sym));
								new_db_sym->name=strdup(name);
								new_db_sym->type=type;
								switch(type) {
								  case TYPE_STR:
									STR_UTF8ToLAT1(xml_value, value);
									new_db_sym->str_value=strdup(value);
									//printf("Key Name:[%s] Str:[%s]\n",new_db_sym->name,new_db_sym->str_value);
									break;
								  case TYPE_INT:
									new_db_sym->int_value=xmlXPathCastStringToNumber(xml_value);
									// printf("Key Name:[%s] Int:[%d]\n",new_db_sym->name,new_db_sym->int_value);
									break;
								  case TYPE_FLOAT:
									new_db_sym->float_value=xmlXPathCastStringToNumber(xml_value);
									// printf("Key Name:[%s] Float:[%f]\n",new_db_sym->name,new_db_sym->float_value);
									break;
								}
								g_hash_table_insert(db_hash_table, g_strdup(name), new_db_sym);
								xmlFree(xml_value);
							}
						}	
						xmlFree(xml_type);
					}
				}
				xmlFree(xml_name);
			}
		}
		xmlXPathFreeObject(search);
	}
	HL_UNLOCK_WRITE(db_table_lock);
}

void replace_dom() {
	/* load the new file in memory */
	xmlDocPtr new_db_doc=loadXMLFile(db_xml_file);
	if (new_db_doc) {			
		/* replace old DOM with the new one */
		if (db_doc) {
			xmlFreeDoc(db_doc);
			printf ("Configuration file %s reloaded.\n",db_xml_file);
		}
		db_doc=new_db_doc;
		refresh_db_hash_table();
		time_db_file_loaded=time(NULL);
	}
}

gboolean refresh_db_doc(gboolean LOCK) {
	struct stat db_file_stat;
	if (!stat(db_xml_file, &db_file_stat)) {
		if (time_db_file_loaded < db_file_stat.st_mtime) {
			if (LOCK) {
				HL_LOCK_READ(db_doc_lock);
				/* Recheck the time: avoid multiple reload by different processus */
				if (!stat(db_xml_file, &db_file_stat) &&
					(time_db_file_loaded < db_file_stat.st_atime)) {
					replace_dom();
				}
				HL_UNLOCK_READ(db_doc_lock);
			} else {
				/* don't need to lock and recheck because LOCK == FALSE */
				replace_dom();
			}
		}
	}
	return db_doc ? TRUE : FALSE; /* ok if a DOM already in memory */
}

gboolean loadDBFile(const char* filename) {
	gboolean result=FALSE;
	if (filename) {
		HL_LOCK_WRITE(db_doc_lock);
		if (db_xml_file)
			free(db_xml_file);
		db_xml_file=strdup(filename);
		result=refresh_db_doc(FALSE);
		HL_UNLOCK_WRITE(db_doc_lock);
		if (result)
			printf("Configuration file %s loaded.\n",filename);
	}
	return result;
}

gboolean xml_save_db() {
	gboolean result=FALSE;
	if (xmlSaveFormatFile(db_xml_file, db_doc, 1) == -1) {
		/* an error occured */
		fprintf(stderr,"Error occured saving file %s: %s\n",db_xml_file, strerror(errno));
	} else {
		time_db_file_loaded=time(NULL);
		result=TRUE;
	}
	return result;
}

/****************************/
/* Find a key in XML tree   */
/**********************************************/
/* note: only use internaly (No lock, etc...) */
/**********************************************/
xmlNodePtr find_xml_key(const char* keytosearch) {
	xmlXPathObjectPtr search;
	xmlNodePtr result=NULL;
	DEFINE_BUF_CONV(keytosearchUTF8);
	GString* xpath_expr=g_string_new("");
	STR_LAT1ToUTF8(keytosearch, keytosearchUTF8);
	g_string_sprintf(xpath_expr,"//%s/%s[@%s=\"%s\"]", ENT_DB_DATABASE, ENT_DB_KEY, ATTR_DB_NAME, keytosearchUTF8);
	// printf("XPATH_EXPR:%s\n",xpath_expr->str);
	
	search = getNodeSet (db_doc,xpath_expr->str); /* retrieve all nick */
	if (search) {
		xmlNodeSetPtr nodeset = search->nodesetval;
		if (nodeset->nodeNr != 0)
			result=nodeset->nodeTab[0]; /* Only one key with a given name must exists */
		xmlXPathFreeObject(search);
	}
	g_string_free(xpath_expr,TRUE);
	return result;
}

/************************/
/* add or modify a key  */
/********************************/
/* output: TRUE=ok, FALSE=error */
/****************************************************************************/
/* note: this function doesn't perform any check of the given value         */
/* the db_doc_lock MUST be set in WRITTEN mode before calling this function */
/****************************************************************************/
gboolean xml_create_key(const char *name, const xmlChar* type, const xmlChar* valueUTF8) {
	xmlNodePtr node;
	xmlNodePtr parent;
	xmlNodePtr oldCopyNode=NULL;
	DEFINE_BUF_CONV(strUTF8);
	
	node=find_xml_key(name);
	if (node) {
		parent=node->parent;
		oldCopyNode=xmlCopyNode(node,1);
		xmlSetProp(node, ATTR_DB_TYPE, type);
		xmlSetProp(node, ATTR_DB_VALUE, valueUTF8);
	} else {
		/* create a new node */
		parent = xmlDocGetRootElement(db_doc)->children;
		if (parent != NULL) {
			while (parent != NULL && (xmlStrcmp(parent->name, (const xmlChar *) ENT_DB_DATABASE))) {
				parent=parent->next;
			}	
			if (parent != NULL) {
				/* I'm on the database node */
				node=xmlNewChild(parent, NULL, ENT_DB_KEY, NULL);
				STR_LAT1ToUTF8(name, strUTF8);
				xmlNewProp(node, ATTR_DB_NAME, strUTF8);
				xmlNewProp(node, ATTR_DB_TYPE, type);
				xmlNewProp(node, ATTR_DB_VALUE, valueUTF8 ? valueUTF8 : NULL);
			}
		}
	}
	if (node) {
		if (!xml_save_db()) { /* save the file */
			/* an error occured -> Revert */
			xmlUnlinkNode(node); /* delete the new created node */
			xmlFreeNode(node);
			node=NULL; /* return -> FALSE */
			if (oldCopyNode) {
				xmlAddChild(parent, oldCopyNode); /* recreate the node */
				oldCopyNode=NULL; /* don't free it at exit */
			}
		}
	}
	if (oldCopyNode)
		xmlFreeNode(oldCopyNode);
	
	return node ? TRUE : FALSE;
}

/*****************/
/* remove a key  */
/********************************/
/* output: TRUE=ok, FALSE=error */
/********************************************************************/
/* note: this function doesn't perform any check of the given value */
/********************************************************************/
gboolean xml_delete_key(const char *name) {
	gboolean result=TRUE;

	xmlNodePtr node=find_xml_key(name);
	if (node) { /* return TRUE if the key is not found */
		xmlNodePtr parent=node->parent;
		xmlNodePtr oldCopyNode=xmlCopyNode(node,1);
		xmlUnlinkNode(node); /* delete the node */
		xmlFree(node);
		if (!xml_save_db()) { /* save the file */
			/* an error occured -> Revert */
			xmlAddChild(parent, oldCopyNode); /* recreate the node */
			result=FALSE;
		} else {
			xmlFree(oldCopyNode);
		}
	}
	return result;
}	

t_db_sym* create_new_key(const char* name) {
	t_db_sym* result;
	result=g_hash_table_lookup(db_hash_table, name);
	if (result) {
		/* Destroy the old key */
		g_hash_table_remove(db_hash_table, name);
	}
	/* Create a new key */
	result=malloc(sizeof(t_db_sym));
	result->name=strdup(name);
	g_hash_table_insert(db_hash_table, g_strdup(name), result);
	return result;
}

gboolean db_del(const char* name) {
	gboolean result=FALSE;
	if (name && strlen(name) > 0) {
		HL_LOCK_WRITE(db_doc_lock);
		if (refresh_db_doc(FALSE)) {		
			if (xml_delete_key(name)) {
				HL_LOCK_WRITE(db_table_lock);
				/* Destroy the key */
				result=g_hash_table_remove(db_hash_table, name);
				HL_UNLOCK_WRITE(db_table_lock);
			}
		}
		HL_UNLOCK_WRITE(db_doc_lock);
	}
	return result;
}

gboolean db_str_set(const char* name, const char* value) {
	t_db_sym* key=NULL;
	if (name && strlen(name) > 0) {
		DEFINE_BUF_CONV(valueUTF8);
		STR_LAT1ToUTF8(value, valueUTF8); /* translate value to UTF8 */
		HL_LOCK_WRITE(db_doc_lock);
		if (refresh_db_doc(FALSE)) {	
			if (xml_create_key(name, DB_TYPE[TYPE_STR], valueUTF8)) {
				HL_LOCK_WRITE(db_table_lock);
				key=create_new_key(name);
				if (key) {
					key->type=TYPE_STR;
					if (value)
						key->str_value=strdup(value);
					else
						key->str_value=strdup("");
				}
				HL_UNLOCK_WRITE(db_table_lock);
			}
		}
		HL_UNLOCK_WRITE(db_doc_lock);
	}
	return key ? TRUE : FALSE;
}

char* db_str_get(const char* name) {
	t_db_sym* key;
	char* result=NULL;
	if (name != NULL && refresh_db_doc(TRUE)) {
		// printf("DB_STR_GET for: %s\n",name);
		HL_LOCK_READ(db_table_lock);
		key=g_hash_table_lookup(db_hash_table, name);
		if (key && key->type==TYPE_STR) {
			if (key->str_value)
				result=strdup(key->str_value);
		}
		HL_UNLOCK_READ(db_table_lock);
	}
	return result;
}

gboolean db_int_set(const char* name, int value) {
	t_db_sym* key=NULL;
	if (name && strlen(name) > 0) {
		xmlChar* valueUTF8=xmlXPathCastNumberToString(value);
		HL_LOCK_WRITE(db_doc_lock);
		if (refresh_db_doc(FALSE)) {	
			if (xml_create_key(name, DB_TYPE[TYPE_INT], valueUTF8)) {
				HL_LOCK_WRITE(db_table_lock);
				key=create_new_key(name);
				if (key) {
					key->type=TYPE_INT;
					key->int_value=value;
				}
				HL_UNLOCK_WRITE(db_table_lock);
			}
		}
		HL_UNLOCK_WRITE(db_doc_lock);
		xmlFree(valueUTF8);
	}
	return key ? TRUE : FALSE;
}

gboolean db_int_get(const char* name, int* value) {
	t_db_sym* key;
	gboolean result=FALSE;
	if (name != NULL && refresh_db_doc(TRUE)) {
		// printf("DB_INT_GET for: %s\n",name);
		HL_LOCK_READ(db_table_lock);
		key=g_hash_table_lookup(db_hash_table, name);
		if (key && key->type==TYPE_INT) {
			*value=key->int_value;
			result=TRUE;
		}
		HL_UNLOCK_READ(db_table_lock);
	}
	return result;
}

gboolean db_float_set(const char* name, double value) {
	t_db_sym* key=NULL;
	if (name && strlen(name) > 0) {
		xmlChar* valueUTF8=xmlXPathCastNumberToString(value);
		HL_LOCK_WRITE(db_doc_lock);
		if (refresh_db_doc(FALSE)) {	
			if (xml_create_key(name, DB_TYPE[TYPE_FLOAT], valueUTF8)) {
				HL_LOCK_WRITE(db_table_lock);
				key=create_new_key(name);
				if (key) {
					key->type=TYPE_FLOAT;
					key->float_value=value;
				}
				HL_UNLOCK_WRITE(db_table_lock);
			}
		}
		HL_UNLOCK_WRITE(db_doc_lock);
		xmlFree(valueUTF8);
	}
	return key ? TRUE : FALSE;
}

gboolean db_float_get(const char* name, float* value) {
	t_db_sym* key;
	gboolean result=FALSE;
	if (name != NULL && refresh_db_doc(TRUE)) {
		// printf("DB_FLOAT_GET for: %s\n",name);
		HL_LOCK_READ(db_table_lock);
		key=g_hash_table_lookup(db_hash_table, name);
		if (key && key->type==TYPE_FLOAT) {
			*value=key->float_value;
			result=TRUE;
		}
		HL_UNLOCK_READ(db_table_lock);
	}
	return result;
}

int db_type_get(const char *name) {
	t_db_sym* key;
	int result=-1;
	if (name != NULL && refresh_db_doc(TRUE)) {
		HL_LOCK_READ(db_table_lock);
		key=g_hash_table_lookup(db_hash_table, name);
		if (key) {
			result=key->type;
		}
		HL_UNLOCK_READ(db_table_lock);
	}
	return result;	
}

void dump_db_keys (gpointer hash_key, gpointer hash_value, gpointer pass_data) {
	const char* keyname=hash_key;
	t_db_sym* key=hash_value;
	dump_db_datas* datas=pass_data;
	char buf[128];
	
	/* add the name to the array */
	g_ptr_array_add(datas->name, g_string_chunk_insert(datas->chunks, keyname));
	/* add the type to the array */
	g_ptr_array_add(datas->type, g_string_chunk_insert(datas->chunks, DB_TYPE[key->type]));

	switch (key->type) {
		case TYPE_STR:
			g_ptr_array_add(datas->value, g_string_chunk_insert(datas->chunks, key->str_value));
			break;
		case TYPE_INT:
			sprintf(buf,"%d", key->int_value);
			g_ptr_array_add(datas->value, g_string_chunk_insert(datas->chunks, buf));
			break;
		case TYPE_FLOAT:
			sprintf(buf,"%f", key->float_value);
			g_ptr_array_add(datas->value, g_string_chunk_insert(datas->chunks, buf));
			break;
		default:
			g_ptr_array_add(datas->value,NULL);
			break;
	}
}

/****************************************************************/
/* dump the db content into name, type, value string array      */
/* if any input value==NULL, the value is created               */
/****************************************************************/
/* return: TRUE=ok, FALSE=error */
/********************************/
gboolean db_get_all_keys(GPtrArray **name, GPtrArray **type, GPtrArray **value, GStringChunk **gsc) {
	dump_db_datas datas;
	gboolean result=FALSE;

	if(*name==NULL)
		*name=g_ptr_array_new();

	if(*type==NULL)
		*type=g_ptr_array_new();

	if(*value==NULL)
		*value=g_ptr_array_new();

	if(*gsc==NULL)
		*gsc=g_string_chunk_new(32);

	HL_LOCK_READ(db_table_lock);
	if (refresh_db_doc(TRUE)) {
		datas.name=*name;
		datas.type=*type;
		datas.value=*value;
		datas.chunks=*gsc;
		g_hash_table_foreach(db_hash_table, dump_db_keys, &datas);
		result=TRUE;
	}
	HL_UNLOCK_READ(db_table_lock);
	return result;
}

void xml_dump_db(const char *filename) {
	xmlSaveFormatFile(filename, db_doc,1);
}

void xml_free_db_resource() {
	xmlFreeDoc(db_doc);
	if (db_hash_table)
		g_hash_table_destroy(db_hash_table);
	if (db_xml_file)
		free(db_xml_file);
}

