/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * cnx_if_detect.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: cnx_if_detect.c,v 2.1 2003/10/19 12:48:09 blusseau Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <glib.h>

#ifdef WIN32
	#include <winsock.h>
#else
	#include <netinet/in.h>
#endif


#include "cnx_if_detect.h"
#include "db_xml.h"
#include "gvar.h"

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

static GArray *dyn_if_ip=NULL;		/* array of struct in_addr (IPv4 address) */
G_LOCK_DEFINE(dyn_if_ip);

/*********************************************************/
/* take a list of comma separated network interface name */
/*********************************************************/
/* output: a sorted array of struct in_addr   */
/*         output can be empty but never NULL */
/*         unless a network error occurs      */
/*         in this case, assume everything is */
/*         unchanged                          */
/**********************************************/
static GArray *build_dyn_ip_array(const char *if_name_list)
{
#if 0
	struct ifconf ifc;
#define TMPBUFSIZ 8192
	char buf[TMPBUFSIZ];
	GArray *ip_array=NULL;
	int s;
	int n;
	gchar **if_fields;

	s=socket(AF_INET, SOCK_DGRAM, 0);
	if(s<0)
	{
		perror("build_dyn_ip_array: socket creation fail");
		return ip_array;
	}
	if_fields=g_strsplit(if_name_list,",",0);
	ip_array=g_array_new(FALSE,FALSE,sizeof(struct in_addr));

	ifc.ifc_len = sizeof (buf);
	ifc.ifc_buf = buf;

	if (ioctl(s, SIOCGIFCONF, (char *)&ifc)<0)
		perror("build_dyn_ip_array: ioctl fail");
	else
	{
		struct ifreq *ifr;
		ifr = ifc.ifc_req;

		for (n = ifc. ifc_len / sizeof (struct ifreq); --n >= 0; ifr++)
		{
			if(ifr->ifr_name;
			/*
			 * We must be careful that we don't use an interface
			 * devoted to an address family other than those intended;
			*/

			if(ifr->ifr_addr.sa_family != AF_INET)
				continue;

			if(ioctl(s, SIOCGIFFLAGS, (char *) ifr)<0)
			{
				perror("build_dyn_ip_array: ioctl on if fail");
				continue;
			}

			/*
			* Skip boring cases.
			*/
			if ( (ifr->ifr_flags & IFF_UP) == 0 )	/* keep only up interface */
				continue;
		}
	} 

	g_strfreev(if_fields);
	close(s);
	return ip_array;
#endif
	return NULL;
}

/***********************************************************/
/* remove existing values of vals_to_remove from to_update */
/***********************************************************/
/* input: 2 arrays of struct in_addr */
/*************************************/
static void array_substract(GArray *to_update, const GArray *vals_to_remove)
{
	int i,j;
	struct in_addr *s, *d;

	for(i=0;i<vals_to_remove->len;i++)
	{
		d=&(g_array_index(vals_to_remove,struct in_addr,i));

		for(j=0;j<to_update->len;j++)
		{
			s=&(g_array_index(to_update,struct in_addr,j));

			if(d->s_addr<s->s_addr)
				break;		/* we are above the value to remove */

			if(d->s_addr==s->s_addr)
			{
				g_array_remove_index(to_update,j);
				break;
			}
		}
	}
}

/*************************************************************************/
/* check the KEY 'DYN_IF' and retrieve the IP of these interface to keep */
/* their information accurate.                                           */
/*************************************************************************/
/* if the IP of an interface has changed, all the connections with the */
/* DYNAMIC flag set are closed (shutdown).                             */
/***********************************************************************/
void update_dynamic_if_ip(void)
{
	char *dyn_if;
	GArray *changed_ip_array=NULL;

	dyn_if=db_str_get("DYN_IF");
	
	G_LOCK(dyn_if_ip);
	if((dyn_if==NULL)||(strlen(dyn_if)==0))
	{
		/* the key does not exist or is empty */
		if(dyn_if_ip!=NULL)
		{
			/* but previously, there was dynamic IP Interface, just wipe them */
			changed_ip_array=dyn_if_ip;
			dyn_if_ip=NULL;
		}
	}
	else
	{
		/* the key exists and is not empty */
		if(dyn_if_ip!=NULL)
		{
			/* the array previously does not exist */
			GArray *new_if_array=build_dyn_ip_array(dyn_if);

			if(new_if_array!=NULL)
			{
				if((new_if_array->len==dyn_if_ip->len)&&
					(!memcmp(new_if_array->data,dyn_if_ip->data,sizeof(struct in_addr)*dyn_if_ip->len)))
				{
					/* the array is unchanged */
					g_array_free(new_if_array,TRUE);
				}
				else
				{	/* the array has changed */
					changed_ip_array=dyn_if_ip;
					dyn_if_ip=new_if_array;
		
					/* remove still existing IP of the changed_ip array */
					array_substract(changed_ip_array,dyn_if_ip);
				}
			}
		}
		else
		{
			dyn_if_ip=build_dyn_ip_array(dyn_if);
		}
	}
	G_UNLOCK(dyn_if_ip);

	if(dyn_if)
		free(dyn_if);

	/* if an array of changing IP exists, close dynamic connection using them */
	if(changed_ip_array!=NULL)
	{
		if(changed_ip_array->len)
		{
		}

		g_array_free(changed_ip_array,TRUE);
	}
}

/********************************************************************/
/* check if the given address is the address of a dynamic interface */
/********************************************************************/
static gboolean is_a_dynamic_address(struct sockaddr_in *ad)
{
	int ret=FALSE;
	int i;

	G_LOCK(dyn_if_ip);
	if((dyn_if_ip!=NULL)&&(dyn_if_ip->len))
	{
		for(i=0;i<dyn_if_ip->len;i++)
		{
			if(g_array_index(dyn_if_ip,struct in_addr,i).s_addr==ad->sin_addr.s_addr)
			{
				ret=TRUE;
				break;
			}
		}
	}
	G_UNLOCK(dyn_if_ip);

	return ret;
}

/*************************************/
/* identify the type of a connection */
/*************************************/
CNX_IF_TYPE identify_if_of_cnx(int socket_fd)
{
	struct sockaddr ad;
#ifdef WIN32
	int	lad;
#else
	socklen_t lad;
#endif
	if(getsockname(socket_fd,&ad,&lad)==0)
	{
		if(ad.sa_family==AF_INET)
		{
			/* we only handle IPv4 address */
			if(is_a_dynamic_address((struct sockaddr_in*)&ad))
				return CIT_DYNAMIC;
			return CIT_STATIC;
		}
	}

	return CIT_UNKNOWN;		/* unknown type of connection */
}

/***************************************************/
/* retrieve the remote IP of an AF_INET connection */
/***************************************************/
void get_remote_if_ip(int sock_fd,struct in_addr *remote_ip)
{
	struct sockaddr ad;
#ifdef WIN32
	int	lad;
#else
	socklen_t lad;
#endif
	if(getpeername(sock_fd,&ad,&lad)==0)
	{
		if(ad.sa_family==AF_INET)
		{
			/* we only handle IPv4 address */
			*remote_ip=*((struct in_addr*)&ad);
		}
	}
	else
	{
		perror("get_remote_if_ip");
		memset(remote_ip,0,sizeof(struct in_addr));
	}
}

