/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * chunk.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: chunk.c,v 2.9 2003/02/25 14:35:16 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <string.h>
#include <stdarg.h>
#include <glib.h>

#include "chunk.h"
#include "chunk_codes.h"
#include "toolkit.h"

/*
Chunk format:

 1 guint16: chunk_type
 1 guint16: nb_subchunk (=N)
 1 guint32: chunk_length  	(the size does not includes these header, only the size users by sub-chunks)
 1 guint32: CRC 
 N sub-chunks

Sub-Chunk format:
 1 guint16: subchunk_type
 1 guint16: subchunk_length (=M) (the size is the size of the data of the sub-chunk without this header)
 M bytes.
 
All Values are in network byte order.
 */

static void compute_header_crc(guint8 *header_crc)
{
	int i;
	int crc_offset=HCHUNK_SIZE-sizeof(guint32);

	header_crc[crc_offset]=0;
	header_crc[crc_offset+1]=0;
	header_crc[crc_offset+2]=0;
	header_crc[crc_offset+3]=0;

	for(i=0;i<(HCHUNK_SIZE-sizeof(guint32));i++)
	{
		header_crc[crc_offset+(i%sizeof(guint32))]^=header_crc[i];
	}
}

/************************************************************/
/* convert a chunk into its planar structure (to save/send) */
/************************************************************/
GByteArray *chunk_convert_chunk_to_mem(CHUNK_CORE *cc)
{
	GByteArray *gba;
	guint16 tmp_16;
	guint32 tmp_32;
	int i;
	guint32 tmp_size=0;

	if(cc==NULL)
		return NULL;

	gba=g_byte_array_new();
	/* first, build the subchunk part */
	for(i=0;i<cc->nb_subchunk;i++)
	{
		/* subchunk type */
		tmp_16=htons(cc->subchunk[i].chunk_type);
		g_byte_array_append(gba,(void*)&tmp_16,sizeof(tmp_16));

		/* subchunk length */
		tmp_16=htons(cc->subchunk[i].chunk_size);
		g_byte_array_append(gba,(void*)&tmp_16,sizeof(tmp_16));

		/* subchunk content */
		if(cc->subchunk[i].chunk_size)
			g_byte_array_append(gba,cc->subchunk[i].chunk_content,cc->subchunk[i].chunk_size);

		tmp_size+=(SHCHUNK_SIZE+cc->subchunk[i].chunk_size);
	}

	/* now, prepend the chunk header */
	/* let's start with a dummy crc */
	tmp_32=0;
	g_byte_array_prepend(gba,(void*)&tmp_32,sizeof(tmp_32));

	/* the size */
	tmp_32=htonl(tmp_size);
	g_byte_array_prepend(gba,(void*)&tmp_32,sizeof(tmp_32));

	/* the number of chunk */
	tmp_16=htons(cc->nb_subchunk);
	g_byte_array_prepend(gba,(void*)&tmp_16,sizeof(tmp_16));
	
	/* and the chunk type */
	tmp_16=htons(cc->chunk_type);
	g_byte_array_prepend(gba,(void*)&tmp_16,sizeof(tmp_16));

	compute_header_crc(gba->data);

	return gba;
}

/******************************************************/
/* convert parameters into a planar structure         */
/* use SC_PARAM_* to provide parameters without error */
/******************************************************/
GByteArray *chunk_convert_param_to_mem(guint16 chunk_type, guint16 nb_subchunk, ...)
{
	GByteArray *gba;
	va_list va;
	guint32 tmp_size=0;
	guint16 tmp_16;
	guint16 size;
	guint32 tmp_32;
	int i;

	va_start(va,nb_subchunk);
	gba=g_byte_array_new();

	/* scan all provided chunk ands copy them in gba */
	for(i=0;i<nb_subchunk;i++)
	{
		/* subchunk type */
		tmp_16=htons(va_arg(va,int));
		g_byte_array_append(gba,(void*)&tmp_16,sizeof(tmp_16));

		/* subchunk length */
		size=va_arg(va,int);
		tmp_16=htons(size);
		g_byte_array_append(gba,(void*)&tmp_16,sizeof(tmp_16));

		/* subchunk content */
		g_byte_array_append(gba,va_arg(va,void*), size);

		tmp_size+=(SHCHUNK_SIZE+size);
	}

	/* now, prepend the chunk header */
	/* let's start with a dummy crc */
	tmp_32=0;
	g_byte_array_prepend(gba,(void*)&tmp_32,sizeof(tmp_32));

	/* the size */
	tmp_32=htonl(tmp_size);
	g_byte_array_prepend(gba,(void*)&tmp_32,sizeof(tmp_32));

	/* the number of chunk */
	tmp_16=htons(nb_subchunk);
	g_byte_array_prepend(gba,(void*)&tmp_16,sizeof(tmp_16));
	
	/* and the chunk type */
	tmp_16=htons(chunk_type);
	g_byte_array_prepend(gba,(void*)&tmp_16,sizeof(tmp_16));

	compute_header_crc(gba->data);

	va_end(va);

	return gba;
}

/*************************************************************/
/* check if header (always HCHUNK_SIZE byte) has a valid CRC */
/*************************************************************/
/* output: 0=no, 1=yes */
/***********************/
static int header_has_a_valid_crc(guint8 *header_crc)
{
	int i;
	int crc_offset=HCHUNK_SIZE-sizeof(guint32);
	guint8 tbl[4];

	tbl[0]=header_crc[crc_offset];
	tbl[1]=header_crc[crc_offset+1];
	tbl[2]=header_crc[crc_offset+2];
	tbl[3]=header_crc[crc_offset+3];

	for(i=0;i<(HCHUNK_SIZE-sizeof(guint32));i++)
	{
		tbl[(i%sizeof(guint32))]^=header_crc[i];
	}

	if((tbl[0]|tbl[1]|tbl[2]|tbl[3])!=0)
		return 0;
	return 1;
}

/*******************************************/
/* convert a planar structure into a chunk */
/************************************************************************************/
/* output: the decoded chunk_core and data used by the decoder are removed from gba */
/*         else NULL and too_small is set to TRUE if there is not enough data       */
/*         available.                                                               */
/*         if *too_small==FALSE and the output is NULL, is_invalid is set to TRUE   */
/*         if the buffer has no meaning                                             */
/************************************************************************************/
CHUNK_CORE *chunk_convert_mem_to_chunk(GByteArray *gba, gboolean *too_small, gboolean *is_invalid)
{
	guint32 chunk_length;
	CHUNK_CORE *nw;
	unsigned int offset;
	int i,mx;
	guint8 *dest, *src;

	if(gba==NULL)
		return NULL;

	/* do we have at least a header ? */
	if(gba->len<HCHUNK_SIZE)
	{
		*too_small=TRUE;
		return NULL;
	}

	if(!header_has_a_valid_crc(gba->data))
	{
		*is_invalid=TRUE;
		*too_small=FALSE;
		return NULL;
	}

	chunk_length=ntohl(HCHUNKF_CHUNK_LEN(gba));

	if((chunk_length+HCHUNK_SIZE)>gba->len)
	{
		printf("not enough data. Want: %d have: %d\n",chunk_length+HCHUNK_SIZE,gba->len);
		*too_small=TRUE;
		return NULL;
	}

	*too_small=FALSE;
	nw=malloc(sizeof(CHUNK_CORE));
	if(nw==NULL)
		return NULL;

	/* initialize chunk core */
	nw->chunk_type=htons(HCHUNKF_CTYPE(gba));
	nw->nb_subchunk=htons(HCHUNKF_NB_CHUNK(gba));
	if(nw->nb_subchunk==0)
	{
		/* no subchunk ? */
		nw->subchunk=NULL;
		offset=HCHUNK_SIZE;
		goto no_subchunk;
	}

	nw->subchunk=calloc(sizeof(SUBCHUNK_CORE),nw->nb_subchunk);
	if(nw==NULL)
	{
		chunk_free(nw);
		return NULL;
	}

	/* start subchunk decoding */
	offset=HCHUNK_SIZE;
	for(i=0;i<nw->nb_subchunk;i++)
	{
		nw->subchunk[i].chunk_type=ntohs(SHCHUNKF_CTYPE(gba,offset));
		nw->subchunk[i].chunk_size=ntohs(SHCHUNKF_SCHUNK_LEN(gba,offset));
		if(nw->subchunk[i].chunk_size==0)
		{
			nw->subchunk[i].chunk_content=NULL;
		}
		else
		{
			nw->subchunk[i].chunk_content=malloc(nw->subchunk[i].chunk_size);
			if(nw->subchunk[i].chunk_content==NULL)
			{
				chunk_free(nw);
				return NULL;
			}
			memcpy(nw->subchunk[i].chunk_content,SHCHUNKF_DATA_ADR(gba,offset),nw->subchunk[i].chunk_size);
		}
		offset+=(SHCHUNK_SIZE+nw->subchunk[i].chunk_size);	/* move to the next start of chunk */
	}

	no_subchunk:
	/* now, offset is at the end of the chunk */
	/* unfortunatelly, there is no g_byte_array function to remove several bytes at the same time */
	dest=&(gba->data[0]);		/* dest position */
	src=&(gba->data[offset]);	/* source position */
	mx=gba->len-offset;			/* amount of bytes to move */
	i=0;
	while(i<mx)
	{
		*dest++=*src++;
		i++;
	}
	g_byte_array_set_size(gba,mx);	/* resize the array */

	return nw;
}

/*******************************/
/* free memory used by a chunk */
/*******************************/
void chunk_free(CHUNK_CORE *cc)
{
	int i;

	if(cc==NULL)
		return;

	/* first, build the subchunk part */
	if(cc->subchunk!=NULL)
	{
		for(i=0;i<cc->nb_subchunk;i++)
		{
			/* subchunk content */
			if((cc->subchunk[i].chunk_size)&&(cc->subchunk[i].chunk_content!=NULL))
				free(cc->subchunk[i].chunk_content);
		}

		free(cc->subchunk);
	}

	free(cc);
}

/* -------------- misc functions -------------- */
/********************************************/
/* create a new chunk core without subchunk */
/********************************************/
CHUNK_CORE *chunk_core_new(guint16 chunk_type)
{
	CHUNK_CORE *cc;

	cc=malloc(sizeof(CHUNK_CORE));
	if(cc==NULL)
		return cc;

	cc->chunk_type=chunk_type;
	cc->nb_subchunk=0;
	cc->subchunk=NULL;
	return cc;
}

/********************************/
/* append a subchunk to a chunk */
/*************************************************************************/
/* note: The chunk content is duplicated and a copy is used in the chunk */
/*************************************************************************/
void chunk_core_append_subchunk(CHUNK_CORE *cc,guint16 chunk_type, guint16 chunk_size, const void *chunk_content)
{
	SUBCHUNK_CORE *tmp;
	SUBCHUNK_CORE *sc;

	tmp=realloc(cc->subchunk,(cc->nb_subchunk+1)*sizeof(SUBCHUNK_CORE));
	if(tmp==NULL)	/* fail to realloc */
		return;

	cc->subchunk=tmp;
	sc=&(cc->subchunk[cc->nb_subchunk]);

	sc->chunk_type=chunk_type;
	sc->chunk_size=chunk_size;
	sc->chunk_content=malloc(chunk_size);
	if(sc->chunk_content==NULL)
	{
		fprintf(stderr,"chunk_core_append_subchunk: out of memory, subchunk discarded.\n");
		return;	/* abort */
	}

	memcpy(sc->chunk_content, chunk_content, sc->chunk_size);
	cc->nb_subchunk++;
}

/**********************************************************************/
/* search inside a CHUNK_CORE for a SUBCHUNK_CORE having the given ID */
/**********************************************************************/
/* output: NULL if not found, else a pointer on the SUBCHUNK_CORE */
/******************************************************************/
SUBCHUNK_CORE *chunk_get_subcore(CHUNK_CORE *cc, guint16 chunk_type)
{
	int i;
	if(cc->subchunk==NULL)
		return NULL;

	for(i=0;i<cc->nb_subchunk;i++)
	{
		if(cc->subchunk[i].chunk_type==chunk_type)
			return &(cc->subchunk[i]);
	}
	return NULL;
}

/*******************************************************************/
/* extract gint or guint from a subchunk core. SC size must be the */
/* sizeof g*int8, g*int16 or g*int32                               */
/*******************************************************************/
gint32 subchunk_get_gint(SUBCHUNK_CORE *sc)
{
	gint32 res;

	switch(sc->chunk_size)
	{
		case 1:	/* 1 gint8 */	
					res=*((gint8*)sc->chunk_content);
					break;
	
		case 2:	/* 1 gint16 */
					res=GINT16_FROM_BE(GET_UAA_GINT16(sc->chunk_content));
					break;

		case 4:	/* 1 gint32 */
					res=GINT32_FROM_BE(GET_UAA_GINT32(sc->chunk_content));
					break;

		default:	/* unknown size */
					fprintf(stderr,"suchunk_get_gint: invalid size (%d) for subchunk %d\n",sc->chunk_size,sc->chunk_type);
					res=0;
					break;
	}
	return res;
}

guint32 subchunk_get_guint(SUBCHUNK_CORE *sc)
{
	guint32 res;

	switch(sc->chunk_size)
	{
		case 1:	/* 1 guint8 */	
					res=*((guint8*)sc->chunk_content);
					break;
	
		case 2:	/* 1 guint16 */
					res=GUINT16_FROM_BE(GET_UAA_GUINT16(sc->chunk_content));
					break;

		case 4:	/* 1 guint32 */
					res=GUINT32_FROM_BE(GET_UAA_GUINT32(sc->chunk_content));
					break;

		default:	/* unknown size */
					fprintf(stderr,"suchunk_get_guint: invalid size (%d) for subchunk %d\n",sc->chunk_size,sc->chunk_type);
					res=0;
					break;
	}
	return res;
}

guint64 subchunk_get_guint64(SUBCHUNK_CORE *sc)
{
	guint64 res;

	switch(sc->chunk_size)
	{
		case 1:	/* 1 guint8 */	
					res=*((guint8*)sc->chunk_content);
					break;
	
		case 2:	/* 1 guint16 */
					res=GUINT16_FROM_BE(GET_UAA_GUINT16(sc->chunk_content));
					break;

		case 4:	/* 1 guint32 */
					res=GUINT32_FROM_BE(GET_UAA_GUINT32(sc->chunk_content));
					break;

		case 8:	/* 1 guint64 */
					res=GUINT64_FROM_BE(GET_UAA_GUINT64(sc->chunk_content));
					break;

		default:	/* unknown size */
					fprintf(stderr,"suchunk_get_guint: invalid size (%d) for subchunk %d\n",sc->chunk_size,sc->chunk_type);
					res=0;
					break;
	}
	return res;
}

/************************************************************************************************/
/* extract a string from a subchunk and ASSIGN it to the given gstring (the gstring must exist) */
/************************************************************************************************/
/* output: TRUE if the previous value is different of the new one, else FALSE */
/******************************************************************************/
gboolean subchunk_assign_gstring(GString *str,SUBCHUNK_CORE *sc)
{
	if( (!strncmp(str->str,sc->chunk_content,sc->chunk_size)) &&
	    (str->str[sc->chunk_size]=='\0') )
		return FALSE;	/* same value */

	g_string_truncate(str,0);

	if(sc->chunk_size)
	{
		int i;

		for(i=0;i<sc->chunk_size;i++)
		{
			g_string_append_c(str,sc->chunk_content[i]);
		}
	}

	return TRUE;
}

/****************************************/
/* convert chunk error code into string */
/****************************************/
const char *sc_num_err_to_str(gint err_code)
{
	switch(err_code)
	{
		case ERR_INVALID_CHALLENGE_REPLY:
				return "Invalid challenge reply";
	}
	return "Unknown error code";
}


