/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * plg_slots.c: Copyright (C) Dirk Rieger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: plugin_slots.c,v 2.2 2003/11/28 14:50:36 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include "dchub_plugin_api.h"

typedef struct
{
	GString* source_nick;
	int free_slot;
	int max_slot;
} SLOTS_ENTRY;

/***********************/
/* arry of SLOTS_ENTRY */
/***********************/
static GArray *slots_array=NULL;

G_LOCK_DEFINE_STATIC(slots_array);

static void plg_find_free_slots(const char *source_nick,const char *dest_nick)
{
	int i;
	int z=0;

	GString *reply;
	GString *list;

	reply=g_string_new("");
	if(dest_nick!=NULL)
	{
		g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
	}
	g_string_append(reply,"<INFO> ");

	list=g_string_new("");

	G_LOCK(slots_array);

	if(slots_array!=NULL)
	{
		for(i=slots_array->len-1;i>=0;i--)
		{
			SLOTS_ENTRY *sle;
			sle=&(g_array_index(slots_array,SLOTS_ENTRY,i));

			if(sle->free_slot>0)
			{
				g_string_append(list,sle->source_nick->str);
				g_string_sprintfa(list," (%d/%d)\r\n",sle->free_slot,sle->max_slot);
				z++;
			}
		}
		g_string_sprintfa(reply,"*** Free Slots (without warranty) - stored users: %i/%i ***\r\n\r\n%s",z,slots_array->len,list->str);
	}
	else
	{
		g_string_sprintfa(reply,"*** Free Slots (without warranty) - stored users: 0/0 ***\r\n");
	}
	G_UNLOCK(slots_array);

	if(z==0)
		g_string_append(reply,"...hmm - seems nobody has a free slot\r\n");

	g_string_append_c(reply,'|');

	plugin_send_to_named_user(source_nick, reply->str);

	g_string_free(reply,TRUE);
	g_string_free(list,TRUE);

	return;
}

static inline void plg_delete_slots_entry(int num)
{
	SLOTS_ENTRY *sld;

	sld=&(g_array_index(slots_array,SLOTS_ENTRY,num));
	if(sld->source_nick)
		g_string_free(sld->source_nick,TRUE);
	slots_array=g_array_remove_index_fast(slots_array,num);
}

static void plg_find_slots_by_nickname_uniq(const char *source_nick)
{
	if(slots_array!=NULL)
	{
		int i;
		for(i=slots_array->len-1;i>=0;i--)
		{
			SLOTS_ENTRY *slo;
			slo=&(g_array_index(slots_array,SLOTS_ENTRY,i));
			if(!strcmp(slo->source_nick->str,source_nick))
			{
				plg_delete_slots_entry(i);
				break;
			}
		}
	}
	return;
}

static void plg_del_slots_on_quit(const char *source_nick)
{
	G_LOCK(slots_array);
	plg_find_slots_by_nickname_uniq(source_nick);
	G_UNLOCK(slots_array);
}

static void plg_add_slots(const char *source_nick, const char *message)
{
	SLOTS_ENTRY sle;

	/* admire the power of scanf (beware, there is ctrl-e in the pattern */
	if(sscanf(message,"$SR %*[^]%*[^ ] %d/%d",&sle.free_slot,&sle.max_slot)!=2)
		return;

	G_LOCK(slots_array);
	if(slots_array!=NULL)
		plg_find_slots_by_nickname_uniq(source_nick);

	sle.source_nick=g_string_new(source_nick);

	if(slots_array==NULL)
		slots_array=g_array_new(FALSE,FALSE,sizeof(SLOTS_ENTRY));

	slots_array=g_array_append_val(slots_array,sle);

	G_UNLOCK(slots_array);
	return;
}


/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
/* ############################## hub interface ############################# */
/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
static void process_command(const char *source_nick, const char *dest_nick, const char *chat)
{
	if(chat[0]=='+')		/* it is a command */
	{
		GString *cmd;
		char *a;
		gchar *param=NULL;

		cmd=g_string_new(chat);
		g_string_truncate(cmd,cmd->len-1);	/* remove the trailing '|' */

		a=strchr(cmd->str,' ');
		if(a!=NULL)
		{
			if(strlen(a+1)!=0)
			{
				param=g_strdup(a+1);
			}
			g_string_truncate(cmd,a-cmd->str);	/* truncate where the space is */
		}

		if(!strcmp(cmd->str,"+slots"))
			plg_find_free_slots(source_nick,dest_nick);

		if(param)
			g_free(param);
		g_string_free(cmd,TRUE);
	}
}

static void globalchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,3).varvalue;

		if(strlen(fullmsg)>0)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				ptr++;
				process_command(source_nick,NULL,ptr);
			}
      }
	}
	return;
}

static void privchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *dest_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		dest_nick=g_array_index(param,PLUGIN_PARAM,3).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,4).varvalue;
		if(fullmsg!=NULL)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				*ptr++;
				process_command(source_nick,dest_nick,ptr);
			}
		}
	}
	return;
}

static void sr_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *nickname;
		const char *message;

		nickname=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		message=g_array_index(param,PLUGIN_PARAM,3).varvalue;
		plg_add_slots(nickname,message);
	}
	return;
}

static void quit_fnc(const GArray *param)
{
   if(param!=NULL)
	{
		const char *nickname;
		nickname=g_array_index(param,PLUGIN_PARAM,1).varvalue;

		plg_del_slots_on_quit(nickname);
	}
   return;
}

/*****************************************************/
/* this function is called when the plugin is loaded */
/*****************************************************/
int plugin_beginning(PLUGIN *plug)
{
	printf("beginning of the SLOTS plugin\n");

	register_dchub_plugin_event(plug,"sr",sr_fnc);
	register_dchub_plugin_event(plug,"globalchat", globalchat_fnc);
	register_dchub_plugin_event(plug,"privchat", privchat_fnc);
	register_dchub_plugin_event(plug,"quit", quit_fnc);

	return 0;		/* success */
}

/************************************************************/
/* this function is called just before unloading the plugin */
/************************************************************/
int plugin_end(PLUGIN *plug)
{
	printf("end of the SLOTS plugin\n");
	return 0;		/* success */
}

