/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * plg_seen.c: Copyright (C) Dirk Rieger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: plugin_seen.c,v 2.3 2003/11/23 06:59:54 ericprev Exp $
*/

#include <stdio.h>
#include <glib.h>

#include <time.h>
#include <string.h>

#include "dchub_plugin_api.h"
#include "../src/users_xml.h"

typedef struct
{
	GString* seen_nick;
	GString* nick_ip;
	int seen_event;
	time_t seen_time;
} SEEN_ENTRY;

/**********************/
/* arry of SEEN_ENTRY */
/**********************/
static GArray *seen_array=NULL;

G_LOCK_DEFINE_STATIC(seen_array);

static void plg_find_seen_by_nickname(const char *source_nick,const char *dest_nick,const char *nickname)
{
	struct tm *stime;
	char tmp_date_buf[512];
	int i;
	int f=FALSE;
	GString *reply;

	reply=g_string_new("");
	if(dest_nick!=NULL)
	{
		g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
	}

	if(nickname==NULL || strlen(nickname)>40)
	{
		g_string_append(reply,"<INFO> Please use this command the following way:\r\n+seen search_name");
	}
	else
	{
		g_string_append(reply,"<INFO> Hmmm - let times think...\r\n\r\n");

		G_LOCK(seen_array);
		if(seen_array!=NULL)
		{
			for(i=seen_array->len-1;i>=0;i--)
			{
				SEEN_ENTRY *nw;
				nw=&(g_array_index(seen_array,SEEN_ENTRY,i));
				if(strcmp(nw->seen_nick->str,nickname) == 0)
				{
					stime=localtime(&nw->seen_time);

					g_string_append(reply,nw->seen_nick->str);
					g_string_append_c(reply,' ');

					if(nw->seen_event == 1)
						strftime(tmp_date_buf,sizeof(tmp_date_buf)-1,"is online Hub since %A, %d %B %Y, %X",stime);
					else
						strftime(tmp_date_buf,sizeof(tmp_date_buf)-1,"left this hub at %A, the %d %B %Y, %X",stime);

					g_string_append(reply,tmp_date_buf);
					g_string_append(reply,"\r\n");

					f = TRUE;
					break;
				}
			}
			if (!f)
			{
				int first_hit = TRUE;
				for(i=seen_array->len-1;i>=0;i--)
				{
					SEEN_ENTRY *nw;
					nw=&(g_array_index(seen_array,SEEN_ENTRY,i));
					if(strstr(nw->seen_nick->str,nickname))
					{
						if (first_hit)
						{
							g_string_sprintfa(reply,"I've not seen <%s> , but maybe you ment...\r\n",nickname);
							first_hit = FALSE;
							f = TRUE;
						}
						stime=localtime(&nw->seen_time);
						g_string_append(reply,nw->seen_nick->str);
						if (nw->seen_event==1)
							strftime(tmp_date_buf,sizeof(tmp_date_buf)-1," - Online since %A %x, %X",stime);
						else
							strftime(tmp_date_buf,sizeof(tmp_date_buf)-1," - Offline since %A %x, %X",stime);
						g_string_append(reply,tmp_date_buf);
						g_string_append(reply,"\r\n");
					}
				}
			}
			if (!f)
			{
				g_string_sprintfa(reply,"\r\n Sorry I've not seen <%s> ...\r\n",nickname);
			}
		}
		else
			g_string_append(reply,"\r\n Sorry, this list was just restarted and is empty\r\n");

		G_UNLOCK(seen_array);
	}
	g_string_append_c(reply,'|');
	plugin_send_to_named_user(source_nick, reply->str);
	g_string_free(reply,TRUE);

	return;
}

static inline void plg_delete_seen_entry(int num)
{
	SEEN_ENTRY *sd;

	sd=&(g_array_index(seen_array,SEEN_ENTRY,num));
	if(sd->seen_nick)
		g_string_free(sd->seen_nick,TRUE);
	if(sd->nick_ip)
		g_string_free(sd->nick_ip,TRUE);
	seen_array=g_array_remove_index_fast(seen_array,num);
}

static void plg_find_seen_by_nickname_uniq(const char *nickname)
{
	if(seen_array!=NULL)
	{
		time_t t;
		struct tm *stime;
		int i;
		for(i=seen_array->len-1;i>=0;i--)
		{
			SEEN_ENTRY *nw;
			nw=&(g_array_index(seen_array,SEEN_ENTRY,i));
			if(!strcmp(nw->seen_nick->str,nickname))
			{
				t=nw->seen_time;
				stime=localtime(&t);
				plg_delete_seen_entry(i);
				break;
			}
		}
	}
}

static void plg_add_seen_entry(int event, const char *nickname, const char *nick_ip)
{
	SEEN_ENTRY nw;
	G_LOCK(seen_array);
	if(seen_array!=NULL)
		plg_find_seen_by_nickname_uniq(nickname);

	nw.seen_nick=g_string_new(nickname);
	nw.nick_ip=g_string_new(nick_ip);
	nw.seen_event=event;
	time(&nw.seen_time);

	if(seen_array==NULL)
		seen_array=g_array_new(FALSE,FALSE,sizeof(SEEN_ENTRY));

	seen_array=g_array_append_val(seen_array,nw);
	G_UNLOCK(seen_array);
}

static void plg_do_users(const char *source_nick,const char *dest_nick)
{
	int userlevel;

	userlevel=plugin_nick_type(source_nick);

	if(userlevel>=get_right_level(OPERATOR))
	{
		int i=0;
		GString *reply;	
		reply=g_string_new("");
		G_LOCK(seen_array);
		if(seen_array!=NULL)
			i=seen_array->len;
		G_UNLOCK(seen_array);
		if(dest_nick!=NULL)
		{
			g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
		}
		g_string_sprintfa(reply,"<INFO> %u User%s since plugin-start|",i,((i>1)?"s":""));
		plugin_send_to_named_user(source_nick, reply->str);
		g_string_free(reply,TRUE);
	}
	return;
}

static void plg_do_revip(const char *source_nick,const char *dest_nick,const char *ip)
{
	int userlevel;

	userlevel=plugin_nick_type(source_nick);

	if(userlevel>=get_right_level(OPERATOR))
	{
		GString *reply;
		reply=g_string_new("");
		if(dest_nick!=NULL)
		{
			g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
		}
		g_string_append(reply,"<INFO> ");

		if(ip==NULL || strspn(ip,"1234567890.")!=strlen(ip))
			g_string_append(reply,"Please use this command the following way:\r\n+searchip ip-adress\r\n");
		else
		{
			g_string_sprintfa(reply,"...search ip %s\r\n",ip);
			G_LOCK(seen_array);
			if(seen_array!=NULL)	
			{
				int i;
				SEEN_ENTRY *nw;
				for(i=seen_array->len-1;i>=0;i--)
				{
					nw=&(g_array_index(seen_array,SEEN_ENTRY,i));
					if(nw->nick_ip!=NULL)
					{
						if(!strncmp(nw->nick_ip->str,ip,strlen(ip)))
						{
							g_string_append(reply,nw->nick_ip->str);
							g_string_append(reply,"\t");
							g_string_append(reply,nw->seen_nick->str);
							g_string_append(reply,"\r\n");
						}
					}
				}
			}
			G_UNLOCK(seen_array);
		}
		g_string_append_c(reply,'|');
		plugin_send_to_named_user(source_nick, reply->str);
		g_string_free(reply,TRUE);
	}
	return;
}

static void plg_do_myip(const char *source_nick,const char *dest_nick)
{
	int i;
	int f=FALSE;

	G_LOCK(seen_array);
	if(seen_array!=NULL)
	{
		GString *reply;
		reply=g_string_new("");
		if(dest_nick!=NULL)
		{
			g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
		}
		g_string_append(reply,"<INFO> ");

		for(i=seen_array->len-1;i>=0;i--)
		{
			SEEN_ENTRY *nw;

			nw=&(g_array_index(seen_array,SEEN_ENTRY,i));
			if(strcmp(nw->seen_nick->str,source_nick) == 0)
			{
				g_string_sprintfa(reply,"Your IP is: %s|",nw->nick_ip->str);
				f = TRUE;
				break;
			}
		}
		if(f==FALSE)
		{
			g_string_append(reply,"<INFO> Hmm, something went wrong|");
		}
		plugin_send_to_named_user(source_nick,reply->str);
		g_string_free(reply,TRUE);
	}
	G_UNLOCK(seen_array);
	
	return;
}

/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
/* ############################## hub interface ############################# */
/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
static void process_command(const char *source_nick, const char *dest_nick, const char *chat)
{
	if(chat[0]=='+')		/* it is a command */
	{
		GString *cmd;
		char *a;
		gchar *param=NULL;

		cmd=g_string_new(chat);
		g_string_truncate(cmd,cmd->len-1);	/* remove the trailing '|' */

		a=strchr(cmd->str,' ');
		if(a!=NULL)
		{
			if(strlen(a+1)!=0)
			{
				param=g_strdup(a+1);
			}
			g_string_truncate(cmd,a-cmd->str);	/* truncate where the space is */
		}

		if(!strcmp(cmd->str,"+seen"))
			plg_find_seen_by_nickname(source_nick,dest_nick,param);
		else if(!strcmp(cmd->str,"+users"))
			plg_do_users(source_nick,dest_nick);
		else if(!strcmp(cmd->str,"+myip"))
			plg_do_myip(source_nick,dest_nick);
		else if(!strcmp(cmd->str,"+searchip"))
			plg_do_revip(source_nick,dest_nick,param);

		if(param)
			g_free(param);
		g_string_free(cmd,TRUE);
	}
}

static void globalchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,3).varvalue;

		if(strlen(fullmsg)>0)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				ptr++;
				process_command(source_nick,NULL,ptr);
			}
      }
	}
	return;
}

static void privchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *dest_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		dest_nick=g_array_index(param,PLUGIN_PARAM,3).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,4).varvalue;
		if(fullmsg!=NULL)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				*ptr++;
				process_command(source_nick,dest_nick,ptr);
			}
		}
	}
	return;
}

static void quit_fnc(const GArray *param)
{
   if(param!=NULL)
	{
		PLUGIN_PARAM *pp;
		pp=&(g_array_index(param,PLUGIN_PARAM,1));

	   plg_add_seen_entry(0,pp->varvalue,NULL);
	}
   return;
}

static void login_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		plg_add_seen_entry(1,g_array_index(param,PLUGIN_PARAM,1).varvalue,
									g_array_index(param,PLUGIN_PARAM,3).varvalue);
	}
	return;
}

/*****************************************************/
/* this function is called when the plugin is loaded */
/*****************************************************/
int plugin_beginning(PLUGIN *plug)
{
	printf("beginning of the SEEN plugin\n");

	register_dchub_plugin_event(plug,"globalchat", globalchat_fnc);
	register_dchub_plugin_event(plug,"privchat", privchat_fnc);
	register_dchub_plugin_event(plug,"quit", quit_fnc);
	register_dchub_plugin_event(plug,"login", login_fnc);

	return 0;		/* success */
}

/************************************************************/
/* this function is called just before unloading the plugin */
/************************************************************/
int plugin_end(PLUGIN *plug)
{
	printf("end of the SEEN plugin\n");
	return 0;		/* success */
}

