/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * plg_msg.c: Copyright (C) Dirk Rieger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: plugin_msg.c,v 2.3 2003/11/23 06:56:42 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include "dchub_plugin_api.h"

typedef struct
{
	GString* source_nick;
	GString* dest_nick;
	GString* msg_message;
	time_t msg_time;
} MSG_ENTRY;

/**********************/
/* arry of MSG_ENTRY */
/**********************/
static GArray *msg_array=NULL;

G_LOCK_DEFINE_STATIC(msg_array);

void plg_add_message(const char *source_nick, const char *dest_nick, const char *message)
{
	MSG_ENTRY nw;
	GString *reply;

	reply=g_string_new("");
	if(dest_nick!=NULL)
	{
		g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
	}
	g_string_append(reply,"<INFO> ");

	if(plugin_nick_duration(dest_nick)!=-1)
		g_string_sprintfa(reply,"Are you too shy to talk to this user directly ? - %s is online\r\n|",dest_nick);
	else if(message!=NULL)
	{
		char *dp=NULL;
		char *ptr=NULL;

		dp=strdup(message);
		ptr=strchr(dp,' ');

		if(ptr!=NULL)
		{
			*ptr++='\0';

			g_string_sprintfa(reply,"Your Message for <%s> ",dp);

			if(strlen(ptr)>255)
				g_string_append(reply,"was too long! - 255 Characters max|");
			else if(strlen(ptr)<5)
				g_string_append(reply,"was too short! - 5 Characters min|");
			else
			{
				/* input initialisation */
				nw.source_nick=g_string_new(source_nick);
				nw.dest_nick=g_string_new(dp);
				nw.msg_message=g_string_new(ptr);
				time(&nw.msg_time);

				G_LOCK(msg_array);
				if(msg_array==NULL)
					msg_array=g_array_new(FALSE,FALSE,sizeof(MSG_ENTRY)); /* Array erstellen */

				/* append entry */
				msg_array=g_array_append_val(msg_array,nw);
				G_UNLOCK(msg_array);
				g_string_append(reply,"stored succesfully!|");
			}
		}
		free(dp);
	}
	else
	{
		g_string_append(reply,"Please use this command the following way:\r\n+msg recipient Message\r\n|");
	}
	plugin_send_to_named_user(source_nick,reply->str);

	return;
}

void plg_find_left_messages(const char *nickname)
{
	G_LOCK(msg_array);
	if(msg_array!=NULL)
	{
		GString *reply;
		int i;
		int z=FALSE;
		reply=g_string_new("");

		for(i=msg_array->len-1;i>=0;i--)
		{
			MSG_ENTRY *nw;
			nw=&(g_array_index(msg_array,MSG_ENTRY,i));
			if(!strcasecmp(nw->dest_nick->str,nickname))
			{
				if (z==FALSE)
				{
					g_string_sprintf(reply,"$To: %s From: INFO $<INFO> Messages have been left for you: (You've got Mail!) :-)\r\n",nickname);
					z=TRUE;
				}
				g_string_append(reply,"\r\n* <");
				g_string_append(reply,nw->source_nick->str);
				g_string_append(reply,"> ");
				g_string_append(reply,nw->msg_message->str);

				if(reply->str[reply->len-1]=='|')	/* replace trailing | if it exists */
					reply->str[reply->len-1]=' ';
			}
		}
		if (z==TRUE)
		{
			g_string_append(reply,"\r\n\r\nUse the command '+clearmsg' to delete the message(s)!\r\n|");
			plugin_send_to_named_user(nickname,reply->str);
		}
		g_string_free(reply,TRUE);
	}
	G_UNLOCK(msg_array);

	return;
}

static inline void plg_delete_msg_entry(int num)
{
	/* msg_array must be lock before calling this function */
	MSG_ENTRY *ad;

	ad=&(g_array_index(msg_array,MSG_ENTRY,num));
	if(ad->dest_nick)
		g_string_free(ad->dest_nick,TRUE);
	if(ad->source_nick)
		g_string_free(ad->source_nick,TRUE);
	if(ad->msg_message)
		g_string_free(ad->msg_message,TRUE);

	g_array_remove_index_fast(msg_array,num);
}

void plg_do_clearmsg(const char *source_nick,const char *dest_nick)
{
	int z=0;
	GString *reply;
	
	reply=g_string_new("");
	if(dest_nick!=NULL)
	{
		g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
	}
	g_string_append(reply,"<INFO> ");

	G_LOCK(msg_array);
	if(msg_array!=NULL)
	{
		int i;
		for(i=msg_array->len-1;i>=0;i--)
		{
			MSG_ENTRY *nw;
			nw=&(g_array_index(msg_array,MSG_ENTRY,i));
			if(!strcasecmp(nw->dest_nick->str,source_nick))
			{
				plg_delete_msg_entry(i);
				z++;
			}
		}
	}
	G_UNLOCK(msg_array);
	if(z==0)
		g_string_append(reply,"no messages found to delete\r\n|");
	else
		g_string_sprintfa(reply,"%i message(s) found and deleted\r\n|",z);

	plugin_send_to_named_user(source_nick,reply->str);
	g_string_free(reply,TRUE);
	return;
}

/******************************************/
/* scan tos array to find expired entries */
/******************************************/
void plg_timeout_message(void)
{
	G_LOCK(msg_array);

	if(msg_array!=NULL)
	{
		int i;
		time_t now;

		now=time(NULL);

		for(i=msg_array->len-1;i>=0;i--)
		{
			MSG_ENTRY *te;

			te=&(g_array_index(msg_array,MSG_ENTRY,i));

			/* remove afk-messages after 24h */
			if(te->msg_time<=(now-(3600*24)))
				plg_delete_msg_entry(i);
		}
	}
	G_UNLOCK(msg_array);
}


/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
/* ############################## hub interface ############################# */
/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
static void process_command(const char *source_nick, const char *dest_nick, const char *chat)
{
	if(chat[0]=='+')		/* it is a command */
	{
		GString *cmd;
		char *a;
		gchar *param=NULL;

		cmd=g_string_new(chat);
		g_string_truncate(cmd,cmd->len-1);	/* remove the trailing '|' */

		a=strchr(cmd->str,' ');
		if(a!=NULL)
		{
			if(strlen(a+1)!=0)
			{
				param=g_strdup(a+1);
			}
			g_string_truncate(cmd,a-cmd->str);	/* truncate where the space is */
		}

		if(!strcmp(cmd->str,"+msg"))
			plg_add_message(source_nick,dest_nick,param);
		else if(!strcmp(cmd->str,"+clearmsg"))
			plg_do_clearmsg(source_nick,dest_nick);

		if(param)
			g_free(param);
		g_string_free(cmd,TRUE);
	}
	plg_timeout_message();
}

static void globalchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,3).varvalue;

		if(strlen(fullmsg)>0)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				ptr++;
				process_command(source_nick,NULL,ptr);
			}
      }
	}
	return;
}

static void privchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *dest_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		dest_nick=g_array_index(param,PLUGIN_PARAM,3).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,4).varvalue;
		if(fullmsg!=NULL)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				*ptr++;
				process_command(source_nick,dest_nick,ptr);
			}
		}
	}
	return;
}

static void login_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *nickname;
		nickname=g_array_index(param,PLUGIN_PARAM,1).varvalue;

		plg_find_left_messages(nickname);
	}
	return;
}

/*****************************************************/
/* this function is called when the plugin is loaded */
/*****************************************************/
int plugin_beginning(PLUGIN *plug)
{
	printf("beginning of the MSG plugin\n");

	register_dchub_plugin_event(plug,"globalchat", globalchat_fnc);
	register_dchub_plugin_event(plug,"privchat", privchat_fnc);
	register_dchub_plugin_event(plug,"login", login_fnc);

	return 0;		/* success */
}

/************************************************************/
/* this function is called just before unloading the plugin */
/************************************************************/
int plugin_end(PLUGIN *plug)
{
	printf("end of the MSG plugin\n");
	return 0;		/* success */
}

