/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * plg_login.c: Copyright (C) Dirk Rieger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: plugin_login.c,v 2.1 2003/11/22 18:52:51 ericprev Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <glib.h>

#include "dchub_plugin_api.h"
#include "generic_help.h"
#include "../src/global_user_if.h"
#include "../src/db_xml.h"
#include "../src/users_xml.h"


typedef struct
{
   char *login_nick;
} LOGIN_ENTRY;

/**********************/
/* arry of LOGIN_ENTRY */
/**********************/
static GArray *login_array=NULL;
G_LOCK_DEFINE_STATIC(login_array);

int maxuser=0;

char *wday[]={"Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"};


static void plg_send_login_msg(const char *source_nick)
{
	GString *reply;

	GString *h_minshare;
	GString *h_maxshare;
	GString *login_date;
	GString *lists[2];
	char *hubname;
	char *hubaddr;
	char *byte;
	int x;
	int users=0;
	int gmaxuser=0;
	int minshare=0;
	int minslots=0;
	int usertype=-1;
	float gmaxshare=0;
	guint64 shared_size=0;
	guint64 shared_mb=0;
	time_t ltime;
	struct tm *ptm;

	h_minshare=g_string_new("");
	h_maxshare=g_string_new("");
	reply=g_string_new("");
	login_date=g_string_new("");

	time(&ltime);
	ptm=localtime(&ltime);
	g_string_sprintf(login_date,"%s the ",wday[ptm->tm_wday]);
	g_string_sprintfa(login_date,"%02i.%02i.%i /%02i:%02i:%02i Hour",ptm->tm_mday,ptm->tm_mon+1,ptm->tm_year+1900,
							ptm->tm_hour,ptm->tm_min,ptm->tm_sec);

	hubname=(void*)db_str_get("HUBNAME");
	hubaddr=(void*)db_str_get("HUBADDR");
	if(!db_int_get("MINSHARE",&minshare))
		minshare=0;
	if(!db_int_get("MINSLOTS",&minslots))
		minslots=0;
	if(!db_int_get("GMAXUSER",&gmaxuser))
		gmaxuser=0;
	if(!db_float_get("GMAXSHARE",&gmaxshare))
		gmaxshare=0.0;

	usertype=plugin_nick_type(source_nick);

	glus_get_nick_lists(&lists[0],&lists[1]);
	/* we don't need the userlist so use the lists[0] to create the oplist to send in welcome */
	g_string_assign(lists[0]," ");

	{
		gchar **opnames;
		int i;
	
		opnames=g_strsplit(lists[1]->str,"$$",0);
		i=0;
		while(opnames[i]!=NULL)
		{
			/* exclude bots here */
			if((strcmp(opnames[i],"+++++++++++++++"))&&(strlen(opnames[i])))
				g_string_append_printf(lists[0],"<%s> ",opnames[i]);
			i++;
		}
		g_strfreev(opnames);
	}
	
	h_minshare=g_string_new("");
	if(minshare>=1024)
	{
		minshare=minshare / 1024;
		g_string_sprintf(h_minshare,"%i GB",minshare);
	}
	else
		g_string_sprintf(h_minshare,"%i MB",minshare);

	plugin_get_hub_user_start(&shared_size,&users);

	if(gmaxshare>(1024.0*1024.0))
	{
		x=1024*1024;
		byte="TB";
	}
	else if(gmaxshare>1024.0)
	{
		x=1024;
		byte="GB";
	}
	else
	{
		x=1;
		byte="MB";
	}

	shared_mb=shared_size/(1024*1024);
	if(gmaxshare<shared_mb)
	{
		gmaxshare=shared_mb;
		db_float_set("GMAXSHARE",(double)gmaxshare);
	}

	g_string_sprintf(h_maxshare,"%.2f %s",gmaxshare/x,byte);
	
	if(users>maxuser)
		maxuser=users;

	if(users>gmaxuser)
	{
		db_int_set("GMAXUSER",users);
		gmaxuser=users;
	}

	g_string_sprintf(reply,"<WELCOME> Nice to meet you %s.	WELCOME TO...\r\n\r\n"
					"- <-- %s --> [%s] --- Minimum Share = [%s + %i Slots] ---\r\n"
					"--- Max. user since hubstart = %i --- Max. user ever = %i --- Best share ever %s ---\r\n\r\n"
					"--- Hubrules (as from: %s)\r\n\r\n%s"
					,source_nick,hubname,hubaddr,h_minshare->str,minslots,maxuser,gmaxuser,h_maxshare->str
					,login_date->str,befehle
					);
	/* add help message according to user level */
	{
		gint16 level;

		if(xml_get_user(source_nick, &level, NULL, NULL, NULL))	/* is a registered user ? */
		{
			g_string_append_printf(reply,"%s",regbefehle);		/* help for all registered users */

			if(level>=get_right_level(OPERATOR))
				g_string_append_printf(reply,"%s",opbefehle);	/* and for OP and above */
		}
	}
	g_string_append_printf(reply,"*The following ops are available: %s\r\n\r\n"
					"Copyright \xA9\x32\x30\x30\x33 by hhgoth - Deutschland C@fe Crew\r\n\r\n",lists[0]->str);

	g_string_append_c(reply,'|');

	plugin_send_to_named_user(source_nick,reply->str);

	g_string_free(reply,TRUE);
	g_string_free(h_minshare,TRUE);
	g_string_free(h_maxshare,TRUE);
	g_string_free(login_date,TRUE);
	g_string_free(lists[0],TRUE);
	g_string_free(lists[1],TRUE);
	return;
}

static inline void plg_delete_login_entry(int num)
{
   LOGIN_ENTRY *dl;

   dl=&(g_array_index(login_array,LOGIN_ENTRY,num));
   if(dl->login_nick)
      free(dl->login_nick);
   login_array=g_array_remove_index_fast(login_array,num);
}

static int plg_find_login_by_nickname(const char *source_nick)
{
	int f=FALSE;
	G_LOCK(login_array);
	if(login_array!=NULL)
	{
		int i;
		for(i=login_array->len-1;i>=0;i--)
		{
			LOGIN_ENTRY *ln;
			ln=&(g_array_index(login_array,LOGIN_ENTRY,i));
			if(strcmp(ln->login_nick,source_nick) == 0)
			{
				plg_delete_login_entry(i);
				f=TRUE;
				break;
			}
		}
	}
	G_UNLOCK(login_array);
	return f;
}

static void plg_find_login_by_nickname_uniq(const char *nickname)
{
   if(login_array!=NULL)
   {
   	int i;
      for(i=login_array->len-1;i>=0;i--)
      {
         LOGIN_ENTRY *ln;
         ln=&(g_array_index(login_array,LOGIN_ENTRY,i));
         if(!strcmp(ln->login_nick,nickname))
         {
            plg_delete_login_entry(i);
            break;
         }
      }
   }
	return;
}

static void plg_add_login_entry(const char *nickname)
{
   LOGIN_ENTRY ln;
   G_LOCK(login_array);

   if(login_array!=NULL)
      plg_find_login_by_nickname_uniq(nickname);

   ln.login_nick=strdup(nickname);

   if(login_array==NULL)
      login_array=g_array_new(FALSE,FALSE,sizeof(LOGIN_ENTRY));

   login_array=g_array_append_val(login_array,ln);
   G_UNLOCK(login_array);

	return;
}

static void plg_del_login_on_quit(const char *nickname)
{
	G_LOCK(login_array);
	if(login_array!=NULL)
		plg_find_login_by_nickname_uniq(nickname);
	G_UNLOCK(login_array);

	return;
}


/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
/* ############################## hub interface ############################# */
/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
static void myinfo_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;

		if(plg_find_login_by_nickname(source_nick))
			plg_send_login_msg(source_nick);
	}
	return;
}

static void login_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *nickname;
		const char *nick_ip;
		nickname=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		nick_ip=g_array_index(param,PLUGIN_PARAM,3).varvalue;

		plg_add_login_entry(nickname);
	}
	return;
}

static void quit_fnc(const GArray *param)
{
   if(param!=NULL)
	{
		const char *source_nick;
		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;

		plg_del_login_on_quit(source_nick);
	}
   return;
}

/*****************************************************/
/* this function is called when the plugin is loaded */
/*****************************************************/
int plugin_beginning(PLUGIN *plug)
{
	printf("beginning of the LOGIN plugin\n");

	register_dchub_plugin_event(plug,"myinfo", myinfo_fnc);
	register_dchub_plugin_event(plug,"login", login_fnc);
	register_dchub_plugin_event(plug,"quit", quit_fnc);

	return 0;		/* success */
}

/************************************************************/
/* this function is called just before unloading the plugin */
/************************************************************/
int plugin_end(PLUGIN *plug)
{
	printf("end of the LOGIN plugin\n");
	return 0;		/* success */
}

