/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * plg_afk.c: Copyright (C) Dirk Rieger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: plugin_afk.c,v 2.3 2003/11/23 12:35:35 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <time.h>
#include <string.h>
#include <glib.h>

#include "dchub_plugin_api.h"
#include "../src/users_xml.h"

typedef struct
{
	GString* afk_nick;
	GString* afk_message;
	time_t afk_time;
} AFK_ENTRY;

/**********************/
/* arry of AFK_ENTRY */
/**********************/
static GArray *afk_array=NULL;

G_LOCK_DEFINE_STATIC(afk_array);

static void plg_find_afk_by_nickname(const char *pm_sender, const char *pm_dest)
{
	G_LOCK(afk_array);
	if(afk_array!=NULL)
	{
#warning	EXCLUDE BOTS HERE - very important for e.g. ClientBot, or an other bot used as op-chat to prevent loops 
		/*** and to reduce hub-load ***/
		if(strcmp(pm_sender,"ClientBot") && strcmp(pm_sender,"some_other_bot_to_exclude"))
		{
			int i;
			for(i=afk_array->len-1;i>=0;i--)
			{
				AFK_ENTRY *nw;
				nw=&(g_array_index(afk_array,AFK_ENTRY,i));
				if(!strcmp(nw->afk_nick->str,pm_dest))
				{
					GString *reply;
					reply=g_string_new("");
					g_string_sprintf(reply,"$To: %s From: %s $<INFO> The user %s is afk (away from keyboard)!\r\nMessage: %s\r\n|",pm_sender,pm_dest,pm_dest,nw->afk_message->str);
					plugin_send_to_named_user(pm_sender,reply->str);
					g_string_free(reply,TRUE);
					break;
				}
				else if(strcmp(nw->afk_nick->str,pm_sender)==0)
				{
					GString *reply;
					reply=g_string_new("");
					g_string_sprintf(reply,"$To: %s From: %s $<INFO> ATTENTION - please remove your afk-message !!!\r\n|",pm_sender,pm_dest);
					plugin_send_to_named_user(pm_sender,reply->str);
					g_string_free(reply,TRUE);
					break;
				}
			}
		}
	}
	G_UNLOCK(afk_array);
}

static inline void plg_delete_afk_entry






(int num)
{
	/* Attention: this function is "unsafe", */
	/* the afk_array has to be locked before !!! */
	AFK_ENTRY *ad;

	ad=&(g_array_index(afk_array,AFK_ENTRY,num));
	if(ad->afk_nick)
		g_string_free(ad->afk_nick,TRUE);
	if(ad->afk_message)
		g_string_free(ad->afk_message,TRUE);

	afk_array=g_array_remove_index_fast(afk_array,num);
}

static void plg_find_afk_by_nickname_uniq(const char *nickname)
{
	if(afk_array!=NULL)
	{
		time_t t;
		struct tm *stime;
		int i;

		for(i=afk_array->len-1;i>=0;i--)
		{
			AFK_ENTRY *nw;
			nw=&(g_array_index(afk_array,AFK_ENTRY,i));
			if(!strcmp(nw->afk_nick->str,nickname))
			{
				t=nw->afk_time;
				stime=localtime(&t);
				plg_delete_afk_entry(i);
				break;
			}
		}
	}
}

static void plg_add_afk(const char *source_nick, const char *dest_nick, const char *message)
{
	AFK_ENTRY nw;
	GString *reply;

	reply=g_string_new("");
	if(dest_nick!=NULL)
	{
		g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
	}
	g_string_append(reply,"<INFO> ");

	if(message==NULL)
	{
		g_string_append(reply,"Please use this command the following way:\r\n+afk your_message\r\n|");
	}
	else if(strlen(message)>255)
	{
		g_string_append(reply,"Sorry your Message was to long - 255 Characters max!\r\n|");
	}
	else
	{
		G_LOCK(afk_array);
		if(afk_array!=NULL)
			plg_find_afk_by_nickname_uniq(source_nick);  /* remove old entry... */

		/* initialise new entry */
		nw.afk_nick=g_string_new(source_nick);
		nw.afk_message=g_string_new(message);
		time(&nw.afk_time);

		if(afk_array==NULL)
			afk_array=g_array_new(FALSE,FALSE,sizeof(AFK_ENTRY)); /* creating Array */

		afk_array=g_array_append_val(afk_array,nw);
		G_UNLOCK(afk_array);

		g_string_sprintfa(reply,"Your afk/away - Message was stored successfully!!! - cu later %s\r\n\r\n"
										"...Please don't forget to tell me if you're back. Command: '+back' !\r\n|",source_nick);

		if (plugin_nick_type(source_nick)>=get_right_level(OPERATOR))
		{
			GString *afk_reply;
			afk_reply=g_string_new("");
			g_string_sprintf(afk_reply,"<INFO> The operator %s is not available|",source_nick);
			plugin_send_to_all_users(afk_reply->str);
			g_string_free(afk_reply,TRUE);
		}
	}
	plugin_send_to_named_user(source_nick,reply->str);
	g_string_free(reply,TRUE);

	return;
}

static void plg_del_afk_on_quit(const char *source_nick)
{
	G_LOCK(afk_array);
	if(afk_array!=NULL)
		plg_find_afk_by_nickname_uniq(source_nick);
	G_UNLOCK(afk_array);
}

static void plg_del_afk_entry(const char *source_nick, const char *dest_nick)
{
	G_LOCK(afk_array);
	if(afk_array!=NULL)
	{
		int i;
		gboolean f=FALSE;
		for(i=0;i<afk_array->len;i++)
		{
			AFK_ENTRY *nw;
			nw=&(g_array_index(afk_array,AFK_ENTRY,i));
			if(!strcmp(nw->afk_nick->str,source_nick))
			{
				int userlevel;
				GString *reply;

				userlevel=plugin_nick_type(source_nick);
				plg_delete_afk_entry(i);

				reply=g_string_new("");
				if(dest_nick!=NULL)
				{
					g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
				}
				g_string_sprintfa(reply,"<INFO> Your afk/away - Message was removed successfully!\r\n- Welcome back %s\r\n|",source_nick);
				plugin_send_to_named_user(source_nick,reply->str);
				
				if (userlevel >=get_right_level(OPERATOR))
				{
					g_string_sprintf(reply,"<INFO> The operator %s is available again|",source_nick);
					plugin_send_to_all_users(reply->str);
				}
				g_string_free(reply,TRUE);
				f=TRUE;
				break;
			}
		}
		if(f==FALSE)
		{
			GString *reply;
			reply=g_string_new("");
			if(dest_nick!=NULL)
			{
				g_string_sprintf(reply,"$To: %s From: %s $",source_nick,dest_nick);
			}
			g_string_sprintfa(reply,"<INFO> No afk message was stored for you, anyway - Welcome back ;)\r\n|");
			plugin_send_to_named_user(source_nick,reply->str);
			g_string_free(reply,TRUE);
		}
	}
	G_UNLOCK(afk_array);
}

/******************************************/
/* scan tos array to find expired entries */
/******************************************/
static void plg_timeout_afk(void)
{
	G_LOCK(afk_array);
	if(afk_array!=NULL)
	{
		int i;
		time_t now;

		now=time(NULL);

		for(i=afk_array->len-1;i>=0;i--)
		{
			AFK_ENTRY *te;

			te=&(g_array_index(afk_array,AFK_ENTRY,i));

			/* remove afk-messages after 24h */
			if(te->afk_time<=(now-(3600*24)))
			{
				plg_delete_afk_entry(i);
			}
		}
	}
	G_UNLOCK(afk_array);
}


/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
/* ############################## hub interface ############################# */
/* ########################################################################## */
/* ########################################################################## */
/* ########################################################################## */
static void process_command(const char *source_nick, const char *dest_nick, const char *chat)
{
	if(chat[0]=='+')		/* it is a command */
	{
		GString *cmd;
		char *a;
		gchar *param=NULL;

		cmd=g_string_new(chat);
		g_string_truncate(cmd,cmd->len-1);	/* remove the trailing '|' */

		a=strchr(cmd->str,' ');
		if(a!=NULL)
		{
			if(strlen(a+1)!=0)
			{
				param=g_strdup(a+1);
			}
			g_string_truncate(cmd,a-cmd->str);	/* truncate where the space is */
		}

		if(!strcmp(cmd->str,"+afk"))
			plg_add_afk(source_nick,dest_nick,param);
		else if(!strcmp(cmd->str,"+back"))
			plg_del_afk_entry(source_nick,dest_nick);

		if(param)
			g_free(param);
		g_string_free(cmd,TRUE);
	}
	else if(dest_nick!=NULL)
	{
		plg_find_afk_by_nickname(source_nick,dest_nick);
	}
}

static void globalchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,3).varvalue;

		if(strlen(fullmsg)>0)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				ptr++;
				process_command(source_nick,NULL,ptr);
			}
      }
		plg_timeout_afk();
	}
	return;
}

static void privchat_fnc(const GArray *param)
{
	if(param!=NULL)
	{
		const char *source_nick;
		const char *dest_nick;
		const char *fullmsg;

		source_nick=g_array_index(param,PLUGIN_PARAM,1).varvalue;
		dest_nick=g_array_index(param,PLUGIN_PARAM,3).varvalue;
		fullmsg=g_array_index(param,PLUGIN_PARAM,4).varvalue;
		if(fullmsg!=NULL)
		{
			char *ptr;

			ptr=strchr(fullmsg,' ');

			if(ptr!=NULL)
			{
				*ptr++;
				process_command(source_nick,dest_nick,ptr);
			}
		}
	}
	return;
}

static void quit_fnc(const GArray *param)
{
   if(param!=NULL)
		plg_del_afk_on_quit(g_array_index(param,PLUGIN_PARAM,1).varvalue);
   return;
}

/*****************************************************/
/* this function is called when the plugin is loaded */
/*****************************************************/
int plugin_beginning(PLUGIN *plug)
{
	printf("beginning of the AFK plugin\n");

	register_dchub_plugin_event(plug,"globalchat", globalchat_fnc);
	register_dchub_plugin_event(plug,"privchat", privchat_fnc);
	register_dchub_plugin_event(plug,"quit", quit_fnc);

	return 0;		/* success */
}

/************************************************************/
/* this function is called just before unloading the plugin */
/************************************************************/
int plugin_end(PLUGIN *plug)
{
	printf("end of the AFK plugin\n");
	return 0;		/* success */
}

