/*
** $Id: audio-entropyd.c,v 0.5 2003/01/27 17:39:31 folkert Exp folkert $
**
** Simple program to reseed kernel random number generator using
** data read from soundcard.
**
** Copyright 1999 Damien Miller <djm@mindrot.org>
**
** This code is licensed under the GNU Public License version 2
** Please see the file COPYING for more details.
**
** 27nov2000, Folkert van Heusden <folkert@vanheusden.com>
**            Added code which locks the buffers into physical
**            memory.
**
** 03sep2002, Udo van den Heuvel <udovdh@yahoo.com>
**	      After noticing that the kernel complained about
**	      recording overruns while the daemon was sleeping
**	      I made the daemon close the dsp_fd before sleeping
**	      and reopening it after sleeping.
**	      Changed the click fix to a constant.
**	      I also bumped the version number to 0.0.1
**
** 10sep2002, Udo van den Heuvel <udovdh@yahoo.com>
**	      Thought of hopefully more intelligent behaviour
**	      for adding entropy:
**	      Let the daemon wait until the kernel signals that
**	      the lower treshold in 
**	      /proc/sys/kernel/random/write_wakeup_threshold
**	      (number of bits) has been reached; 
**	      then start adding until value of
**	      /proc/sys/kernel/random/poolsize (in bytes) has
**	      been reached.
**	      Also after a timeout still some entropy is added.
** 
** 16nov2002, Folkert van Heusden <folkert@vanheusden.com>
**            Lowered the sample-rate; the larger the sample-
**            rate, the more biased the data gets.
**            Removed the mixer-code; we should not crank-up
**            the volume since that also increases the bias.
**            Changed the daemonise-function to use 'daemon(...)
**            which is a lot less code.
**            Removed hashing-code since the kernel already
**            takes care of doing such things.
**            Removed lots of code that dumped all kinds of
**            data to files. Not needed anyway in my opinion.
**            Added code that determines the amount of entropy
**            in a buffer of data so that we actually say
**            something good to the kernel when submitting that
**            data.
**            Removed sleeping-code; now only data gets
**            submitted when the kernel-entropy-spool gets
**            empty.
**            Rewrote the method of entropy-gathering; some
**            code has been added to remove bias in the data.
**
** 22dec2002  Folkert van Heusden <folkert@vanheusden.com>
**            Locking buffers in memory
**
** 04jan2003  Folkert van Heusden <folkert@vanheusden.com>
**            Moved some code to a library
**
** 02Apr2007  Folkert van Heusden <folkert@vanheusden.com>
**            Added entropy-tester
*
* $Log: audio-entropyd.c,v $
* Revision 0.5  2003/01/27 17:39:31  folkert
* *** empty log message ***
*
** 
*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <getopt.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/time.h>

#include <linux/soundcard.h>
#include <asm/types.h>
#include <linux/random.h>
#include <errno.h>

#include "snd_dev.h"
#include "proc.h"
#include "val.h"
#include "RNGTEST.h"
#include "error.h"

#define DEFAULT_DSP_DEV				"/dev/dsp"
#define RANDOM_DEVICE				"/dev/random"
#define URANDOM_DEVICE				"/dev/urandom"
#define DEFAULT_INPUT_BUFFER_SIZE		16384
#define DEFAULT_SAMPLE_RATE			11025
#define PID_FILE				"/var/run/audio-entropyd.pid"
#define DEFAULT_CLICK_READ			(DEFAULT_SAMPLE_RATE * 2)
#define DEFAULT_POOLSIZE_FN                     "/proc/sys/kernel/random/poolsize"
#define	RNGTEST_PENALTY				(20000 / 8) /* how many bytes to skip when the rng-test fails */

int loggingstate = 1;
int treshold = 0;
char skip_test = 0;

#define max(x, y)	((x)>(y)?(x):(y))

/* Prototypes */
void main_loop(const char *dsp_device, int read_size, int sample_rate);
int setup_dsp_device(const char *dsp_device, int sample_rate);
void usage(void);
void credit_krng(int random_fd, struct rand_pool_info *entropy);
void daemonise(void);
void gracefully_exit(int signum);
void logging_handler(int signum);
void get_random_data(char *input_buffer, int read_size, int sample_rate, int *n_output_bytes, char *output_buffer);
int add_to_kernel_entropyspool(int handle, char *buffer, int nbytes);

char *dsp_device = DEFAULT_DSP_DEV;

/* Functions */

int main(int argc, char **argv)
{
	int read_size = DEFAULT_INPUT_BUFFER_SIZE;
	int sample_rate = DEFAULT_SAMPLE_RATE;
	int c;
	static struct option long_options[] =
	{
		{"read-size",		1, NULL, 'r'},
		{"dsp-device",		1, NULL, 'd'},
		{"sample-rate",	1, NULL, 'N'},
		{"skip-test", 0, NULL, 's' },
		{"help",				0, NULL, 'h'},
		{NULL,				0, NULL, 0}
	};

	/* Process commandline options */
	while(1)
	{
		c = getopt_long (argc, argv, "sr:d:w:N:Kh", long_options, NULL);
		if (c == -1)
			break;
		
		switch(c)
		{
			case 's':
				skip_test = 1;
				break;
			case 'r':
				read_size = atoi(optarg);
				break;
			case 'd':
				dsp_device = strdup(optarg);
				break;
			case 'N':
				sample_rate = atoi(optarg);
				break;
			case '?':
			case 'h':
				usage();
				exit(0);
			default:
				fprintf(stderr, "Invalid commandline options.\n\n");
				usage();
				exit(1);
		}
	}
	
	if (read_size == 0)
	{
		fprintf(stderr, "Error: Read size must be a greater than zero.\n");
		exit(1);
	}
	if (read_size & (read_size - 1))
	{
		fprintf(stderr, "Error: Read size must be a power of two.\n");
		exit(1);
	}

	RNGTEST_init();

	signal(SIGHUP, gracefully_exit);
	signal(SIGINT, gracefully_exit);
	signal(SIGTERM, gracefully_exit);
	signal(SIGUSR1, logging_handler);
	signal(SIGUSR2, logging_handler);

	openlog("audio-entropyd", LOG_CONS, LOG_DAEMON);

	syslog(LOG_NOTICE, "audio-entropyd starting up");

	daemonise();

	main_loop(dsp_device, read_size, sample_rate);
	
	exit(0);
}

int setup_dsp_device(const char *dsp_device, int sample_rate)
{
	int dsp_fd, setting;

	/* Open sound device */
	dsp_fd = open(dsp_device, O_RDONLY);
	if (dsp_fd == -1)
		error_exit("Could not open %s for reading: %m", dsp_device);

	/* Set sample rate */
	if (set_soundcard_sample_rate(dsp_fd, sample_rate) == -1)
		error_exit("Could not set %i sample rate", sample_rate);
	
	/* Set sample format */
	setting = AFMT_S16_LE; /* 16 bit little endian */
	if (ioctl(dsp_fd, SNDCTL_DSP_SETFMT, &setting) == -1)
		error_exit("Could not set 16 bit sample mode: %m");
	
	/* Set stereo */
	if (set_soundcard_number_of_channels(dsp_fd, 2) == -1)
		error_exit("Could not set stereo sample mode: %m");
	
	return(dsp_fd);
}

void main_loop(const char *dsp_device, int read_size, int sample_rate)
{
	unsigned char output_buffer[read_size];
	char input_buffer[max(read_size, DEFAULT_CLICK_READ)];
	int n_output_bytes;
	int random_fd = -1, max_bits;
	FILE *poolsize_fh;

	/* lock buffers in core */
	if (mlock(output_buffer, sizeof(output_buffer)) == -1 || mlock(input_buffer, sizeof(input_buffer)) == -1)
		error_exit("Could not lock buffers in core");

	/* Open kernel random device */
	random_fd = open(RANDOM_DEVICE, O_RDWR);
	if (random_fd == -1)
		error_exit("Couldn't open random device: %m");

        /* find out poolsize */
        poolsize_fh = fopen(DEFAULT_POOLSIZE_FN, "rb");
        if (!poolsize_fh)
                error_exit("Couldn't open poolsize file: %m");
        fscanf(poolsize_fh, "%d", &max_bits);
        fclose(poolsize_fh);

	/* first get some data so that we can immediately submit something when the
	 * kernel entropy-buffer gets below some limit
	 */
	get_random_data(input_buffer, read_size, sample_rate, &n_output_bytes, output_buffer);

	/* Main read loop */
	for(;;)
	{	
		int added = 0, before, after;
		fd_set write_fd;
		FD_ZERO(&write_fd);
		FD_SET(random_fd, &write_fd);
	
		for(;;) 
	        { 
			int rc = select(random_fd+1, NULL, &write_fd, NULL, NULL); /* wait for krng */ 
			if (rc >= 0) break; 
			if (errno != EINTR) 
				error_exit("Select error: %m"); 
       		}

		/* find out how many bits to add */
		if (ioctl(random_fd, RNDGETENTCNT, &before) == -1)
			error_exit("Couldn't query entropy-level from kernel");

		if (loggingstate == 1)
			syslog(LOG_DEBUG, "woke up due to low entropy state (%d)", before);

		/* loop until the buffer is (supposed to be) full: we do NOT check the number of bits
		 * currently in the buffer each iteration, since (on a heavily used random-driver)
		 * audio-entropyd might run constantly, using a lot of cpu-usage
		 */
		for(after=0; after != max_bits;)
		{
			if (n_output_bytes > 0)
			{
				added += add_to_kernel_entropyspool(random_fd, output_buffer, n_output_bytes);
			}

			/* Get number of bits in KRNG after credit */
			if (ioctl(random_fd, RNDGETENTCNT, &after) == -1)
				error_exit("Coundn't query entropy-level from kernel: %m");

			get_random_data(input_buffer, read_size, sample_rate, &n_output_bytes, output_buffer);
		}

		if (loggingstate == 1)
		{
			syslog(LOG_INFO, "Entropy credit of %i bits made (%i bits before, %i bits after)",
				added, before, after);
		}
	}
}

int add_to_kernel_entropyspool(int handle, char *buffer, int nbytes)
{
        double nbits;
	struct rand_pool_info *output;

	output = (struct rand_pool_info *)malloc(sizeof(struct rand_pool_info) + nbytes);
	if (!output)
		error_exit("malloc failure in add_to_kernel_entropyspool");

        // calculate number of bits in the block of
        // data. put in structure
        nbits = calc_nbits_in_data((unsigned char *)buffer, nbytes);
        if (nbits >= 1.0)
        {
                output -> entropy_count = (int)nbits;
                output -> buf_size      = nbytes;
		memcpy(output -> buf, buffer, nbytes);

                if (ioctl(handle, RNDADDENTROPY, output) == -1)
			error_exit("RNDADDENTROPY failed!");
        }

	free(output);

	return (int)nbits;
}

#define order(a, b)     (((a) == (b)) ? -1 : (((a) > (b)) ? 1 : 0))

void get_random_data(char *input_buffer, int read_size, int sample_rate, int *n_output_bytes, char *output_buffer)
{
	int dsp_fd;
	int n_to_do, bits_out=0, loop;
	char *dummy;
	static short psl=0, psr=0; /* previous samples */
	static char a=1; /* alternater */
	unsigned char byte_out=0;
	static int error_state = 0;

	*n_output_bytes=0;

	/* Open and set up DSP for reading */
	dsp_fd = setup_dsp_device(dsp_device, sample_rate);	
		
	/* Discard the first data read */
	/* it often contains weird looking data - probably a click from */
	/* driver loading / card initialisation */
	read(dsp_fd, input_buffer, DEFAULT_CLICK_READ);
		
	/* Read a buffer of audio */
	n_to_do = read_size;
	dummy = input_buffer;
	while (n_to_do)
	{
		int bytes_read = read(dsp_fd, dummy, n_to_do);
		if (bytes_read == -1) 
		{
			if (errno != EINTR)
				error_exit("Read error: %m");
		}
		else
		{
			n_to_do -= bytes_read;
			dummy += bytes_read;	
		}
	}

	/* de-biase the data */
	for(loop=0; loop<read_size; loop+=8)
	{
		int w1 = (input_buffer[loop+0]<<8) + input_buffer[loop+1];
		int w2 = (input_buffer[loop+2]<<8) + input_buffer[loop+3];
		int w3 = (input_buffer[loop+4]<<8) + input_buffer[loop+5];
		int w4 = (input_buffer[loop+6]<<8) + input_buffer[loop+7];
		int o1, o2;

		/* Determine order of channels for each sample, subtract previous sample
		 * to compensate for unbalanced audio devices */
		o1 = order(w1-psl, w2-psr);
		o2 = order(w3-psl, w4-psr);
		if (a > 0)
		{
			psl = w3;
			psr = w4;
		}
		else
		{
			psl = w1;
			psr = w2;
		}

		/* If both samples have the same order, there is bias in the samples, so we
		 * discard them; if both channels are equal on either sample, we discard
		 * them too; additionally, alternate the sample we'll use next (even more
		 * bias removal) */
		if (o1 == o2 || o1 < 0 || o2 < 0)
		{
			a = -a;
		}
		else
		{
			/* We've got a random bit; the bit is either the order from the first or
			 * the second sample, determined by the alternator 'a' */
			char bit = (a > 0) ? o1 : o2;

			byte_out <<= 1;
			byte_out += bit;

			bits_out++;

			if (bits_out==8)
			{
				if (error_state == 0)
				{
					output_buffer[*n_output_bytes]=byte_out;
					(*n_output_bytes)++;
				}
				bits_out=0;

				RNGTEST_add(byte_out);
				if (skip_test == 0 && RNGTEST() == -1)
				{
					if (error_state == 0)
						syslog(LOG_CRIT, "test of random data failed, skipping %d bytes before re-using data-stream (%d bytes in flush)", RNGTEST_PENALTY, error_state);
					error_state = RNGTEST_PENALTY;
					*n_output_bytes = 0;
				}
				else
				{
					if (error_state > 0)
					{
						error_state--;
						
						if (error_state == 0)
							syslog(LOG_INFO, "Restarting fetching of entropy data");
					}
				}
			}
		}
	}

	close(dsp_fd);
}

void usage(void)
{
	fprintf(stderr, "Usage: audio-entropyd [options]\n\n");
	fprintf(stderr, "Collect entropy from a soundcard and feed it into the kernel random pool.\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "--read-size,    -r []  Number of bytes to read at a time. (Default %i)\n", DEFAULT_INPUT_BUFFER_SIZE);
	fprintf(stderr, "--dsp-device,   -d []  Specify sound device to use. (Default %s)\n", DEFAULT_DSP_DEV);
	fprintf(stderr, "--sample-rate,  -N []  Audio sampling rate. (default %i)\n", DEFAULT_SAMPLE_RATE);
	fprintf(stderr, "\n");
}

void daemonise(void)
{
	if (become_daemon() == -1)
		error_exit("cannot fork into the background");

	if (write_pidfile(PID_FILE) == -1)
		error_exit("Couldn't open PID file \"%s\" for writing: %m.", PID_FILE);
}

void gracefully_exit(int signum)
{
      if (munlockall() == -1)
              error_exit("problem unlocking pages");
	unlink(PID_FILE);
	syslog(LOG_NOTICE, "audio-entropyd stopping due to signal %d", signum);
	exit(0);
}

void logging_handler(int signum)
{
      if (signum == SIGUSR1) 
            loggingstate = 1; 
      if (signum == SIGUSR2) 
            loggingstate = 0; 
}
