/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: arch_posix.c,v 1.3 2000/01/09 15:26:37 rich Exp $
 */

/* This file contains architecture specific code which should compile
 * and run correctly on most platforms which implement a reasonable
 * subset of the POSIX.1 standard, and most SVR4- and BSD-derived
 * operating systems.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifndef MAP_FAILED
#define MAP_FAILED ((void *)-1)
#endif

#include "xracer.h"
#include "xracer-arch.h"
#include "xracer-log.h"

/* Any program-level initialization may be done here. */
void
xrArchInit ()
{
}

/* Create a shared memory segment. */
void *
xrArchCreateSharedMemory (size_t size)
{
  FILE *fp;
  void *shm;
  int fd;

  /* Linux doesn't support anonymous shared memory mappings, more's
   * the shame, so use a file instead.
   */
  fp = tmpfile ();
  if (fp == NULL)
    {
      xrLogPerror ("temporary file");
      return 0;
    }

  fd = fileno (fp);
  if (ftruncate (fd, size) < 0)
    {
      xrLogPerror ("ftruncate");
      fclose (fp);
      return 0;
    }

  /* Create the shared memory segment. */
  shm = (void *) mmap (0, size, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);
  if (shm == MAP_FAILED)
    {
      xrLogPerror ("mmap");
      fclose (fp);
      return 0;
    }

  /* Is this necessary? */
  memset (shm, 0, size);

  return shm;
}

/* Delete a shared memory segment. */
int
xrArchDeleteSharedMemory (void *ptr, size_t size)
{
  return munmap (ptr, size);
}

/* Arrange for a signal to be sent when a child dies. */
int
xrArchSetCallbackOnChildTerminationSignal (void (*fn) (void))
{
  struct sigaction sa;

  memset (&sa, 0, sizeof sa);
  sa.sa_handler = fn ? (void (*)(int)) fn : SIG_IGN;

  if (sigaction (SIGCHLD, &sa, 0) < 0)
    {
      xrLogPerror ("sigaction");
      return -1;
    }

  return 0;
}

/* Arrange for a signal to be sent when a SIGUSR1 is received. */
int
xrArchSetCallbackOnUserSignal (void (*fn) (void))
{
  struct sigaction sa;

  memset (&sa, 0, sizeof sa);
  sa.sa_handler = fn ? (void (*)(int)) fn : SIG_IGN;

  if (sigaction (SIGUSR1, &sa, 0) < 0)
    {
      xrLogPerror ("sigaction");
      return -1;
    }

  return 0;
}

/* Create a child process. */
int
xrArchCreateChild (void (*entry_point) (void *), void *args)
{
  int pid = fork ();

  if (pid > 0)
    {
      /* Parent. */
      return pid;
    }
  else if (pid == 0)
    {
      /* Child. */
      entry_point (args);
      _exit (0);
    }
  else
    {
      /* Error. */
      xrLogPerror ("fork");
      return -1;
    }
}

/* Wait for a child to die. */
int
xrArchWaitChild ()
{
  if (wait (0) == -1)
    {
      xrLogPerror ("wait");
      return -1;
    }
  return 0;
}

/* Send a signal. */
int
xrArchSendUserSignal (int pid)
{
  if (kill (pid, SIGUSR1) == -1)
    {
      xrLogPerror ("kill");
      return -1;
    }
  return 0;
}

/* Recalculate current time. */
void
xrArchRecalculateCurrentTime ()
{
  double last_time = xrCurrentTime;
  struct timeval tv;

  gettimeofday (&tv, 0);

  xrCurrentTime = (double) tv.tv_sec + (double) tv.tv_usec * 1e-6;
  xrTimeInterval = xrCurrentTime - last_time;
}
