﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lightsail/LightsailRequest.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Lightsail {
namespace Model {

/**
 */
class AttachInstancesToLoadBalancerRequest : public LightsailRequest {
 public:
  AWS_LIGHTSAIL_API AttachInstancesToLoadBalancerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AttachInstancesToLoadBalancer"; }

  AWS_LIGHTSAIL_API Aws::String SerializePayload() const override;

  AWS_LIGHTSAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the load balancer.</p>
   */
  inline const Aws::String& GetLoadBalancerName() const { return m_loadBalancerName; }
  inline bool LoadBalancerNameHasBeenSet() const { return m_loadBalancerNameHasBeenSet; }
  template <typename LoadBalancerNameT = Aws::String>
  void SetLoadBalancerName(LoadBalancerNameT&& value) {
    m_loadBalancerNameHasBeenSet = true;
    m_loadBalancerName = std::forward<LoadBalancerNameT>(value);
  }
  template <typename LoadBalancerNameT = Aws::String>
  AttachInstancesToLoadBalancerRequest& WithLoadBalancerName(LoadBalancerNameT&& value) {
    SetLoadBalancerName(std::forward<LoadBalancerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of strings representing the instance name(s) you want to attach to
   * your load balancer.</p> <p>An instance must be <code>running</code> before you
   * can attach it to your load balancer.</p> <p>There are no additional limits on
   * the number of instances you can attach to your load balancer, aside from the
   * limit of Lightsail instances you can create in your account (20).</p>
   */
  inline const Aws::Vector<Aws::String>& GetInstanceNames() const { return m_instanceNames; }
  inline bool InstanceNamesHasBeenSet() const { return m_instanceNamesHasBeenSet; }
  template <typename InstanceNamesT = Aws::Vector<Aws::String>>
  void SetInstanceNames(InstanceNamesT&& value) {
    m_instanceNamesHasBeenSet = true;
    m_instanceNames = std::forward<InstanceNamesT>(value);
  }
  template <typename InstanceNamesT = Aws::Vector<Aws::String>>
  AttachInstancesToLoadBalancerRequest& WithInstanceNames(InstanceNamesT&& value) {
    SetInstanceNames(std::forward<InstanceNamesT>(value));
    return *this;
  }
  template <typename InstanceNamesT = Aws::String>
  AttachInstancesToLoadBalancerRequest& AddInstanceNames(InstanceNamesT&& value) {
    m_instanceNamesHasBeenSet = true;
    m_instanceNames.emplace_back(std::forward<InstanceNamesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_loadBalancerName;

  Aws::Vector<Aws::String> m_instanceNames;
  bool m_loadBalancerNameHasBeenSet = false;
  bool m_instanceNamesHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
