/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertPath;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.hadoop.hdds.protocol.proto.SCMSecurityProtocolProtos;
import org.apache.hadoop.hdds.protocolPB.SCMSecurityProtocolClientSideTranslatorPB;
import org.apache.hadoop.hdds.scm.client.ClientTrustManager;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.ssl.ReloadingX509KeyManager;
import org.apache.hadoop.hdds.security.ssl.ReloadingX509TrustManager;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CAType;
import org.apache.hadoop.hdds.security.x509.certificate.client.CACertificateProvider;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateClient;
import org.apache.hadoop.hdds.security.x509.certificate.client.CertificateNotification;
import org.apache.hadoop.hdds.security.x509.certificate.client.RootCaRotationPoller;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateCodec;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.security.x509.exception.CertificateException;
import org.apache.hadoop.hdds.security.x509.keys.HDDSKeyGenerator;
import org.apache.hadoop.hdds.security.x509.keys.KeyStorage;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.slf4j.Logger;

public abstract class DefaultCertificateClient
implements CertificateClient {
    private static final String CERT_FILE_EXTENSION = ".crt";
    public static final String CERT_FILE_NAME_FORMAT = "%s.crt";
    private final Logger logger;
    private final SecurityConfig securityConfig;
    private KeyStorage keyStorage;
    private PrivateKey privateKey;
    private PublicKey publicKey;
    private CertPath certPath;
    private Map<String, CertPath> certificateMap;
    private Set<X509Certificate> rootCaCertificates;
    private Set<X509Certificate> caCertificates;
    private String certSerialId;
    private String caCertId;
    private String rootCaCertId;
    private String component;
    private final String threadNamePrefix;
    private ReloadingX509KeyManager keyManager;
    private ReloadingX509TrustManager trustManager;
    private ScheduledExecutorService executorService;
    private Consumer<String> certIdSaveCallback;
    private Runnable shutdownCallback;
    private SCMSecurityProtocolClientSideTranslatorPB scmSecurityClient;
    private final Set<CertificateNotification> notificationReceivers;
    private RootCaRotationPoller rootCaRotationPoller;

    protected DefaultCertificateClient(SecurityConfig securityConfig, SCMSecurityProtocolClientSideTranslatorPB scmSecurityClient, Logger log, String certSerialId, String component, String threadNamePrefix, Consumer<String> saveCertId, Runnable shutdown) {
        Objects.requireNonNull(securityConfig);
        this.securityConfig = securityConfig;
        this.scmSecurityClient = scmSecurityClient;
        this.logger = log;
        this.certificateMap = new ConcurrentHashMap<String, CertPath>();
        this.component = component;
        this.threadNamePrefix = threadNamePrefix;
        this.certIdSaveCallback = saveCertId;
        this.shutdownCallback = shutdown;
        this.notificationReceivers = new HashSet<CertificateNotification>();
        this.rootCaCertificates = ConcurrentHashMap.newKeySet();
        this.caCertificates = ConcurrentHashMap.newKeySet();
        this.updateCertSerialId(certSerialId);
    }

    private KeyStorage keyStorage() throws IOException {
        if (this.keyStorage == null) {
            this.keyStorage = new KeyStorage(this.securityConfig, this.component);
        }
        return this.keyStorage;
    }

    private synchronized void loadAllCertificates() {
        Path path = this.securityConfig.getCertificateLocation(this.component);
        if (!path.toFile().exists() && this.certSerialId == null) {
            return;
        }
        try (Stream<Path> certFiles = Files.list(path);){
            certFiles.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(this::readCertificateFile);
        }
        catch (IOException e) {
            this.getLogger().warn("Certificates could not be loaded.", (Throwable)e);
            return;
        }
        if (this.shouldStartCertificateRenewerService()) {
            if (this.securityConfig.isAutoCARotationEnabled()) {
                this.startRootCaRotationPoller();
            }
            if (this.certPath != null && this.executorService == null) {
                this.startCertificateRenewerService();
            } else if (this.executorService != null) {
                this.getLogger().debug("CertificateRenewerService is already started.");
            } else {
                this.getLogger().warn("Component certificate was not loaded.");
            }
        } else {
            this.getLogger().info("CertificateRenewerService and root ca rotation polling is disabled for {}", (Object)this.component);
        }
    }

    protected String threadNamePrefix() {
        return this.threadNamePrefix;
    }

    private void startRootCaRotationPoller() {
        if (this.rootCaRotationPoller == null) {
            this.rootCaRotationPoller = new RootCaRotationPoller(this.securityConfig, new HashSet<X509Certificate>(this.rootCaCertificates), this.scmSecurityClient, this.threadNamePrefix);
            this.rootCaRotationPoller.addRootCARotationProcessor(this::getRootCaRotationListener);
            this.rootCaRotationPoller.run();
        } else {
            this.getLogger().debug("Root CA certificate rotation poller is already started.");
        }
    }

    public synchronized void registerRootCARotationListener(Function<List<X509Certificate>, CompletableFuture<Void>> listener) {
        if (this.securityConfig.isAutoCARotationEnabled()) {
            this.rootCaRotationPoller.addRootCARotationProcessor(listener);
        }
    }

    private synchronized void readCertificateFile(Path filePath) {
        CertificateCodec codec = new CertificateCodec(this.securityConfig, this.component);
        String fileName = filePath.getFileName().toString();
        try {
            CertPath allCertificates = codec.getCertPath(fileName);
            X509Certificate cert = this.firstCertificateFrom(allCertificates);
            String readCertSerialId = cert.getSerialNumber().toString();
            if (readCertSerialId.equals(this.certSerialId)) {
                this.certPath = allCertificates;
            }
            this.certificateMap.put(readCertSerialId, allCertificates);
            this.addCertsToSubCaMapIfNeeded(fileName, allCertificates);
            this.addCertToRootCaMapIfNeeded(fileName, allCertificates);
            this.updateCachedData(fileName, CAType.SUBORDINATE, this::updateCachedSubCAId);
            this.updateCachedData(fileName, CAType.ROOT, this::updateCachedRootCAId);
            this.getLogger().info("Added certificate {} from file: {}.", (Object)readCertSerialId, (Object)filePath.toAbsolutePath());
            if (this.getLogger().isDebugEnabled()) {
                this.getLogger().debug("Certificate: {}", (Object)cert);
            }
        }
        catch (IOException | IndexOutOfBoundsException | java.security.cert.CertificateException e) {
            this.getLogger().error("Error reading certificate from file: {}.", (Object)filePath.toAbsolutePath(), (Object)e);
        }
    }

    private void updateCachedData(String fileName, CAType tryCAType, Consumer<String> updateCachedId) throws IOException {
        String caTypePrefix = tryCAType.getFileNamePrefix();
        if (fileName.startsWith(caTypePrefix)) {
            updateCachedId.accept(fileName.substring(caTypePrefix.length(), fileName.length() - CERT_FILE_EXTENSION.length()));
        }
    }

    private synchronized void updateCachedRootCAId(String s) {
        BigInteger candidateNewId = new BigInteger(s);
        if (this.rootCaCertId == null || new BigInteger(this.rootCaCertId).compareTo(candidateNewId) < 0) {
            this.rootCaCertId = s;
        }
    }

    private synchronized void updateCachedSubCAId(String s) {
        BigInteger candidateNewId = new BigInteger(s);
        if (this.caCertId == null || new BigInteger(this.caCertId).compareTo(candidateNewId) < 0) {
            this.caCertId = s;
        }
    }

    private void addCertsToSubCaMapIfNeeded(String fileName, CertPath certs) {
        if (fileName.startsWith(CAType.SUBORDINATE.getFileNamePrefix())) {
            this.caCertificates.addAll(certs.getCertificates().stream().map(x -> (X509Certificate)x).collect(Collectors.toSet()));
        }
    }

    private void addCertToRootCaMapIfNeeded(String fileName, CertPath certs) {
        if (fileName.startsWith(CAType.ROOT.getFileNamePrefix())) {
            this.rootCaCertificates.add(this.firstCertificateFrom(certs));
        }
    }

    public synchronized PrivateKey getPrivateKey() {
        if (this.privateKey != null) {
            return this.privateKey;
        }
        Path keyPath = this.securityConfig.getKeyLocation(this.component);
        if (OzoneSecurityUtil.checkIfFileExist((Path)keyPath, (String)this.securityConfig.getPrivateKeyFileName())) {
            try {
                this.privateKey = this.keyStorage().readPrivateKey();
            }
            catch (IOException e) {
                this.getLogger().error("Error while getting private key.", (Throwable)e);
            }
        }
        return this.privateKey;
    }

    public synchronized PublicKey getPublicKey() {
        if (this.publicKey != null) {
            return this.publicKey;
        }
        Path keyPath = this.securityConfig.getKeyLocation(this.component);
        if (OzoneSecurityUtil.checkIfFileExist((Path)keyPath, (String)this.securityConfig.getPublicKeyFileName())) {
            try {
                this.publicKey = this.keyStorage().readPublicKey();
            }
            catch (IOException e) {
                this.getLogger().error("Error while getting public key.", (Throwable)e);
            }
        }
        return this.publicKey;
    }

    public X509Certificate getCertificate() {
        CertPath currentCertPath = this.getCertPath();
        if (currentCertPath == null || currentCertPath.getCertificates() == null) {
            return null;
        }
        return this.firstCertificateFrom(currentCertPath);
    }

    public synchronized CertPath getCertPath() {
        if (this.certPath != null) {
            return this.certPath;
        }
        if (this.certSerialId == null) {
            this.getLogger().error("Default certificate serial id is not set. Can't locate the default certificate for this client.");
            return null;
        }
        this.loadAllCertificates();
        if (this.certificateMap.containsKey(this.certSerialId)) {
            this.certPath = this.certificateMap.get(this.certSerialId);
        }
        return this.certPath;
    }

    public synchronized X509Certificate getCACertificate() {
        CertPath caCertPath = this.getCACertPath();
        if (caCertPath == null || caCertPath.getCertificates() == null) {
            return null;
        }
        return this.firstCertificateFrom(caCertPath);
    }

    public synchronized List<X509Certificate> getTrustChain() throws IOException {
        CertPath path = this.getCertPath();
        if (path == null || path.getCertificates() == null) {
            return null;
        }
        ArrayList<X509Certificate> chain = new ArrayList<X509Certificate>();
        if (path.getCertificates().size() > 1) {
            for (int i = 0; i < path.getCertificates().size(); ++i) {
                chain.add((X509Certificate)path.getCertificates().get(i));
            }
        } else {
            X509Certificate cert = this.getCertificate();
            if (cert != null) {
                chain.add(this.getCertificate());
            }
            if ((cert = this.getCACertificate()) != null) {
                chain.add(this.getCACertificate());
            }
            if ((cert = this.getRootCACertificate()) != null) {
                chain.add(cert);
            }
            Preconditions.checkState((!chain.isEmpty() ? 1 : 0) != 0, (Object)"Empty trust chain");
        }
        return chain;
    }

    public synchronized CertPath getCACertPath() {
        if (this.caCertId != null) {
            return this.certificateMap.get(this.caCertId);
        }
        return null;
    }

    public synchronized X509Certificate getCertificate(String certId) throws CertificateException {
        if (this.certificateMap.containsKey(certId) && this.certificateMap.get(certId).getCertificates() != null) {
            return this.firstCertificateFrom(this.certificateMap.get(certId));
        }
        return this.getCertificateFromScm(certId);
    }

    private X509Certificate getCertificateFromScm(String certId) throws CertificateException {
        this.getLogger().info("Getting certificate with certSerialId:{}.", (Object)certId);
        try {
            String pemEncodedCert = this.getScmSecureClient().getCertificate(certId);
            this.storeCertificate(pemEncodedCert, CAType.NONE);
            return CertificateCodec.getX509Certificate((String)pemEncodedCert);
        }
        catch (Exception e) {
            this.getLogger().error("Error while getting Certificate with certSerialId:{} from scm.", (Object)certId, (Object)e);
            throw new CertificateException("Error while getting certificate for certSerialId:" + certId, (Throwable)e, CertificateException.ErrorCode.CERTIFICATE_ERROR);
        }
    }

    public byte[] signData(byte[] data) throws CertificateException {
        try {
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initSign(this.getPrivateKey());
            sign.update(data);
            return sign.sign();
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", (Throwable)e, CertificateException.ErrorCode.CRYPTO_SIGN_ERROR);
        }
    }

    public boolean verifySignature(byte[] data, byte[] signature, X509Certificate cert) throws CertificateException {
        try {
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initVerify(cert);
            sign.update(data);
            return sign.verify(signature);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", (Throwable)e, CertificateException.ErrorCode.CRYPTO_SIGNATURE_VERIFICATION_ERROR);
        }
    }

    private boolean verifySignature(byte[] data, byte[] signature, PublicKey pubKey) throws CertificateException {
        try {
            Signature sign = Signature.getInstance(this.securityConfig.getSignatureAlgo(), this.securityConfig.getProvider());
            sign.initVerify(pubKey);
            sign.update(data);
            return sign.verify(signature);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            this.getLogger().error("Error while signing the stream", (Throwable)e);
            throw new CertificateException("Error while signing the stream", (Throwable)e, CertificateException.ErrorCode.CRYPTO_SIGNATURE_VERIFICATION_ERROR);
        }
    }

    public CertificateSignRequest.Builder configureCSRBuilder() throws SCMSecurityException {
        return new CertificateSignRequest.Builder().setConfiguration(this.securityConfig).addInetAddresses().setDigitalEncryption(true).setDigitalSignature(true);
    }

    public void storeCertificate(String pemEncodedCert, CAType caType) throws CertificateException {
        CertificateCodec certificateCodec = new CertificateCodec(this.securityConfig, this.component);
        this.storeCertificate(pemEncodedCert, caType, certificateCodec, true, false);
    }

    public synchronized void storeCertificate(String pemEncodedCert, CAType caType, CertificateCodec codec, boolean addToCertMap, boolean updateCA) throws CertificateException {
        try {
            CertPath certificatePath = CertificateCodec.getCertPathFromPemEncodedString((String)pemEncodedCert);
            X509Certificate cert = this.firstCertificateFrom(certificatePath);
            String certName = String.format(CERT_FILE_NAME_FORMAT, caType.getFileNamePrefix() + cert.getSerialNumber().toString());
            if (updateCA) {
                if (caType == CAType.SUBORDINATE) {
                    this.caCertId = cert.getSerialNumber().toString();
                }
                if (caType == CAType.ROOT) {
                    this.rootCaCertId = cert.getSerialNumber().toString();
                }
            }
            codec.writeCertificate(certName, pemEncodedCert);
            if (addToCertMap) {
                this.certificateMap.put(cert.getSerialNumber().toString(), certificatePath);
                if (caType == CAType.SUBORDINATE) {
                    this.caCertificates.add(cert);
                }
                if (caType == CAType.ROOT) {
                    this.rootCaCertificates.add(cert);
                }
            }
        }
        catch (IOException | java.security.cert.CertificateException e) {
            throw new CertificateException("Error while storing certificate.", (Throwable)e, CertificateException.ErrorCode.CERTIFICATE_ERROR);
        }
    }

    public synchronized void initWithRecovery() throws IOException {
        this.recoverStateIfNeeded(this.init());
    }

    @VisibleForTesting
    public synchronized CertificateClient.InitResponse init() throws IOException {
        int initCase = 0;
        PrivateKey pvtKey = this.getPrivateKey();
        PublicKey pubKey = this.getPublicKey();
        X509Certificate certificate = this.getCertificate();
        if (pvtKey != null) {
            initCase |= 4;
        }
        if (pubKey != null) {
            initCase |= 2;
        }
        if (certificate != null) {
            initCase |= 1;
        }
        this.getLogger().info("Certificate client init case: {}", (Object)initCase);
        Preconditions.checkArgument((initCase < InitCase.values().length ? 1 : 0) != 0, (Object)"Not a valid case.");
        InitCase init = InitCase.values()[initCase];
        return this.handleCase(init);
    }

    private X509Certificate firstCertificateFrom(CertPath certificatePath) {
        return CertificateCodec.firstCertificateFrom((CertPath)certificatePath);
    }

    protected CertificateClient.InitResponse handleCase(InitCase init) throws IOException {
        switch (init) {
            case NONE: {
                this.getLogger().info("Creating keypair for client as keypair and certificate not found.");
                this.bootstrapClientKeys();
                return CertificateClient.InitResponse.GETCERT;
            }
            case CERT: {
                this.getLogger().error("Private key not found, while certificate is still present. Delete keypair and try again.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PUBLIC_KEY: {
                this.getLogger().error("Found public key but private key and certificate missing.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PRIVATE_KEY: {
                this.getLogger().info("Found private key but public key and certificate is missing.");
                if (this.recoverPublicKeyFromPrivateKey()) {
                    return CertificateClient.InitResponse.GETCERT;
                }
                return CertificateClient.InitResponse.FAILURE;
            }
            case PUBLICKEY_CERT: {
                this.getLogger().error("Found public key and certificate but private key is missing.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PRIVATEKEY_CERT: {
                this.getLogger().info("Found private key and certificate but public key missing.");
                if (this.recoverPublicKey()) {
                    return CertificateClient.InitResponse.SUCCESS;
                }
                this.getLogger().error("Public key recovery failed.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case PUBLICKEY_PRIVATEKEY: {
                this.getLogger().info("Found private and public key but certificate is missing.");
                if (this.validateKeyPair(this.getPublicKey())) {
                    return CertificateClient.InitResponse.GETCERT;
                }
                this.getLogger().info("Keypair validation failed.");
                return CertificateClient.InitResponse.FAILURE;
            }
            case ALL: {
                this.getLogger().info("Found certificate file along with KeyPair.");
                if (this.validateKeyPairAndCertificate()) {
                    return CertificateClient.InitResponse.SUCCESS;
                }
                return CertificateClient.InitResponse.FAILURE;
            }
        }
        this.getLogger().error("Unexpected case: {} (private/public/cert)", (Object)Integer.toBinaryString(init.ordinal()));
        return CertificateClient.InitResponse.FAILURE;
    }

    protected void recoverStateIfNeeded(CertificateClient.InitResponse state) throws IOException {
        String upperCaseComponent = this.component.toUpperCase();
        this.getLogger().info("Init response: {}", (Object)state);
        switch (state) {
            case SUCCESS: {
                this.getLogger().info("Initialization successful, case:{}.", (Object)state);
                break;
            }
            case GETCERT: {
                Path certLocation = this.securityConfig.getCertificateLocation(this.getComponentName());
                String certId = this.signAndStoreCertificate(this.configureCSRBuilder().build(), certLocation, false);
                if (this.certIdSaveCallback == null) {
                    throw new RuntimeException(upperCaseComponent + " doesn't have the certIdSaveCallback set. The new certificate ID " + certId + " cannot be persisted to the VERSION file");
                }
                this.certIdSaveCallback.accept(certId);
                this.getLogger().info("Successfully stored {} signed certificate, case:{}.", (Object)upperCaseComponent, (Object)state);
                break;
            }
            default: {
                this.getLogger().error("{} security initialization failed. Init response: {}", (Object)upperCaseComponent, (Object)state);
                throw new RuntimeException(upperCaseComponent + " security initialization failed.");
            }
        }
    }

    protected boolean validateKeyPairAndCertificate() throws CertificateException {
        if (this.validateKeyPair(this.getPublicKey())) {
            this.getLogger().info("Keypair validated.");
            if (!this.validateKeyPair(this.getCertificate().getPublicKey())) {
                this.getLogger().error("Stored certificate is generated with different private key.");
                return false;
            }
        } else {
            this.getLogger().error("Keypair validation failed.");
            return false;
        }
        this.getLogger().info("Keypair validated with certificate.");
        return true;
    }

    protected boolean recoverPublicKey() throws CertificateException {
        PublicKey pubKey = this.getCertificate().getPublicKey();
        try {
            if (!this.validateKeyPair(pubKey)) {
                this.getLogger().error("Can't recover public key corresponding to private key.");
                return false;
            }
            this.keyStorage().storePublicKey(pubKey);
            this.publicKey = pubKey;
        }
        catch (IOException e) {
            throw new CertificateException("Error while trying to recover public key.", (Throwable)e, CertificateException.ErrorCode.BOOTSTRAP_ERROR);
        }
        return true;
    }

    protected boolean recoverPublicKeyFromPrivateKey() throws CertificateException {
        PrivateKey priKey = this.getPrivateKey();
        try {
            if (priKey != null && priKey instanceof RSAPrivateCrtKey) {
                RSAPrivateCrtKey rsaCrtKey = (RSAPrivateCrtKey)priKey;
                RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(rsaCrtKey.getModulus(), rsaCrtKey.getPublicExponent());
                PublicKey pubKey = KeyFactory.getInstance(this.securityConfig.getKeyAlgo()).generatePublic(rsaPublicKeySpec);
                if (this.validateKeyPair(pubKey)) {
                    this.keyStorage().storePublicKey(pubKey);
                    this.publicKey = pubKey;
                    this.getLogger().info("Public key is recovered from the private key.");
                    return true;
                }
            }
        }
        catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new CertificateException("Error while trying to recover public key.", (Throwable)e, CertificateException.ErrorCode.BOOTSTRAP_ERROR);
        }
        this.getLogger().error("Can't recover public key corresponding to private key.");
        return false;
    }

    protected boolean validateKeyPair(PublicKey pubKey) throws CertificateException {
        byte[] challenge = RandomStringUtils.random((int)1000, (int)0, (int)0, (boolean)false, (boolean)false, null, (Random)new SecureRandom()).getBytes(StandardCharsets.UTF_8);
        return this.verifySignature(challenge, this.signData(challenge), pubKey);
    }

    protected void bootstrapClientKeys() throws IOException {
        Path keyPath = this.securityConfig.getKeyLocation(this.component);
        if (Files.notExists(keyPath, new LinkOption[0])) {
            try {
                Files.createDirectories(keyPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new CertificateException("Error while creating directories for certificate storage.", CertificateException.ErrorCode.BOOTSTRAP_ERROR);
            }
        }
        KeyPair keyPair = this.createKeyPair(this.keyStorage());
        this.privateKey = keyPair.getPrivate();
        this.publicKey = keyPair.getPublic();
    }

    protected KeyPair createKeyPair(KeyStorage storage) throws CertificateException {
        KeyPair keyPair;
        HDDSKeyGenerator keyGenerator = new HDDSKeyGenerator(this.securityConfig);
        try {
            KeyStorage keyStorageToUse = storage == null ? this.keyStorage() : storage;
            keyPair = keyGenerator.generateKey();
            keyStorageToUse.storeKeyPair(keyPair);
        }
        catch (IOException | NoSuchAlgorithmException | NoSuchProviderException e) {
            this.getLogger().error("Error while bootstrapping certificate client.", (Throwable)e);
            throw new CertificateException("Error while bootstrapping certificate.", CertificateException.ErrorCode.BOOTSTRAP_ERROR);
        }
        return keyPair;
    }

    public Logger getLogger() {
        return this.logger;
    }

    public String getComponentName() {
        return this.component;
    }

    public synchronized X509Certificate getRootCACertificate() {
        if (this.rootCaCertId != null) {
            return this.firstCertificateFrom(this.certificateMap.get(this.rootCaCertId));
        }
        return null;
    }

    public Set<X509Certificate> getAllRootCaCerts() {
        Set<X509Certificate> certs = Collections.unmodifiableSet(this.rootCaCertificates);
        this.getLogger().info("{} has {} Root CA certificates", (Object)this.component, (Object)certs.size());
        return certs;
    }

    public Set<X509Certificate> getAllCaCerts() {
        Set<X509Certificate> certs = Collections.unmodifiableSet(this.caCertificates);
        this.getLogger().info("{} has {} CA certificates", (Object)this.component, (Object)certs.size());
        return certs;
    }

    public ReloadingX509TrustManager getTrustManager() throws CertificateException {
        try {
            if (this.trustManager == null) {
                Set<X509Certificate> newRootCaCerts = this.rootCaCertificates.isEmpty() ? this.caCertificates : this.rootCaCertificates;
                this.trustManager = new ReloadingX509TrustManager(KeyStore.getDefaultType(), new ArrayList<X509Certificate>(newRootCaCerts));
                this.notificationReceivers.add((CertificateNotification)this.trustManager);
            }
            return this.trustManager;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new CertificateException("Failed to init trustManager", (Throwable)e, CertificateException.ErrorCode.KEYSTORE_ERROR);
        }
    }

    public ReloadingX509KeyManager getKeyManager() throws CertificateException {
        try {
            if (this.keyManager == null) {
                this.keyManager = new ReloadingX509KeyManager(KeyStore.getDefaultType(), this.getComponentName(), this.getPrivateKey(), this.getTrustChain());
                this.notificationReceivers.add((CertificateNotification)this.keyManager);
            }
            return this.keyManager;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new CertificateException("Failed to init keyManager", (Throwable)e, CertificateException.ErrorCode.KEYSTORE_ERROR);
        }
    }

    public ClientTrustManager createClientTrustManager() throws IOException {
        CACertificateProvider caCertificateProvider = () -> {
            ArrayList<X509Certificate> caCerts = new ArrayList<X509Certificate>();
            caCerts.addAll(this.getAllCaCerts());
            caCerts.addAll(this.getAllRootCaCerts());
            return caCerts;
        };
        return new ClientTrustManager(caCertificateProvider, caCertificateProvider);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerNotificationReceiver(CertificateNotification receiver) {
        Set<CertificateNotification> set = this.notificationReceivers;
        synchronized (set) {
            this.notificationReceivers.add(receiver);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void notifyNotificationReceivers(String oldCaCertId, String newCaCertId) {
        Set<CertificateNotification> set = this.notificationReceivers;
        synchronized (set) {
            this.notificationReceivers.forEach(r -> r.notifyCertificateRenewed((CertificateClient)this, oldCaCertId, newCaCertId));
        }
    }

    public synchronized void close() throws IOException {
        if (this.executorService != null) {
            this.executorService.shutdownNow();
            this.executorService = null;
        }
        if (this.rootCaRotationPoller != null) {
            this.rootCaRotationPoller.close();
        }
    }

    public Duration timeBeforeExpiryGracePeriod(X509Certificate certificate) {
        LocalDateTime currentTime;
        Duration gracePeriod = this.securityConfig.getRenewalGracePeriod();
        Date expireDate = certificate.getNotAfter();
        LocalDateTime gracePeriodStart = expireDate.toInstant().minus(gracePeriod).atZone(ZoneId.systemDefault()).toLocalDateTime();
        if (gracePeriodStart.isBefore(currentTime = LocalDateTime.now())) {
            return Duration.ZERO;
        }
        return Duration.between(currentTime, gracePeriodStart);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String renewAndStoreKeyAndCertificate(boolean force) throws CertificateException {
        String newCertSerialId;
        KeyPair newKeyPair;
        if (!force) {
            DefaultCertificateClient defaultCertificateClient = this;
            synchronized (defaultCertificateClient) {
                Preconditions.checkArgument((boolean)this.timeBeforeExpiryGracePeriod(this.firstCertificateFrom(this.certPath)).isZero());
            }
        }
        String newKeyPath = this.securityConfig.getKeyLocation(this.component).toString() + "-next";
        String newCertPath = this.securityConfig.getCertificateLocation(this.component).toString() + "-next";
        File newKeyDir = new File(newKeyPath);
        File newCertDir = new File(newCertPath);
        try {
            FileUtils.deleteDirectory((File)newKeyDir);
            FileUtils.deleteDirectory((File)newCertDir);
            Files.createDirectories(newKeyDir.toPath(), new FileAttribute[0]);
            Files.createDirectories(newCertDir.toPath(), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new CertificateException("Error while deleting/creating " + newKeyPath + " or " + newCertPath + " directories to cleanup  certificate storage. ", (Throwable)e, CertificateException.ErrorCode.RENEW_ERROR);
        }
        try {
            KeyStorage newKeyStorage = new KeyStorage(this.securityConfig, this.component, "-next");
            newKeyPair = this.createKeyPair(newKeyStorage);
        }
        catch (IOException e) {
            throw new CertificateException("Error while creating new key pair.", (Throwable)e, CertificateException.ErrorCode.RENEW_ERROR);
        }
        try {
            CertificateSignRequest.Builder csrBuilder = this.configureCSRBuilder();
            csrBuilder.setKey(newKeyPair);
            newCertSerialId = this.signAndStoreCertificate(csrBuilder.build(), Paths.get(newCertPath, new String[0]), true);
        }
        catch (Exception e) {
            throw new CertificateException("Error while signing and storing new certificates.", (Throwable)e, CertificateException.ErrorCode.RENEW_ERROR);
        }
        File currentKeyDir = new File(this.securityConfig.getKeyLocation(this.component).toString());
        File currentCertDir = new File(this.securityConfig.getCertificateLocation(this.component).toString());
        File backupKeyDir = new File(this.securityConfig.getKeyLocation(this.component).toString() + "-previous");
        File backupCertDir = new File(this.securityConfig.getCertificateLocation(this.component).toString() + "-previous");
        try {
            Files.move(currentKeyDir.toPath(), backupKeyDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            throw new CertificateException("Failed to move " + currentKeyDir.getAbsolutePath() + " to " + backupKeyDir.getAbsolutePath() + " during certificate renew.", CertificateException.ErrorCode.RENEW_ERROR);
        }
        try {
            Files.move(currentCertDir.toPath(), backupCertDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            this.rollbackBackupDir(currentKeyDir, currentCertDir, backupKeyDir, backupCertDir);
            throw new CertificateException("Failed to move " + currentCertDir.getAbsolutePath() + " to " + backupCertDir.getAbsolutePath() + " during certificate renew.", CertificateException.ErrorCode.RENEW_ERROR);
        }
        try {
            Files.move(newKeyDir.toPath(), currentKeyDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            String msg = "Failed to move " + newKeyDir.getAbsolutePath() + " to " + currentKeyDir.getAbsolutePath() + " during certificate renew.";
            this.rollbackBackupDir(currentKeyDir, currentCertDir, backupKeyDir, backupCertDir);
            throw new CertificateException(msg, CertificateException.ErrorCode.RENEW_ERROR);
        }
        try {
            Files.move(newCertDir.toPath(), currentCertDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            String msg = "Failed to move " + newCertDir.getAbsolutePath() + " to " + currentCertDir.getAbsolutePath() + " during certificate renew.";
            try {
                FileUtils.deleteDirectory((File)new File(currentKeyDir.toString()));
            }
            catch (IOException e1) {
                this.getLogger().error("Failed to delete current KeyDir {} which is moved  from the newly generated KeyDir {}", new Object[]{currentKeyDir, newKeyDir, e});
                throw new CertificateException(msg, CertificateException.ErrorCode.RENEW_ERROR);
            }
            this.rollbackBackupDir(currentKeyDir, currentCertDir, backupKeyDir, backupCertDir);
            throw new CertificateException(msg, CertificateException.ErrorCode.RENEW_ERROR);
        }
        this.getLogger().info("Successful renew key and certificate. New certificate {}.", (Object)newCertSerialId);
        return newCertSerialId;
    }

    private void rollbackBackupDir(File currentKeyDir, File currentCertDir, File backupKeyDir, File backupCertDir) throws CertificateException {
        try {
            Files.move(backupKeyDir.toPath(), currentKeyDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            String msg = "Failed to move " + backupKeyDir.getAbsolutePath() + " back to " + currentKeyDir.getAbsolutePath() + " during rollback.";
            throw new CertificateException(msg, CertificateException.ErrorCode.ROLLBACK_ERROR);
        }
        try {
            Files.move(backupCertDir.toPath(), currentCertDir.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            String msg = "Failed to move " + backupCertDir.getAbsolutePath() + " back to " + currentCertDir.getAbsolutePath() + " during rollback.";
            throw new CertificateException(msg, CertificateException.ErrorCode.ROLLBACK_ERROR);
        }
        Preconditions.checkArgument((boolean)currentCertDir.exists());
        Preconditions.checkArgument((boolean)currentKeyDir.exists());
    }

    public void cleanBackupDir() {
        File backupKeyDir = new File(this.securityConfig.getKeyLocation(this.component).toString() + "-previous");
        File backupCertDir = new File(this.securityConfig.getCertificateLocation(this.component).toString() + "-previous");
        if (backupKeyDir.exists()) {
            try {
                FileUtils.deleteDirectory((File)backupKeyDir);
            }
            catch (IOException e) {
                this.getLogger().error("Error while deleting {} directories for certificate storage cleanup.", (Object)backupKeyDir, (Object)e);
            }
        }
        if (backupCertDir.exists()) {
            try {
                FileUtils.deleteDirectory((File)backupCertDir);
            }
            catch (IOException e) {
                this.getLogger().error("Error while deleting {} directories for certificate storage cleanup.", (Object)backupCertDir, (Object)e);
            }
        }
    }

    public synchronized void reloadKeyAndCertificate(String newCertId) {
        this.privateKey = null;
        this.publicKey = null;
        this.certPath = null;
        this.caCertId = null;
        this.rootCaCertId = null;
        String oldCaCertId = this.updateCertSerialId(newCertId);
        this.getLogger().info("Reset and reloaded key and all certificates for new certificate {}.", (Object)newCertId);
        this.notifyNotificationReceivers(oldCaCertId, newCertId);
    }

    public SecurityConfig getSecurityConfig() {
        return this.securityConfig;
    }

    private synchronized String updateCertSerialId(String newCertSerialId) {
        this.certSerialId = newCertSerialId;
        this.getLogger().info("Certificate serial ID set to {}", (Object)this.certSerialId);
        this.loadAllCertificates();
        return this.certSerialId;
    }

    protected abstract SCMSecurityProtocolProtos.SCMGetCertResponseProto sign(CertificateSignRequest var1) throws IOException;

    protected String signAndStoreCertificate(CertificateSignRequest csr, Path certificatePath, boolean renew) throws CertificateException {
        try {
            SCMSecurityProtocolProtos.SCMGetCertResponseProto response = this.sign(csr);
            if (response.hasX509CACertificate()) {
                String pemEncodedCert = response.getX509Certificate();
                CertificateCodec certCodec = new CertificateCodec(this.getSecurityConfig(), certificatePath);
                this.storeCertificate(pemEncodedCert, CAType.NONE, certCodec, false, !renew);
                this.storeCertificate(response.getX509CACertificate(), CAType.SUBORDINATE, certCodec, false, !renew);
                this.getAndStoreAllRootCAs(certCodec, renew);
                return this.updateCertSerialId(CertificateCodec.getX509Certificate((String)pemEncodedCert).getSerialNumber().toString());
            }
            throw new CertificateException("Unable to retrieve certificate chain.");
        }
        catch (IOException | java.security.cert.CertificateException e) {
            this.logger.error("Error while signing and storing SCM signed certificate.", (Throwable)e);
            throw new CertificateException("Error while signing and storing SCM signed certificate.", (Throwable)e);
        }
    }

    private void getAndStoreAllRootCAs(CertificateCodec certCodec, boolean renew) throws IOException {
        List<String> rootCAPems = this.scmSecurityClient.getAllRootCaCertificates();
        for (String rootCAPem : rootCAPems) {
            this.storeCertificate(rootCAPem, CAType.ROOT, certCodec, false, !renew);
        }
    }

    public SCMSecurityProtocolClientSideTranslatorPB getScmSecureClient() {
        return this.scmSecurityClient;
    }

    protected boolean shouldStartCertificateRenewerService() {
        return true;
    }

    public synchronized CompletableFuture<Void> getRootCaRotationListener(List<X509Certificate> rootCAs) {
        if (this.rootCaCertificates.containsAll(rootCAs)) {
            return CompletableFuture.completedFuture(null);
        }
        CertificateRenewerService renewerService = new CertificateRenewerService(true, this.rootCaRotationPoller::setCertificateRenewalError);
        return CompletableFuture.runAsync(renewerService, this.executorService);
    }

    public synchronized void startCertificateRenewerService() {
        Preconditions.checkNotNull((Object)this.getCertificate(), (Object)"Component certificate should not be empty");
        Duration gracePeriod = this.securityConfig.getRenewalGracePeriod();
        long timeBeforeGracePeriod = this.timeBeforeExpiryGracePeriod(this.firstCertificateFrom(this.certPath)).toMillis();
        long interval = Math.min(gracePeriod.toMillis() / 3L, TimeUnit.DAYS.toMillis(1L));
        if (this.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(this.threadNamePrefix + this.getComponentName() + "-CertificateRenewerService").setDaemon(true).build());
        }
        this.executorService.scheduleAtFixedRate(new CertificateRenewerService(false, () -> {}), timeBeforeGracePeriod + 1L, interval, TimeUnit.MILLISECONDS);
        this.getLogger().info("CertificateRenewerService for {} is started with first delay {} ms and interval {} ms.", new Object[]{this.component, timeBeforeGracePeriod, interval});
    }

    @VisibleForTesting
    public synchronized void setCACertificate(X509Certificate cert) throws Exception {
        this.caCertId = cert.getSerialNumber().toString();
        String pemCert = CertificateCodec.getPEMEncodedString((X509Certificate)cert);
        this.certificateMap.put(this.caCertId, CertificateCodec.getCertPathFromPemEncodedString((String)pemCert));
    }

    public class CertificateRenewerService
    implements Runnable {
        private boolean forceRenewal;
        private Runnable rotationErrorCallback;

        public CertificateRenewerService(boolean forceRenewal, Runnable rotationErrorCallback) {
            this.forceRenewal = forceRenewal;
            this.rotationErrorCallback = rotationErrorCallback;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Class<DefaultCertificateClient> clazz = DefaultCertificateClient.class;
            synchronized (DefaultCertificateClient.class) {
                String newCertId;
                X509Certificate currentCert = DefaultCertificateClient.this.getCertificate();
                Duration timeLeft = DefaultCertificateClient.this.timeBeforeExpiryGracePeriod(currentCert);
                if (!this.forceRenewal && !timeLeft.isZero()) {
                    DefaultCertificateClient.this.getLogger().info("Current certificate {} hasn't entered the renew grace period. Remaining period is {}. ", (Object)currentCert.getSerialNumber().toString(), (Object)timeLeft);
                    // ** MonitorExit[var1_1] (shouldn't be in output)
                    return;
                }
                try {
                    DefaultCertificateClient.this.getLogger().info("Current certificate {} needs to be renewed remaining grace period {}. Forced renewal due to root ca rotation: {}.", new Object[]{currentCert.getSerialNumber().toString(), timeLeft, this.forceRenewal});
                    newCertId = DefaultCertificateClient.this.renewAndStoreKeyAndCertificate(this.forceRenewal);
                }
                catch (CertificateException e) {
                    this.rotationErrorCallback.run();
                    if (e.errorCode() == CertificateException.ErrorCode.ROLLBACK_ERROR && DefaultCertificateClient.this.shutdownCallback != null) {
                        DefaultCertificateClient.this.getLogger().error("Failed to rollback key and cert after an unsuccessful renew try.", (Throwable)e);
                        DefaultCertificateClient.this.shutdownCallback.run();
                    }
                    DefaultCertificateClient.this.getLogger().error("Failed to renew and store key and cert. Keep using existing certificates.", (Throwable)e);
                    // ** MonitorExit[var1_1] (shouldn't be in output)
                    return;
                }
                if (DefaultCertificateClient.this.certIdSaveCallback != null) {
                    DefaultCertificateClient.this.certIdSaveCallback.accept(newCertId);
                }
                DefaultCertificateClient.this.reloadKeyAndCertificate(newCertId);
                DefaultCertificateClient.this.cleanBackupDir();
                // ** MonitorExit[var1_1] (shouldn't be in output)
                return;
            }
        }
    }

    protected static enum InitCase {
        NONE,
        CERT,
        PUBLIC_KEY,
        PUBLICKEY_CERT,
        PRIVATE_KEY,
        PRIVATEKEY_CERT,
        PUBLICKEY_PRIVATEKEY,
        ALL;

    }
}

