/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.ha;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.hdds.protocol.proto.SCMRatisProtocol;
import org.apache.hadoop.hdds.scm.block.DeletedBlockLog;
import org.apache.hadoop.hdds.scm.block.DeletedBlockLogImpl;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.ha.SCMHADBTransactionBuffer;
import org.apache.hadoop.hdds.scm.ha.SCMNodeDetails;
import org.apache.hadoop.hdds.scm.ha.SCMRatisRequest;
import org.apache.hadoop.hdds.scm.ha.SCMRatisResponse;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.security.symmetric.ManagedSecretKey;
import org.apache.hadoop.hdds.utils.TransactionInfo;
import org.apache.hadoop.hdds.utils.db.DBCheckpoint;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.ExitUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LifeCycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMStateMachine
extends BaseStateMachine {
    private static final Logger LOG = LoggerFactory.getLogger(SCMStateMachine.class);
    private StorageContainerManager scm;
    private Map<SCMRatisProtocol.RequestType, Object> handlers;
    private SCMHADBTransactionBuffer transactionBuffer;
    private final SimpleStateMachineStorage storage = new SimpleStateMachineStorage();
    private final boolean isInitialized;
    private ExecutorService installSnapshotExecutor;
    private DBCheckpoint installingDBCheckpoint = null;
    private List<ManagedSecretKey> installingSecretKeys = null;
    private AtomicLong currentLeaderTerm = new AtomicLong(-1L);
    private AtomicBoolean refreshedAfterLeaderReady = new AtomicBoolean();

    public SCMStateMachine(StorageContainerManager scm, SCMHADBTransactionBuffer buffer) {
        this.scm = scm;
        this.handlers = new EnumMap<SCMRatisProtocol.RequestType, Object>(SCMRatisProtocol.RequestType.class);
        this.transactionBuffer = buffer;
        TransactionInfo latestTrxInfo = this.transactionBuffer.getLatestTrxInfo();
        if (!latestTrxInfo.isDefault()) {
            this.updateLastAppliedTermIndex(latestTrxInfo.getTerm(), latestTrxInfo.getTransactionIndex());
            LOG.info("Updated lastAppliedTermIndex {} with transactionInfo term andIndex", (Object)latestTrxInfo);
        }
        this.installSnapshotExecutor = HadoopExecutors.newSingleThreadExecutor((ThreadFactory)new ThreadFactoryBuilder().setNameFormat(scm.threadNamePrefix() + "SCMInstallSnapshot-%d").build());
        this.isInitialized = true;
    }

    public SCMStateMachine() {
        this.isInitialized = false;
    }

    public void registerHandler(SCMRatisProtocol.RequestType type, Object handler) {
        this.handlers.put(type, handler);
    }

    public SnapshotInfo getLatestSnapshot() {
        return this.transactionBuffer == null ? null : this.transactionBuffer.getLatestSnapshot();
    }

    public void initialize(RaftServer server, RaftGroupId id, RaftStorage raftStorage) throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            super.initialize(server, id, raftStorage);
            this.storage.init(raftStorage);
            LOG.info("{}: initialize {}", (Object)server.getId(), (Object)id);
        }, new Class[0]);
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        CompletableFuture<Message> applyTransactionFuture = new CompletableFuture<Message>();
        try {
            SCMRatisRequest request = SCMRatisRequest.decode(Message.valueOf((ByteString)trx.getStateMachineLogEntry().getLogData()));
            if (LOG.isDebugEnabled()) {
                LOG.debug("{}: applyTransaction {}", (Object)this.getId(), (Object)TermIndex.valueOf((RaftProtos.LogEntryProto)trx.getLogEntry()));
            }
            try {
                applyTransactionFuture.complete(this.process(request));
            }
            catch (SCMException ex) {
                if (ex.getResult() == SCMException.ResultCodes.INTERNAL_ERROR || ex.getResult() == SCMException.ResultCodes.IO_EXCEPTION) {
                    throw ex;
                }
                applyTransactionFuture.completeExceptionally(ex);
            }
            if (this.scm.isInSafeMode() && this.refreshedAfterLeaderReady.get()) {
                this.scm.getScmSafeModeManager().refreshAndValidate();
            }
            this.transactionBuffer.updateLatestTrxInfo(TransactionInfo.valueOf((TermIndex)TermIndex.valueOf((RaftProtos.LogEntryProto)trx.getLogEntry())));
        }
        catch (Exception ex) {
            applyTransactionFuture.completeExceptionally(ex);
            ExitUtils.terminate((int)1, (String)ex.getMessage(), (Throwable)ex, (Logger)StateMachine.LOG);
        }
        return applyTransactionFuture;
    }

    private Message process(SCMRatisRequest request) throws Exception {
        try {
            Object handler = this.handlers.get(request.getType());
            if (handler == null) {
                throw new IOException("No handler found for request type " + request.getType());
            }
            Object result = handler.getClass().getMethod(request.getOperation(), request.getParameterTypes()).invoke(handler, request.getArguments());
            return SCMRatisResponse.encode(result);
        }
        catch (NoSuchMethodException | SecurityException ex) {
            throw new InvalidProtocolBufferException(ex.getMessage());
        }
        catch (InvocationTargetException e) {
            Exception targetEx = (Exception)e.getTargetException();
            throw targetEx != null ? targetEx : e;
        }
    }

    public void notifyLogFailed(Throwable ex, RaftProtos.LogEntryProto failedEntry) {
        LOG.error("SCM statemachine appendLog failed, entry: {}", (Object)failedEntry);
        ExitUtils.terminate((int)1, (String)ex.getMessage(), (Throwable)ex, (Logger)StateMachine.LOG);
    }

    public void notifyNotLeader(Collection<TransactionContext> pendingEntries) {
        if (!this.isInitialized) {
            return;
        }
        LOG.info("current leader SCM steps down.");
        this.scm.getScmContext().updateLeaderAndTerm(false, 0L);
        this.scm.getSCMServiceManager().notifyStatusChanged();
    }

    public CompletableFuture<TermIndex> notifyInstallSnapshotFromLeader(RaftProtos.RoleInfoProto roleInfoProto, TermIndex firstTermIndexInLog) {
        if (!roleInfoProto.getFollowerInfo().hasLeaderInfo()) {
            return JavaUtils.completeExceptionally((Throwable)new IOException("Failed to notifyInstallSnapshotFromLeader due to missing leader info"));
        }
        String leaderAddress = roleInfoProto.getFollowerInfo().getLeaderInfo().getId().getAddress();
        Optional<SCMNodeDetails> leaderDetails = this.scm.getSCMHANodeDetails().getPeerNodeDetails().stream().filter(p -> p.getRatisHostPortStr().equals(leaderAddress)).findFirst();
        Preconditions.checkState((boolean)leaderDetails.isPresent());
        String leaderNodeId = leaderDetails.get().getNodeId();
        LOG.info("Received install snapshot notification from SCM leader: {} with term index: {}", (Object)leaderAddress, (Object)firstTermIndexInLog);
        CompletableFuture<TermIndex> future = CompletableFuture.supplyAsync(() -> {
            List<ManagedSecretKey> secretKeys;
            DBCheckpoint checkpoint = this.scm.getScmHAManager().downloadCheckpointFromLeader(leaderNodeId);
            if (checkpoint == null) {
                return null;
            }
            try {
                secretKeys = this.scm.getScmHAManager().getSecretKeysFromLeader(leaderNodeId);
                LOG.info("Got secret keys from leaders {}", secretKeys);
            }
            catch (IOException ex) {
                LOG.error("Failed to get secret keys from SCM leader {}", (Object)leaderNodeId, (Object)ex);
                return null;
            }
            TermIndex termIndex = this.scm.getScmHAManager().verifyCheckpointFromLeader(leaderNodeId, checkpoint);
            if (termIndex != null) {
                this.setInstallingSnapshotData(checkpoint, secretKeys);
            }
            return termIndex;
        }, this.installSnapshotExecutor);
        return future;
    }

    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId newLeaderId) {
        if (!this.isInitialized) {
            return;
        }
        this.currentLeaderTerm.set(this.scm.getScmHAManager().getRatisServer().getDivision().getInfo().getCurrentTerm());
        if (!groupMemberId.getPeerId().equals((Object)newLeaderId)) {
            LOG.info("leader changed, yet current SCM is still follower.");
            return;
        }
        LOG.info("current SCM becomes leader of term {}.", (Object)this.currentLeaderTerm);
        this.scm.getScmContext().updateLeaderAndTerm(true, this.currentLeaderTerm.get());
        this.scm.getSequenceIdGen().invalidateBatch();
        DeletedBlockLog deletedBlockLog = this.scm.getScmBlockManager().getDeletedBlockLog();
        Preconditions.checkArgument((boolean)(deletedBlockLog instanceof DeletedBlockLogImpl));
        ((DeletedBlockLogImpl)deletedBlockLog).onBecomeLeader();
        this.scm.getScmDecommissionManager().onBecomeLeader();
        this.scm.scmHAMetricsUpdate(newLeaderId.toString());
    }

    public long takeSnapshot() throws IOException {
        TransactionInfo lastAppliedTrxInfo;
        TermIndex lastTermIndex = this.getLastAppliedTermIndex();
        long lastAppliedIndex = lastTermIndex.getIndex();
        if (!this.isInitialized) {
            return lastAppliedIndex;
        }
        long startTime = Time.monotonicNow();
        TransactionInfo latestTrxInfo = this.transactionBuffer.getLatestTrxInfo();
        if (latestTrxInfo.compareTo(lastAppliedTrxInfo = TransactionInfo.valueOf((TermIndex)lastTermIndex)) < 0) {
            this.transactionBuffer.updateLatestTrxInfo(lastAppliedTrxInfo);
            this.transactionBuffer.setLatestSnapshot(lastAppliedTrxInfo.toSnapshotInfo());
        } else {
            lastAppliedIndex = latestTrxInfo.getTransactionIndex();
        }
        this.transactionBuffer.flush();
        LOG.info("Current Snapshot Index {}, takeSnapshot took {} ms", (Object)lastAppliedIndex, (Object)(Time.monotonicNow() - startTime));
        return lastAppliedIndex;
    }

    public void notifyTermIndexUpdated(long term, long index) {
        this.updateLastAppliedTermIndex(term, index);
        if (!this.isInitialized) {
            return;
        }
        if (this.transactionBuffer != null) {
            this.transactionBuffer.updateLatestTrxInfo(TransactionInfo.valueOf((long)term, (long)index));
        }
        if (this.currentLeaderTerm.get() == term) {
            if (!this.refreshedAfterLeaderReady.get()) {
                this.scm.getScmSafeModeManager().refresh();
                this.scm.getDatanodeProtocolServer().start();
                this.refreshedAfterLeaderReady.set(true);
            }
            this.currentLeaderTerm.set(-1L);
        }
    }

    public void notifyLeaderReady() {
        if (!this.isInitialized) {
            return;
        }
        this.scm.getScmContext().setLeaderReady();
        this.scm.getSCMServiceManager().notifyStatusChanged();
        this.scm.getFinalizationManager().onLeaderReady();
    }

    public void notifyConfigurationChanged(long term, long index, RaftProtos.RaftConfigurationProto newRaftConfiguration) {
    }

    public void pause() {
        LifeCycle lc = this.getLifeCycle();
        LOG.info("{}: Try to pause from current LifeCycle state {}", (Object)this.getId(), (Object)lc);
        if (lc.getCurrentState() != LifeCycle.State.NEW) {
            lc.transition(LifeCycle.State.PAUSING);
            lc.transition(LifeCycle.State.PAUSED);
        }
    }

    public void reinitialize() throws IOException {
        Preconditions.checkNotNull((Object)this.installingDBCheckpoint);
        DBCheckpoint checkpoint = this.installingDBCheckpoint;
        List<ManagedSecretKey> secretKeys = this.installingSecretKeys;
        this.installingDBCheckpoint = null;
        this.installingSecretKeys = null;
        TermIndex termIndex = null;
        try {
            termIndex = this.scm.getScmHAManager().installCheckpoint(checkpoint);
        }
        catch (Exception e) {
            LOG.error("Failed to reinitialize SCMStateMachine.", (Throwable)e);
            throw new IOException(e);
        }
        LOG.info("{}: SCMStateMachine is reinitializing. newTermIndex = {}", (Object)this.getId(), (Object)termIndex);
        try {
            this.transactionBuffer.init();
            this.setLastAppliedTermIndex(termIndex);
        }
        catch (IOException ioe) {
            LOG.error("Failed to unpause ", (Throwable)ioe);
        }
        if (secretKeys != null) {
            Objects.requireNonNull(this.scm.getSecretKeyManager()).reinitialize(secretKeys);
        }
        this.getLifeCycle().transition(LifeCycle.State.STARTING);
        this.getLifeCycle().transition(LifeCycle.State.RUNNING);
    }

    public void close() throws IOException {
        if (!this.isInitialized) {
            return;
        }
        if (this.scm.getScmHAManager().getRatisServer().isStopped()) {
            super.close();
            this.transactionBuffer.close();
            HadoopExecutors.shutdown((ExecutorService)this.installSnapshotExecutor, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        } else if (!this.scm.isStopped()) {
            this.scm.shutDown("scm statemachine is closed by ratis, terminate SCM");
        }
    }

    @VisibleForTesting
    public void setInstallingSnapshotData(DBCheckpoint checkpoint, List<ManagedSecretKey> secretKeys) {
        Preconditions.checkArgument((this.installingDBCheckpoint == null ? 1 : 0) != 0);
        this.installingDBCheckpoint = checkpoint;
        this.installingSecretKeys = secretKeys;
    }
}

