/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.block;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.block.DeletedBlockLogStateManager;
import org.apache.hadoop.hdds.scm.block.ScmBlockDeletingServiceMetrics;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMDeletedBlockTransactionStatusManager {
    public static final Logger LOG = LoggerFactory.getLogger(SCMDeletedBlockTransactionStatusManager.class);
    private final Map<Long, Set<UUID>> transactionToDNsCommitMap;
    private final Map<Long, Integer> transactionToRetryCountMap;
    private final DeletedBlockLogStateManager deletedBlockLogStateManager;
    private final ContainerManager containerManager;
    private final ScmBlockDeletingServiceMetrics metrics;
    private final long scmCommandTimeoutMs;
    private final SCMDeleteBlocksCommandStatusManager scmDeleteBlocksCommandStatusManager;

    public SCMDeletedBlockTransactionStatusManager(DeletedBlockLogStateManager deletedBlockLogStateManager, ContainerManager containerManager, ScmBlockDeletingServiceMetrics metrics, long scmCommandTimeoutMs) {
        this.deletedBlockLogStateManager = deletedBlockLogStateManager;
        this.metrics = metrics;
        this.containerManager = containerManager;
        this.scmCommandTimeoutMs = scmCommandTimeoutMs;
        this.transactionToDNsCommitMap = new ConcurrentHashMap<Long, Set<UUID>>();
        this.transactionToRetryCountMap = new ConcurrentHashMap<Long, Integer>();
        this.scmDeleteBlocksCommandStatusManager = new SCMDeleteBlocksCommandStatusManager();
    }

    public void incrementRetryCount(List<Long> txIDs, long maxRetry) throws IOException {
        ArrayList<Long> txIDsToUpdate = new ArrayList<Long>();
        for (Long txID : txIDs) {
            int currentCount = this.transactionToRetryCountMap.getOrDefault(txID, 0);
            if ((long)currentCount > maxRetry) continue;
            if ((long)(++currentCount) > maxRetry) {
                txIDsToUpdate.add(txID);
            }
            this.transactionToRetryCountMap.put(txID, currentCount);
        }
        if (!txIDsToUpdate.isEmpty()) {
            this.deletedBlockLogStateManager.increaseRetryCountOfTransactionInDB(txIDsToUpdate);
        }
    }

    public void resetRetryCount(List<Long> txIDs) throws IOException {
        for (Long txID : txIDs) {
            this.transactionToRetryCountMap.computeIfPresent(txID, (key, value) -> 0);
        }
    }

    public int getOrDefaultRetryCount(long txID, int defaultValue) {
        return this.transactionToRetryCountMap.getOrDefault(txID, defaultValue);
    }

    public void onSent(DatanodeDetails dnId, SCMCommand<?> scmCommand) {
        this.scmDeleteBlocksCommandStatusManager.onSent(dnId.getUuid(), scmCommand.getId());
    }

    public Map<UUID, Map<Long, SCMDeleteBlocksCommandStatusManager.CmdStatus>> getCommandStatusByTxId(Set<UUID> dnIds) {
        return this.scmDeleteBlocksCommandStatusManager.getCommandStatusByTxId(dnIds);
    }

    public void recordTransactionCreated(UUID dnId, long scmCmdId, Set<Long> dnTxSet) {
        this.scmDeleteBlocksCommandStatusManager.recordScmCommand(SCMDeleteBlocksCommandStatusManager.createScmCmdStatusData(dnId, scmCmdId, dnTxSet));
        dnTxSet.forEach(txId -> {
            Set cfr_ignored_0 = this.transactionToDNsCommitMap.putIfAbsent((Long)txId, new LinkedHashSet());
        });
    }

    public void clear() {
        this.transactionToRetryCountMap.clear();
        this.scmDeleteBlocksCommandStatusManager.clear();
        this.transactionToDNsCommitMap.clear();
    }

    public void cleanAllTimeoutSCMCommand(long timeoutMs) {
        this.scmDeleteBlocksCommandStatusManager.cleanAllTimeoutSCMCommand(timeoutMs);
    }

    public void onDatanodeDead(UUID dnId) {
        this.scmDeleteBlocksCommandStatusManager.onDatanodeDead(dnId);
    }

    public boolean isDuplication(DatanodeDetails dnDetail, long tx, Map<UUID, Map<Long, SCMDeleteBlocksCommandStatusManager.CmdStatus>> commandStatus) {
        if (this.alreadyExecuted(dnDetail.getUuid(), tx)) {
            return true;
        }
        return this.inProcessing(dnDetail.getUuid(), tx, commandStatus);
    }

    public boolean alreadyExecuted(UUID dnId, long txId) {
        Set<UUID> dnsWithTransactionCommitted = this.transactionToDNsCommitMap.get(txId);
        return dnsWithTransactionCommitted != null && dnsWithTransactionCommitted.contains(dnId);
    }

    @VisibleForTesting
    public void commitTransactions(List<StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult> transactionResults, UUID dnId) {
        ArrayList<Long> txIDsToBeDeleted = new ArrayList<Long>();
        for (StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult transactionResult : transactionResults) {
            if (this.isTransactionFailed(transactionResult)) {
                this.metrics.incrBlockDeletionTransactionFailure();
                continue;
            }
            try {
                List containerDns;
                this.metrics.incrBlockDeletionTransactionSuccess();
                long txID = transactionResult.getTxID();
                Set<UUID> dnsWithCommittedTxn = this.transactionToDNsCommitMap.get(txID);
                ContainerID containerId = ContainerID.valueOf((long)transactionResult.getContainerID());
                if (dnsWithCommittedTxn == null) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("Transaction txId: {} commit by Datanode: {} for ContainerId: {} failed. Corresponding entry not found.", new Object[]{txID, dnId, containerId});
                    continue;
                }
                dnsWithCommittedTxn.add(dnId);
                ContainerInfo container = this.containerManager.getContainer(containerId);
                Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(containerId);
                if (Math.min(replicas.size(), dnsWithCommittedTxn.size()) >= container.getReplicationConfig().getRequiredNodes() && dnsWithCommittedTxn.containsAll(containerDns = replicas.stream().map(ContainerReplica::getDatanodeDetails).map(DatanodeDetails::getUuid).collect(Collectors.toList()))) {
                    this.transactionToDNsCommitMap.remove(txID);
                    this.transactionToRetryCountMap.remove(txID);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Purging txId: {} from block deletion log", (Object)txID);
                    }
                    txIDsToBeDeleted.add(txID);
                }
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Datanode txId: {} ContainerId: {} committed by Datanode: {}", new Object[]{txID, containerId, dnId});
            }
            catch (IOException e) {
                LOG.warn("Could not commit delete block transaction: " + transactionResult.getTxID(), (Throwable)e);
            }
        }
        try {
            this.deletedBlockLogStateManager.removeTransactionsFromDB(txIDsToBeDeleted);
            this.metrics.incrBlockDeletionTransactionCompleted(txIDsToBeDeleted.size());
        }
        catch (IOException e) {
            LOG.warn("Could not commit delete block transactions: " + txIDsToBeDeleted, (Throwable)e);
        }
    }

    @VisibleForTesting
    public void commitSCMCommandStatus(List<StorageContainerDatanodeProtocolProtos.CommandStatus> deleteBlockStatus, UUID dnId) {
        this.processSCMCommandStatus(deleteBlockStatus, dnId);
        this.scmDeleteBlocksCommandStatusManager.cleanTimeoutSCMCommand(dnId, this.scmCommandTimeoutMs);
    }

    private boolean inProcessing(UUID dnId, long deletedBlocksTxId, Map<UUID, Map<Long, SCMDeleteBlocksCommandStatusManager.CmdStatus>> commandStatus) {
        Map<Long, SCMDeleteBlocksCommandStatusManager.CmdStatus> deletedBlocksTxStatus = commandStatus.get(dnId);
        return deletedBlocksTxStatus != null && deletedBlocksTxStatus.get(deletedBlocksTxId) != null;
    }

    private void processSCMCommandStatus(List<StorageContainerDatanodeProtocolProtos.CommandStatus> deleteBlockStatus, UUID dnID) {
        HashMap lastStatus = new HashMap();
        HashMap summary = new HashMap();
        deleteBlockStatus.forEach(cmdStatus -> {
            lastStatus.put(cmdStatus.getCmdId(), cmdStatus);
            summary.put(cmdStatus.getCmdId(), cmdStatus.getStatus());
        });
        LOG.debug("CommandStatus {} from Datanode: {} ", summary, (Object)dnID);
        for (Map.Entry entry : lastStatus.entrySet()) {
            StorageContainerDatanodeProtocolProtos.CommandStatus.Status status = ((StorageContainerDatanodeProtocolProtos.CommandStatus)entry.getValue()).getStatus();
            this.scmDeleteBlocksCommandStatusManager.updateStatusByDNCommandStatus(dnID, (Long)entry.getKey(), status);
        }
    }

    private boolean isTransactionFailed(StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult result) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Got block deletion ACK from datanode, TXIDs {}, success {}", (Object)result.getTxID(), (Object)result.getSuccess());
        }
        if (!result.getSuccess()) {
            LOG.warn("Got failed ACK for TXID {}, prepare to resend the TX in next interval", (Object)result.getTxID());
            return true;
        }
        return false;
    }

    protected static class SCMDeleteBlocksCommandStatusManager {
        public static final Logger LOG = LoggerFactory.getLogger(SCMDeleteBlocksCommandStatusManager.class);
        private final Map<UUID, Map<Long, CmdStatusData>> scmCmdStatusRecord = new ConcurrentHashMap<UUID, Map<Long, CmdStatusData>>();
        private static final CmdStatus DEFAULT_STATUS = CmdStatus.TO_BE_SENT;
        private static final Set<CmdStatus> STATUSES_REQUIRING_TIMEOUT = new HashSet<CmdStatus>(Arrays.asList(CmdStatus.SENT));

        protected static CmdStatusData createScmCmdStatusData(UUID dnId, long scmCmdId, Set<Long> deletedBlocksTxIds) {
            return new CmdStatusData(dnId, scmCmdId, deletedBlocksTxIds);
        }

        protected void recordScmCommand(CmdStatusData statusData) {
            LOG.debug("Record ScmCommand: {}", (Object)statusData);
            this.scmCmdStatusRecord.computeIfAbsent(statusData.getDnId(), k -> new ConcurrentHashMap()).put(statusData.getScmCmdId(), statusData);
        }

        protected void onSent(UUID dnId, long scmCmdId) {
            this.updateStatus(dnId, scmCmdId, StorageContainerDatanodeProtocolProtos.CommandStatus.Status.PENDING);
        }

        protected void onDatanodeDead(UUID dnId) {
            LOG.info("Clean SCMCommand record for Datanode: {}", (Object)dnId);
            this.scmCmdStatusRecord.remove(dnId);
        }

        protected void updateStatusByDNCommandStatus(UUID dnId, long scmCmdId, StorageContainerDatanodeProtocolProtos.CommandStatus.Status newState) {
            this.updateStatus(dnId, scmCmdId, newState);
        }

        protected void cleanAllTimeoutSCMCommand(long timeoutMs) {
            for (UUID dnId : this.scmCmdStatusRecord.keySet()) {
                for (CmdStatus status : STATUSES_REQUIRING_TIMEOUT) {
                    this.removeTimeoutScmCommand(dnId, this.getScmCommandIds(dnId, status), timeoutMs);
                }
            }
        }

        public void cleanTimeoutSCMCommand(UUID dnId, long timeoutMs) {
            for (CmdStatus status : STATUSES_REQUIRING_TIMEOUT) {
                this.removeTimeoutScmCommand(dnId, this.getScmCommandIds(dnId, status), timeoutMs);
            }
        }

        private Set<Long> getScmCommandIds(UUID dnId, CmdStatus status) {
            HashSet<Long> scmCmdIds = new HashSet<Long>();
            Map<Long, CmdStatusData> record = this.scmCmdStatusRecord.get(dnId);
            if (record == null) {
                return scmCmdIds;
            }
            for (CmdStatusData statusData : record.values()) {
                if (!statusData.getStatus().equals((Object)status)) continue;
                scmCmdIds.add(statusData.getScmCmdId());
            }
            return scmCmdIds;
        }

        private Instant getUpdateTime(UUID dnId, long scmCmdId) {
            Map<Long, CmdStatusData> record = this.scmCmdStatusRecord.get(dnId);
            if (record == null || record.get(scmCmdId) == null) {
                return null;
            }
            return record.get(scmCmdId).getUpdateTime();
        }

        private void updateStatus(UUID dnId, long scmCmdId, StorageContainerDatanodeProtocolProtos.CommandStatus.Status newStatus) {
            Map<Long, CmdStatusData> recordForDn = this.scmCmdStatusRecord.get(dnId);
            if (recordForDn == null) {
                LOG.warn("Unknown Datanode: {} Scm Command ID: {} report status {}", new Object[]{dnId, scmCmdId, newStatus});
                return;
            }
            if (recordForDn.get(scmCmdId) == null) {
                LOG.debug("Unknown SCM Command ID: {} Datanode: {} report status {}", new Object[]{scmCmdId, dnId, newStatus});
                return;
            }
            boolean changed = false;
            CmdStatusData statusData = recordForDn.get(scmCmdId);
            CmdStatus oldStatus = statusData.getStatus();
            switch (newStatus) {
                case PENDING: {
                    if (oldStatus != CmdStatus.TO_BE_SENT && oldStatus != CmdStatus.SENT) break;
                    statusData.setStatus(CmdStatus.SENT);
                    changed = true;
                    break;
                }
                case EXECUTED: 
                case FAILED: {
                    if (oldStatus == CmdStatus.SENT) {
                        this.removeScmCommand(dnId, scmCmdId);
                        changed = true;
                    }
                    if (oldStatus != CmdStatus.TO_BE_SENT) break;
                    LOG.error("Received {} status for a command marked TO_BE_SENT. This indicates a potential issue in command handling. SCM Command ID: {}, Datanode: {}, Current status: {}", new Object[]{newStatus, scmCmdId, dnId, oldStatus});
                    this.removeScmCommand(dnId, scmCmdId);
                    changed = true;
                    break;
                }
                default: {
                    LOG.error("Unexpected status from Datanode: {}. SCM Command ID: {} with status: {}.", new Object[]{dnId, scmCmdId, newStatus});
                }
            }
            if (!changed) {
                LOG.warn("Cannot update illegal status for Datanode: {} SCM Command ID: {} status {} by DN report status {}", new Object[]{dnId, scmCmdId, oldStatus, newStatus});
            } else {
                LOG.debug("Successful update Datanode: {} SCM Command ID: {} status From {} to {}, DN report status {}", new Object[]{dnId, scmCmdId, oldStatus, statusData.getStatus(), newStatus});
            }
        }

        private void removeTimeoutScmCommand(UUID dnId, Set<Long> scmCmdIds, long timeoutMs) {
            Instant now = Instant.now();
            for (Long scmCmdId : scmCmdIds) {
                Instant updateTime = this.getUpdateTime(dnId, scmCmdId);
                if (updateTime == null || Duration.between(updateTime, now).toMillis() <= timeoutMs) continue;
                CmdStatusData state = this.removeScmCommand(dnId, scmCmdId);
                LOG.warn("SCM BlockDeletionCommand {} for Datanode: {} was removed after {}ms without update", new Object[]{state, dnId, timeoutMs});
            }
        }

        private CmdStatusData removeScmCommand(UUID dnId, long scmCmdId) {
            Map<Long, CmdStatusData> record = this.scmCmdStatusRecord.get(dnId);
            if (record == null || record.get(scmCmdId) == null) {
                return null;
            }
            CmdStatusData statusData = record.remove(scmCmdId);
            LOG.debug("Remove ScmCommand {} for Datanode: {} ", (Object)statusData, (Object)dnId);
            return statusData;
        }

        public Map<UUID, Map<Long, CmdStatus>> getCommandStatusByTxId(Set<UUID> dnIds) {
            HashMap<UUID, Map<Long, CmdStatus>> result = new HashMap<UUID, Map<Long, CmdStatus>>(this.scmCmdStatusRecord.size());
            for (UUID dnId : dnIds) {
                Map<Long, CmdStatusData> record = this.scmCmdStatusRecord.get(dnId);
                if (record == null) continue;
                HashMap<Long, CmdStatus> dnStatusMap = new HashMap<Long, CmdStatus>();
                for (CmdStatusData statusData : record.values()) {
                    CmdStatus status = statusData.getStatus();
                    for (Long deletedBlocksTxId : statusData.getDeletedBlocksTxIds()) {
                        dnStatusMap.put(deletedBlocksTxId, status);
                    }
                }
                result.put(dnId, dnStatusMap);
            }
            return result;
        }

        private void clear() {
            this.scmCmdStatusRecord.clear();
        }

        @VisibleForTesting
        Map<UUID, Map<Long, CmdStatusData>> getScmCmdStatusRecord() {
            return this.scmCmdStatusRecord;
        }

        protected static final class CmdStatusData {
            private final UUID dnId;
            private final long scmCmdId;
            private final Set<Long> deletedBlocksTxIds;
            private Instant updateTime;
            private CmdStatus status;

            private CmdStatusData(UUID dnId, long scmTxID, Set<Long> deletedBlocksTxIds) {
                this.dnId = dnId;
                this.scmCmdId = scmTxID;
                this.deletedBlocksTxIds = deletedBlocksTxIds;
                this.setStatus(DEFAULT_STATUS);
            }

            public Set<Long> getDeletedBlocksTxIds() {
                return Collections.unmodifiableSet(this.deletedBlocksTxIds);
            }

            public UUID getDnId() {
                return this.dnId;
            }

            public long getScmCmdId() {
                return this.scmCmdId;
            }

            public CmdStatus getStatus() {
                return this.status;
            }

            public void setStatus(CmdStatus status) {
                this.updateTime = Instant.now();
                this.status = status;
            }

            public Instant getUpdateTime() {
                return this.updateTime;
            }

            public String toString() {
                return "ScmTxStateMachine{dnId=" + this.dnId + ", scmTxID=" + this.scmCmdId + ", deletedBlocksTxIds=" + this.deletedBlocksTxIds + ", updateTime=" + this.updateTime + ", status=" + (Object)((Object)this.status) + '}';
            }
        }

        public static enum CmdStatus {
            TO_BE_SENT,
            SENT;

        }
    }
}

