/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.utils;

import java.io.IOException;
import java.math.BigInteger;
import java.net.InetAddress;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.security.x509.exception.CertificateException;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.apache.hadoop.ozone.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.ozone.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.ozone.shaded.org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.ozone.shaded.org.apache.commons.validator.routines.DomainValidator;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.ASN1Encodable;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.ASN1EncodableVector;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.ASN1Object;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.DERSequence;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.DERTaggedObject;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.DERUTF8String;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x500.X500Name;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x509.BasicConstraints;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x509.Extension;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x509.GeneralName;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x509.GeneralNames;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x509.KeyUsage;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.cert.CertIOException;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.cert.X509v3CertificateBuilder;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.operator.ContentSigner;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.operator.OperatorCreationException;
import org.apache.hadoop.ozone.shaded.org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SelfSignedCertificate {
    private String subject;
    private String clusterID;
    private String scmID;
    private LocalDateTime beginDate;
    private LocalDateTime endDate;
    private KeyPair key;
    private SecurityConfig config;
    private List<GeneralName> altNames;
    private static final Logger LOG = LoggerFactory.getLogger(SelfSignedCertificate.class);

    private SelfSignedCertificate(Builder builder) {
        this.subject = builder.subject;
        this.clusterID = builder.clusterID;
        this.scmID = builder.scmID;
        this.beginDate = builder.beginDate;
        this.endDate = builder.endDate;
        this.config = builder.config;
        this.key = builder.key;
        this.altNames = builder.altNames;
    }

    @VisibleForTesting
    public static String getNameFormat() {
        return CertificateSignRequest.getDistinguishedNameFormatWithSN();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private X509Certificate generateCertificate(BigInteger caCertSerialId) throws OperatorCreationException, IOException {
        byte[] encoded = this.key.getPublic().getEncoded();
        SubjectPublicKeyInfo publicKeyInfo = SubjectPublicKeyInfo.getInstance(encoded);
        ContentSigner contentSigner = new JcaContentSignerBuilder(this.config.getSignatureAlgo()).setProvider(this.config.getProvider()).build(this.key.getPrivate());
        BigInteger serial = caCertSerialId == null ? new BigInteger(Long.toString(Time.monotonicNow())) : caCertSerialId;
        String dnName = String.format(SelfSignedCertificate.getNameFormat(), this.subject, this.scmID, this.clusterID, serial);
        X500Name name = new X500Name(dnName);
        Date validFrom = Date.from(this.beginDate.atZone(ZoneId.systemDefault()).toInstant());
        Date validTill = Date.from(this.endDate.atZone(ZoneId.systemDefault()).toInstant());
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(name, serial, validFrom, validTill, name, publicKeyInfo);
        if (caCertSerialId != null) {
            builder.addExtension(Extension.basicConstraints, true, new BasicConstraints(true));
            int keyUsageFlag = 6;
            KeyUsage keyUsage = new KeyUsage(keyUsageFlag);
            builder.addExtension(Extension.keyUsage, true, keyUsage);
            if (this.altNames != null && !this.altNames.isEmpty()) {
                builder.addExtension(new Extension(Extension.subjectAlternativeName, false, new GeneralNames(this.altNames.toArray(new GeneralName[this.altNames.size()])).getEncoded()));
            }
        }
        try {
            X509Certificate cert = new JcaX509CertificateConverter().getCertificate(builder.build(contentSigner));
            LOG.info("Certificate {} is issued by {} to {}, valid from {} to {}", new Object[]{cert.getSerialNumber(), cert.getIssuerDN(), cert.getSubjectDN(), cert.getNotBefore(), cert.getNotAfter()});
            return cert;
        }
        catch (java.security.cert.CertificateException e) {
            throw new CertificateException("Could not create self-signed X509Certificate.", (Throwable)e, CertificateException.ErrorCode.CERTIFICATE_ERROR);
        }
    }

    public static class Builder {
        private String subject;
        private String clusterID;
        private String scmID;
        private LocalDateTime beginDate;
        private LocalDateTime endDate;
        private KeyPair key;
        private SecurityConfig config;
        private BigInteger caCertSerialId;
        private List<GeneralName> altNames;

        public Builder setConfiguration(SecurityConfig configuration) {
            this.config = configuration;
            return this;
        }

        public Builder setKey(KeyPair keyPair) {
            this.key = keyPair;
            return this;
        }

        public Builder setSubject(String subjectString) {
            this.subject = subjectString;
            return this;
        }

        public Builder setClusterID(String s2) {
            this.clusterID = s2;
            return this;
        }

        public Builder setScmID(String s2) {
            this.scmID = s2;
            return this;
        }

        public Builder setBeginDate(LocalDateTime date) {
            this.beginDate = date;
            return this;
        }

        public Builder setEndDate(LocalDateTime date) {
            this.endDate = date;
            return this;
        }

        public Builder makeCA() {
            return this.makeCA(BigInteger.ONE);
        }

        public Builder makeCA(BigInteger serialId) {
            this.caCertSerialId = serialId;
            return this;
        }

        public Builder addInetAddresses() throws CertificateException {
            try {
                DomainValidator validator = DomainValidator.getInstance();
                List<InetAddress> inetAddresses = OzoneSecurityUtil.getValidInetsForCurrentHost();
                this.addInetAddresses(inetAddresses, validator);
            }
            catch (IOException e) {
                throw new CertificateException("Error while getting Inet addresses for the CSR builder", (Throwable)e, CertificateException.ErrorCode.CSR_ERROR);
            }
            return this;
        }

        public Builder addInetAddresses(List<InetAddress> addresses, DomainValidator validator) {
            addresses.forEach(ip -> {
                this.addIpAddress(ip.getHostAddress());
                if (validator.isValid(ip.getCanonicalHostName())) {
                    this.addDnsName(ip.getCanonicalHostName());
                } else {
                    LOG.error("Invalid domain {}", (Object)ip.getCanonicalHostName());
                }
            });
            return this;
        }

        public Builder addDnsName(String dnsName) {
            Preconditions.checkNotNull(dnsName, "dnsName cannot be null");
            this.addAltName(2, dnsName);
            return this;
        }

        public Builder addIpAddress(String ip) {
            Preconditions.checkNotNull(ip, "Ip address cannot be null");
            this.addAltName(7, ip);
            return this;
        }

        private Builder addAltName(int tag, String name) {
            if (this.altNames == null) {
                this.altNames = new ArrayList<GeneralName>();
            }
            if (tag == 0) {
                ASN1Object ono = this.addOtherNameAsn1Object(name);
                this.altNames.add(new GeneralName(tag, ono));
            } else {
                this.altNames.add(new GeneralName(tag, name));
            }
            return this;
        }

        private ASN1Object addOtherNameAsn1Object(String name) {
            String otherNameOID = "2.16.840.1.113730.3.1.34";
            ASN1EncodableVector otherName = new ASN1EncodableVector();
            otherName.add(new ASN1ObjectIdentifier("2.16.840.1.113730.3.1.34"));
            otherName.add(new DERTaggedObject(true, 0, (ASN1Encodable)new DERUTF8String(name)));
            return new DERTaggedObject(false, 0, (ASN1Encodable)new DERSequence(otherName));
        }

        public X509Certificate build() throws SCMSecurityException, IOException {
            Preconditions.checkNotNull(this.key, "Key cannot be null");
            Preconditions.checkArgument(StringUtils.isNotBlank(this.subject), "Subject cannot be blank");
            Preconditions.checkArgument(StringUtils.isNotBlank(this.clusterID), "Cluster ID cannot be blank");
            Preconditions.checkArgument(StringUtils.isNotBlank(this.scmID), "SCM ID cannot be blank");
            Preconditions.checkArgument(this.beginDate.isBefore(this.endDate), "Certificate begin date should be before end date");
            Duration certDuration = Duration.between(this.beginDate, this.endDate);
            Duration maxDuration = this.config.getMaxCertificateDuration();
            if (certDuration.compareTo(maxDuration) > 0) {
                throw new SCMSecurityException("The cert duration violates the maximum configured value. Please check the hdds.x509.max.duration config key. Current Value: " + certDuration + " config: " + maxDuration);
            }
            SelfSignedCertificate rootCertificate = new SelfSignedCertificate(this);
            try {
                return rootCertificate.generateCertificate(this.caCertSerialId);
            }
            catch (CertIOException | OperatorCreationException e) {
                throw new CertificateException("Unable to create root certificate.", e.getCause());
            }
        }
    }
}

