/****************************************************************************
 * sched/signal/sig_dispatch.c
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <inttypes.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sched.h>
#include <assert.h>
#include <errno.h>
#include <debug.h>

#include <nuttx/irq.h>
#include <nuttx/arch.h>
#include <nuttx/signal.h>
#include <nuttx/queue.h>

#include "sched/sched.h"
#include "group/group.h"
#include "semaphore/semaphore.h"
#include "signal/signal.h"
#include "mqueue/mqueue.h"

/****************************************************************************
 * Private Types
 ****************************************************************************/

struct sig_arg_s
{
  pid_t pid;
  bool need_restore;
};

/****************************************************************************
 * Private Functions
 ****************************************************************************/

#ifdef CONFIG_SMP
static int sig_handler(FAR void *cookie)
{
  FAR struct sig_arg_s *arg = cookie;
  FAR struct tcb_s *tcb;
  irqstate_t flags;

  flags = enter_critical_section();
  tcb = nxsched_get_tcb(arg->pid);

  if (!tcb || tcb->task_state == TSTATE_TASK_INVALID ||
      (tcb->flags & TCB_FLAG_EXIT_PROCESSING) != 0)
    {
      /* There is no TCB with this pid or, if there is, it is not a task. */

      leave_critical_section(flags);
      return -ESRCH;
    }

  if (arg->need_restore)
    {
      tcb->flags &= ~TCB_FLAG_CPU_LOCKED;
    }

  if (tcb->sigdeliver)
    {
      up_schedule_sigaction(tcb);
    }

  leave_critical_section(flags);
  return OK;
}
#endif

/****************************************************************************
 * Name: nxsig_queue_action
 *
 * Description:
 *   Queue a signal action for delivery to a task.
 *
 * Returned Value:
 *   Returns 0 (OK) on success or a negated errno value on failure.
 *
 * Assumptions:
 *   Called in critical section
 *
 ****************************************************************************/

static int nxsig_queue_action(FAR struct tcb_s *stcb,
                              FAR sigactq_t *sigact,
                              FAR siginfo_t *info)
{
  FAR sigq_t    *sigq;
  int            ret = OK;

  DEBUGASSERT(stcb != NULL && stcb->group != NULL);

  /* Check if a valid signal handler is available and if the signal is
   * unblocked. NOTE: There is no default action.
   */

  if ((sigact) && (sigact->act.sa_u._sa_sigaction))
    {
      /* Allocate a new element for the signal queue. NOTE:
       * nxsig_alloc_pendingsigaction will force a system crash if it is
       * unable to allocate memory for the signal data.
       */

      sigq = nxsig_alloc_pendingsigaction();
      if (!sigq)
        {
          ret = -ENOMEM;
        }
      else
        {
          /* Populate the new signal queue element */

          sigq->action.sighandler = sigact->act.sa_u._sa_sigaction;
          sigq->mask = sigact->act.sa_mask;
          if ((sigact->act.sa_flags & SA_NODEFER) == 0)
            {
              sigaddset(&sigq->mask, info->si_signo);
            }

          memcpy(&sigq->info, info, sizeof(siginfo_t));
          sigq->info.si_user = sigact->act.sa_user;

          /* Put it at the end of the pending signals list */

          sq_addlast((FAR sq_entry_t *)sigq, &(stcb->sigpendactionq));

          /* Then schedule execution of the signal handling action on the
           * recipient's thread. SMP related handling will be done in
           * up_schedule_sigaction()
           */

          if (!stcb->sigdeliver)
            {
#ifdef CONFIG_SMP
              int cpu = stcb->cpu;
              int me  = this_cpu();

              stcb->sigdeliver = nxsig_deliver;
              if (cpu != me && stcb->task_state == TSTATE_TASK_RUNNING)
                {
                  struct sig_arg_s arg;

                  if ((stcb->flags & TCB_FLAG_CPU_LOCKED) != 0)
                    {
                      arg.need_restore   = false;
                    }
                  else
                    {
                      arg.need_restore   = true;
                      stcb->flags        |= TCB_FLAG_CPU_LOCKED;
                    }

                  arg.pid = stcb->pid;
                  nxsched_smp_call_single(stcb->cpu, sig_handler, &arg);
                }
              else
#endif
                {
                  stcb->sigdeliver = nxsig_deliver;
                  up_schedule_sigaction(stcb);
                }
            }
        }
    }

  return ret;
}

/****************************************************************************
 * Name: nxsig_alloc_pendingsignal
 *
 * Description:
 *   Allocate a pending signal list entry
 *
 * Assumptions:
 *   Called with g_sigpendingsignal locked
 *
 ****************************************************************************/

static FAR sigpendq_t *nxsig_alloc_pendingsignal(void)
{
  FAR sigpendq_t *sigpend;

  /* Try to get the pending signal structure from the free list */

  sigpend = (FAR sigpendq_t *)sq_remfirst(&g_sigpendingsignal);
  if (!sigpend && up_interrupt_context())
    {
      /* If no pending signal structure is available in the free list,
       * then try the special list of structures reserved for
       * interrupt handlers
       */

      sigpend = (FAR sigpendq_t *)sq_remfirst(&g_sigpendingirqsignal);
    }

  return sigpend;
}

/****************************************************************************
 * Name: nxsig_find_pendingsignal
 *
 * Description:
 *   Find a specified element in the pending signal list
 *
 * Assumptions:
 *   Called with group->tg_sigpendingq locked
 *
 ****************************************************************************/

static FAR sigpendq_t *
nxsig_find_pendingsignal(FAR struct task_group_s *group, int signo)
{
  FAR sigpendq_t *sigpend = NULL;

  DEBUGASSERT(group != NULL);

  /* Determining whether a signal is reliable or unreliable */

  if (SIGRTMIN <= signo && signo <= SIGRTMAX)
    {
      return sigpend;
    }

  /* Search the list for a action pending on this signal */

  for (sigpend = (FAR sigpendq_t *)group->tg_sigpendingq.head;
       (sigpend && sigpend->info.si_signo != signo);
       sigpend = sigpend->flink);

  return sigpend;
}

/****************************************************************************
 * Name: nxsig_dispatch_kernel_action
 ****************************************************************************/

static void nxsig_dispatch_kernel_action(FAR struct tcb_s *stcb,
                                         FAR siginfo_t *info)
{
  FAR struct task_group_s *group = stcb->group;
  FAR sigactq_t *sigact;

  sigact = nxsig_find_action(group, info->si_signo);
  if (sigact && (sigact->act.sa_flags & SA_KERNELHAND))
    {
      info->si_user = sigact->act.sa_user;
      (sigact->act.sa_sigaction)(info->si_signo, info, NULL);
    }
}

/****************************************************************************
 * Name: nxsig_add_pendingsignal
 *
 * Description:
 *   Add the specified signal to the signal pending list. NOTE: This
 *   function will queue only one entry for each pending signal. This
 *   was done intentionally so that a run-away sender cannot consume
 *   all of memory.
 *
 * Assumptions:
 *   Called with tg_sigpendingq locked
 *
 ****************************************************************************/

static FAR sigpendq_t *nxsig_add_pendingsignal(FAR struct tcb_s *stcb,
                                               FAR siginfo_t *info,
                                               bool group_dispatch)
{
  FAR struct task_group_s *group;
  FAR sigpendq_t *sigpend;

  DEBUGASSERT(stcb != NULL && stcb->group != NULL);
  group = stcb->group;

  /* Check if the signal is already pending for the group */

  sigpend = nxsig_find_pendingsignal(group, info->si_signo);
  if (sigpend != NULL)
    {
      /* The signal is already pending... retain only one copy */

      memcpy(&sigpend->info, info, sizeof(siginfo_t));
    }

  /* No... There is nothing pending in the group for this signo */

  else
    {
      /* Allocate a new pending signal entry */

      sigpend = nxsig_alloc_pendingsignal();
      if (sigpend != NULL)
        {
          /* Put the signal information into the allocated structure */

          memcpy(&sigpend->info, info, sizeof(siginfo_t));

          /* Mark the tcb which need to receive the signal. If any
           * thread in the group may receive it, set it to NULL
           */

          sigpend->tcb = group_dispatch ? NULL : stcb;

          /* Add the structure to the group pending signal list */

          sq_addlast((FAR sq_entry_t *)sigpend, &group->tg_sigpendingq);
        }
    }

  DEBUGASSERT(sigpend);

  return sigpend;
}

/****************************************************************************
 * Name: nxsig_alloc_dyn_pending
 *
 * Description:
 *   Dynamically allocate more pending signal and pending sigaction
 *   structures, if there are no more left. Note that this leaves the
 *   the critical section for the allocation. During that time it is
 *   it is possible that structures are freed, or another signalling thread
 *   allocates more structures. This is not an issue, any extra pending
 *   structures are freed after they get used.
 *
 * Assumptions:
 *   Called with g_sigpendingsignal and g_sigpendingaction locked by
 *   critical section.
 *
 ****************************************************************************/

static int nxsig_alloc_dyn_pending(FAR irqstate_t *flags)
{
  int ret = OK;
  bool alloc_signal = sq_empty(&g_sigpendingsignal);
  bool alloc_sigact = sq_empty(&g_sigpendingaction);

  if (alloc_signal || alloc_sigact)
    {
      FAR sigpendq_t *sigpend = NULL;
      FAR sigq_t *sigq = NULL;

      /* We can't do memory allocations in idle task or interrupt */

      if (up_interrupt_context() || sched_idletask())
        {
          return -EAGAIN;
        }

      /* Leave critical section for the duration of heap operations */

      leave_critical_section(*flags);

      /* Allocate more pending signals if there are no more */

      if (alloc_signal)
        {
          sigpend = kmm_malloc(sizeof(sigpendq_t));
        }

      /* Allocate more pending signal actions if there are no more */

      if (alloc_sigact)
        {
          sigq = kmm_malloc(sizeof(sigq_t));
        }

      /* Restore critical section and add the allocated structures to
       * the free pending queues
       */

      *flags = enter_critical_section();

      if (alloc_signal)
        {
          if (sigpend)
            {
              sigpend->type = SIG_ALLOC_DYN;
              sq_addfirst((sq_entry_t *)sigpend, &g_sigpendingsignal);
            }
          else
            {
              ret = -EAGAIN;
            }
        }

      if (alloc_sigact)
        {
          if (sigq)
            {
              sigq->type = SIG_ALLOC_DYN;
              sq_addfirst((sq_entry_t *)sigq, &g_sigpendingaction);
            }
          else
            {
              ret = -EAGAIN;
            }
        }
    }

  return ret;
}

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: nxsig_tcbdispatch
 *
 * Description:
 *   All signals received the task (whatever the source) go through this
 *   function to be processed. This function is responsible for:
 *
 *   - Determining if the signal is blocked.
 *   - Queuing and dispatching signal actions
 *   - Unblocking tasks that are waiting for signals
 *   - Queuing pending signals.
 *
 *   This function will deliver the signal to the task associated with
 *   the specified TCB. This function should *not* typically be used
 *   to dispatch signals since it will *not* follow the group signal
 *   deliver algorithms.
 *
 * Returned Value:
 *   Returns 0 (OK) on success or a negated errno value on failure.
 *
 ****************************************************************************/

int nxsig_tcbdispatch(FAR struct tcb_s *stcb, siginfo_t *info,
                      bool group_dispatch)
{
  FAR struct tcb_s *rtcb = this_task();
  FAR sigactq_t *sigact;
  irqstate_t flags;
  int masked;
  int ret = OK;
  FAR sigpendq_t *sigpend = NULL;

  sinfo("TCB=%p pid=%d signo=%d code=%d value=%d masked=%s\n",
        stcb, stcb->pid, info->si_signo, info->si_code,
        info->si_value.sival_int,
        sigismember(&stcb->sigprocmask, info->si_signo) == 1 ? "YES" : "NO");

  DEBUGASSERT(stcb != NULL && info != NULL);

  /* Return ESRCH when thread was in exit processing */

  if ((stcb->flags & TCB_FLAG_EXIT_PROCESSING) != 0)
    {
      return -ESRCH;
    }

  /* Don't actually send a signal for signo 0. */

  if (info->si_signo == 0)
    {
      return OK;
    }

  /************************** MASKED SIGNAL ACTIONS *************************/

  /* Find if there is a group sigaction associated with this signal */

  sigact = nxsig_find_action(stcb->group, info->si_signo);

  flags = enter_critical_section();

  /* Make sure that there is always at least one sigpednq and sigq structure
   * available, in case one needs to be queued later. Note that this breaks
   * the critical section if it needs to allocate any new structures. So it
   * needs to be done here before using the task state or sigprocmask.
   */

  ret = nxsig_alloc_dyn_pending(&flags);
  if (ret < 0)
    {
      leave_critical_section(flags);
      return ret;
    }

  masked = nxsig_ismember(&stcb->sigprocmask, info->si_signo);

#ifdef CONFIG_LIB_SYSCALL
  /* Check if the signal is masked OR if the signal is received while we are
   * processing a system call -- in either case, it will be added to the
   * list of pending signals. Unmasked user signal actions will be deferred
   * while we process the system call.
   *
   * If a thread calls a blocking system call, the thread will still be
   * unblocked when the signal occurs (see OTHER SIGNAL HANDLING below), but
   * any associated user signal action will be deferred until the system
   * call returns. For example, if the application calls sem_wait(), the
   * following would occur:
   *
   *   1. System call entry logic will block user signal handling and call
   *      sem_wait() in kernel mode.
   *   2. sem_wait() will block,
   *   3. The receipt of the signal will cause any signal action to pend
   *      but will unblock sem_wait(),
   *   4. The sem_wait() system call will awaken and return EINTR,
   *   5. The pending signal action will occur after the sem_wait() system
   *      call returns to user mode.
   *
   * Syscall handlers (and logic-in-general within the OS) should not use
   * signal handlers.
   */

  if ((masked == 1) || (stcb->flags & TCB_FLAG_SYSCALL) != 0)
#else
  /* Check if the signal is masked. In that case, it will be added to the
   * list of pending signals.
   */

  if (masked == 1)
#endif
    {
      /* Check if the task is waiting for this pending signal. If so, then
       * unblock it. This must be performed in a critical section because
       * signals can be queued from the interrupt level.
       */

      if (stcb->task_state == TSTATE_WAIT_SIG &&
          (masked == 0 ||
           (nxsig_ismember(&stcb->sigwaitmask, info->si_signo) == 1)))
        {
          if (stcb->sigunbinfo != NULL)
            {
              memcpy(stcb->sigunbinfo, info, sizeof(siginfo_t));
            }

          sigemptyset(&stcb->sigwaitmask);
          wd_cancel(&stcb->waitdog);

          /* Remove the task from waiting list */

          dq_rem((FAR dq_entry_t *)stcb, list_waitingforsignal());

          /* Add the task to ready-to-run task list and
           * perform the context switch if one is needed
           */

          if (nxsched_add_readytorun(stcb))
            {
              up_switch_context(this_task(), rtcb);
            }

#ifdef CONFIG_LIB_SYSCALL
          /* Must also add signal action if in system call */

          if (masked == 0)
            {
              sigpend = nxsig_add_pendingsignal(stcb, info, group_dispatch);
            }
#endif
        }

      /* Its not one we are waiting for... Add it to the list of pending
       * signals.
       */

      else
        {
          sigpend = nxsig_add_pendingsignal(stcb, info, group_dispatch);
        }
    }

  /************************* UNMASKED SIGNAL ACTIONS ************************/

  else
    {
      /* Queue any sigaction's requested by this task. */

      ret = nxsig_queue_action(stcb, sigact, info);

      /* Deliver of the signal must be performed in a critical section */

      /* Check if the task is waiting for an unmasked signal. If so, then
       * unblock it. This must be performed in a critical section because
       * signals can be queued from the interrupt level.
       */

      if (stcb->task_state == TSTATE_WAIT_SIG)
        {
          if (stcb->sigunbinfo != NULL)
            {
              memcpy(stcb->sigunbinfo, info, sizeof(siginfo_t));
            }

          sigemptyset(&stcb->sigwaitmask);
          wd_cancel(&stcb->waitdog);

          /* Remove the task from waiting list */

          dq_rem((FAR dq_entry_t *)stcb, list_waitingforsignal());

          /* Add the task to ready-to-run task list and
           * perform the context switch if one is needed
           */

          if (nxsched_add_readytorun(stcb))
            {
              up_switch_context(this_task(), rtcb);
            }
        }

      /* If the task neither was waiting for the signal nor had a signal
       * handler attached to the signal, then the default action is
       * simply to ignore the signal
       */
    }

  /************************* OTHER SIGNAL HANDLING **************************/

  /* Performed only if the signal is unmasked. These actions also must
   * happen within a system call.
   */

  if (masked == 0)
    {
      /* If the task is blocked waiting for a semaphore, then that task must
       * be unblocked when a signal is received.
       */

      if (stcb->task_state == TSTATE_WAIT_SEM)
        {
          nxsem_wait_irq(stcb, EINTR);
        }

#if !defined(CONFIG_DISABLE_MQUEUE) || !defined(CONFIG_DISABLE_MQUEUE_SYSV)
      /* If the task is blocked waiting on a message queue, then that task
       * must be unblocked when a signal is received.
       */

      else if (stcb->task_state == TSTATE_WAIT_MQNOTEMPTY ||
          stcb->task_state == TSTATE_WAIT_MQNOTFULL)
        {
          nxmq_wait_irq(stcb, EINTR);
        }
#endif

#ifdef CONFIG_SIG_SIGSTOP_ACTION
      /* If the task was stopped by SIGSTOP or SIGTSTP, then unblock the task
       * if SIGCONT is received.
       */

      else if (stcb->task_state == TSTATE_TASK_STOPPED &&
          info->si_signo == SIGCONT)
        {
#ifdef HAVE_GROUP_MEMBERS
          group_continue(stcb);
#else
          /* Remove the task from waiting list */

          dq_rem((FAR dq_entry_t *)stcb, list_stoppedtasks());

          /* Add the task to ready-to-run task list and
           * perform the context switch if one is needed
           */

          if (nxsched_add_readytorun(stcb))
            {
              up_switch_context(this_task(), rtcb);
            }
#endif
        }
#endif
    }

  leave_critical_section(flags);

  /* Dispatch kernel action, if needed, in case a pending signal was added */

  if (sigpend != NULL)
    {
      nxsig_dispatch_kernel_action(stcb, &sigpend->info);
    }

  /* In case nxsig_ismember failed due to an invalid signal number */

  if (masked < 0)
    {
      ret = -EINVAL;
    }

  return ret;
}

/****************************************************************************
 * Name: nxsig_dispatch
 *
 * Description:
 *   This is the front-end for nxsig_tcbdispatch that should be typically
 *   be used to dispatch a signal. If HAVE_GROUP_MEMBERS is defined,
 *   then function will follow the group signal delivery algorithms:
 *
 *   This front-end does the following things before calling
 *   nxsig_tcbdispatch.
 *
 *     With HAVE_GROUP_MEMBERS defined:
 *     - Get the TCB associated with the pid.
 *     - If the TCB was found, get the group from the TCB.
 *     - If the PID has already exited, lookup the group that that was
 *       started by this task.
 *     - Use the group to pick the TCB to receive the signal
 *     - Call nxsig_tcbdispatch with the TCB
 *
 *     With HAVE_GROUP_MEMBERS *not* defined
 *     - Get the TCB associated with the pid.
 *     - Call nxsig_tcbdispatch with the TCB
 *
 * Returned Value:
 *   Returns 0 (OK) on success or a negated errno value on failure.
 *
 ****************************************************************************/

int nxsig_dispatch(pid_t pid, FAR siginfo_t *info, bool thread)
{
#ifdef HAVE_GROUP_MEMBERS
  if (!thread)
    {
      /* Find the group by process PID and call group signal() to send the
       * signal to the correct group member.
       */

      FAR struct task_group_s *group = task_getgroup(pid);
      if (group != NULL)
        {
          return group_signal(group, info);
        }
    }
  else
#endif
    {
      /* Get the TCB associated with the thread TID */

      FAR struct tcb_s *stcb = nxsched_get_tcb(pid);
      if (stcb != NULL)
        {
          return nxsig_tcbdispatch(stcb, info, false);
        }
    }

  return -ESRCH;
}
