import { EnvelopeBusMessagePurpose, } from "../api";
export class EnvelopeBusMessageManager {
    get server() {
        return {
            receive: (m, apiImpl) => {
                console.debug(m);
                this.receive(m, apiImpl);
            },
        };
    }
    constructor(send, name = `${new Date().getMilliseconds()}`) {
        this.send = send;
        this.name = name;
        this.requestHandlers = new Map();
        this.localNotificationsSubscriptions = new Map();
        this.remoteNotificationsSubscriptions = [];
        this.localSharedValueSubscriptions = new Map();
        this.localSharedValuesStore = new Map();
        this.clientApi = {
            requests: cachedProxy(new Map(), {
                get: (target, name) => {
                    return (...args) => this.request(name, ...args);
                },
            }),
            notifications: cachedProxy(new Map(), {
                get: (target, name) => ({
                    subscribe: (callback) => this.subscribeToNotification(name, callback),
                    unsubscribe: (callback) => this.unsubscribeFromNotification(name, callback),
                    send: (...args) => this.notify(name, ...args),
                }),
            }),
            shared: cachedProxy(new Map(), {
                get: (target, name) => ({
                    set: (value) => this.setSharedValue(name, value),
                    subscribe: (callback) => this.subscribeToSharedValue(name, callback, { owned: false }),
                    unsubscribe: (callback) => this.unsubscribeFromSharedValue(name, callback),
                }),
            }),
        };
        this.shared = cachedProxy(new Map(), {
            get: (target, name) => ({
                set: (value) => this.setSharedValue(name, value),
                subscribe: (callback) => this.subscribeToSharedValue(name, callback, { owned: true }),
                unsubscribe: (callback) => this.unsubscribeFromSharedValue(name, callback),
            }),
        });
        this.requestIdCounter = 0;
    }
    setSharedValue(method, value) {
        var _a;
        this.localSharedValuesStore.set(method, value);
        (_a = this.localSharedValueSubscriptions.get(method)) === null || _a === void 0 ? void 0 : _a.forEach((callback) => callback(value));
        this.send({
            type: method,
            purpose: EnvelopeBusMessagePurpose.SHARED_VALUE_UPDATE,
            data: value,
        });
    }
    subscribeToSharedValue(method, callback, config) {
        var _a;
        const activeSubscriptions = (_a = this.localSharedValueSubscriptions.get(method)) !== null && _a !== void 0 ? _a : [];
        this.localSharedValueSubscriptions.set(method, [...activeSubscriptions, callback]);
        if (config.owned || this.localSharedValuesStore.get(method)) {
            callback(this.getCurrentStoredSharedValueOrDefault(method, this.currentApiImpl));
        }
        else {
            this.send({
                type: method,
                purpose: EnvelopeBusMessagePurpose.SHARED_VALUE_GET_DEFAULT,
                data: [],
            });
        }
        return callback;
    }
    unsubscribeFromSharedValue(name, callback) {
        const activeSubscriptions = this.localSharedValueSubscriptions.get(name);
        if (!activeSubscriptions) {
            return;
        }
        const index = activeSubscriptions.indexOf(callback);
        if (index < 0) {
            return;
        }
        activeSubscriptions.splice(index, 1);
    }
    getCurrentStoredSharedValueOrDefault(method, apiImpl) {
        var _a, _b;
        const m = method;
        return ((_a = this.localSharedValuesStore.get(m)) !== null && _a !== void 0 ? _a : this.localSharedValuesStore.set(m, (_b = apiImpl === null || apiImpl === void 0 ? void 0 : apiImpl[m]) === null || _b === void 0 ? void 0 : _b.apply(apiImpl).defaultValue).get(method));
    }
    subscribeToNotification(method, callback) {
        var _a;
        const activeSubscriptions = (_a = this.localNotificationsSubscriptions.get(method)) !== null && _a !== void 0 ? _a : [];
        this.localNotificationsSubscriptions.set(method, [...activeSubscriptions, callback]);
        this.send({
            type: method,
            purpose: EnvelopeBusMessagePurpose.NOTIFICATION_SUBSCRIPTION,
            data: [],
        });
        return callback;
    }
    unsubscribeFromNotification(method, callback) {
        const activeSubscriptions = this.localNotificationsSubscriptions.get(method);
        if (!activeSubscriptions) {
            return;
        }
        const index = activeSubscriptions.indexOf(callback);
        if (index < 0) {
            return;
        }
        activeSubscriptions.splice(index, 1);
        this.send({
            type: method,
            purpose: EnvelopeBusMessagePurpose.NOTIFICATION_UNSUBSCRIPTION,
            data: [],
        });
    }
    request(method, ...args) {
        const requestId = this.getNextRequestId();
        this.send({
            requestId: requestId,
            type: method,
            data: args,
            purpose: EnvelopeBusMessagePurpose.REQUEST,
        });
        return new Promise((resolve, reject) => {
            this.requestHandlers.set(requestId, { resolve, reject });
        });
    }
    notify(method, ...args) {
        this.send({
            type: method,
            data: args,
            purpose: EnvelopeBusMessagePurpose.NOTIFICATION,
        });
    }
    respond(request, data, error) {
        if (request.purpose !== EnvelopeBusMessagePurpose.REQUEST) {
            throw new Error("Cannot respond a message that is not a request");
        }
        if (!request.requestId) {
            throw new Error("Cannot respond a request without a requestId");
        }
        this.send({
            requestId: request.requestId,
            purpose: EnvelopeBusMessagePurpose.RESPONSE,
            type: request.type,
            data: data,
            error: error instanceof Error ? error.message : JSON.stringify(error),
        });
    }
    callback(response) {
        if (response.purpose !== EnvelopeBusMessagePurpose.RESPONSE) {
            throw new Error("Cannot invoke callback with a message that is not a response");
        }
        if (!response.requestId) {
            throw new Error("Cannot acknowledge a response without a requestId");
        }
        const callback = this.requestHandlers.get(response.requestId);
        if (!callback) {
            throw new Error("Callback not found for " + response);
        }
        this.requestHandlers.delete(response.requestId);
        if (!response.error) {
            callback.resolve(response.data);
        }
        else {
            callback.reject(new Error(response.error));
        }
    }
    receive(message, apiImpl) {
        var _a, _b;
        this.currentApiImpl = apiImpl;
        if (message.purpose === EnvelopeBusMessagePurpose.RESPONSE) {
            this.callback(message);
            return;
        }
        if (message.purpose === EnvelopeBusMessagePurpose.REQUEST) {
            const request = message;
            let response;
            try {
                const api = apiImpl[request.type];
                if (api !== undefined) {
                    response = api.apply(apiImpl, request.data);
                }
                else {
                    console.warn(`API '${String(request.type)}' was not found. Request will be ignored.`);
                    return;
                }
            }
            catch (err) {
                console.error(err);
                this.respond(request, undefined, err);
                return;
            }
            if (!(response instanceof Promise)) {
                throw new Error(`Cannot make a request to '${String(request.type)}' because it does not return a Promise`);
            }
            response
                .then((data) => {
                this.respond(request, data);
            })
                .catch((err) => {
                console.error(err);
                this.respond(request, undefined, err);
            });
            return;
        }
        if (message.purpose === EnvelopeBusMessagePurpose.NOTIFICATION) {
            const method = message.type;
            (_a = apiImpl[method]) === null || _a === void 0 ? void 0 : _a.apply(apiImpl, message.data);
            if (this.remoteNotificationsSubscriptions.indexOf(method) >= 0) {
                this.send({
                    type: method,
                    purpose: EnvelopeBusMessagePurpose.NOTIFICATION,
                    data: message.data,
                });
            }
            const localSubscriptionMethod = message.type;
            (_b = this.localNotificationsSubscriptions.get(localSubscriptionMethod)) === null || _b === void 0 ? void 0 : _b.forEach((callback) => {
                callback(...message.data);
            });
            return;
        }
        if (message.purpose === EnvelopeBusMessagePurpose.NOTIFICATION_SUBSCRIPTION) {
            const method = message.type;
            if (this.remoteNotificationsSubscriptions.indexOf(method) < 0) {
                this.remoteNotificationsSubscriptions.push(method);
            }
            return;
        }
        if (message.purpose === EnvelopeBusMessagePurpose.NOTIFICATION_UNSUBSCRIPTION) {
            const method = message.type;
            const index = this.remoteNotificationsSubscriptions.indexOf(method);
            if (index >= 0) {
                this.remoteNotificationsSubscriptions.splice(index, 1);
            }
            return;
        }
        if (message.purpose === EnvelopeBusMessagePurpose.SHARED_VALUE_GET_DEFAULT) {
            const method = message.type;
            this.send({
                type: method,
                purpose: EnvelopeBusMessagePurpose.SHARED_VALUE_UPDATE,
                data: this.getCurrentStoredSharedValueOrDefault(method, apiImpl),
            });
            return;
        }
        if (message.purpose === EnvelopeBusMessagePurpose.SHARED_VALUE_UPDATE) {
            const method = message.type;
            const subscriptions = this.localSharedValueSubscriptions.get(method);
            this.localSharedValuesStore.set(method, message.data);
            subscriptions === null || subscriptions === void 0 ? void 0 : subscriptions.forEach((callback) => callback(message.data));
            return;
        }
    }
    getNextRequestId() {
        return `${this.name}_${this.requestIdCounter++}`;
    }
}
function cachedProxy(cache, p) {
    return new Proxy({}, {
        set: (target, name, value) => {
            cache.set(name, value);
            return true;
        },
        get: (target, name) => {
            var _a, _b;
            return (_a = cache.get(name)) !== null && _a !== void 0 ? _a : cache.set(name, (_b = p.get) === null || _b === void 0 ? void 0 : _b.call(p, target, name)).get(name);
        },
    });
}
//# sourceMappingURL=EnvelopeBusMessageManager.js.map