import axios from "axios";
import Keycloak from "keycloak-js";
import { ANONYMOUS_USER, KeycloakUserContext } from "../contexts/KogitoAppContext";
export const isAuthEnabled = () => {
    return window["KOGITO_ENV_MODE"] !== "DEV";
};
export const isKeycloakHealthCheckDisabled = () => {
    return window["KOGITO_CONSOLES_KEYCLOAK_DISABLE_HEALTH_CHECK"];
};
export const getUpdateTokenValidity = () => {
    const updateTokenValidity = window["KOGITO_CONSOLES_KEYCLOAK_UPDATE_TOKEN_VALIDITY"];
    if (typeof updateTokenValidity !== "number") {
        return 30;
    }
    return updateTokenValidity;
};
let currentSecurityContext;
let keycloak;
export const getLoadedSecurityContext = () => {
    if (!currentSecurityContext) {
        if (isAuthEnabled()) {
            throw Error("Cannot load security context! Please reload screen and log in again.");
        }
        currentSecurityContext = getNonAuthUserContext();
    }
    return currentSecurityContext;
};
export const checkAuthServerHealth = () => {
    return new Promise((resolve, reject) => {
        fetch(window["KOGITO_CONSOLES_KEYCLOAK_HEALTH_CHECK_URL"])
            .then((response) => {
            if (response.status === 200) {
                resolve();
            }
        })
            .catch(() => {
            reject();
        });
    });
};
export const getKeycloakClient = () => {
    return new Keycloak({
        realm: window["KOGITO_CONSOLES_KEYCLOAK_REALM"],
        url: window["KOGITO_CONSOLES_KEYCLOAK_URL"],
        clientId: window["KOGITO_CONSOLES_KEYCLOAK_CLIENT_ID"],
    });
};
export const initializeKeycloak = (onloadSuccess) => {
    keycloak = getKeycloakClient();
    return keycloak
        .init({
        onLoad: "login-required",
    })
        .then((authenticated) => {
        if (authenticated) {
            currentSecurityContext = new KeycloakUserContext({
                userName: keycloak.tokenParsed["preferred_username"],
                roles: keycloak.tokenParsed["groups"],
                token: keycloak.token,
                tokenMinValidity: getUpdateTokenValidity(),
                logout: () => handleLogout(),
            });
            onloadSuccess();
        }
    });
};
export const loadSecurityContext = (onloadSuccess, onLoadFailure) => {
    if (isAuthEnabled()) {
        if (isKeycloakHealthCheckDisabled()) {
            return initializeKeycloak(onloadSuccess);
        }
        else {
            return checkAuthServerHealth()
                .then(() => {
                return initializeKeycloak(onloadSuccess);
            })
                .catch((_error) => {
                onLoadFailure();
            });
        }
    }
    else {
        currentSecurityContext = getNonAuthUserContext();
        onloadSuccess();
        return Promise.resolve();
    }
};
const getNonAuthUserContext = () => {
    return {
        getCurrentUser() {
            return ANONYMOUS_USER;
        },
    };
};
export const getToken = () => {
    if (isAuthEnabled()) {
        const ctx = getLoadedSecurityContext();
        return ctx.getToken();
    }
};
export const updateKeycloakToken = () => {
    if (!isAuthEnabled()) {
        return Promise.resolve();
    }
    return new Promise((resolve, reject) => {
        const ctx = getLoadedSecurityContext();
        keycloak
            .updateToken(getUpdateTokenValidity())
            .then(() => {
            ctx.setToken(keycloak.token);
            resolve();
        })
            .catch((error) => {
            reject(error);
        });
    });
};
export const setBearerToken = (config) => {
    if (!isAuthEnabled()) {
        return Promise.resolve(config);
    }
    return new Promise((resolve, reject) => {
        updateKeycloakToken()
            .then(() => {
            config.headers.Authorization = "Bearer " + keycloak.token;
            resolve(config);
        })
            .catch((error) => reject(error));
    });
};
export const appRenderWithAxiosInterceptorConfig = async (appRender, onLoadFailure) => {
    await loadSecurityContext(() => {
        appRender(getLoadedSecurityContext());
    }, onLoadFailure);
    if (isAuthEnabled()) {
        axios.interceptors.response.use((response) => response, (error) => {
            if (error.response.status === 401) {
                handleLogout();
            }
            return Promise.reject(error);
        });
        axios.interceptors.request.use((config) => setBearerToken(config), (error) => {
            Promise.reject(error);
        });
    }
};
export const handleLogout = () => {
    currentSecurityContext = undefined;
    if (keycloak) {
        keycloak.logout();
    }
};
//# sourceMappingURL=KeycloakClient.js.map