import { jsx as _jsx } from "react/jsx-runtime";
import * as Monaco from "@kie-tools-core/monaco-editor";
import * as React from "react";
import { useCallback, useEffect, useImperativeHandle, useMemo, useRef, useState } from "react";
import { feelDefaultConfig, feelDefaultSuggestions, feelTheme, feelTokensConfig, MONACO_FEEL_LANGUAGE, MONACO_FEEL_THEME, } from "./FeelConfigs";
import { FeelSyntacticSymbolNature } from "@kie-tools/dmn-feel-antlr4-parser";
import { SemanticTokensProvider } from "./semanticTokensProvider";
export const EXPRESSION_PROPERTIES_SEPARATOR = ".";
Monaco.languages.register({
    aliases: [MONACO_FEEL_LANGUAGE, "feel", "feel-dmn"],
    id: MONACO_FEEL_LANGUAGE,
    mimetypes: ["text/feel"],
});
Monaco.languages.setMonarchTokensProvider(MONACO_FEEL_LANGUAGE, feelTokensConfig());
Monaco.editor.defineTheme(MONACO_FEEL_THEME, feelTheme());
let __firstTimeInitializingMonacoToEnableColorizingCorrectly = true;
export const FeelInput = React.forwardRef(({ enabled, value, suggestionProvider, onBlur, onPreviewChanged, onKeyDown, onChange, options, feelIdentifiers, expressionId, }, forwardRef) => {
    const monacoContainer = useRef(null);
    const monacoRef = useRef();
    const [currentParsedExpression, setCurrentParsedExpression] = useState();
    const semanticTokensProvider = useMemo(() => new SemanticTokensProvider(feelIdentifiers, expressionId, setCurrentParsedExpression), [expressionId, feelIdentifiers]);
    const getLastValidSymbolAtPosition = useCallback((currentParsedExpression, position) => {
        let lastValidSymbol;
        for (let i = 0; i < currentParsedExpression.feelIdentifiedSymbols.length; i++) {
            const feelVariable = currentParsedExpression.feelIdentifiedSymbols[i];
            if (feelVariable.startIndex < position && position <= feelVariable.startIndex + feelVariable.length) {
                lastValidSymbol = feelVariable;
                const target = i - 1;
                if (target < currentParsedExpression.feelIdentifiedSymbols.length &&
                    0 <= target &&
                    lastValidSymbol.feelSymbolNature === FeelSyntacticSymbolNature.Unknown) {
                    lastValidSymbol = currentParsedExpression.feelIdentifiedSymbols[target];
                }
                break;
            }
        }
        return lastValidSymbol;
    }, []);
    const getSymbolAtPosition = useCallback((currentParsedExpression, position) => {
        for (const feelVariable of currentParsedExpression.feelIdentifiedSymbols) {
            if (feelVariable.startIndex < position && position <= feelVariable.startIndex + feelVariable.length) {
                return feelVariable;
            }
        }
        return undefined;
    }, []);
    const getDefaultCompletionItems = useCallback((suggestionProvider, model, position) => {
        if (suggestionProvider) {
            const items = suggestionProvider(model.getValue(), position.lineNumber, position.column - 1);
            if (items.length > 0) {
                return items;
            }
        }
        return feelDefaultSuggestions();
    }, []);
    useEffect(() => {
        if (!__firstTimeInitializingMonacoToEnableColorizingCorrectly) {
            return;
        }
        console.info("Registered FEEL language on Monaco Editor to enable correct 'colorize' call.");
        Monaco.editor
            .create(monacoContainer.current, {
            theme: MONACO_FEEL_THEME,
            language: MONACO_FEEL_LANGUAGE,
            "semanticHighlighting.enabled": true,
        })
            .dispose();
        __firstTimeInitializingMonacoToEnableColorizingCorrectly = false;
    }, []);
    const completionItemProvider = useCallback(() => {
        return {
            triggerCharacters: [EXPRESSION_PROPERTIES_SEPARATOR],
            provideCompletionItems: (model, position) => {
                const completionItems = getDefaultCompletionItems(suggestionProvider, model, position);
                const variablesSuggestions = new Array();
                if (currentParsedExpression) {
                    let pos = position.column - 1;
                    const expression = model.getValue();
                    const currentChar = expression.charAt(pos - 1);
                    if (currentChar === EXPRESSION_PROPERTIES_SEPARATOR || currentChar === " ") {
                        pos--;
                    }
                    const lastValidSymbol = getLastValidSymbolAtPosition(currentParsedExpression, pos);
                    if (lastValidSymbol &&
                        lastValidSymbol.feelSymbolNature !== FeelSyntacticSymbolNature.Unknown &&
                        expression.charAt(lastValidSymbol.startIndex + lastValidSymbol.length) === ".") {
                        for (const scopeSymbol of lastValidSymbol.scopeSymbols) {
                            variablesSuggestions.push({
                                kind: Monaco.languages.CompletionItemKind.Variable,
                                label: scopeSymbol.name,
                                insertText: scopeSymbol.name,
                                detail: scopeSymbol.type,
                                range: {
                                    startLineNumber: lastValidSymbol.startLine + 1,
                                    endLineNumber: lastValidSymbol.endLine + 1,
                                    startColumn: lastValidSymbol.startIndex + lastValidSymbol.length + 2,
                                    endColumn: lastValidSymbol.startIndex + lastValidSymbol.length + 2 + scopeSymbol.name.length,
                                },
                            });
                        }
                    }
                    else {
                        const currentSymbol = getSymbolAtPosition(currentParsedExpression, pos);
                        for (const scopeSymbol of currentParsedExpression.availableSymbols) {
                            if (currentSymbol && scopeSymbol.name.startsWith(currentSymbol.text)) {
                                variablesSuggestions.push({
                                    kind: Monaco.languages.CompletionItemKind.Variable,
                                    label: scopeSymbol.name,
                                    insertText: scopeSymbol.name,
                                    detail: scopeSymbol.type,
                                    sortText: "1",
                                    range: {
                                        startLineNumber: currentSymbol.startLine + 1,
                                        endLineNumber: currentSymbol.endLine + 1,
                                        startColumn: currentSymbol.startIndex + 1,
                                        endColumn: currentSymbol.startIndex + 1 + scopeSymbol.name.length,
                                    },
                                });
                            }
                            else {
                                variablesSuggestions.push({
                                    kind: Monaco.languages.CompletionItemKind.Variable,
                                    label: scopeSymbol.name,
                                    insertText: scopeSymbol.name,
                                    detail: scopeSymbol.type,
                                    sortText: "2",
                                });
                            }
                        }
                        variablesSuggestions.push(...completionItems);
                    }
                    return {
                        suggestions: variablesSuggestions,
                    };
                }
                return {
                    suggestions: completionItems,
                };
            },
        };
    }, [
        currentParsedExpression,
        getDefaultCompletionItems,
        getLastValidSymbolAtPosition,
        getSymbolAtPosition,
        suggestionProvider,
    ]);
    useEffect(() => {
        if (!enabled) {
            return;
        }
        const disposable = Monaco.languages.registerCompletionItemProvider(MONACO_FEEL_LANGUAGE, completionItemProvider());
        return () => {
            disposable.dispose();
        };
    }, [completionItemProvider, currentParsedExpression, enabled, suggestionProvider]);
    useEffect(() => {
        if (!enabled) {
            return;
        }
        const disposable = Monaco.languages.registerDocumentSemanticTokensProvider({ language: MONACO_FEEL_LANGUAGE }, semanticTokensProvider);
        return () => {
            disposable.dispose();
        };
    }, [enabled, expressionId, feelIdentifiers, semanticTokensProvider]);
    const config = useMemo(() => {
        return feelDefaultConfig(options);
    }, [options]);
    useEffect(() => {
        if (enabled && !monacoRef.current) {
            const element = monacoContainer.current;
            const editor = Monaco.editor.create(element, config);
            editor.onDidChangeModelContent((event) => {
                const value = editor.getValue();
                Monaco.editor.colorize(value, MONACO_FEEL_LANGUAGE, {}).then((colorizedValue) => {
                    onChange === null || onChange === void 0 ? void 0 : onChange(event, value, colorizedValue);
                });
            });
            editor.onDidBlurEditorWidget(() => {
                onBlur === null || onBlur === void 0 ? void 0 : onBlur(editor.getValue());
            });
            editor.onKeyDown((e) => {
                onKeyDown === null || onKeyDown === void 0 ? void 0 : onKeyDown(e, editor.getValue());
            });
            editor.focus();
            monacoRef.current = editor;
        }
    }, [config, enabled, onBlur, onChange, onKeyDown]);
    useEffect(() => {
        var _a, _b;
        if (enabled) {
            (_a = monacoRef.current) === null || _a === void 0 ? void 0 : _a.setValue(value !== null && value !== void 0 ? value : "");
            (_b = monacoRef.current) === null || _b === void 0 ? void 0 : _b.setPosition(calculatePosition(value !== null && value !== void 0 ? value : ""));
        }
    }, [value, enabled]);
    useEffect(() => {
        var _a, _b;
        if (!enabled && monacoRef.current) {
            onBlur === null || onBlur === void 0 ? void 0 : onBlur((_a = monacoRef.current) === null || _a === void 0 ? void 0 : _a.getValue());
            (_b = monacoRef.current) === null || _b === void 0 ? void 0 : _b.dispose();
            monacoRef.current = undefined;
        }
    }, [enabled, onBlur]);
    useEffect(() => {
        Monaco.editor.colorize(value !== null && value !== void 0 ? value : "", MONACO_FEEL_LANGUAGE, {}).then((colorizedValue) => {
            onPreviewChanged === null || onPreviewChanged === void 0 ? void 0 : onPreviewChanged(colorizedValue);
        });
    }, [onPreviewChanged, value]);
    useImperativeHandle(forwardRef, () => ({
        setMonacoValue: (newValue) => { var _a; return (_a = monacoRef.current) === null || _a === void 0 ? void 0 : _a.setValue(newValue); },
        getMonacoValue: () => { var _a; return (_a = monacoRef.current) === null || _a === void 0 ? void 0 : _a.getValue(); },
        insertNewLineToMonaco: () => { var _a; return (_a = monacoRef.current) === null || _a === void 0 ? void 0 : _a.trigger("keyboard", "type", { text: "\n" }); },
        isSuggestionWidgetOpen: () => { var _a, _b; return (_b = (_a = monacoRef.current) === null || _a === void 0 ? void 0 : _a._contentWidgets["editor.widget.suggestWidget"]) === null || _b === void 0 ? void 0 : _b.position; },
    }));
    return (_jsx("div", { className: "feel-input", children: _jsx("div", { ref: monacoContainer, role: "textbox", "data-testid": "monaco-container" }) }));
});
function calculatePosition(value) {
    const lines = value.split("\n");
    const lineNumber = lines.length;
    const column = lines[lineNumber - 1].length + 1;
    return { lineNumber, column };
}
//# sourceMappingURL=FeelInput.js.map