/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.spark.sql.execution

import org.apache.gluten.exception.GlutenException
import org.apache.gluten.execution.WholeStageTransformer
import org.apache.gluten.sql.shims.SparkShimLoader

import org.apache.spark.sql.catalyst.rules.Rule
import org.apache.spark.sql.execution.adaptive.{AdaptiveSparkPlanExec, AdaptiveSparkPlanHelper, BroadcastQueryStageExec, ShuffleQueryStageExec}
import org.apache.spark.sql.execution.exchange.{BroadcastExchangeLike, ReusedExchangeExec, ShuffleExchangeLike}

import java.util
import java.util.Collections.newSetFromMap
import java.util.concurrent.atomic.AtomicInteger

/**
 * Generate `transformStageId` for `WholeStageTransformerExec`. This rule updates the whole plan
 * tree with * incremental and unique transform stage id before the final execution.
 *
 * In Spark, the whole stage id is generated by incrementing a global counter. In Gluten, it's not
 * possible to use global counter for id generation, especially in the case of AQE.
 */
case class GenerateTransformStageId() extends Rule[SparkPlan] with AdaptiveSparkPlanHelper {
  private val transformStageCounter: AtomicInteger = new AtomicInteger(0)

  private val wholeStageTransformerCache =
    newSetFromMap[WholeStageTransformer](new util.IdentityHashMap())

  def apply(plan: SparkPlan): SparkPlan = {
    updateStageId(plan)
    plan
  }

  private def updateStageId(plan: SparkPlan): Unit = {
    plan match {
      case b: BroadcastQueryStageExec =>
        b.plan match {
          case b: BroadcastExchangeLike => updateStageId(b)
          case _: ReusedExchangeExec =>
          case _ =>
            throw new GlutenException(s"wrong plan for broadcast stage:\n ${plan.treeString}")
        }
      case s: ShuffleQueryStageExec =>
        s.plan match {
          case s: ShuffleExchangeLike => updateStageId(s)
          case _: ReusedExchangeExec =>
          case _ =>
            throw new GlutenException(s"wrong plan for shuffle stage:\n ${plan.treeString}")
        }
      case aqe: AdaptiveSparkPlanExec if SparkShimLoader.getSparkShims.isFinalAdaptivePlan(aqe) =>
        updateStageId(stripAQEPlan(aqe))
      case wst: WholeStageTransformer if !wholeStageTransformerCache.contains(wst) =>
        updateStageId(wst.child)
        wst.transformStageId = transformStageCounter.incrementAndGet()
        wholeStageTransformerCache.add(wst)
      case plan =>
        plan.subqueries.foreach(updateStageId)
        plan.children.foreach(updateStageId)
    }
  }
}
