/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.spark.sql.delta

import org.apache.gluten.backendsapi.BackendsApiManager
import org.apache.gluten.execution.datasource.GlutenFormatFactory

import org.apache.spark.internal.Logging
import org.apache.spark.sql.SparkSession
import org.apache.spark.sql.execution.datasources.{OutputWriter, OutputWriterFactory}
import org.apache.spark.sql.execution.datasources.parquet.{ParquetFileFormat, ParquetOptions}
import org.apache.spark.sql.sources.DataSourceRegister
import org.apache.spark.sql.types.StructType

import org.apache.hadoop.fs.FileStatus
import org.apache.hadoop.mapreduce.{Job, TaskAttemptContext}
import org.apache.parquet.hadoop.ParquetOutputFormat
import org.apache.parquet.hadoop.codec.CodecConfig
import org.apache.parquet.hadoop.util.ContextUtil
import org.slf4j.LoggerFactory

class GlutenParquetFileFormat
  extends ParquetFileFormat
  with DataSourceRegister
  with Logging
  with Serializable {
  import GlutenParquetFileFormat._

  private val logger = LoggerFactory.getLogger(classOf[GlutenParquetFileFormat])

  override def shortName(): String = "gluten-parquet"

  override def toString: String = "GlutenParquet"

  override def hashCode(): Int = getClass.hashCode()

  override def equals(other: Any): Boolean = other.isInstanceOf[GlutenParquetFileFormat]

  override def inferSchema(
      sparkSession: SparkSession,
      options: Map[String, String],
      files: Seq[FileStatus]): Option[StructType] = {
    super.inferSchema(sparkSession, options, files)
  }

  override def prepareWrite(
      sparkSession: SparkSession,
      job: Job,
      options: Map[String, String],
      dataSchema: StructType): OutputWriterFactory = {
    if (isNativeWritable(dataSchema)) {
      // Pass compression to job conf so that the file extension can be aware of it.
      val conf = ContextUtil.getConfiguration(job)
      val parquetOptions = new ParquetOptions(options, sparkSession.sessionState.conf)
      conf.set(ParquetOutputFormat.COMPRESSION, parquetOptions.compressionCodecClassName)
      val nativeConf =
        GlutenFormatFactory("parquet")
          .nativeConf(options, parquetOptions.compressionCodecClassName)

      return new OutputWriterFactory {
        override def getFileExtension(context: TaskAttemptContext): String = {
          CodecConfig.from(context).getCodec.getExtension + ".parquet"
        }

        override def newInstance(
            path: String,
            dataSchema: StructType,
            context: TaskAttemptContext): OutputWriter = {
          GlutenFormatFactory("parquet")
            .createOutputWriter(path, dataSchema, context, nativeConf)

        }
      }
    }
    logger.warn(
      s"Data schema is unsupported by Gluten Parquet writer: $dataSchema, " +
        s"falling back to the vanilla Spark Parquet writer")
    super.prepareWrite(sparkSession, job, options, dataSchema)
  }
}

object GlutenParquetFileFormat {
  def isNativeWritable(schema: StructType): Boolean = {
    BackendsApiManager.getSettings.supportNativeWrite(schema.fields)
  }
}
