/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.gluten.execution.iceberg

import org.apache.gluten.execution.{GlutenClickHouseWholeStageTransformerSuite, IcebergScanTransformer}

import org.apache.spark.SparkConf

import org.apache.commons.io.FileUtils

import java.io.File

class ClickHouseIcebergMOREqualityDeletionSuite extends GlutenClickHouseWholeStageTransformerSuite {

  protected val ICEBERG_GENERATED_PATH = "junit6640909127060857423"
  protected val ICEBERG_WAREHOUSE = s"/tmp/$ICEBERG_GENERATED_PATH"
  protected val equalityDeletedDataPath: String =
    "../../../../gluten-iceberg/src/test/resources"

  override protected def sparkConf: SparkConf = {
    super.sparkConf
      .set("spark.shuffle.manager", "org.apache.spark.shuffle.sort.ColumnarShuffleManager")
      .set("spark.sql.adaptive.enabled", "true")
      .set("spark.sql.shuffle.partitions", "2")
      .set("spark.memory.offHeap.size", "2g")
      .set("spark.unsafe.exceptionOnMemoryLeak", "true")
      .set("spark.sql.autoBroadcastJoinThreshold", "-1")
      .set(
        "spark.sql.extensions",
        "org.apache.iceberg.spark.extensions.IcebergSparkSessionExtensions")
      .set("spark.sql.catalog.local", "org.apache.iceberg.spark.SparkCatalog")
      .set("spark.sql.catalog.local.type", "hadoop")
      .set("spark.sql.catalog.local.warehouse", s"file://$ICEBERG_WAREHOUSE/default")
  }

  override def beforeAll(): Unit = {
    super.beforeAll()

    val icebergPathDir = new File(ICEBERG_WAREHOUSE)
    if (icebergPathDir.exists()) {
      FileUtils.forceDelete(icebergPathDir)
    }
    FileUtils.forceMkdir(icebergPathDir)
    val equalityDeletedData =
      new File(s"$resPath$equalityDeletedDataPath/$ICEBERG_GENERATED_PATH")
    FileUtils.copyDirectory(equalityDeletedData, icebergPathDir)
  }

  testWithSpecifiedSparkVersion("iceberg read mor table with equality deletion", "3.3", "3.5") {
    // The table 'test_upsert_query' was generated by Flink + Iceberg from the iceberg ut,
    // the root path must be the '/tmp/junit6640909127060857423/default'
    val testTableName = "local.db.test_upsert_query"
    runQueryAndCompare(s"""
                          |select * from $testTableName;
                          |""".stripMargin) {
      checkGlutenPlan[IcebergScanTransformer]
    }
  }
}
