/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.server.kv.rowmerger.aggregate.factory;

/* This file is based on source code of Apache Paimon Project (https://paimon.apache.org/), licensed by the Apache
 * Software Foundation (ASF) under the Apache License, Version 2.0. See the NOTICE file distributed with this work for
 * additional information regarding copyright ownership. */

import org.apache.fluss.metadata.AggFunction;
import org.apache.fluss.metadata.AggFunctionType;
import org.apache.fluss.metadata.AggFunctions;
import org.apache.fluss.server.kv.rowmerger.aggregate.functions.FieldListaggAgg;
import org.apache.fluss.types.DataType;
import org.apache.fluss.types.StringType;

import static org.apache.fluss.utils.Preconditions.checkArgument;

/**
 * Factory for {@link FieldListaggAgg} with identifier "string_agg".
 *
 * <p>This is an alias factory for listagg aggregation function. It creates the same aggregator as
 * {@link FieldListaggAggFactory} but uses "string_agg" as the identifier, providing an alternative
 * name for the same functionality.
 */
public class FieldStringAggFactory implements FieldAggregatorFactory {

    @Override
    public FieldListaggAgg create(DataType fieldType, AggFunction aggFunction) {
        checkArgument(
                fieldType instanceof StringType,
                "Data type for string_agg column must be 'StringType' but was '%s'.",
                fieldType);

        // Get delimiter from function parameters, default to comma
        String delimiter = aggFunction.getParameter(AggFunctions.PARAM_DELIMITER);
        if (delimiter == null) {
            delimiter = AggFunctions.DEFAULT_LISTAGG_DELIMITER;
        }

        return new FieldListaggAgg((StringType) fieldType, delimiter);
    }

    @Override
    public String identifier() {
        return AggFunctionType.STRING_AGG.toString();
    }
}
