/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.cosn;

import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.endpoint.EndpointBuilder;
import com.qcloud.cos.endpoint.SuffixEndpointBuilder;
import com.qcloud.cos.exception.CosClientException;
import com.qcloud.cos.exception.CosServiceException;
import com.qcloud.cos.http.HttpProtocol;
import com.qcloud.cos.model.AbortMultipartUploadRequest;
import com.qcloud.cos.model.COSObject;
import com.qcloud.cos.model.CompleteMultipartUploadRequest;
import com.qcloud.cos.model.CompleteMultipartUploadResult;
import com.qcloud.cos.model.CopyObjectRequest;
import com.qcloud.cos.model.DeleteObjectRequest;
import com.qcloud.cos.model.GetObjectMetadataRequest;
import com.qcloud.cos.model.GetObjectRequest;
import com.qcloud.cos.model.InitiateMultipartUploadRequest;
import com.qcloud.cos.model.InitiateMultipartUploadResult;
import com.qcloud.cos.model.ListObjectsRequest;
import com.qcloud.cos.model.ObjectListing;
import com.qcloud.cos.model.ObjectMetadata;
import com.qcloud.cos.model.PartETag;
import com.qcloud.cos.model.PutObjectRequest;
import com.qcloud.cos.model.PutObjectResult;
import com.qcloud.cos.model.UploadPartRequest;
import com.qcloud.cos.model.UploadPartResult;
import com.qcloud.cos.region.Region;
import com.qcloud.cos.utils.Base64;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.cosn.ByteBufferInputStream;
import org.apache.hadoop.fs.cosn.CosNUtils;
import org.apache.hadoop.fs.cosn.FileMetadata;
import org.apache.hadoop.fs.cosn.NativeFileSystemStore;
import org.apache.hadoop.fs.cosn.PartialListing;
import org.apache.hadoop.fs.cosn.auth.COSCredentialsProviderList;
import org.apache.hadoop.util.VersionInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class CosNativeFileSystemStore
implements NativeFileSystemStore {
    private COSClient cosClient;
    private String bucketName;
    private int maxRetryTimes;
    public static final Logger LOG = LoggerFactory.getLogger(CosNativeFileSystemStore.class);

    CosNativeFileSystemStore() {
    }

    private void initCOSClient(URI uri, Configuration conf) throws IOException {
        ClientConfig config;
        COSCredentialsProviderList credentialProviderList = CosNUtils.createCosCredentialsProviderSet(uri, conf);
        String region = conf.get("fs.cosn.bucket.region");
        String endpointSuffix = conf.get("fs.cosn.bucket.endpoint_suffix");
        if (null == region && null == endpointSuffix) {
            String exceptionMsg = String.format("config %s and %s at least one", "fs.cosn.bucket.region", "fs.cosn.bucket.endpoint_suffix");
            throw new IOException(exceptionMsg);
        }
        boolean useHttps = conf.getBoolean("fs.cosn.useHttps", false);
        if (null == region) {
            config = new ClientConfig(new Region(""));
            config.setEndpointBuilder((EndpointBuilder)new SuffixEndpointBuilder(endpointSuffix));
        } else {
            config = new ClientConfig(new Region(region));
        }
        if (useHttps) {
            config.setHttpProtocol(HttpProtocol.https);
        }
        config.setUserAgent(conf.get("fs.cosn.user.agent", "cos-hadoop-plugin-v5.3") + " For  Hadoop " + VersionInfo.getVersion());
        this.maxRetryTimes = conf.getInt("fs.cosn.maxRetries", 3);
        config.setMaxConnectionsCount(conf.getInt("fs.cosn.max.connection.num", 2048));
        this.cosClient = new COSClient(credentialProviderList.getCredentials(), config);
    }

    @Override
    public void initialize(URI uri, Configuration conf) throws IOException {
        try {
            this.initCOSClient(uri, conf);
            this.bucketName = uri.getHost();
        }
        catch (Exception e) {
            this.handleException(e, "");
        }
    }

    private void storeFileWithRetry(String key, InputStream inputStream, byte[] md5Hash, long length) throws IOException {
        try {
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentMD5(Base64.encodeAsString((byte[])md5Hash));
            objectMetadata.setContentLength(length);
            PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucketName, key, inputStream, objectMetadata);
            PutObjectResult putObjectResult = (PutObjectResult)this.callCOSClientWithRetry(putObjectRequest);
            LOG.debug("Store file successfully. COS key: [{}], ETag: [{}].", (Object)key, (Object)putObjectResult.getETag());
        }
        catch (Exception e) {
            String errMsg = String.format("Store file failed. COS key: [%s], exception: [%s]", key, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), key);
        }
    }

    @Override
    public void storeFile(String key, File file, byte[] md5Hash) throws IOException {
        LOG.info("Store file from local path: [{}]. file length: [{}] COS key: [{}]", new Object[]{file.getCanonicalPath(), file.length(), key});
        this.storeFileWithRetry(key, new BufferedInputStream(new FileInputStream(file)), md5Hash, file.length());
    }

    @Override
    public void storeFile(String key, InputStream inputStream, byte[] md5Hash, long contentLength) throws IOException {
        LOG.info("Store file from input stream. COS key: [{}], length: [{}].", (Object)key, (Object)contentLength);
        this.storeFileWithRetry(key, inputStream, md5Hash, contentLength);
    }

    @Override
    public void storeEmptyFile(String key) throws IOException {
        if (!key.endsWith("/")) {
            key = key + "/";
        }
        ObjectMetadata objectMetadata = new ObjectMetadata();
        objectMetadata.setContentLength(0L);
        ByteArrayInputStream input = new ByteArrayInputStream(new byte[0]);
        PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucketName, key, (InputStream)input, objectMetadata);
        try {
            PutObjectResult putObjectResult = (PutObjectResult)this.callCOSClientWithRetry(putObjectRequest);
            LOG.debug("Store empty file successfully. COS key: [{}], ETag: [{}].", (Object)key, (Object)putObjectResult.getETag());
        }
        catch (Exception e) {
            String errMsg = String.format("Store empty file failed. COS key: [%s], exception: [%s]", key, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public PartETag uploadPart(File file, String key, String uploadId, int partNum) throws IOException {
        try (FileInputStream inputStream = new FileInputStream(file);){
            PartETag partETag = this.uploadPart(inputStream, key, uploadId, partNum, file.length());
            return partETag;
        }
    }

    @Override
    public PartETag uploadPart(InputStream inputStream, String key, String uploadId, int partNum, long partSize) throws IOException {
        UploadPartRequest uploadPartRequest = new UploadPartRequest();
        uploadPartRequest.setBucketName(this.bucketName);
        uploadPartRequest.setUploadId(uploadId);
        uploadPartRequest.setInputStream(inputStream);
        uploadPartRequest.setPartNumber(partNum);
        uploadPartRequest.setPartSize(partSize);
        uploadPartRequest.setKey(key);
        try {
            UploadPartResult uploadPartResult = (UploadPartResult)this.callCOSClientWithRetry(uploadPartRequest);
            return uploadPartResult.getPartETag();
        }
        catch (Exception e) {
            String errMsg = String.format("Current thread: [%d], COS key: [%s], upload id: [%s], part num: [%d], exception: [%s]", Thread.currentThread().getId(), key, uploadId, partNum, e.toString());
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public void abortMultipartUpload(String key, String uploadId) {
        LOG.info("Abort the multipart upload. COS key: [{}], upload id: [{}].", (Object)key, (Object)uploadId);
        AbortMultipartUploadRequest abortMultipartUploadRequest = new AbortMultipartUploadRequest(this.bucketName, key, uploadId);
        this.cosClient.abortMultipartUpload(abortMultipartUploadRequest);
    }

    @Override
    public String getUploadId(String key) {
        if (null == key || key.length() == 0) {
            return "";
        }
        LOG.info("Initiate a multipart upload. bucket: [{}], COS key: [{}].", (Object)this.bucketName, (Object)key);
        InitiateMultipartUploadRequest initiateMultipartUploadRequest = new InitiateMultipartUploadRequest(this.bucketName, key);
        InitiateMultipartUploadResult initiateMultipartUploadResult = this.cosClient.initiateMultipartUpload(initiateMultipartUploadRequest);
        return initiateMultipartUploadResult.getUploadId();
    }

    @Override
    public CompleteMultipartUploadResult completeMultipartUpload(String key, String uploadId, List<PartETag> partETagList) {
        Collections.sort(partETagList, new Comparator<PartETag>(){

            @Override
            public int compare(PartETag o1, PartETag o2) {
                return o1.getPartNumber() - o2.getPartNumber();
            }
        });
        LOG.info("Complete the multipart upload. bucket: [{}], COS key: [{}], upload id: [{}].", new Object[]{this.bucketName, key, uploadId});
        CompleteMultipartUploadRequest completeMultipartUploadRequest = new CompleteMultipartUploadRequest(this.bucketName, key, uploadId, partETagList);
        return this.cosClient.completeMultipartUpload(completeMultipartUploadRequest);
    }

    private FileMetadata queryObjectMetadata(String key) throws IOException {
        GetObjectMetadataRequest getObjectMetadataRequest = new GetObjectMetadataRequest(this.bucketName, key);
        try {
            ObjectMetadata objectMetadata = (ObjectMetadata)this.callCOSClientWithRetry(getObjectMetadataRequest);
            long mtime = 0L;
            if (objectMetadata.getLastModified() != null) {
                mtime = objectMetadata.getLastModified().getTime();
            }
            long fileSize = objectMetadata.getContentLength();
            FileMetadata fileMetadata = new FileMetadata(key, fileSize, mtime, !key.endsWith("/"));
            LOG.debug("Retrieve file metadata. COS key: [{}], ETag: [{}], length: [{}].", new Object[]{key, objectMetadata.getETag(), objectMetadata.getContentLength()});
            return fileMetadata;
        }
        catch (CosServiceException e) {
            if (e.getStatusCode() != 404) {
                String errorMsg = String.format("Retrieve file metadata file failed. COS key: [%s], CosServiceException: [%s].", key, e.toString());
                LOG.error(errorMsg);
                this.handleException(new Exception(errorMsg), key);
            }
            return null;
        }
    }

    @Override
    public FileMetadata retrieveMetadata(String key) throws IOException {
        FileMetadata fileMetadata;
        if (key.endsWith("/")) {
            key = key.substring(0, key.length() - 1);
        }
        if (!key.isEmpty() && (fileMetadata = this.queryObjectMetadata(key)) != null) {
            return fileMetadata;
        }
        key = key + "/";
        return this.queryObjectMetadata(key);
    }

    @Override
    public InputStream retrieve(String key) throws IOException {
        LOG.debug("Retrieve object key: [{}].", (Object)key);
        GetObjectRequest getObjectRequest = new GetObjectRequest(this.bucketName, key);
        try {
            COSObject cosObject = (COSObject)this.callCOSClientWithRetry(getObjectRequest);
            return cosObject.getObjectContent();
        }
        catch (Exception e) {
            String errMsg = String.format("Retrieving key: [%s] occurs an exception: [%s].", key, e.toString());
            LOG.error("Retrieving COS key: [{}] occurs an exception: [{}].", (Object)key, (Object)e);
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public InputStream retrieve(String key, long byteRangeStart) throws IOException {
        try {
            LOG.debug("Retrieve COS key:[{}]. range start:[{}].", (Object)key, (Object)byteRangeStart);
            long fileSize = this.getFileLength(key);
            long byteRangeEnd = fileSize - 1L;
            GetObjectRequest getObjectRequest = new GetObjectRequest(this.bucketName, key);
            if (byteRangeEnd >= byteRangeStart) {
                getObjectRequest.setRange(byteRangeStart, fileSize - 1L);
            }
            COSObject cosObject = (COSObject)this.callCOSClientWithRetry(getObjectRequest);
            return cosObject.getObjectContent();
        }
        catch (Exception e) {
            String errMsg = String.format("Retrieving COS key: [%s] occurs an exception. byte range start: [%s], exception: [%s].", key, byteRangeStart, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public InputStream retrieveBlock(String key, long byteRangeStart, long byteRangeEnd) throws IOException {
        try {
            GetObjectRequest request = new GetObjectRequest(this.bucketName, key);
            request.setRange(byteRangeStart, byteRangeEnd);
            COSObject cosObject = (COSObject)this.callCOSClientWithRetry(request);
            return cosObject.getObjectContent();
        }
        catch (CosServiceException e) {
            String errMsg = String.format("Retrieving key [%s] with byteRangeStart [%d] occurs an CosServiceException: [%s].", key, byteRangeStart, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), key);
            return null;
        }
        catch (CosClientException e) {
            String errMsg = String.format("Retrieving key [%s] with byteRangeStart [%d] occurs an exception: [%s].", key, byteRangeStart, e.toString());
            LOG.error("Retrieving COS key: [{}] with byteRangeStart: [{}] occurs an exception: [{}].", new Object[]{key, byteRangeStart, e});
            this.handleException(new Exception(errMsg), key);
            return null;
        }
    }

    @Override
    public PartialListing list(String prefix, int maxListingLength) throws IOException {
        return this.list(prefix, maxListingLength, null, false);
    }

    @Override
    public PartialListing list(String prefix, int maxListingLength, String priorLastKey, boolean recurse) throws IOException {
        return this.list(prefix, recurse ? null : "/", maxListingLength, priorLastKey);
    }

    private PartialListing list(String prefix, String delimiter, int maxListingLength, String priorLastKey) throws IOException {
        LOG.debug("List objects. prefix: [{}], delimiter: [{}], maxListLength: [{}], priorLastKey: [{}].", new Object[]{prefix, delimiter, maxListingLength, priorLastKey});
        if (!prefix.startsWith("/")) {
            prefix = prefix + "/";
        }
        ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
        listObjectsRequest.setBucketName(this.bucketName);
        listObjectsRequest.setPrefix(prefix);
        listObjectsRequest.setDelimiter(delimiter);
        listObjectsRequest.setMarker(priorLastKey);
        listObjectsRequest.setMaxKeys(Integer.valueOf(maxListingLength));
        ObjectListing objectListing = null;
        try {
            objectListing = (ObjectListing)this.callCOSClientWithRetry(listObjectsRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("prefix: [%s], delimiter: [%s], maxListingLength: [%d], priorLastKey: [%s]. List objects occur an exception: [%s].", prefix, delimiter == null ? "" : delimiter, maxListingLength, priorLastKey, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), prefix);
        }
        ArrayList<FileMetadata> fileMetadataArray = new ArrayList<FileMetadata>();
        ArrayList<FileMetadata> commonPrefixArray = new ArrayList<FileMetadata>();
        if (null == objectListing) {
            String errMsg = String.format("List the prefix: [%s] failed. delimiter: [%s], max listing length: [%s], prior last key: [%s]", prefix, delimiter, maxListingLength, priorLastKey);
            this.handleException(new Exception(errMsg), prefix);
        }
        List summaries = objectListing.getObjectSummaries();
        for (Object cosObjectSummary : summaries) {
            String filePath = cosObjectSummary.getKey();
            if (!filePath.startsWith("/")) {
                filePath = "/" + filePath;
            }
            if (filePath.equals(prefix)) continue;
            long mtime = 0L;
            if (cosObjectSummary.getLastModified() != null) {
                mtime = cosObjectSummary.getLastModified().getTime();
            }
            long fileLen = cosObjectSummary.getSize();
            fileMetadataArray.add(new FileMetadata(filePath, fileLen, mtime, true));
        }
        List commonPrefixes = objectListing.getCommonPrefixes();
        for (String commonPrefix : commonPrefixes) {
            if (!commonPrefix.startsWith("/")) {
                commonPrefix = "/" + commonPrefix;
            }
            commonPrefixArray.add(new FileMetadata(commonPrefix, 0L, 0L, false));
        }
        FileMetadata[] fileMetadata = new FileMetadata[fileMetadataArray.size()];
        for (int i = 0; i < fileMetadataArray.size(); ++i) {
            fileMetadata[i] = (FileMetadata)fileMetadataArray.get(i);
        }
        FileMetadata[] commonPrefixMetaData = new FileMetadata[commonPrefixArray.size()];
        for (int i = 0; i < commonPrefixArray.size(); ++i) {
            commonPrefixMetaData[i] = (FileMetadata)commonPrefixArray.get(i);
        }
        if (!objectListing.isTruncated()) {
            return new PartialListing(null, fileMetadata, commonPrefixMetaData);
        }
        return new PartialListing(objectListing.getNextMarker(), fileMetadata, commonPrefixMetaData);
    }

    @Override
    public void delete(String key) throws IOException {
        LOG.debug("Delete object key: [{}] from bucket: {}.", (Object)key, (Object)this.bucketName);
        try {
            DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.bucketName, key);
            this.callCOSClientWithRetry(deleteObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Delete key: [%s] occurs an exception: [%s].", key, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), key);
        }
    }

    public void rename(String srcKey, String dstKey) throws IOException {
        LOG.debug("Rename source key: [{}] to dest key: [{}].", (Object)srcKey, (Object)dstKey);
        try {
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(this.bucketName, srcKey, this.bucketName, dstKey);
            this.callCOSClientWithRetry(copyObjectRequest);
            DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.bucketName, srcKey);
            this.callCOSClientWithRetry(deleteObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Rename object unsuccessfully. source cos key: [%s], dest COS key: [%s], exception: [%s]", srcKey, dstKey, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), srcKey);
        }
    }

    @Override
    public void copy(String srcKey, String dstKey) throws IOException {
        LOG.debug("Copy source key: [{}] to dest key: [{}].", (Object)srcKey, (Object)dstKey);
        try {
            CopyObjectRequest copyObjectRequest = new CopyObjectRequest(this.bucketName, srcKey, this.bucketName, dstKey);
            this.callCOSClientWithRetry(copyObjectRequest);
        }
        catch (Exception e) {
            String errMsg = String.format("Copy object unsuccessfully. source COS key: %s, dest COS key: %s, exception: %s", srcKey, dstKey, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), srcKey);
        }
    }

    @Override
    public void purge(String prefix) throws IOException {
        throw new IOException("purge not supported");
    }

    @Override
    public void dump() throws IOException {
        throw new IOException("dump not supported");
    }

    private void handleException(Exception e, String key) throws IOException {
        String cosPath = "cosn://" + this.bucketName + key;
        String exceptInfo = String.format("%s : %s", cosPath, e.toString());
        throw new IOException(exceptInfo);
    }

    @Override
    public long getFileLength(String key) throws IOException {
        LOG.debug("Get file length. COS key: {}", (Object)key);
        GetObjectMetadataRequest getObjectMetadataRequest = new GetObjectMetadataRequest(this.bucketName, key);
        try {
            ObjectMetadata objectMetadata = (ObjectMetadata)this.callCOSClientWithRetry(getObjectMetadataRequest);
            return objectMetadata.getContentLength();
        }
        catch (Exception e) {
            String errMsg = String.format("Getting file length occurs an exception.COS key: %s, exception: %s", key, e.toString());
            LOG.error(errMsg);
            this.handleException(new Exception(errMsg), key);
            return 0L;
        }
    }

    private <X> Object callCOSClientWithRetry(X request) throws CosServiceException, IOException {
        String sdkMethod = "";
        int retryIndex = 1;
        while (true) {
            String errMsg;
            try {
                if (request instanceof PutObjectRequest) {
                    sdkMethod = "putObject";
                    return this.cosClient.putObject((PutObjectRequest)request);
                }
                if (request instanceof UploadPartRequest) {
                    sdkMethod = "uploadPart";
                    if (((UploadPartRequest)request).getInputStream() instanceof ByteBufferInputStream) {
                        ((UploadPartRequest)request).getInputStream().mark((int)((UploadPartRequest)request).getPartSize());
                    }
                    return this.cosClient.uploadPart((UploadPartRequest)request);
                }
                if (request instanceof GetObjectMetadataRequest) {
                    sdkMethod = "queryObjectMeta";
                    return this.cosClient.getObjectMetadata((GetObjectMetadataRequest)request);
                }
                if (request instanceof DeleteObjectRequest) {
                    sdkMethod = "deleteObject";
                    this.cosClient.deleteObject((DeleteObjectRequest)request);
                    return new Object();
                }
                if (request instanceof CopyObjectRequest) {
                    sdkMethod = "copyFile";
                    return this.cosClient.copyObject((CopyObjectRequest)request);
                }
                if (request instanceof GetObjectRequest) {
                    sdkMethod = "getObject";
                    return this.cosClient.getObject((GetObjectRequest)request);
                }
                if (request instanceof ListObjectsRequest) {
                    sdkMethod = "listObjects";
                    return this.cosClient.listObjects((ListObjectsRequest)request);
                }
                throw new IOException("no such method");
            }
            catch (CosServiceException cse) {
                errMsg = String.format("Call cos sdk failed, retryIndex: [%d / %d], call method: %s, exception: %s", retryIndex, this.maxRetryTimes, sdkMethod, cse.toString());
                int statusCode = cse.getStatusCode();
                if (statusCode / 100 == 5) {
                    if (retryIndex <= this.maxRetryTimes) {
                        LOG.info(errMsg);
                        long sleepLeast = (long)retryIndex * 300L;
                        long sleepBound = (long)retryIndex * 500L;
                        try {
                            if (request instanceof UploadPartRequest && ((UploadPartRequest)request).getInputStream() instanceof ByteBufferInputStream) {
                                ((UploadPartRequest)request).getInputStream().reset();
                            }
                            Thread.sleep(ThreadLocalRandom.current().nextLong(sleepLeast, sleepBound));
                            ++retryIndex;
                        }
                        catch (InterruptedException e) {
                            throw new IOException(e.toString());
                        }
                        continue;
                    }
                    LOG.error(errMsg);
                    throw new IOException(errMsg);
                }
                throw cse;
            }
            catch (Exception e) {
                errMsg = String.format("Call cos sdk failed, call method: %s, exception: %s", sdkMethod, e.toString());
                LOG.error(errMsg);
                throw new IOException(errMsg);
            }
            break;
        }
    }

    @Override
    public void close() {
        if (null != this.cosClient) {
            this.cosClient.shutdown();
        }
    }
}

