/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/*
 *
 * putil.c
 *
 */


#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#define MAXBUFSIZE 1024

#include "putil.h"

int
Lock (FILE * f)
{
#ifdef LOCK
  struct flock lock;

  lock.l_type = F_WRLCK;
  lock.l_whence = SEEK_SET;
  lock.l_start = 0;
  lock.l_len = 0;
  return fcntl (fileno (f), F_SETLKW, &lock);
#else
  return 0;
#endif
}


int
UnLock (FILE * f)
{
#ifdef LOCK
  struct flock lock;

  lock.l_type = F_UNLCK;
  lock.l_whence = SEEK_SET;
  lock.l_start = 0;
  lock.l_len = 0;
  return fcntl (fileno (f), F_SETLKW, &lock);
#else
  return 0;
#endif
}


/*
 * 
 * list.c
 *
 */

/* list.c
**/

#include "list.h"


void *
ListInsert (Listll * list, Listll * member, Listll * at)
{
  Listll *this;
  Listll *prev = NULL;

  Listll *mlast;

  for (this = list; this != at; this = this->next)
    prev = this;

  mlast = member;
  if (mlast)
    while (mlast->next != NULL)
      mlast = mlast->next;

  if (prev)
    {
      prev->next = member;
      mlast->next = this;
    }
  else
    {
      if (member)
	{
	  mlast->next = list;
	  list = member;
	};
    }

  return (void *) list;
}


void *
ListDelete (Listll * list, Listll * member)
{
  Listll *this;
  Listll *prev = NULL;

  if (list == NULL)
    return NULL;

  for (this = list; this != member; this = this->next)
    prev = this;

  if (prev)
    prev->next = this->next;
  else
    list = this->next;

  member->next = NULL;
  return (void *) list;
}

/*
 *
 * pargs.c
 *
 */

#include "pargs.h"

ArgStream
OpenArgv (int nth, char separator, char *argv[], int argc)
{
  ArgStream a;

  a = (ArgStream) malloc (sizeof (struct _ArgStream));

  a->ungotten = 0;

  a->here = argv[nth];
  a->this = nth;
  a->argv = argv;
  a->argc = argc;
  a->sep = separator;

  return a;
}


void
CloseArgv (ArgStream a)
{
  free (a);
  a = NULL;
}

char *
CXCNewString (const char *str)
{
  char *s;
  if (str == NULL)
    return (NULL);
  s = (char *) malloc ((unsigned) strlen (str) + 1);
  strcpy (s, str);
  return (s);
}

int
PSkipWhite (FILE * file)
{
  char ch = getc (stdin);

  while (ch == ' '
	 || ch == '\n' || ch == '\t' || ch == ':' || ch == '\0' || ch == EOF)
    {
      if (ch == '\0')
	return -1;
      if (ch == EOF)
	return -1;
      ch = getc (stdin);
    }

  ungetc (ch, stdin);

  return 1;
}


int
PReadTok (FILE * file, char *token, int max)
{
  int i;

  for (i = 0; i < max; i++)
    {

      token[i] = getc (file);

      if (!i && token[i] == '\0')
	return -1;

      if (token[i] == ' '
	  || token[i] == '\n'
	  || token[i] == '\t' || token[i] == ':' || token[i] == '\0')
	break;
    }

  token[i] = '\0';

  return 1;
}

/*
 *
 * Word -- cheap spaced parser
 *
 */
int
Word (char *lbuf, char *tbuf, int *lptr)
{
  int ip;
  int i;
  char quotes;

  /* if no string was specified, just return */
  if (lbuf == NULL)
    return (0);

  /* just a more convenient pointer ... */
  ip = *lptr;

  /* if we are at the end of string, just return */
  if (lbuf[ip] == '\0')
    return (0);

  /* skip over white space */
  while (isspace (lbuf[ip]))
    {
      if (lbuf[ip] == '\0')
	return (0);
      else
	ip++;
    }

  /* check for an explicit quote */
  quotes = '\0';
  if (lbuf[ip] == '"')
    {
      quotes = '"';
    }
  if (lbuf[ip] == '\'')
    {
      quotes = '\'';
    }

  /* grab next token */
  if (quotes != '\0')
    {
      /* bump past quotes */
      ip++;
      /* grab up to next quotes -- but skip escaped quotes */
      for (i = 0; lbuf[ip] != 0; i++, ip++)
	{
	  if ((lbuf[ip] == quotes) && (lbuf[ip - 1] != '\\'))
	    break;
	  else
	    tbuf[i] = lbuf[ip];
	}
    }
  else
    {
      /* grab up to next whitespace */
      for (i = 0; lbuf[ip] && !isspace (lbuf[ip]); i++, ip++)
	tbuf[i] = lbuf[ip];
    }
  /* bump past delimiter (but not null terminator) */
  if (lbuf[ip])
    ip++;

  /* null terminate */
  tbuf[i] = '\0';

  /* return number of chars in parsed string (might be 0) */
  *lptr = ip;
  return (1);
}


/*
** temp procedure to replace non-ansii/posix strdup 
*/
char *
str_duplicate (const char *in)
{
  char *val = NULL;

  if (in)
    {
      int length = (int) strlen (in) + 1;

      if ((val = malloc (length * sizeof (char))) != NULL)
	strcpy (val, in);
    }

  return val;
}
