use util:serialize;

/**
 * This class is the base class for all requests (sent from the client to the server). We rely on
 * the built-in serialization for sending and receiving them.
 */
class Request : serializable {
	// Called by the server to execute the request. Returns a suitable message, or "null" if the
	// client should be disconnected.
	Response? execute(ServerConn server, Database db) {
		error("Unknown message");
	}

	// Helper to create an error message.
	protected Response error(Str message) {
		ErrorResponse(message);
	}
}


/**
 * Authentication request. Expected to be the first message sent by a client.
 *
 * Returns either an AuthReply, AuthLoginReply, or an error.
 */
class AuthRequest : extends Request, serializable {
	// The client key.
	Str key;

	init(Str key) {
		init { key = key; }
	}

	// NOTE: This is handled as a special case, so we don't override "execute" here.
}

/**
 * Log out from this client. No additional data needed.
 */
class LogoutRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		db.logout(server.clientId);
		LogoutResponse();
	}
}

/**
 * Change nickname.
 */
class ChangeNicknameRequest : extends Request, serializable {
	Str newName;

	init(Str newName) {
		init { newName = newName; }
	}

	Response? execute(ServerConn server, Database db) : override {
		db.changeName(server.userId, newName);
		AuthResponse(newName);
	}
}

/**
 * Ask for the leaderboard.
 */
class LeaderboardRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		ScorePair[] scores;
		for (user, score in db.allScores()) {
			scores << ScorePair(user, score);
		}

		scores.sort();

		// Pick the top 10, and the current user.
		Bool currentFound = false;
		Score[] result;
		for (i, score in scores) {
			Bool add = (score.user == server.userId);
			currentFound |= add;
			add |= i < 10;

			if (currentFound & !add)
				break;

			if (add) {
				if (user = db.findUserName(score.user))
					result << Score(user, score.score, i.int + 1);
			}
		}

		LeaderboardResponse(result);
	}
}

// Pair for use when working with the scoreboard.
private value ScorePair {
	Int user;
	Int score;

	init(Int user, Int score) {
		init { user = user; score = score; }
	}

	Bool <(ScorePair o) {
		score > o.score;
	}
}

/**
 * Ask for challenges.
 */
class ChallengesRequest : extends Request, serializable {
	// Only solved problems?
	Bool solved;

	// Only improved problems? Ignored if "solved" is true.
	Bool improved;

	init(Bool wantSolved, Bool wantImproved) {
		init { solved = wantSolved; improved = wantImproved; }
	}

	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.userChallenges(server.userId, solved, improved));
	}
}

/**
 * Ask for a user's own problems.
 */
class OwnProblemsRequest : extends Request, serializable {
	Response? execute(ServerConn server, Database db) : override {
		ProblemResponse(db.ownProblems(server.userId));
	}
}

/**
 * Ask for more detailed stats of a problem.
 */
class StatsRequest : extends Request, serializable {
	// Problem we are interested in.
	Int problem;

	// Shall solutions be included in the response?
	Bool solutions;

	init(Int problem, Bool solutions) {
		init {
			problem = problem;
			solutions = solutions;
		}
	}

	init(Problem problem, Bool solutions) {
		init {
			problem = problem.id;
			solutions = solutions;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		Solution[] sol;
		if (solutions)
			sol = db.solutionsTo(problem);
		Problem? parent = db.parentTo(problem);
		StatsResponse(parent, sol);
	}
}

/**
 * Ask for details about a problem (including source).
 */
class DetailsRequest : extends Request, serializable {
	Int problemId;

	init(Int id) {
		init { problemId = id; }
	}

	Response? execute(ServerConn server, Database db) : override {
		db.problemDetails(problemId, server.userId);
	}
}

/**
 * Submit a solution to a problem.
 */
class PostSolutionRequest : extends Request, serializable {
	// Solution to this problem.
	Int problemId;

	// Type of solution.
	Str type;

	// Solution data (if any).
	Str? solution;

	init(Int problemId, Str type, Str? solution) {
		init {
			problemId = problemId;
			type = type;
			solution = solution;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		unless (problemAuthor = db.authorTo(problemId))
			throw ServerError("No problem with the id ${problemId}!");

		if (problemAuthor == server.userId)
			throw ServerError("Can not solve your own problem!");

		var id = db.newSolution(problemId, server.userId, type, solution);
		PostSolutionResponse(id);
	}
}

/**
 * Publish a new problem.
 */
class NewProblemRequest : extends Request, serializable {
	Str title;
	Str code;
	Str language;

	init(Str title, Str code, Str language) {
		init {
			title = title;
			code = code;
			language = language;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		var id = db.createProblem(server.userId, title, code, language);
		NewProblemResponse(id);
	}
}

/**
 * Publish an improvement to an existing solution..
 */
class NewImprovementRequest : extends NewProblemRequest, serializable {
	Int solutionId;

	init(Str title, Str code, Str language, Int solutionId) {
		init(title, code, language) {
			solutionId = solutionId;
		}
	}

	Response? execute(ServerConn server, Database db) : override {
		var id = db.createImprovement(server.userId, title, code, language, solutionId);
		NewProblemResponse(id);
	}
}
