# Sorune Libraries
# Copyright (C) 2004-2005 Darren Smith
# All Rights Reserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# e-mail: sorune2004@yahoo.com

use strict;
use FileHandle;
use Errno qw(EAGAIN);
use File::Path;
use File::Basename;
use File::Spec;
use Cwd;

# Globals
my @currentCmd = ();
my @currentItem = ();
my %cfg = ();
my %info = ();
my %musicDb = ();
my %playlistDb = ();
my %recordDb = ();
my %existFiles = ();
my %addedFiles = ();
my %deletedFiles = ();
my ($neurosHome,$mainSerial,$fwVersion,$msg,$soruneDb,$neurosDrive);
my ($locBar,$mainBar,$bkpkBar,$fwBar,$capBar,$freeBar,$progDg,$progDgButton,
    $playlistName,$ucBalloon,$ucBalloonMsg,$fsBalloon,$fsBalloonMsg,
    $exportFilename,$dirText,$bcText);
my ($addButton,$delButton,$syncButton,$playButton,$playlistButton,
    $rescanButton,$rebuildButton,$resetButton,$clearButton,$exportButton,
    $dupeButton,$backupButton,$restoreButton,$optionsButton,$aboutButton,
    $exitButton);
my ($artistD,$albumD,$genreD,$titleD,$yearD,$tracknumD,$formatD,$lengthD,
    $sizeD,$random) = (1,0,0,1,0,0,0,0,0,0);
my ($albumdiffD,$artistdiffD,$genrediffD,$titlediffD) = (0,0,0,0);
my ($font,$fontBold,$fontSm);
my ($info0,$info1,$info2);
my ($info0Frame,$info1Frame,$info2Frame);
my $syncNeeded = 0;
my %themes = (
    'Black' => {
        'mainbg' => 'black', 'mainfg' => 'orange',
        'filebg' => 'black', 'filefg' => 'orange',
        'filebgsel' => 'orange', 'filefgsel' => 'black',
        'audiobg' => 'black', 'audiofg' => 'orange',
        'audiobgsel' => 'grey45', 'audiofgsel' => 'white',
        'audiobgdel' => 'darkred', 'audiobgadd' => 'darkgreen',
        'statusbarbg' => 'black', 'statusbarfg' => 'orange',
        'progressbg' => 'grey45', 'progressfg' => 'orange',
        'menubg' => 'black', 'menufg' => 'orange',
        'buttonbg' => 'black', 'buttonfg' => 'orange',
        'scrollbg' => 'black', 'scrollfg' => 'orange',
        'toolbarbg' => 'orange',
    },
    'Cordial' => {
        'mainbg' => 'grey93', 'mainfg' => 'black',
        'filebg' => 'darkred', 'filefg' => 'white',
        'filebgsel' => 'grey55', 'filefgsel' => 'black',
        'audiobg' => 'black', 'audiofg' => 'white',
        'audiobgsel' => 'grey55', 'audiofgsel' => 'black',
        'audiobgdel' => 'darkred', 'audiobgadd' => 'darkgreen',
        'statusbarbg' => 'grey70', 'statusbarfg' => 'black',
        'progressbg' => 'grey45', 'progressfg' => 'darkblue',
        'menubg' => 'grey85', 'menufg' => 'black',
        'buttonbg' => 'grey85', 'buttonfg' => 'black',
        'scrollbg' => 'grey85', 'scrollfg' => 'grey85',
        'toolbarbg' => 'grey85',
    },
    'Earth' => {
        'mainbg' => 'grey93', 'mainfg' => 'black',
        'filebg' => 'darkgreen', 'filefg' => 'white',
        'filebgsel' => 'grey55', 'filefgsel' => 'black',
        'audiobg' => 'darkblue', 'audiofg' => 'white',
        'audiobgsel' => 'grey55', 'audiofgsel' => 'black',
        'audiobgdel' => 'darkred', 'audiobgadd' => 'darkgreen',
        'statusbarbg' => 'grey70', 'statusbarfg' => 'black',
        'progressbg' => 'grey45', 'progressfg' => 'darkblue',
        'menubg' => 'grey85', 'menufg' => 'black',
        'buttonbg' => 'grey85', 'buttonfg' => 'black',
        'scrollbg' => 'grey85', 'scrollfg' => 'grey85',
        'toolbarbg' => 'grey85',
    },
    'Neuros I' => {
        'mainbg' => 'grey93', 'mainfg' => 'black',
        'filebg' => 'darkgreen', 'filefg' => 'white',
        'filebgsel' => 'grey', 'filefgsel' => 'black',
        'audiobg' => 'orange', 'audiofg' => 'black',
        'audiobgsel' => 'grey', 'audiofgsel' => 'black',
        'audiobgdel' => 'red', 'audiobgadd' => 'green',
        'statusbarbg' => 'grey', 'statusbarfg' => 'black',
        'progressbg' => 'grey55', 'progressfg' => 'orange',
        'menubg' => 'grey80', 'menufg' => 'black',
        'buttonbg' => 'grey80', 'buttonfg' => 'black',
        'scrollbg' => 'grey80', 'scrollfg' => 'grey80',
        'toolbarbg' => 'grey80',
    },
    'Neuros II' => {
        'mainbg' => 'black', 'mainfg' => 'orange',
        'filebg' => 'black', 'filefg' => 'orange',
        'filebgsel' => 'orange', 'filefgsel' => 'black',
        'audiobg' => 'orange', 'audiofg' => 'black',
        'audiobgsel' => 'grey45', 'audiofgsel' => 'white',
        'audiobgdel' => 'red', 'audiobgadd' => 'green',
        'statusbarbg' => 'orange', 'statusbarfg' => 'black',
        'progressbg' => 'grey45', 'progressfg' => 'orange',
        'menubg' => 'black', 'menufg' => 'orange',
        'buttonbg' => 'black', 'buttonfg' => 'orange',
        'scrollbg' => 'black', 'scrollfg' => 'orange',
        'toolbarbg' => 'orange',
    },
    'Purple' => {
        'mainbg' => '#440044', 'mainfg' => 'white',
        'filebg' => '#440044', 'filefg' => 'white',
        'filebgsel' => 'grey55', 'filefgsel' => 'black',
        'audiobg' => '#440044', 'audiofg' => 'white',
        'audiobgsel' => 'grey55', 'audiofgsel' => 'black',
        'audiobgdel' => 'darkred', 'audiobgadd' => 'darkgreen',
        'statusbarbg' => '#440044', 'statusbarfg' => 'white',
        'progressbg' => 'grey45', 'progressfg' => '#440044',
        'menubg' => '#440044', 'menufg' => 'white',
        'buttonbg' => '#440044', 'buttonfg' => 'white',
        'scrollbg' => '#440044', 'scrollfg' => '#440044',
        'toolbarbg' => '#440044',
    },
    'U.S.A.' => {
        'mainbg' => 'grey93', 'mainfg' => 'black',
        'filebg' => 'darkred', 'filefg' => 'white',
        'filebgsel' => 'grey55', 'filefgsel' => 'black',
        'audiobg' => 'darkblue', 'audiofg' => 'white',
        'audiobgsel' => 'grey55', 'audiofgsel' => 'black',
        'audiobgdel' => 'darkred', 'audiobgadd' => 'darkgreen',
        'statusbarbg' => 'grey70', 'statusbarfg' => 'black',
        'progressbg' => 'grey45', 'progressfg' => 'darkblue',
        'menubg' => 'grey85', 'menufg' => 'black',
        'buttonbg' => 'grey85', 'buttonfg' => 'black',
        'scrollbg' => 'grey85', 'scrollfg' => 'grey85',
        'toolbarbg' => 'grey85',
    },
);
if ($^O eq "MSWin32") {
    $themes{'System'} = {
        'mainbg' => 'grey93', 'mainfg' => 'black',
        'filebg' => 'darkred', 'filefg' => 'white',
        'filebgsel' => 'SystemButtonFace', 'filefgsel' => 'black',
        'audiobg' => 'black', 'audiofg' => 'white',
        'audiobgsel' => 'SystemButtonFace', 'audiofgsel' => 'black',
        'audiobgdel' => 'darkred', 'audiobgadd' => 'darkgreen',
        'statusbarbg' => 'SystemButtonFace', 'statusbarfg' => 'black',
        'progressbg' => 'SystemScrollbar', 'progressfg' => 'darkblue',
        'menubg' => 'SystemButtonFace', 'menufg' => 'black',
        'buttonbg' => 'SystemButtonFace', 'buttonfg' => 'black',
        'scrollbg' => 'SystemScrollbar', 'scrollfg' => 'SystemScrollbar',
        'toolbarbg' => 'SystemButtonFace',
    };
}

sub scheme
{
    my ($type, $w) = @_;

    if (scalar @_ % 2 > 0) {
        message('ERR',"Internal: scheme called with wrong # of arguments.\n");
        return;
    }

    while (scalar @_) {
        my $type = shift;
        my $w = shift;
       
        if ($type eq 'arrow') {
            $w->configure(
                -background => $cfg{'colors'}{'buttonbg'},
                -foreground => $cfg{'colors'}{'buttonfg'},
                -activebackground => $cfg{'colors'}{'buttonbg'},
                -activeforeground => $cfg{'colors'}{'buttonfg'},
                -highlightthickness => 0,
            );
        } elsif ($type eq 'balloon') {
            $w->configure(
                -font => $fontSm,
            );
            if ($^O eq "MSWin32") { eval { $w->attributes(-topmost => 1); }; }
        } elsif ($type eq 'browseentry') {
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
                -selectbackground => $cfg{'colors'}{'menubg'},
                -highlightthickness => 0,
            );
            my $arrow = $w->Subwidget('arrow');
            scheme('arrow',$arrow);
            my $slistbox = $w->Subwidget('slistbox');
            scheme('slistbox',$slistbox);
        } elsif ($type eq 'button') {
            $w->configure(
                -font => $font,
                -width => 6,
                -background => $cfg{'colors'}{'buttonbg'},
                -activebackground => $cfg{'colors'}{'buttonbg'},
                -foreground => $cfg{'colors'}{'buttonfg'},
                -activeforeground => $cfg{'colors'}{'buttonfg'},
            );
            if ($^O ne "MSWin32") {
                $w->configure(
                    -highlightthickness => 0,
                );
            }
        } elsif ($type eq 'canvas') {
            $w->configure(
                -background => $cfg{'colors'}{'audiobg'},
                -highlightthickness => 0,
            );
            my $xscroll = $w->Subwidget('xscrollbar');
            my $yscroll = $w->Subwidget('yscrollbar');
            scheme('scrollbar' => $xscroll, 'scrollbar' => $yscroll);
        } elsif ($type eq 'checkbutton') {
            $w->configure(
                -activebackground => $cfg{'colors'}{'menubg'},
                -background => $cfg{'colors'}{'menubg'},
                -selectcolor => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
                -activeforeground => $cfg{'colors'}{'menufg'},
            );
            if ($^O ne "MSWin32") {
                $w->configure(
                    -highlightthickness => 0,
                );
            }
        } elsif ($type eq 'entry') { 
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
                -selectbackground => $cfg{'colors'}{'menubg'},
                -highlightthickness => 0,
            );
        } elsif ($type eq 'frame') {
            $w->configure(
                -background => $cfg{'colors'}{'menubg'},
            );
        } elsif ($type eq 'label') {
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
            );
        } elsif ($type eq 'listbox') {
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
                -selectbackground => $cfg{'colors'}{'menufg'},
                -selectforeground => $cfg{'colors'}{'menubg'},
                -highlightthickness => 0,
            );
        } elsif ($type eq 'menu') {
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'filebg'},
                -activebackground => $cfg{'colors'}{'filebgsel'},
                -foreground => $cfg{'colors'}{'filefg'},
                -activeforeground => $cfg{'colors'}{'filefgsel'},
                -relief => 'raised',
                -tearoff => 0,
            );
        } elsif ($type eq 'notebook') {
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
                -inactivebackground => 'grey50',
                -disabledforeground => $cfg{'colors'}{'menufg'},
                -backpagecolor => $cfg{'colors'}{'menubg'},
            );
        } elsif ($type eq 'progressbar') {
            $w->configure(
                -width => 30,
                -from => 0,
                -to => 100,
                -troughcolor => $cfg{'colors'}{'progressbg'},
                -colors => [0, $cfg{'colors'}{'progressfg'}],
            );
        } elsif ($type eq 'scrollbar') {
            my $width = 10;
            if ($^O eq "MSWin32") { $width = 15; }

            $w->configure(
                -width => $width,
                -background => $cfg{'colors'}{'scrollfg'},
                -activebackground => $cfg{'colors'}{'scrollfg'},
                -highlightthickness => 0,
                -troughcolor => $cfg{'colors'}{'scrollbg'},
                -highlightthickness => 0,
            );
        } elsif ($type eq 'slistbox') {
            $w->configure(
                -font => $font,
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
                -selectbackground => $cfg{'colors'}{'menufg'},
                -selectforeground => $cfg{'colors'}{'menubg'},
                -highlightthickness => 0,
            );
            my $scrollbar = $w->Subwidget('yscrollbar');
            scheme('scrollbar',$scrollbar);
            if ($^O eq 'MSWin32') {
                $w->bind('<MouseWheel>', [sub {
                    $_[0]->yview(scroll => -($_[1]/120) * 5, 'units')
                },Ev('D')]);
            } else {
                $w->bind('<Button-4>', sub {
                    $w->yview(scroll => -5, 'units');
                });
                $w->bind('<Button-5>', sub {
                    $w->yview(scroll => 5, 'units');
                });
            }
        } elsif ($type eq 'toplevel') {
            $w->configure(
                -background => $cfg{'colors'}{'menubg'},
                -foreground => $cfg{'colors'}{'menufg'},
            );
        } elsif ($type eq 'tree') {
            $w->configure(
                -background => $cfg{'colors'}{'filebg'},
                -foreground => $cfg{'colors'}{'filefg'},
                -selectbackground => $cfg{'colors'}{'filebgsel'},
                -selectforeground => $cfg{'colors'}{'filefgsel'},
                -highlightthickness => 0,
            );
            my $xscroll = $w->Subwidget('xscrollbar');
            my $yscroll = $w->Subwidget('yscrollbar');
            scheme('scrollbar' => $xscroll, 'scrollbar' => $yscroll);
            if ($^O eq 'MSWin32') {
                $w->bind('<MouseWheel>',
                    [sub {$_[0]->yview(scroll => -($_[1]/120) * 5, 'units')},Ev('D')]);
            } else {
                $w->bind('<Button-4>', sub { $w->yview(scroll => -5, 'units'); });
                $w->bind('<Button-5>', sub { $w->yview(scroll => 5, 'units'); });
            }
        }
    }
}

sub soruneGUI($)
{
    my $cfgFile = shift;

    eval {
        require Tk;
        import Tk qw /Ev/;
        require Tk::Entry;
        require Tk::Pane;
        require Tk::Tree;
        require Tk::DialogBox;
        require Tk::ROText;
        require Tk::ProgressBar;
        require Tk::BrowseEntry;
        require Tk::Balloon;
        require Tk::Photo;
        require Tk::Adjuster;
        require Tk::ItemStyle;
        require Tk::Wm;
        require Tk::NoteBook;
        push @INC, File::Spec->rel2abs(dirname($0));
        require "toolbar.pm";
        require "icons.pm";
    };
    if ($@) {
        message('ERR',"The Sorune gui requires the Perl Tk module.\n");
        return 1;
    }

    my $splash = createSplash();

    # Read the Sorune configuration file
    readSoruneCfg($cfgFile,\%cfg);

    if (defined $cfg{'general'}{'font'}) {
        $font = $cfg{'general'}{'font'};
        my ($fam,$size) = split / /, $font;
        if ($size <= 30) { $fontBold = "$fam $size bold"; }
        if ($size >= 5)  { $fontSm = "$fam " . ($size-2); }
    } else {
        $font     = 'helvetica 10';
        $fontBold = 'helvetica 10 bold';
        $fontSm   = 'helvetica 8';
    }

    # Setup the main, top, left, right and bottom windows
    my $main = MainWindow->new(
        -title => "Sorune",
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    );

    # Read icons
    my @icons = getSoruneIcons();
    while (scalar @icons) {
        my $name = shift @icons;
        my $data = shift @icons;
        $main->Photo($name, -data => $data);
    }

    $main->withdraw;
    $main->focusFollowsMouse;
    if (defined $cfg{'general'}{'geometry'}) {
        $main->geometry($cfg{'general'}{'geometry'});
    }

    my $top = $main->Frame(
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => 'top', -expand => 1, -fill => 'both');
    my $left = $top->Frame(
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => 'left',
        -anchor => 'w',
        -padx => 5,
        -expand => 1,
        -fill => 'both',
    );
   
    # Setup directory frame and text
    my $dirFrame = $left->Frame->pack(-fill => 'x',-side => 'top');
    $dirText = $dirFrame->ROText(
        -font => $font,
        -height => 1,
        -relief => "flat",
        -borderwidth => 0,
        -insertwidth => 0,
        -takefocus => 0,
        -highlightthickness => 0,
        -exportselection => 1,
        -background => $cfg{'colors'}{'mainbg'},
        -selectbackground => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
        -selectforeground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => 'left',-fill => 'x',-expand => 1);

    # Setup adjuster
    my $adj = $top->Adjuster(-background => $cfg{'colors'}{'mainbg'});
    $adj->packAfter($left, -side => 'left');
    my $right = $top->Frame(
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => 'right',
        -anchor => 'n',
        -expand => 1,
        -fill => 'y',
    );
    
    my $bottom = $main->Frame(
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'} 
    )->pack(-side => 'bottom',
        -anchor => 's',
        -expand => 0,
        -fill => 'x',
    );

    # Setup message window
    $msg = $bottom->Scrolled('ROText',
        -scrollbars => 'osoe',
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
        -highlightthickness => 0,
        -height => 5,
        -wrap => 'word',
        -font => $font,
    )->pack(
        -side => 'top',
        -expand => 1,
        -fill => 'x',
    );
    my $yscroll = $msg->Subwidget('yscrollbar');
    scheme('scrollbar',$yscroll);
    if ($^O eq 'MSWin32') {
        $msg->bind('<MouseWheel>',
            [sub {$_[0]->yview(scroll => -($_[1]/120) * 5, 'units')},Ev('D')]);
    } else {
        $msg->bind('<Button-4>', sub { $msg->yview(scroll => -5, 'units'); });
        $msg->bind('<Button-5>', sub { $msg->yview(scroll => 5, 'units'); });
    }
    enableGuiMessaging($msg);

    # Setup breadcrumb frame and text
    my $bcFrame = $right->Frame->pack(-fill => 'x', -side => 'top');
    $bcText = $bcFrame->ROText(
        -font => $font,
        -height => 1,
        -relief => "flat",
        -borderwidth => 0,
        -insertwidth => 0,
        -takefocus => 0,
        -highlightthickness => 0,
        -exportselection => 1,
        -background => $cfg{'colors'}{'mainbg'},
        -selectbackground => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
        -selectforeground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => 'left',-fill => 'x',-expand => 1);

    # Setup the audio browser
    my $canvas = $right->Scrolled('Canvas',
        -scrollbars => 'oe',
        -height => 176,
        -width => 401,
        -borderwidth => 5,
        -relief => 'ridge',
        -takefocus => 1,
    )->pack(-expand => 1, -fill => 'y', -pady => 5);
    scheme('canvas' => $canvas);

    # Setup song information label frames
    setInfo($right);

    # Setup toolbar (reads in images)
    my $tb = $main->ToolBar(-movable => 1,
        -side => 'top',
        -indicatorcolor => $cfg{'colors'}{'mainfg'},
    );

    # Setup the file browser
    my $tree;
    if (defined $cfg{'general'}{'musichome'} and 
        -r $cfg{'general'}{'musichome'}) {
        $tree = dirTree($main,$left,$cfg{'general'}{'musichome'},0);
    } else {
        $tree = dirTree($main,$left,getcwd(),0);
    }

    # Setup the right click menu
    my $menu = createRightClickMenu($main,$tree,$canvas,$bottom);

    # Clear anchor when we lose focus
    $tree->bind('<FocusOut>' => sub { $tree->anchorClear; });

    # Set the initial directory text (above the tree)
    setDirText($tree->selectionGet);

    # Setup canvas bindings
    $canvas->CanvasBind('<Button-1>' => sub {getLeftClick($tree,$canvas)});
    $canvas->CanvasBind('<Double-Button-1>' =>
        sub {getDoubleLeftClick($tree,$canvas)});
    $canvas->CanvasBind('<Button-3>' => sub {getRightClick($tree,$canvas)});

    $addButton = $tb->ToolButton(-image => 'add',
        -tip => 'Add To Neuros [Ctrl-a]',
        -accelerator => '<Control-a>',
        -command => [ \&gAdd, $main, $tree, $canvas, $bottom, undef ],
    );
    $delButton = $tb->ToolButton(-image => 'delete',
        -tip => 'Delete From Neuros [Ctrl-d]',
        -accelerator => '<Control-d>',
        -command => [ \&gDelete, $tree, $canvas, $bottom ],
    );
    $syncButton = $tb->ToolButton(-image => 'sync',
        -tip => 'Synchronize [Ctrl-s]',
        -accelerator => '<Control-s>',
        -command => [ \&gSync, $main, $tree, $canvas, $bottom ],
    );
    $tb->separator;
    $playButton = $tb->ToolButton(-image => 'play',
        -tip => 'Play [Ctrl-p]',
        -accelerator => '<Control-p>',
        -command => [ \&gPlay, $tree, $canvas ],
    );
    $playlistButton = $tb->ToolButton(-image => 'playlist',
        -tip => 'Create/Add To Playlist [Ctrl-l]',
        -accelerator => '<Control-l>',
        -command => [ \&gPlaylist, $main, $tree, $canvas, $bottom, 0 ],
    );
    $tb->separator;
    $rescanButton = $tb->ToolButton(-image => 'rescan',
        -tip => 'Rescan [Ctrl-c]',
        -accelerator => '<Control-c>',
        -command => [ \&gScan, $tree, $canvas, $bottom ],
    );
    $rebuildButton = $tb->ToolButton(-image => 'rebuild',
        -tip => 'Rebuild [Ctrl-b]',
        -accelerator => '<Control-b>',
        -command => [ \&gRebuild, $main, $tree, $canvas, $bottom ],
    );
    $resetButton = $tb->ToolButton(-image => 'reset',
        -tip => 'Reset/Clear [Ctrl-t]',
        -accelerator => '<Control-t>',
        -command => [ \&gReset, $main, $tree, $canvas, $bottom ],
    );
    $clearButton = $tb->ToolButton(-image => 'clear',
        -tip => 'Clear Additions/Deletions [Ctrl-r]',
        -accelerator => '<Control-r>',
        -command => sub {
            my $change = 0;
            foreach my $key (keys %musicDb) {
                if (defined $musicDb{$key}{'delete'}) {
                    delete $musicDb{$key}{'delete'};
                    $change = 1;
                }
                if (defined $musicDb{$key}{'add'}) {
                    (my $localFile = $key) =~ s/$neurosDrive/$neurosHome/;
                    if (-r $localFile) {
                        delete $musicDb{$key}{'add'};
                    } else {
                        removeFromPlaylist(\%musicDb,\%playlistDb,undef,$key,
                            $neurosHome);
                        delete $musicDb{$key};
                    }
                    $change = 1;
                }
            }
            foreach my $key (keys %recordDb) {
                if (defined $recordDb{$key}{'delete'}) {
                    delete $recordDb{$key}{'delete'};
                    $change = 1;
                }
            }
            if ($change) {
                %info = ();
                displayMenu($canvas,undef,undef);
                updateTreeIcons($tree,undef);
                configureStatusBar($bottom);
            }
            message('INFO',"Sorune clear additions/deletions complete.\n");
        }
    );
    $dupeButton = $tb->ToolButton(-image => 'duplicates',
        -tip => 'Find Duplicates [Ctrl-u]',
        -accelerator => '<Control-u>',
        -command => [ \&gDuplicates, $main, $canvas ],
    );
    $exportButton = $tb->ToolButton(-image => 'importexport',
        -tip => 'Import/Export [Ctrl-o]',
        -accelerator => '<Control-o>',
        -command => [ \&gExport, $main, $tree, $canvas, $bottom ],
    );
    $backupButton = $tb->ToolButton(-image => 'backup',
        -tip => 'Backup [Ctrl-k]',
        -accelerator => '<Control-k>',
        -command => sub {
            backupDb("$neurosHome/woid_db","$neurosHome/sorune/database");
        },
    );
    $restoreButton = $tb->ToolButton(-image => 'restore',
        -tip => 'Restore [Ctrl-e]',
        -accelerator => '<Control-e>',
        -command => sub {
            restoreDb("$neurosHome/sorune/database",
                "$neurosHome/woid_db");
        },
    );
    $tb->separator;
    $optionsButton = $tb->ToolButton(-image => 'config',
        -tip => 'Configuration Options [Ctrl-g]',
        -accelerator => '<Control-g>',
        -command => [ \&gConfiguration, $main, $top, $tb, $tree,
            $canvas, $right, $bottom, $msg ],
    );
    $aboutButton = $tb->ToolButton(-image => 'info',
        -tip => 'About/Help [Ctrl-h]',
        -accelerator => '<Control-h>',
        -command => [ \&gAbout, $main ],
    );
    $exitButton = $tb->ToolButton(-image => 'exit',
        -tip => 'Exit [Ctrl-x]',
        -accelerator => '<Control-x>',
        -command => [ \&gExit, $main, $tree ],
    );
    $tb->configure(-activebackground => $cfg{'colors'}{'toolbarbg'});
    $main->protocol('WM_DELETE_WINDOW',[ \&gExit, $main, $tree ]);

    # Cfg file error checking
    if ($cfg{'colors'}{'audiobg'} eq $cfg{'colors'}{'audiobgsel'}) {
        message('ERR', "Colors audiobg and audiobgsel must be different.\n");
        if ($cfg{'colors'}{'audiobg'} eq 'grey') {
            $cfg{'colors'}{'audiobgsel'} = 'darkgrey';
            message('INFO', "Setting audiobgsel to darkgrey.\n");
        } else {
            $cfg{'colors'}{'audiobgsel'} = 'grey';
            message('INFO', "Setting audiobgsel to grey.\n");
        }
    }

    # Scan for Neuros and setup audio browser
    gScan($tree,$canvas,$bottom);
    displayMenu($canvas,'main|undef|1|main|Main|undef|undef',undef);
    configureStatusBar($bottom);

    # Set treewidth, main icon and focus
    if (defined $cfg{'general'}{'treewidth'}) {
        $left->configure(-width => $cfg{'general'}{'treewidth'});
    }

    # Bring up sorune
    $splash->after(1000);
    $splash->destroy;
    $main->after(200);
    $main->deiconify;
    $main->iconimage('main32');

    # Make sure "Music Home" is set or at least that the user knows about it
    if ($cfg{'general'}{'version'} ne getVersion() and
        (!defined $cfg{'general'}{'musichome'} or
        $cfg{'general'}{'musichome'} eq "")) {
        warnMusicHome($main,$optionsButton);
    }
    $cfg{'general'}{'version'} = getVersion();

    $main->focus;
    $canvas->focus;

    Tk::MainLoop();
    return 0;
}

sub createSplash()
{
    my $splash = new MainWindow(-title => 'Sorune');
    $splash->withdraw;
    $splash->iconify;
    $splash->overrideredirect(1);

    my $photo = $splash->Photo("splash",
        -data => getSoruneSplash());
    $splash->geometry("+" . 
        int($splash->screenwidth/2 - $photo->width/2) . "+" .
        int($splash->screenheight/2 - $photo->height/2)
    );
    my $label = $splash->Label(
        -image => "splash",
        -borderwidth => 0,
    )->pack(-side => 'top');
    $splash->update;
    $splash->deiconify;
    return $splash;
}

sub warnMusicHome($$)
{
    my ($main,$optionsButton) = @_;

    my $top = $main->Toplevel(-title => 'First Run Notice');
    my $lab1 = $top->Label(-text => 'Music Home should be set before adding songs and syncing with Sorune.')->pack;
    my $lab2 = $top->Label(
        -text => 'You will be taken to the configuration page where you can set Music Home.',
    )->pack;
    my $lab3 = $top->Label(
        -text => '(Press Help on that screen for why this is important.)',
    )->pack;

    my $okButton = $top->Button(
        -text => 'Ok',
        -command => sub {
            $top->destroy;
            $optionsButton->invoke;
        },
    )->pack(-side => 'left', -padx => 10, -expand => 1);

    $top->bind('<Return>', sub { $okButton->invoke; });

    scheme('toplevel' => $top, 'label' => $lab1, 'label' => $lab2, 
        'label' => $lab3, 'button' => $okButton);
    $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; });

    relatePos($main,$top,500,100,1,0);
    $okButton->focus;
}

sub createRightClickMenu($$$$)
{
    my ($main,$tree,$canvas,$bottom) = @_;

    my $menu = $tree->Menu;
    $menu->add('command', -label => 'Add To Neuros',
        -command => [ \&gAdd, $main, $tree, $canvas, $bottom, undef ],
    );
    $menu->add('command', -label => 'Add To Neuros + Sync',
        -command => sub {
            my $status = gAdd($main, $tree, $canvas, $bottom, undef);
            if ($status == 0) { gSync($main, $tree, $canvas, $bottom); }
        }
    );
    $menu->add('command', -label => 'Delete From Neuros',
        -command => [ \&gDelete, $tree, $canvas, $bottom ],
    );
    $menu->add('command', -label => 'Delete From Neuros + Sync',
        -command => sub {
            gDelete($tree, $canvas,$bottom);
            gSync($main, $tree, $canvas, $bottom);
        },
    );
    $menu->add('command', -label => 'Sync',
        -command => [ \&gSync, $main, $tree, $canvas, $bottom ],
    );
    $menu->add('separator');
    $menu->add('command', -label => 'Play',
        -command => [ \&gPlay, $tree, $canvas ],
    );
    $menu->add('command', -label => 'Create/Add To Playlist',
        -command => [ \&gPlaylist, $main, $tree, $canvas, $bottom, 0 ],
    );
    $menu->add('command', -label => 'Create/Add To Playlist + Sync',
        -command => sub {
            gPlaylist($main, $tree, $canvas, $bottom, 1);
        },
    );
    $tree->bind("<ButtonPress-3>", [ \&displayRightClickMenu, $menu,
        Ev('X'), Ev('Y') ]);
    $tree->bind("<ButtonPress-1>" => sub { $menu->unpost; });
    return $menu;
}

sub displayRightClickMenu($$)
{
    my ($tree,$menu,$x,$y) = @_;

    scheme('menu',$menu);
    if (defined $menu) {
        $tree->anchorClear;
        $menu->post($x,$y);
    }
}

sub setButtonState($)
{
    my $state = shift;
    my @buttons = ($addButton,$delButton,$syncButton,$resetButton,
        $rebuildButton,$backupButton,$restoreButton,$clearButton,
        $playlistButton,$exportButton,$dupeButton);

    foreach (@buttons) {
        if ($state eq 'disabled') {
            $_->configure(-state => 'disabled');
            $_->bind('<Enter>' => [$_, 'configure', -relief => 'flat']);
        } elsif ($state eq 'normal') {
            $_->configure(-state => 'normal');
            $_->bind('<Enter>' => [$_, 'configure', -relief => 'raised']);
        }
    }
}

sub gAdd($$$$$)
{
    my ($main,$tree,$canvas,$bottom,$findFilesRef) = @_;
    my $status = "Scanning for audio files";
    if (!scalar ($tree->selectionGet)) {
        message('ERR',"Please make selection from file browser first.\n");
        return -1;
    }
    my $top = $main->Toplevel(-title => 'Add Files To Sorune Database',
        -background => $cfg{'colors'}{'menubg'},
        -foreground => $cfg{'colors'}{'menufg'},
    );
    my $percentDone = 0;
    my $halt = 0;

    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f2 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $progress = $f1->ProgressBar(
        -length => 300,
        -blocks => 300,
        -variable => \$percentDone
    )->pack(-side => 'left', -expand => 1, -fill => 'x');
    my $sLabel = $f2->Label(-textvariable => \$status,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $cancelButton = $top->Button(-text => 'Cancel',
        -command=> sub { $halt = 1; $top->update; }
    )->pack(-side => 'bottom', -pady => 5);

    scheme('frame' => $f1, 'frame' => $f2,
        'label' => $sLabel,
        'progressbar' => $progress,
        'button' => $cancelButton,
    );
    $top->protocol('WM_DELETE_WINDOW', sub { $halt = 1; $top->update; });

    relatePos($main,$top,500,100,1,0);
    $cancelButton->focus;
    $top->update;

    # Find music files
    my $startTime = time;
    my @findFiles = getTreeFiles($tree,$top,$halt,\$status);
    my $fileCount = scalar @findFiles;
    if (defined $findFilesRef) {
        @$findFilesRef = @findFiles;
    }
    $status = "Adding files to database";
    $top->update;
    
    # Add music files
    if (!$halt) {
        my $musicHome;
        if (defined $cfg{'general'}{'musichome'}) {
            $musicHome = $cfg{'general'}{'musichome'};
        }
        for (my $i = 0; $i < $fileCount; $i++) {
            if ($findFiles[$i] =~ /^$neurosHome/) { next; }
            addFile($findFiles[$i],$musicHome,$neurosHome,\%musicDb,
                \%playlistDb,\%cfg,$neurosDrive,0);
            if ($halt) { last; }
            $percentDone = ($i+1) * 100 / $fileCount;
            $top->update;
        }
    }

    %info = ();
    if (defined $cfg{'general'}{'duplicates'} and
        $cfg{'general'}{'duplicates'} eq '1') {
        $status = "Checking for duplicate album names";
        $top->update;
        fixDuplicateAlbums(\%musicDb);
    }
    displayMenu($canvas,undef,undef);
    updateTreeIcons($tree,undef);
    configureStatusBar($bottom);
    $percentDone = 100;
    $top->update;
    if ($halt) {
        message('INFO',"Sorune additions aborted.\n");
    } else {
        if ($fileCount) {
            message('INFO',"Sorune additions complete (" . 
                getTime(time - $startTime) . ")\n");
        }
    }
    $top->destroy;
    $syncNeeded = 1;

    return $halt;
}

sub gScan($$$)
{
    my ($tree,$canvas,$bottom) = @_;
    my $lastLocation = undef;
    if (defined $cfg{'general'}{'lastlocation'}) {
        $lastLocation = $cfg{'general'}{'lastlocation'};
    }
    ($neurosHome,$mainSerial,$fwVersion) = locateNeuros($lastLocation);

    if (defined $neurosHome) {
        $soruneDb = "$neurosHome/sorune/sorune.db";
        eval { require Compress::Zlib; };
        if (!$@) { $soruneDb .= ".gz"; }
        if (-r $soruneDb) {
            readSoruneDb($soruneDb,\%musicDb,\%playlistDb,\%recordDb);
        } else {
            message('ERR',
                "Sorune database not found. Please rebuild or add files.\n");
        }
        if (-f "$neurosHome/bkpk.sn") {
            $neurosDrive = "C:";
        } else {
            $neurosDrive = "D:";
        }
        %info = ();
        displayMenu($canvas,undef,undef);
        updateTreeIcons($tree,undef);
        setButtonState('normal');
        $cfg{'general'}{'lastlocation'} = $neurosHome;
        writeSoruneCfg(\%cfg);
    } else {
        message('ERR',"Neuros not found. Please connect and rescan or check permissions.\n");
        setButtonState('disabled');
    }

    configureStatusBar($bottom);
}

sub getTreePaths
{
    my ($tree,$e) = @_;
    my @paths = ();
    my @items = ();

    if (!defined $e) {
        @items = $tree->info('children');
    } else {
        @items = $tree->info('children',$e);
    }

    foreach my $item (@items) {
        push @paths, $item;
        push @paths, getTreePaths($tree,$item);
    }
    return @paths;
}

sub configureStatusBar($)
{
    my $bottom = shift;
    my $dbSize = getSize(getMusicSize(\%musicDb,\%recordDb));
    my $freeSize = getSize(getFreeSpace($neurosHome));
    my @bars = (\$locBar,\$mainBar,\$fwBar,\$capBar,\$freeBar);
    my @data = ("Location","Serial","Firmware","Audio Size","Free Space");
    my @data2 = ($neurosHome,$mainSerial,$fwVersion,$dbSize,$freeSize);
    my $count = scalar @data;
    my $text;

    for (my $i = 0; $i < $count; $i++) {
        if (defined $data2[$i]) {
            $text = "$data[$i]: $data2[$i]";
        } else {
            $text = "$data[$i]: Unknown";
        }
        if (defined ${$bars[$i]}) {
            ${$bars[$i]}->configure(-text => $text, -font => $fontSm,
                -background => $cfg{'colors'}{'statusbarbg'},
                -foreground => $cfg{'colors'}{'statusbarfg'},
            );
        } else {
            ${$bars[$i]} = $bottom->Label(-relief => 'sunken',
                -borderwidth => 1,
                -font => $fontSm,
                -text => $text,
                -background => $cfg{'colors'}{'statusbarbg'},
                -foreground => $cfg{'colors'}{'statusbarfg'},
            );
            ${$bars[$i]}->pack(-anchor => 's',
                -side => 'left',
                -expand => 1,
                -fill => 'x',
            );
        }
    }

    if (defined $ucBalloon) { $ucBalloon->destroy; $ucBalloon = undef; }
    $ucBalloonMsg = undef;
    my $heightCount = 0;
    foreach my $format ('MP3','OGG','WAV','WMA') {
        if (defined $info{$format}{'count'}) {
            if ($heightCount) { $ucBalloonMsg .= "\n"; }
            $ucBalloonMsg .= "$format: " . $info{$format}{'count'} .
                " Songs (" . getTime($info{$format}{'length'}) . " / " .
                getSize($info{$format}{'size'}) . ")";
            $heightCount++;
        }
    }
    if (defined $ucBalloonMsg and !defined $ucBalloon) {
        $ucBalloon = $bottom->Balloon(
            -postcommand => [\&balloonPos, $bottom, $heightCount],
        );
        $ucBalloon->attach($capBar, -balloonmsg => \$ucBalloonMsg);
        scheme('balloon',$ucBalloon);
    }

    if (defined $fsBalloon) { $fsBalloon->destroy; $fsBalloon = undef; }
    $fsBalloonMsg = undef;
    $heightCount = 0;
    if (defined $info{'all'}{'addsize'}) {
        $fsBalloonMsg .= "Scheduled Additions: " . 
            getSize($info{'all'}{'addsize'});
        $heightCount++;
    }
    if (defined $info{'all'}{'deletesize'}) {
        if ($heightCount) { $fsBalloonMsg .= "\n"; }
        $fsBalloonMsg .= "Scheduled Deletions: " . 
            getSize($info{'all'}{'deletesize'});
        $heightCount++;
    }
    if (defined $fsBalloonMsg and !defined $fsBalloon) {
        $fsBalloon = $bottom->Balloon(
            -postcommand => [\&balloonPos, $bottom, $heightCount],
        );
        $fsBalloon->attach($freeBar, -balloonmsg => \$fsBalloonMsg);
        scheme('balloon',$fsBalloon);
    }
    $bottom->configure(-background => $cfg{'colors'}{'statusbarbg'});
}

sub balloonPos($$)
{
    my ($parent,$count) = @_;
    my $x = $parent->pointerx + 10;
    my $y = $parent->pointery - (20 * $count);
    return "$x,$y";
}

sub gConfiguration($$$$$$$)
{
    my ($main,$parentTop,$tb,$tree,$canvas,$right,$bottom,$msg) = @_;
    my ($musicHome,$recordingsHome,$untagFormat,$mediaPlayer,$colorScheme,
        $aaSort,$abjust,$artistThe,$balloon);
    my ($audio,$songs,$playlists,$artists,$albums,$genres,$years,$recordings);
    my ($artalb,$genart,$genalb,$notify) = (0,0,0,0);
    my ($usetags,$normalize,$md5sum,$duplicates) = (0,0,0,0);
    my ($albumdiff,$artistdiff,$genrediff,$titlediff) = (0,0,0,0);
    my ($family,$size) = split / /, $font;

    if (defined $cfg{'general'}{'musichome'}) {
        $musicHome = $cfg{'general'}{'musichome'};
    }
    if (defined $cfg{'general'}{'recordingshome'}) {
        $recordingsHome = $cfg{'general'}{'recordingshome'};
    }
    if (defined $cfg{'general'}{'untaggedformat'}) {
        $untagFormat = $cfg{'general'}{'untaggedformat'};
    }
    if (defined $cfg{'general'}{'mediaplayer'}) {
        $mediaPlayer = $cfg{'general'}{'mediaplayer'};
    }
    if (defined $cfg{'general'}{'usetags'} and
        $cfg{'general'}{'usetags'} eq "1") {
        $usetags = 1;
    }
    if (defined $cfg{'general'}{'normalize'} and
        $cfg{'general'}{'normalize'} eq "1") {
        $normalize = 1;
    }
    if (defined $cfg{'general'}{'duplicates'} and
        $cfg{'general'}{'duplicates'} eq "1") {
        $duplicates = 1;
    }
    if (defined $cfg{'general'}{'md5sum'} and
        $cfg{'general'}{'md5sum'} eq "1") {
        $md5sum = 1;
    }
    if (defined $cfg{'nam'}{'artistalbum'} and
        $cfg{'nam'}{'artistalbum'} eq "1") {
        $artalb = 1;
    }
    if (defined $cfg{'nam'}{'genreartist'} and
        $cfg{'nam'}{'genreartist'} eq "1") {
        $genart = 1;
    }
    if (defined $cfg{'nam'}{'genrealbum'} and
        $cfg{'nam'}{'genrealbum'} eq "1") {
        $genalb = 1;
    }
    if (defined $cfg{'general'}{'albumdiff'}) {
        $albumdiff = $cfg{'general'}{'albumdiff'};
    }
    if (defined $cfg{'general'}{'artistdiff'}) {
        $artistdiff = $cfg{'general'}{'artistdiff'};
    }
    if (defined $cfg{'general'}{'genrediff'}) {
        $genrediff = $cfg{'general'}{'genrediff'};
    }
    if (defined $cfg{'general'}{'titlediff'}) {
        $titlediff = $cfg{'general'}{'titlediff'};
    }
    if (defined $cfg{'nam'}{'audio'}) { $audio = $cfg{'nam'}{'audio'}; }
    if (defined $cfg{'nam'}{'songs'}) { $songs = $cfg{'nam'}{'songs'}; }
    if (defined $cfg{'nam'}{'playlists'}) { $playlists = $cfg{'nam'}{'playlists'}; }
    if (defined $cfg{'nam'}{'artists'}) { $artists = $cfg{'nam'}{'artists'}; }
    if (defined $cfg{'nam'}{'albums'}) { $albums = $cfg{'nam'}{'albums'}; }
    if (defined $cfg{'nam'}{'genres'}) { $genres = $cfg{'nam'}{'genres'}; }
    if (defined $cfg{'nam'}{'years'}) { $years = $cfg{'nam'}{'years'}; }
    if (defined $cfg{'nam'}{'recordings'}) {
        $recordings = $cfg{'nam'}{'recordings'};
    }

    if (defined $tree) {
        $tree->selectionClear;
    }

    my $top = $main->Toplevel(-title => 'Configuration Options');
    my $book = $top->NoteBook()->pack(-expand => 1, -fill => 'both');

    my $page1 = $book->add('General', -label => 'General');
    my $p1f1 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p1f2 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p1f3 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p1f4 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p1f5 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p1f6 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p1f7 = $page1->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');

    my $page2 = $book->add('Audio', -label => 'Audio');
    my $p2f1 = $page2->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p2f2 = $page2->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p2f3 = $page2->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p2f4 = $page2->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p2f5 = $page2->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');

    my $page3 = $book->add('Text', -label => 'Text');
    my $p3f1 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f2 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f3 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f4 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f5 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f6 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f7 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $p3f8 = $page3->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');

    # Page 1

    my $mhLab = $p1f1->Label(-text => "Music Home:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $mhEntry = $p1f1->Entry(-textvariable => \$musicHome,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $b1 = $p1f1->Button(-text => "Browse",
        -command => sub {
            dirTree($main,$top,$musicHome,\$musicHome,1);
        },
    )->pack(-side => 'left', -padx => 5);
    $balloon = $top->Balloon();
    scheme('balloon',$balloon);
    $balloon->attach($mhEntry,
        -balloonmsg => "Strip this directory from those created on the Neuros.",
    );

    my $rhLab = $p1f2->Label(-text => "Recordings Home:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $rhEntry = $p1f2->Entry(-textvariable => \$recordingsHome,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $b2 = $p1f2->Button(-text => "Browse",
        -command => sub {
            dirTree($main,$top,$recordingsHome,\$recordingsHome,1);
        },
    )->pack(-side => 'left', -padx => 5);
    $balloon = $top->Balloon();
    scheme('balloon',$balloon);
    $balloon->attach($rhEntry,
        -balloonmsg => "Transfer recordings during sync to this directory.",
    );

    my $ufLab = $p1f3->Label(-text => "Untagged Format:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $ufEntry = $p1f3->Entry(-textvariable => \$untagFormat,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    $balloon = $top->Balloon();
    scheme('balloon',$balloon);
    $balloon->attach($ufEntry,
        -balloonmsg => "Specify the directory structure under music home for files w/o id3 or equivalent tags.\n" .
        'Values: %ALBUM%, %ARTIST%, %GENRE%, %TRACKNUMBER%, %TITLE% and %YEAR%'
        . "\n" .
        'Example: %GENRE%/%ARTIST%/%ALBUM%/%TRACKNUMBER%-%TITLE%'
    );

    my $mpLab = $p1f4->Label(-text => "Media Player:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $mpEntry = $p1f4->Entry(-textvariable => \$mediaPlayer,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $b3 = $p1f4->Button(-text => "Browse",
        -command => sub {
            dirTree($main,$top,$mediaPlayer,\$mediaPlayer,1);
        },
    )->pack(-side => 'left', -padx => 5);

    my $themeLab = $p1f5->Label(-text => "Theme:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $csEntry = $p1f5->BrowseEntry(
        -width => 12,
        -variable => \$colorScheme,
    )->pack(-fill => 'x', -side => 'left');
    $csEntry->insert('end', sort keys %themes);
    my $famEntry = $p1f5->BrowseEntry(
        -width => 25,
        -variable => \$family,
    )->pack(-fill => 'x', -side => 'left', -padx => 5);
    $famEntry->insert('end', sort $main->fontFamilies);
    my $sizeEntry = $p1f5->BrowseEntry(
        -width => 5,
        -variable => \$size,
    )->pack(-fill => 'x', -side => 'left');
    $sizeEntry->insert('end', (3 .. 32));
    my $abjEntry = $p1f5->BrowseEntry(
        -width => 12,
        -variable => \$abjust,
    )->pack(-fill => 'x', -side => 'left', -padx => 5);
    $abjEntry->insert('end', ('center','left'));

    # set current values
    my $keys = keys %{$cfg{'colors'}};
    $colorScheme = "Custom";
    foreach my $scheme (sort keys %themes) {
        my $count = 0;
        foreach my $item (sort keys %{$themes{$scheme}}) {
            if ($themes{$scheme}{$item} eq $cfg{'colors'}{$item}) {
                $count++;
            }
        }
        if ($count == $keys) {
            $colorScheme = $scheme;
            last;
        }
    }
    if ($colorScheme eq 'Custom') { $themes{'Custom'} = $cfg{'colors'}; }
    if ($cfg{'general'}{'abjust'} eq 'left') {
        $abjust = 'left';
    } else {
        $abjust = 'center';
    }

    my $tcLab = $p1f6->Label(-text => "Tag Compare:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    my $adEntry = $p1f6->Entry(-textvariable => \$albumdiff,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $adLab = $p1f6->Label(-text => "Album",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $adEntry2 = $p1f6->Entry(-textvariable => \$artistdiff,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $adLab2 = $p1f6->Label(-text => "Artist",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $gdEntry = $p1f6->Entry(-textvariable => \$genrediff,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $gdLab = $p1f6->Label(-text => "Genre",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $tdEntry = $p1f6->Entry(-textvariable => \$titlediff,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $tdLab = $p1f6->Label(-text => "Title",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');

    my $miscLab = $p1f7->Label(-text => "Misc:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    my $cb4 = $p1f7->Checkbutton(-variable => \$usetags)->pack(-side => 'left');
    my $utLab = $p1f7->Label(-text => "Use Tags",
        -width => 10,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $cb5 = $p1f7->Checkbutton(-variable => \$normalize)->pack(-side => 'left');
    my $normLab = $p1f7->Label(-text => "Normalize",
        -width => 10,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $cb6 = $p1f7->Checkbutton(-variable => \$duplicates)->pack(-side => 'left');
    my $duplicatesLab = $p1f7->Label(-text => "Duplicates",
        -width => 10,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $cb7 = $p1f7->Checkbutton(-variable => \$md5sum)->pack(-side => 'left');
    my $vtLab = $p1f7->Label(-text => "Verify Transfer",
        -width => 14,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');

    # Page 2

    my @menuItems = ();
    my $moLab = $p2f1->Label(-text => "Menu Order:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    @menuItems = split /\s*,\s*/, $cfg{'nam'}{'menuorder'};
    my $menuList = $p2f1->Listbox(
        -selectmode => 'single',
        -width => 25,
        -height => 7,
    )->pack(-side => 'left');
    $menuList->insert('end', @menuItems);

    my $blankLab = $p2f2->Label(-text => "",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    my $upButton = $p2f2->Button(-text => 'Up',
        -command => [ \&moveItem, $menuList, \@menuItems, 0 ],
    )->pack(-side => 'left');
    my $downButton = $p2f2->Button(-text => 'Down',
        -command => [ \&moveItem, $menuList, \@menuItems, 1 ],
    )->pack(-side => 'left', -padx => 5);
    my $defaultButton = $p2f2->Button(-text => 'Default',
        -command => sub {
            $menuList->delete(0,'end');
            @menuItems = ('Songs','Playlists','Artists','Albums',
                'Genres','Years','Recordings');
            $menuList->insert('end',@menuItems);
        },
    )->pack(-side => 'left');

    my $nmLab = $p2f3->Label(-text => "Neuros Menus:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $cb1 = $p2f3->Checkbutton(-variable => \$artalb)->pack(-side => 'left');
    my $aaLab = $p2f3->Label(-text => "Album / Artist",
        -width => 15,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $cb2 = $p2f3->Checkbutton(-variable => \$genalb)->pack(-side => 'left');
    my $agLab = $p2f3->Label(-text => "Album / Genre",
        -width => 15,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $cb3 = $p2f3->Checkbutton(-variable => \$genart)->pack(-side => 'left');
    my $agLab2 = $p2f3->Label(-text => "Artist / Genre",
        -width => 15,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $sortLab = $p2f4->Label(-text => "Sort:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    my $aaLab2 = $p2f4->Label(-text => "Album / Artist",
        -width => 12,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $aaSortEntry = $p2f4->BrowseEntry(
        -width => 20,
        -variable => \$aaSort,
    )->pack(-fill => 'x', -side => 'left');
    $aaSortEntry->insert('end', 'Alphabetical', 'Chronological',
        'Reverse Chronological');
    if ($cfg{'nam'}{'aasort'} eq '1') {
        $aaSort = 'Chronological';
    } elsif ($cfg{'nam'}{'aasort'} eq '2') {
        $aaSort = 'Reverse Chronological';
    } else {
        $aaSort = 'Alphabetical';
    }

    my $anLab = $p2f5->Label(-text => "The Artist:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    my $artistTheEntry = $p2f5->BrowseEntry(
        -width => 20,
        -variable => \$artistThe,
    )->pack(-fill => 'x', -side => 'left');
    $artistTheEntry->insert('end', 'Artist',
        'Artist, The', 'The Artist');
    if ($cfg{'nam'}{'artistthe'} eq '1') {
        $artistThe = 'Artist, The';
    } elsif ($cfg{'nam'}{'artistthe'} eq '2') {
        $artistThe = 'Artist';
    } else {
        $artistThe = 'The Artist';
    }

    # Page 3

    my $naLab = $p3f1->Label(-text => "Neuros Audio:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $naEntry = $p3f1->Entry(-textvariable => \$audio,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $songsLab = $p3f2->Label(-text => "Songs:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $songsEntry = $p3f2->Entry(-textvariable => \$songs,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $playlistsLab = $p3f3->Label(-text => "Playlists:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $playlistsEntry = $p3f3->Entry(-textvariable => \$playlists,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $artistsLab = $p3f4->Label(-text => "Artists:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $artistsEntry = $p3f4->Entry(-textvariable => \$artists,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $albumsLab = $p3f5->Label(-text => "Albums:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $albumsEntry = $p3f5->Entry(-textvariable => \$albums,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $genresLab = $p3f6->Label(-text => "Genres:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $genresEntry = $p3f6->Entry(-textvariable => \$genres,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $yearsLab = $p3f7->Label(-text => "Years:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $yearsEntry = $p3f7->Entry(-textvariable => \$years,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');
    my $recordingsLab = $p3f8->Label(-text => "Recordings:",
        -width => 15,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left',-padx => 5);
    my $recordingsEntry = $p3f8->Entry(-textvariable => \$recordings,
    )->pack(-side => 'left',-expand => 1, -fill => 'x');

    my $okButton = $top->Button(
        -text => 'Ok',
        -command => sub {
            if (-d $mhEntry->get) {
                $cfg{'general'}{'musichome'} = $mhEntry->get;
            } else {
                undef $cfg{'general'}{'musichome'};
            }
            if (-d $rhEntry->get) {
                $cfg{'general'}{'recordingshome'} = $rhEntry->get;
            } else {
                undef $cfg{'general'}{'recordingshome'};
            }
            if ($ufEntry->get ne "") {
                $cfg{'general'}{'untaggedformat'} = $ufEntry->get;
            } else {
                undef $cfg{'general'}{'untaggedformat'};
            }
            if (-x $mpEntry->get) {
                $cfg{'general'}{'mediaplayer'} = $mpEntry->get;
            } else {
                undef $cfg{'general'}{'mediaplayer'};
            }
            if ($aaSort eq 'Chronological') {
                $cfg{'nam'}{'aasort'} = 1;
            } elsif ($aaSort eq 'Reverse Chronological') {
                $cfg{'nam'}{'aasort'} = 2;
            } else {
                $cfg{'nam'}{'aasort'} = 0;
            }
            if ($artistThe eq 'Artist, The') {
                $cfg{'nam'}{'artistthe'} = 1;
            } elsif ($artistThe eq 'Artist') {
                $cfg{'nam'}{'artistthe'} = 2;
            } else {
                $cfg{'nam'}{'artistthe'} = 0;
            }
            if ($audio ne "") { $cfg{'nam'}{'audio'} = $audio; }
            if ($songs ne "") { $cfg{'nam'}{'songs'} = $songs; }
            if ($playlists ne "") { $cfg{'nam'}{'playlists'} = $playlists; }
            if ($artists ne "") { $cfg{'nam'}{'artists'} = $artists; }
            if ($albums ne "") { $cfg{'nam'}{'albums'} = $albums; }
            if ($genres ne "") { $cfg{'nam'}{'genres'} = $genres; }
            if ($years ne "") { $cfg{'nam'}{'years'} = $years; }
            if ($recordings ne "") { $cfg{'nam'}{'recordings'} = $recordings; }
            $cfg{'general'}{'font'} = "$family $size";
            $cfg{'general'}{'usetags'} = $usetags;
            $cfg{'general'}{'normalize'} = $normalize;
            $cfg{'general'}{'duplicates'} = $duplicates;
            $cfg{'general'}{'md5sum'} = $md5sum;
            $cfg{'general'}{'abjust'} = $abjust;
            $cfg{'general'}{'albumdiff'} = $albumdiff;
            $cfg{'general'}{'artistdiff'} = $artistdiff;
            $cfg{'general'}{'genrediff'} = $genrediff;
            $cfg{'general'}{'titlediff'} = $titlediff;
            $cfg{'nam'}{'artistalbum'} = $artalb;
            $cfg{'nam'}{'genreartist'} = $genart;
            $cfg{'nam'}{'genrealbum'} = $genalb;
            $cfg{'nam'}{'menuorder'} = join ',',$menuList->get(0,'end');
            applyFont($bottom,$tree,$canvas,$family,$size);
            $cfg{'colors'} = $themes{$colorScheme};
            scheme('tree' => $tree, 'canvas' => $canvas);

            $tb->activebackground($cfg{'colors'}{'toolbarbg'});
            $parentTop->configure(-background => $cfg{'colors'}{'mainbg'});
            foreach my $child ($parentTop->children) {
                if ($child->class eq "Adjuster") {
                    $child->configure(-background => $cfg{'colors'}{'mainbg'});
                } elsif ($child->class eq "Frame") {
                    $child->configure(-background => $cfg{'colors'}{'mainbg'});
                    foreach my $gchild ($child->children) {
                        if ($gchild->class eq "Label") {
                            $gchild->configure(
                                -background => $cfg{'colors'}{'mainbg'},
                                -foreground => $cfg{'colors'}{'mainfg'},
                            );
                        } elsif ($gchild->class eq "Frame") {
                            foreach my $ggchild ($gchild->children) {
                                if ($ggchild->class eq "ROText") {
                                    $ggchild->configure(
                                        -background => $cfg{'colors'}{'mainbg'},
                                        -foreground => $cfg{'colors'}{'mainfg'},
                                    );
                                }
                            }
                        }
                    }
                }
            }

            my $style = $tree->ItemStyle('imagetext',
                -font => $font,
                -background => $cfg{'colors'}{'filebg'},
                -foreground => $cfg{'colors'}{'filefg'},
                -selectbackground => $cfg{'colors'}{'filebgsel'},
                -selectforeground => $cfg{'colors'}{'filefgsel'},
            );
            my @paths = getTreePaths($tree,undef);
            foreach my $path (@paths) {
                $tree->entryconfigure($path, -style => $style);
                $tree->selectionClear($path);
            }

            foreach my $child ($bottom->children) {
                if ($child->class eq "Frame") {
                    $child->configure(-background => $cfg{'colors'}{'mainbg'});
                }
            }
            $msg->configure(
                -background => $cfg{'colors'}{'mainbg'},
                -foreground => $cfg{'colors'}{'mainfg'},
            );
            configureStatusBar($bottom);
            setInfo($right);

            if (writeSoruneCfg(\%cfg) == -1) {
                message('ERR',"Could not save Sorune configuration.\n");
            }
            displayMenu($canvas,undef,undef);
            $top->destroy;
        },
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);
    my $helpButton = $top->Button(
        -text => 'Help',
        -command => [ \&gAbout, $main ],
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);
    my $cancelButton = $top->Button(
        -text => 'Cancel',
        -command => sub { $top->destroy; },
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);

    scheme('toplevel' => $top, 
        'frame' => $page1, 'frame' => $page2, 'frame' => $page3,
        'frame' => $p1f1, 'frame' => $p1f2, 'frame' => $p1f3, 'frame' => $p1f4,
        'frame' => $p1f5, 'frame' => $p1f6, 'frame' => $p1f7,
        'frame' => $p2f1, 'frame' => $p2f2, 'frame' => $p2f3, 'frame' => $p2f4, 
        'frame' => $p2f5, 'frame' => $p3f1, 'frame' => $p3f2, 'frame' => $p3f3,
        'frame' => $p3f4, 'frame' => $p3f5, 'frame' => $p3f6, 'frame' => $p3f7,
        'frame' => $p3f8,
        'button' => $b1, 'button' => $b2, 'button' => $b3, 
        'button' => $okButton, 'button' => $cancelButton,
        'button' => $downButton, 'button' => $upButton,
        'button' => $defaultButton, 'button' => $helpButton,
        'checkbutton' => $cb1, 'checkbutton' => $cb2,
        'checkbutton' => $cb3, 'checkbutton' => $cb4,
        'checkbutton' => $cb5, 'checkbutton' => $cb6,
        'checkbutton' => $cb7,
        'entry' => $mhEntry, 'entry' => $rhEntry, 'entry' => $ufEntry,
        'entry' => $mpEntry, 'entry' => $adEntry, 'entry' => $adEntry2,
        'entry' => $gdEntry, 'entry' => $tdEntry,
        'entry' => $naEntry, 'entry' => $songsEntry,
        'entry' => $playlistsEntry, 'entry' => $artistsEntry,
        'entry' => $albumsEntry, 'entry' => $genresEntry,
        'entry' => $yearsEntry, 'entry' => $recordingsEntry,
        'browseentry' => $csEntry, 'browseentry' => $famEntry,
        'browseentry' => $sizeEntry, 'browseentry' => $aaSortEntry,
        'browseentry' => $abjEntry, 'browseentry' => $artistTheEntry,
        'label' => $mhLab, 'label' => $rhLab, 'label' => $ufLab,
        'label' => $mpLab, 'label' => $themeLab, 'label' => $nmLab,
        'label' => $aaLab, 'label' => $agLab, 'label' => $agLab2,
        'label' => $sortLab, 'label' => $aaLab2, 'label' => $anLab,
        'label' => $miscLab, 'label' => $utLab, 'label' => $normLab,
        'label' => $vtLab, 'label' => $moLab, 'label' => $blankLab,
        'label' => $tcLab, 'label' => $adLab, 'label' => $adLab2,
        'label' => $gdLab, 'label' => $tdLab,
        'label' => $naLab, 'label' => $songsLab,
        'label' => $playlistsLab, 'label' => $artistsLab,
        'label' => $albumsLab, 'label' => $genresLab,
        'label' => $yearsLab, 'label' => $recordingsLab,
        'label' => $duplicatesLab,
        'listbox' => $menuList,
        'notebook' => $book,
    );

    $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; });

    relatePos($main,$top,640,325,1,0);
    $okButton->focus;
}

sub gAbout($)
{
    my $main = shift;
    my $top = $main->Toplevel(-title => 'About/Help');

    my $b = $top->Button(
        -image =>'sorune-logo',
        -background => 'black',
        -activebackground => 'black',
        -relief => 'flat',
        -borderwidth => 0,
        -command => sub { startBrowser('http://www.sorune.com'); },
    );
    $b->pack(-side => 'top', -pady => 5);

    my $l1 = $top->Label(
        -text => 'Version ' . getVersion(),
    )->pack(-side => 'top', -fill => 'x');
    my $l2 = $top->Label(
        -text => 'Copyright  2004-2005 Darren Smith, All Rights Reserved.',
    )->pack(-side => 'top', -fill => 'x');
    my $b2 = $top->Button(
        -font => $font,
        -background => $cfg{'colors'}{'buttonbg'},
        -activebackground => $cfg{'colors'}{'buttonbg'},
        -foreground => $cfg{'colors'}{'buttonfg'},
        -activeforeground => $cfg{'colors'}{'buttonfg'},
        -relief => 'flat',
        -borderwidth => 0,
        -text => 'Sorune Group: http://groups.yahoo.com/group/sorune',
        -command => sub { startBrowser('http://groups.yahoo.com/group/sorune'); },
    );
    $b2->pack(-side => 'top');
    my $b3 = $top->Button(
        -font => $font,
        -background => $cfg{'colors'}{'buttonbg'},
        -activebackground => $cfg{'colors'}{'buttonbg'},
        -foreground => $cfg{'colors'}{'buttonfg'},
        -activeforeground => $cfg{'colors'}{'buttonfg'},
        -relief => 'flat',
        -borderwidth => 0,
        -text => 'Sorune Forums: http://forums.sorune.com',
        -command => sub { startBrowser('http://forums.sorune.com'); },
    );
    $b3->pack(-side => 'top');


    my $help = $top->Scrolled('ROText',
        -scrollbars => 'osoe',
        -font => $font,
        -height => 1,
        -relief => 'flat',
        -borderwidth => 0,
        -insertwidth => 0,
        -takefocus => 0,
        -wrap => 'word',
        -highlightthickness => 0,
        -exportselection => 1,
        -background => $cfg{'colors'}{'menubg'},
        -selectbackground => $cfg{'colors'}{'menubg'},
        -foreground => $cfg{'colors'}{'menufg'},
        -selectforeground => $cfg{'colors'}{'menufg'},
    )->pack(-side => 'top',-fill => 'both',-expand => 1);
    my $scrollbar = $help->Subwidget('yscrollbar');
    scheme('scrollbar' => $scrollbar, 'label' => $l1, 'label' => $l2);
    if ($^O eq 'MSWin32') {
        $help->bind('<MouseWheel>', [sub {
            $_[0]->yview(scroll => -($_[1]/120) * 5, 'units')
        },Ev('D')]);
    } else {
        $help->bind('<Button-4>', sub {
            $help->yview(scroll => -5, 'units');
        });
        $help->bind('<Button-5>', sub {
            $help->yview(scroll => 5, 'units');
        });
    }

    $help->imageCreate('end', -image => 'info');
    $help->insert('end',"  Nomenclature
The file browser is on the left side and the audio browser is on the right side. After a Sorune database is created by either rebuilding or adding songs, you may navigate through your audio collection via the audio browser. Single clicks in the audio browser will select items. Double clicking will take you to the corresponding submenu. Right clicking will take you back one level. You may also navigate backwards via the history trail above the audio browser.

");
    $help->imageCreate('end', -image => 'config');
    $help->insert('end',"  Configuration Options
Music Home - Set this to the directory where your music collection is stored.  Sorune will remove this path from the music directory on your Neuros when transferring files.

Example:
    PC Music Layout: C:/My Documents/My Music/Artist/Album/01-Song.mp3
    Neuros Location: F:

    -- Setting #1 --
    Music Home: <blank>
    Neuros location will be: F:/MUSIC/My Documents/My Music/Artist/Album/01-Song.mp3

    -- Setting #2 --
    Music Home: C:/My Documents/My Music
    Neuros location will be: F:/MUSIC/Artist/Album/01-Song.mp3

Recordings Home - Set this if you want Sorune to move your recordings to the specified directory during a sync.

Untagged Format - Set this if you maintain a logical directory structure and have files that are not tagged. Settings include \%ARTIST\%, \%ALBUM\%, \%TITLE\%, \%GENRE\%, \%TRACKNUMBER\% and \%YEAR\%.

Media Player - Set this to your preferred media player.
 
Theme/Font/Justification - Customize Sorune's interface.

Tag Compare - This feature allows similar tags to be grouped together. For example, say you have the following genres: \"Trip Hop\", \"Trip-Hop\" and \"TripHop\". Setting the Tag Compare Genre to 1 will group all of these genres together since they all differ by 1 character. This comparison is case insensitive.

Use Tags - Read ID3 or equivalent tags from audio files. If you don't want to use tags, uncheck this option and see the Untagged Format specifier above.

Normalize - Titlecase all tags in the database and convert underscores to spaces. (Note: This is done during addition of tracks. Clear and rebuild database if necessary.)

Duplicates - Append artist and if necessary date tags to duplicate album entries in order to make them unique.

Verify Transfer - Confirm that the Neuros copy is identical to the version on your computer.

Menu Order - Change the audio browser menu order. (Note: This does not change the audio menu order on the Neuros as that is not yet supported in firmware.)

Neuros Menus - Choose which sub-menus to create on the Neuros. (Note: Choosing multiple sub-menus causes issues with navigation on the Neuros. These issues are firmware related and are being addressed. It is recommended to only enable one sub-menu at this time.)

Sort - Choose how to sort your albums when navigating through the album under artist menus.

The Artist - Choose how to handle initial \"The\" in artist names. (Note: This is done during addition of tracks. Clear and rebuild database if necessary.)

");
    $help->imageCreate('end', -image => 'add');
    $help->insert('end',"  Add Music
Select items to add from the file browser. Click the Add to Neuros button. After you've finished adding music, click the Synchronize button.

");
    $help->imageCreate('end', -image => 'delete');
    $help->insert('end',"  Delete Music
Select items to delete either from the file browser or the audio browser.  Click the Remove from Neuros button. After you've finished removing music, click the Synchronize button.

");
    $help->imageCreate('end', -image => 'sync');
    $help->insert('end',"  Synchronize
Performs all add and delete operations and then creates the Neuros databases. If you have specified a recordings home directory (see above), synchronize will also move all recordings to that directory.

");
    $help->imageCreate('end', -image => 'play');
    $help->insert('end',"  Play Music
Select items for playing from the audio or file browser and click the Play button.

");
    $help->imageCreate('end', -image => 'playlist');
    $help->insert('end',"  Create or Add to a Playlist
Select items for the playlist in the audio or file browser and click the Create/Add to Playlist button. Type the name of the playlist and whether or not to randomize the playlist. After you've finished creating playlists, click the Synchronize button.

");
    $help->imageCreate('end', -image => 'rescan');
    $help->insert('end',"  Rescan
If you start Sorune without your Neuros attached, most items will be unavailable.  After you've attached your Neuros, click the rescan button to discover it.

");
    $help->imageCreate('end', -image => 'rebuild');
    $help->insert('end',"  Rebuild
Search through the music directory on your Neuros for new or removed audio files to add to the Neuros menus. If your database has become corrupted, you may want to clear it before doing a rebuild (see the Reset/Clear button). This will be sure to update all files and not just those that are new or missing.

");
    $help->imageCreate('end', -image => 'reset');
    $help->insert('end',"  Reset/Clear
Reset removes all files from the Neuros. Clear removes the databases only.

");
    $help->imageCreate('end', -image => 'clear');
    $help->insert('end',"  Clear Additions/Deletions
Clear all pending Add and Delete operations.

");
    $help->imageCreate('end', -image => 'duplicates');
    $help->insert('end',"  Find Duplicates
Searches the database for duplicate songs looking at the keys specified. Select Artist and Title for a more common search or for a more exact match try Size and Length.

");
    $help->imageCreate('end', -image => 'importexport');
    $help->insert('end',"  Import/Export
Import or Export the Sorune database from or to a text file stored in tab separated format.

");
    $help->imageCreate('end', -image => 'backup');
    $help->insert('end',"  Backup
Backup the Neuros databases. Only the last backup is maintained.

");
    $help->imageCreate('end', -image => 'restore');
    $help->insert('end',"  Restore
Restore the Neuros databases from backup.

");

    my $okButton = $top->Button(-text => 'Ok',
        -command => sub { $top->destroy; },
    )->pack(-side => 'bottom', -pady => 5);

    $top->bind('<Return>', sub { $okButton->invoke; });

    scheme('toplevel' => $top, 'button' => $okButton,
        'scrollbar' => $scrollbar);
    $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; });

    relatePos($main,$top,
        int($main->width() - ($main->width() / 10)),
        int($main->height() - ($main->height() / 10)),
        1,0);
    $okButton->focus;
}

sub gExport($$$$)
{
    my ($main,$tree,$canvas,$bottom) = @_;

    my $top = $main->Toplevel(-title => 'Import/Export Database');
    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $lab1 = $f1->Label(-text => 'Filename:')->pack(-side => 'left');
    my $e = $f1->Entry(-textvariable => \$exportFilename,
    )->pack(-side => 'left', -expand => 1, -fill => 'x');
    my $b1 = $f1->Button(-text => "Browse",
        -command => sub {
            (my $dir = getHome()) =~ s/\\/\//g;
            if (defined $tree) {
                $tree->selectionClear;
            }
            dirTree($main,$top,$dir,\$exportFilename,1);
        },
    )->pack(-side => 'left', -padx => 5);
    my $started = 0;

    my $importButton = $top->Button(
        -text => 'Import',
        -command => sub {
            if (defined $exportFilename) {
                if ($started) { return; }
                $started = 1;
                my $musicHome = undef;
                if (defined $cfg{'general'}{'musichome'}) {
                    $musicHome = $cfg{'general'}{'musichome'};
                }
                if (-d $exportFilename) { $exportFilename .= "/sorunedb.txt"; }
                my $status = importDb($exportFilename,\%musicDb,\%playlistDb,
                    \%recordDb,\%cfg,$neurosHome,$musicHome,$neurosDrive);
                if ($status == 0) {
                    %info = ();
                    displayMenu($canvas,undef,undef);
                    updateTreeIcons($tree,undef);
                    configureStatusBar($bottom);
                    message('INFO',"Database imported from $exportFilename.\n");
                } else {
                    message('INFO',"Database import failed.\n");
                }
                $top->destroy;
            }
        },
    )->pack(-side => 'left', -padx => 25, -pady => 5, -expand => 1);

    my $exportButton = $top->Button(
        -text => 'Export',
        -command => sub {
            if (defined $exportFilename) {
                if ($started) { return; }
                $started = 1;
                if (-d $exportFilename) { $exportFilename .= "/sorunedb.txt"; }
                my $status = exportDb($exportFilename,\%musicDb,\%recordDb);
                if ($status == 0) {
                    message('INFO',"Database exported to $exportFilename.\n");
                } else {
                    message('INFO',"Database export failed.\n");
                }
                $top->destroy;
            }
        },
    )->pack(-side => 'left', -padx => 25, -pady => 5, -expand => 1);

    my $cancelButton = $top->Button(
        -text => 'Cancel',
        -command => sub { if (!$started) { $top->destroy; } },
    )->pack(-side => 'right', -padx => 25, -pady => 5, -expand => 1);

    scheme('toplevel' => $top, 'frame' => $f1, 'label' => $lab1,
        'entry' => $e, 'button' => $b1, 'button' => $importButton,
        'button' => $exportButton, 'button' => $cancelButton);
    $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; });

    relatePos($main,$top,500,100,1,0);
    $importButton->focus;
}

sub gExit($$)
{
    my ($main,$tree) = @_;

    my $top = $main->Toplevel(-title => 'Confirmation');
    if ($syncNeeded) {
        $top->Label(-text => 'Synchronization needed.',
            -font => $font,
            -background => $cfg{'colors'}{'menubg'},
            -foreground => 'red',
        )->pack;
    }
    my $lab1 = $top->Label(-text => 'Exit?')->pack;

    my $okButton = $top->Button(
        -text => 'Ok',
        -command => sub {
            $cfg{'general'}{'geometry'} = $main->geometry;
            $cfg{'general'}{'treewidth'} = $tree->width;
            writeSoruneCfg(\%cfg);
            $main->destroy;
        },
    )->pack(-side => 'left', -padx => 10, -expand => 1);

    $top->bind('<Return>', sub { $okButton->invoke; });

    my $cancelButton = $top->Button(
        -text => 'Cancel',
        -command => sub { $top->destroy; },
    )->pack(-side => 'right', -padx => 10, -expand => 1);

    scheme('toplevel' => $top, 'label' => $lab1, 'button' => $okButton,
        'button' => $cancelButton);
    $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; });

    relatePos($main,$top,200,75,1,0);
    $okButton->focus;
}

sub gDuplicates($$)
{
    my ($main,$canvas) = @_;

    my $top = $main->Toplevel(-title => 'Find Duplicates');
    my $percentDone = 0;
    my $halt = 0;
    my $started = 0;

    my $f0 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f2 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f3 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f4 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');

    my $label = $f0->Label(-text =>
        "Find duplicates based on these fields matching:",
        -width => 40, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb1 = $f1->Checkbutton(-variable => \$artistD)->pack(-side => 'left');
    my $lab1 = $f1->Label(-text => "Artist", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb2 = $f1->Checkbutton(-variable => \$albumD)->pack(-side => 'left');
    my $lab2 = $f1->Label(-text => "Album", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb3 = $f1->Checkbutton(-variable => \$genreD)->pack(-side => 'left');
    my $lab3 = $f1->Label(-text => "Genre", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $cb4 = $f2->Checkbutton(-variable => \$titleD)->pack(-side => 'left');
    my $lab4 = $f2->Label(-text => "Title", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb5 = $f2->Checkbutton(-variable => \$yearD)->pack(-side => 'left');
    my $lab5 = $f2->Label(-text => "Year", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb6 = $f2->Checkbutton(-variable => \$tracknumD)->pack(-side => 'left');
    my $lab6 = $f2->Label(-text => "Track Number", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $cb7 = $f3->Checkbutton(-variable => \$formatD)->pack(-side => 'left');
    my $lab7 = $f3->Label(-text => "Format", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb8 = $f3->Checkbutton(-variable => \$lengthD)->pack(-side => 'left');
    my $lab8 = $f3->Label(-text => "Length", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $cb9 = $f3->Checkbutton(-variable => \$sizeD)->pack(-side => 'left');
    my $lab9 = $f3->Label(-text => "Size", -width => 10, -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $tcLab = $f4->Label(-text => "Tag Compare:",
        -width => 12,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left', -padx => 5);
    my $adEntry = $f4->Entry(-textvariable => \$albumdiffD,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $adLab = $f4->Label(-text => "Album",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $adEntry2 = $f4->Entry(-textvariable => \$artistdiffD,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $adLab2 = $f4->Label(-text => "Artist",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $gdEntry = $f4->Entry(-textvariable => \$genrediffD,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $gdLab = $f4->Label(-text => "Genre",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');
    my $tdEntry = $f4->Entry(-textvariable => \$titlediffD,
        -width => 2,
        -justify => 'right',
    )->pack(-side => 'left');
    my $tdLab = $f4->Label(-text => "Title",
        -width => 7,
        -justify => 'left',
        -anchor => 'w',
    )->pack(-side => 'left');

    my $progress = $top->ProgressBar(
        -length => 500,
        -blocks => 500,
        -variable => \$percentDone
    )->pack(-fill => 'x', -side => 'top');

    my $findButton = $top->Button(-text => 'Find',
        -command => sub {
            if ($started) { return; }
            $started = 1;
            my $startTime = time;
            my @fields = ();
            if ($artistD == 1 or $artistdiffD > 0) { push @fields, 'artist'; }
            if ($albumD == 1 or $albumdiffD > 0) { push @fields, 'album'; }
            if ($genreD == 1 or $genrediffD > 0) { push @fields, 'genre'; }
            if ($titleD == 1 or $titlediffD > 0) { push @fields, 'title'; }
            if ($yearD == 1) { push @fields, 'date'; }
            if ($tracknumD == 1) { push @fields, 'tracknumber'; }
            if ($formatD == 1) { push @fields, 'format'; }
            if ($lengthD == 1) { push @fields, 'length'; }
            if ($sizeD == 1) { push @fields, 'size'; }

            # find duplicates
            my @files = sort keys %musicDb;
            my $fileCount = scalar @files;
            my @dups = ();
            my $found;
            for (my $i = 0; $i < $fileCount; $i++) {
                $percentDone = ($i+1) * 100 / $fileCount;
                $top->update;
                if ($halt) { last; }
                for (my $j = $i+1; $j < $fileCount; $j++) {
                    $found = 1;
                    foreach my $field (@fields) {
                        if ($field eq 'album' and $albumdiffD > 0) {
                            if (stridiff($musicDb{$files[$i]}{'album'},
                                $musicDb{$files[$j]}{'album'}) > $albumdiffD) {
                                $found = 0;
                                last;
                            }
                        } elsif ($field eq 'artist' and $artistdiffD > 0) {
                            if (stridiff($musicDb{$files[$i]}{'artist'},
                                $musicDb{$files[$j]}{'artist'}) > $artistdiffD) {
                                $found = 0;
                                last;
                            }
                            if (defined $musicDb{$files[$i]}{'artist2'} and
                                defined $musicDb{$files[$i]}{'artist2'}) {
                                if (stridiff($musicDb{$files[$i]}{'artist2'},
                                    $musicDb{$files[$j]}{'artist2'}) > $artistdiffD) {
                                    $found = 0;
                                    last;
                                }
                            } elsif (defined $musicDb{$files[$i]}{'artist2'} or
                                defined $musicDb{$files[$i]}{'artist2'}) {
                                $found = 0;
                                last;
                            }
                        } elsif ($field eq 'genre' and $genrediffD > 0) {
                            if (stridiff($musicDb{$files[$i]}{'genre'},
                                $musicDb{$files[$j]}{'genre'}) > $genrediffD) {
                                $found = 0;
                                last;
                            }
                        } elsif ($field eq 'title' and $titlediffD > 0) {
                            if (stridiff($musicDb{$files[$i]}{'title'},
                                $musicDb{$files[$j]}{'title'}) > $titlediffD) {
                                $found = 0;
                                last;
                            }
                        } elsif ($musicDb{$files[$i]}{$field} ne 
                            $musicDb{$files[$j]}{$field}) {
                            $found = 0;
                            last;
                        }
                    }
                    if ($found) {
                        my ($foundi,$foundj) = (0,0);
                        foreach my $file (@dups) {
                            if ($file eq $files[$i]) { $foundi = 1; }
                            if ($file eq $files[$j]) { $foundj = 1; }
                        }
                        if (!$foundi) { push @dups, $files[$i]; }
                        if (!$foundj) { push @dups, $files[$j]; }
                    }
                }
            }
           
            # draw menus 
            my @names = ();
            my $name;
            my $dupCount = scalar @dups;
            if (!$halt and $dupCount) {
                foreach my $key (sort {
                    $musicDb{$a}{'title'} cmp $musicDb{$b}{'title'}
                } @dups) {
                    $name = 
                        "title|undef|0|NULL|$musicDb{$key}{'title'}|$key|undef";
                    getColor(\$name);
                    push @names, $name;
                }

                my $cmd = pop @currentCmd;
                my @parts = split /\|/, $cmd;
                if (!($parts[0] eq "title" and $parts[2] eq "0" and 
                    $parts[4] eq "Duplicates")) {
                    push @currentCmd, $cmd;
                }
                push @currentCmd, "title|undef|0|undef|Duplicates|undef|undef";
                drawMenus($canvas,\@names,undef);
                setBcText($canvas);

                message('INFO',"$dupCount possible duplicates found (" . 
                    getTime(time - $startTime) . ")\n");
            } else {
                if ($halt) {
                    message("INFO","Find duplicates aborted.\n");
                } else {
                    message("INFO","No duplicates found.\n");
                }
            }

            $top->destroy;
        },
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);

    my $cancelButton = $top->Button(-text => 'Cancel',
        -command => sub {
            if ($started) {
                $halt = 1;
                $top->update;
            } else {
                $top->destroy;
            }
        },
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);

    scheme('toplevel' => $top,
        'frame' => $f1, 'frame' => $f2, 'frame' => $f3, 'frame' => $f4,
        'label' => $lab1, 'label' => $lab2, 'label' => $lab3,
        'label' => $lab4, 'label' => $lab5, 'label' => $lab6,
        'label' => $lab7, 'label' => $lab8, 'label' => $lab9,
        'label' => $label, 'label' => $tcLab, 'label' => $adLab,
        'label' => $adLab2, 'label' => $gdLab, 'label' => $tdLab,
        'button' => $cancelButton, 'button' => $findButton,
        'checkbutton' => $cb1, 'checkbutton' => $cb2, 'checkbutton' => $cb3,
        'checkbutton' => $cb4, 'checkbutton' => $cb5, 'checkbutton' => $cb6,
        'checkbutton' => $cb7, 'checkbutton' => $cb8, 'checkbutton' => $cb9,
        'entry' => $adEntry, 'entry' => $adEntry2, 'entry' => $gdEntry,
        'entry' => $tdEntry,
        'progressbar' => $progress,
    );
    $top->protocol('WM_DELETE_WINDOW', sub {
        if ($started) {
            $halt = 1;
            $top->update;
        } else {
            $top->destroy;
        }
    });

    relatePos($main,$top,500,200,1,0);
    $cancelButton->focus;
    $top->update;
}

sub gSync($$$$)
{
    my ($main,$tree,$canvas,$bottom) = @_;

    my $freeSpace = getFreeSpace($neurosHome);
    my $xferSize = getXferSize(\%musicDb,\%recordDb);

    # subtract 16MB from free space to account for databases/playlists
    if (($freeSpace - (16 * 1024 * 1024))  < $xferSize) {
        message('ERR',"Not enough free space on Neuros to complete sync, aborting.\n");
        return;
    }

    my $top = $main->Toplevel(-title => 'Synchronize');
    my $percentDone = 0;
    my $halt = 0;
    my $status = "";

    my ($artist,$album,$title,$genre) = ('Artist:','Album:','Title:','Genre:');
    my ($et,$tr,$xr) = ('Elapsed:','Remaining:','Transfer Rate:');
    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f2 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f3 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f4 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f5 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f6 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f7 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');

    my $lab1 = $f1->Label(-textvariable => \$artist,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $lab2 = $f2->Label(-textvariable => \$title,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $lab3 = $f3->Label(-textvariable => \$album,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $lab4 = $f4->Label(-textvariable => \$genre,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $progress = $f5->ProgressBar(
        -length => 500,
        -blocks => 500,
        -variable => \$percentDone
    )->pack(-fill => 'x', -expand => 1, -side => 'top');

    my $lab5 = $f6->Label(-textvariable => \$et,
        -width => 15,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $lab6 = $f6->Label(-textvariable => \$tr,
        -width => 15,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');
    my $lab7 = $f6->Label(-textvariable => \$xr,
        -width => 15,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $sLabel = $f7->Label(-textvariable => \$status,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $cancelButton = $top->Button(-text => 'Cancel',
        -command=> sub { $halt = 1; $top->update; }
    )->pack(-side => 'bottom', -pady => 5);

    scheme('toplevel' => $top, 'frame' => $f1, 'frame' => $f2,
        'frame' => $f3, 'frame' => $f4, 'frame' => $f5, 'frame' => $f6,
        'frame' => $f7,
        'label' => $lab1, 'label' => $lab2, 'label' => $lab3,
        'label' => $lab4, 'label' => $lab5, 'label' => $lab6,
        'label' => $sLabel, 'label' => $lab7,
        'progressbar' => $progress, 'button' => $cancelButton);
    $top->protocol('WM_DELETE_WINDOW', sub { $halt = 1; $top->update; });

    relatePos($main,$top,500,200,1,0);
    $cancelButton->focus;
    $top->update;

    my $startTime = time;
    my @addFiles = sort {
        sprintf("%s-%s-%03d",
            $musicDb{$a}{'artist'},
            $musicDb{$a}{'album'},
            $musicDb{$a}{'tracknumber'},
        )
        cmp
        sprintf("%s-%s-%03d",
            $musicDb{$b}{'artist'},
            $musicDb{$b}{'album'},
            $musicDb{$b}{'tracknumber'},
        )
    } (list(\%musicDb,\%recordDb,$neurosHome,'add'));
    my @delFiles = sort(list(\%musicDb,\%recordDb,$neurosHome,'delete'));
    my $addFileCount = scalar @addFiles;
    my $delFileCount = scalar @delFiles;
    my $totalCount = $addFileCount + $delFileCount;
    my ($xferTime,$xferElapsed,$xferBytesTotal,$xferBytes) = (time,0,0,0);
    for (my $i = 0; $i < $addFileCount; $i++) {
        $xferBytesTotal += $musicDb{$addFiles[$i]}{'size'};
    }
    my $tb;

    # delete files
    if (!$halt) {
        $status = "Removing files from Neuros";
        $top->update;
        for (my $i = 0; $i < $delFileCount; $i++) {
            $tb = "Sorune (" . ($i + 1 + $addFileCount) . "/$totalCount)";
            $main->configure(-title => $tb);
            syncFile('delete',$delFiles[$i],\%musicDb,\%playlistDb,
                \%recordDb,\%cfg,$neurosHome,$top);
            if ($halt) { last; }
            $percentDone = (($i+1) + $addFileCount) * 100 / $totalCount;
            $top->update;
        }
    }
  
    # handle recordings
    my $syncFileCount = 0;
    if (!$halt) {
        $status = "Transferring recordings";
        $top->update;
        $syncFileCount = syncRecordings(\%recordDb,\%cfg,$neurosHome,
            $neurosDrive,$top,\$halt);
    }

    # add files 
    if (!$halt) {
        $status = "Transferring files to Neuros";
        for (my $i = 0; $i < $addFileCount; $i++) {
            $tb = "Sorune (" . ($i + 1) . "/$totalCount)";
            $main->configure(-title => $tb);
            $artist = "Artist: $musicDb{$addFiles[$i]}{'artist'}";
            $album = "Album: $musicDb{$addFiles[$i]}{'album'}";
            $title = "Title: $musicDb{$addFiles[$i]}{'title'}";
            $genre = "Genre: $musicDb{$addFiles[$i]}{'genre'}";
            if ($xferBytes > 0 and $xferElapsed > 0) {
                $tr = "Remaining: " . 
                    getTime(($xferBytesTotal - $xferBytes) /
                    ($xferBytes / $xferElapsed));
                $et = "Elapsed: " . getTime($xferElapsed);
                $xr = "Transfer Rate: " . getSize($xferBytes/$xferElapsed) . "/s";
            }
            if ($halt) { last; }
            $top->update;
            syncFile('add',$addFiles[$i],\%musicDb,\%playlistDb,\%recordDb,
                \%cfg,$neurosHome,$top);
            if ($halt) { last; }
            $percentDone = ($i+1) * 100 / $totalCount;
            $xferBytes += $musicDb{$addFiles[$i]}{'size'};
            $xferElapsed = time - $xferTime;
        }
    }

    # create databases
    if (!$halt) {
        $status = "Creating Neuros databases";
        $top->update;
        my $ndbHome = "$neurosHome/woid_db/audio";
        createNAM(\%cfg,\%musicDb,\%playlistDb,\%recordDb,$ndbHome,
            $fwVersion,$top);
    }

    # export playlists
    if (!$halt) {
        $status = "Exporting playlists";
        $top->update;
        createPlaylists($neurosHome,\%musicDb,\%playlistDb);
    }

    if ($addFileCount or $delFileCount or $syncFileCount) {
        %info = ();
        my $cmd = pop @currentCmd;
        my @parts = split /\|/, $cmd;
        if ($parts[4] eq "Duplicates" and ($delFileCount or $syncFileCount)) {
            $cmd = pop @currentCmd;
        }
        displayMenu($canvas,$cmd,undef);
        updateTreeIcons($tree,undef);
    }
    $status = "Creating Sorune database";
    $top->update;
    writeSoruneDb($soruneDb,\%musicDb,\%playlistDb,\%recordDb,\%cfg);
    configureStatusBar($bottom);
    $percentDone = 100;
    if ($halt) {
        message('INFO',"Neuros sync aborted.\n");
        $syncNeeded = 1;
    } else {
        message('INFO',"Neuros sync complete (" . 
            getTime(time - $startTime) . ")\n");
        $syncNeeded = 0;
    }
    $main->configure(-title => "Sorune");
    $top->destroy;
}

sub gReset($$$$)
{
    my ($main,$tree,$canvas,$bottom) = @_;

    my $top = $main->Toplevel(-title => 'Reset/Clear Neuros');
    my $percentDone = 0;
    my $halt = 0;
    my $started = 0;
    my $ndbHome = "$neurosHome/woid_db/audio";
    my $status = "";

    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f2 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f3 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f4 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $progress = $f1->ProgressBar(
        -length => 300,
        -blocks => 300,
        -variable => \$percentDone
    )->pack(-fill => 'x', -expand => 1);

    my $lab1 = $f2->Label(
        -text => 'Reset will erase all files from the Neuros.',
    )->pack();
    my $lab2 = $f3->Label(-text => 'Clear will only erase the database.',
    )->pack();
    my $sLabel = $f4->Label(-textvariable => \$status,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $resetButton = $top->Button(-text => 'Reset',
        -command => sub {
            if ($started) { return; }
            $started = 1;
            $top->update;
            my @findFiles = ();
            my $startTime = time;
            $status = "Scanning for audio files";
            $top->update;
            finddepth({wanted => sub { push @findFiles, $File::Find::name; },
                no_chdir => 1, postprocess => sub {
                    $status = "Scanning $File::Find::dir";
                    $top->update;
                }},
                $neurosHome);
            $status = "Removing files";
            if (!$halt) {
                my $fileCount = scalar @findFiles;
                for (my $i = 0; $i < $fileCount; $i++) {
                    if (-f $findFiles[$i]) {
                        if ($findFiles[$i] ne "$neurosHome/bkpk.sn" and
                            $findFiles[$i] ne "$neurosHome/sn.txt" and
                            $findFiles[$i] ne "$neurosHome/version.txt") {
                            unlink $findFiles[$i];
                        }
                    } else {
                        rmdir $findFiles[$i];
                    }
                    if ($halt) { last; }
                    $percentDone = ($i+1) * 100 / $fileCount;
                    $top->update;
                }
                %musicDb = ();
                %recordDb = ();
                %playlistDb = ();
                createNAM(\%cfg,\%musicDb,\%playlistDb,\%recordDb,$ndbHome,
                    $fwVersion,$top);
                %info = ();
                displayMenu($canvas,undef,undef);
                updateTreeIcons($tree,undef);
                configureStatusBar($bottom);
                writeSoruneDb($soruneDb,\%musicDb,\%playlistDb,\%recordDb,
                    \%cfg);
                $percentDone = 100;
                $top->update;
            }
            if ($halt) {
                message('INFO',"Neuros reset and sync aborted.\n");
            } else {
                message('INFO',"Neuros reset and sync complete (" . 
                    getTime(time - $startTime) . ")\n");
            }
            $top->destroy;
            $syncNeeded = 0;
        }
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);

    my $clearButton = $top->Button(-text => 'Clear',
        -command => sub {
            my $startTime = time;
            if ($started) { return; }
            $started = 1;
            $status = "Clearing databases";
            $top->update;
            %musicDb = ();
            %recordDb = ();
            %playlistDb = ();
            createNAM(\%cfg,\%musicDb,\%playlistDb,\%recordDb,$ndbHome,
                $fwVersion,$top);
            %info = ();
            displayMenu($canvas,undef,undef);
            updateTreeIcons($tree,undef);
            configureStatusBar($bottom);
            writeSoruneDb($soruneDb,\%musicDb,\%playlistDb,\%recordDb,\%cfg);
            $percentDone = 100;
            $top->update;
            message('INFO',"Neuros clear and sync complete (" . 
                getTime(time - $startTime) . ")\n");
            $top->destroy;
            $syncNeeded = 0;
        }
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);

    my $cancelButton = $top->Button(-text => 'Cancel',
        -font => $font,
        -background => $cfg{'colors'}{'buttonbg'},
        -activebackground => $cfg{'colors'}{'buttonbg'},
        -foreground => $cfg{'colors'}{'buttonfg'},
        -activeforeground => $cfg{'colors'}{'buttonfg'},
        -highlightthickness => 0,
        -command => sub {
            if ($started) {
                $halt = 1;
                $top->update;
            } else {
                $top->destroy;
            }
        },
    )->pack(-side => 'left', -padx => 20, -pady => 5, -expand => 1);

    scheme('frame' => $f1, 'frame' => $f2, 'frame' => $f3, 'frame' => $f4,
        'toplevel' => $top, 'progressbar' => $progress,
        'label' => $lab1, 'label' => $lab2, 'label' => $sLabel,
        'button' => $resetButton,
        'button' => $clearButton, 'button' => $cancelButton);
    $top->protocol('WM_DELETE_WINDOW', sub {
        if ($started) {
            $halt = 1;
            $top->update;
        } else {
            $top->destroy;
        }
    });

    relatePos($main,$top,500,150,1,0);
    $cancelButton->focus;
    $top->update;
}

sub gRebuild($$$$)
{
    my ($main,$tree,$canvas,$bottom) = @_;

    my $top = $main->Toplevel(-title => 'Rebuild Database');
    my $percentDone = 0;
    my $halt = 0;
    my $ndbHome = "$neurosHome/woid_db/audio";
    my @findFiles = ();
    my $status = "Scanning for audio files";

    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f2 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f3 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $progress = $f1->ProgressBar(
        -length => 300,
        -blocks => 300,
        -variable => \$percentDone
    )->pack(-fill => 'x', -expand => 1);

    my $sLabel = $f2->Label(-textvariable => \$status,
        -anchor => 'w',
    )->pack(-side => 'left', -padx => 5, -expand => 1, -fill => 'x');

    my $cancelButton = $f3->Button(-text => 'Cancel',
        -command=> sub { $halt = 1; $top->update; }
    )->pack(-side => 'bottom', -pady => 5);

    scheme('frame' => $f1, 'frame' => $f2, 'frame' => $f3,
        'label' => $sLabel, 'toplevel' => $top, 'progressbar' => $progress,
        'button' => $cancelButton);
    $top->protocol('WM_DELETE_WINDOW', sub { $halt = 1; $top->update; });

    relatePos($main,$top,500,100,1,0);
    $cancelButton->focus;
    $top->update;

    # Add music files
    my $startTime = time;
    my $neurosMusicHome = "$neurosHome/music";
    if (-d $neurosMusicHome) {
        find({no_chdir => 1, wanted => sub {
            if (/\.ogg$/i or /\.mp3$/i or /\.wma$/i or /\.wav$/i or /\.m3u$/i) {
                push @findFiles, $File::Find::name;
            }
        }, postprocess => sub {
            $status = "Scanning $File::Find::dir";
            $top->update;
        }}, $neurosMusicHome);
    }

    $status = "Rebuilding databases";
    $top->update;
    if (!$halt) {
        my $fileCount = scalar @findFiles;
        for (my $i = 0; $i < $fileCount; $i++) {
            addFile($findFiles[$i],$neurosMusicHome,$neurosHome,\%musicDb,
                \%playlistDb,\%cfg,$neurosDrive,0);
            if ($halt) { last; }
            $percentDone = ($i+1) * 100 / $fileCount;
            $top->update;
        }
    }


    # Remove music files that no longer exist from the db
    if (!$halt) {
        my $found;
        foreach my $file (keys %musicDb) {
            $found = 0;
            foreach my $foundFile (@findFiles) {
                $foundFile =~ s/$neurosHome/$neurosDrive/;
                if ($file eq $foundFile) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) { delete $musicDb{$file}; }
        }

        # Handle recordings
        $status = "Transferring recordings";
        $top->update;
        syncRecordings(\%recordDb,\%cfg,$neurosHome,$neurosDrive,$top,\$halt);
    }

    if (defined $cfg{'general'}{'duplicates'} and
        $cfg{'general'}{'duplicates'} eq '1') {
        $status = "Checking for duplicate album names";
        $top->update;
        fixDuplicateAlbums(\%musicDb);
    }

    $status = "Create Neuros databases";
    $top->update;
    createNAM(\%cfg,\%musicDb,\%playlistDb,\%recordDb,$ndbHome,$fwVersion,$top);
    %info = ();
    displayMenu($canvas,undef,undef);
    updateTreeIcons($tree,undef);
    configureStatusBar($bottom);
    writeSoruneDb($soruneDb,\%musicDb,\%playlistDb,\%recordDb,\%cfg);
    $top->update;
    $percentDone = 100;
    if ($halt) {
        message('INFO',"Neuros rebuild and sync aborted.\n");
    } else {
        message('INFO',"Neuros rebuild and sync complete (" . 
            getTime(time - $startTime) . ")\n");
    }
    $top->destroy;
}

sub relatePos($$$$$$)
{
    my ($parent,$child,$childWidth,$childHeight,$trans,$resize) = @_;

    my @mainG = split /x|\+/, $parent->geometry;
    my $x = int($mainG[2] + ($mainG[0] - $childWidth)/2);
    my $y = int($mainG[3] + ($mainG[1] - $childHeight)/2);
    $child->geometry("${childWidth}x$childHeight+$x+$y");

    if ($trans) { $child->transient($parent); }
    if (!$resize) { $child->resizable(0,0); }
    $child->idletasks;
    $child->iconimage('main32');
    $child->focus;
    $child->grab;
}

sub addPaths($$$$)
{
    my ($tree,$all,$new,$path) = @_;
    my @args = ();
    $path =~ s/\\/\//g;

    if (!defined $path or $path eq "" or !$tree->info('exists',$path)) {
        return;
    }
    if ($tree->getmode($path) eq 'none') {
        $path = ascii($path);
        my @files = ();
        if ($path eq "/") {
            @files = File::Glob::glob("$path*");
        } else {
            @files = File::Glob::glob("$path/*");
        }

        # Double click on audio file, play file and return
        if (scalar @files == 0) {
            if (!$new) { gPlay($tree,undef); }
            return;
        }

        my $style = $tree->ItemStyle('imagetext',
            -font => $font,
            -background => $cfg{'colors'}{'filebg'},
            -foreground => $cfg{'colors'}{'filefg'},
            -selectbackground => $cfg{'colors'}{'filebgsel'},
            -selectforeground => $cfg{'colors'}{'filefgsel'},
        );

        # populate directories first
        foreach (@files) {
            if (-d $_) {
                @args = ();
                if (-r $_) {
                    push @args, (-image => 'folderclosed');
                } else {
                    push @args, (-image => 'folderlocked');
                }
                my $text = (split(/\//, $_))[-1];
                if ($_ eq '/') { $text = ""; }
                push @args, (-text => $text, -style => $style);
                $tree->add($_, @args);
            }
        }

        # populate files last
        foreach (@files) {
            if (!-d $_) {
                @args = ();
                if (/\.mp3$/i or /\.ogg$/i or /\.wav$/i or /\.wma$/i or
                    /\.m3u$/i) {
                    push @args, (-image => 'audiofile');
                } elsif ($all != 1) {
                    next;
                }
                my $text = (split(/\//, $_))[-1];
                if ($_ eq '/') { $text = ""; }
                push @args, (-text => $text, -style => $style);
                $tree->add($_, @args);
            }
        }

        if ($^O eq "MSWin32" and $path =~ /^[A-Z]:$/) {
            $tree->entryconfigure($path,-image => 'harddrive');
        } else {
            $tree->entryconfigure($path,-image => 'folderopen');
        }
        $tree->setmode($path,'open');
    } elsif ($tree->getmode($path) eq 'open') {
        $tree->delete('offsprings',$path);
        if ($^O eq "MSWin32" and $path =~ /^[A-Z]:$/) {
            $tree->entryconfigure($path,-image => 'harddrive');
        } else {
            $tree->entryconfigure($path,-image => 'folderclosed');
        }
        $tree->setmode($path,'none');
    }
    updateTreeIcons($tree,$path);
}

sub getCurrent($)
{
    my $canvas = shift;
    my @items = ($canvas->find('withtag', 'current'));
    return pop @items;
}

sub getText($$)
{
    my ($canvas,$item) = @_;
    if (ref($item) eq "ARRAY") {
        $item = @{$item}[0];
    }
    my (@text) = ($canvas->find('withtag', "text-$item"));
    return pop @text;
}

sub getName($$)
{
    my ($canvas,$item) = @_;
    my @taglist = ($canvas->gettags($item));
    my $name = undef;
    foreach (@taglist) {
        next if ($_ eq 'current' or $_ eq 'rectangle');
        $name = $_;
        last;
    }
    return $name;
}

sub getCanvasNames($$)
{
    my ($canvas,$color) = @_;
    my @items = ();
    if (defined $canvas) {
        my @allItems = ($canvas->find('withtag', 'rectangle'));
        foreach (@allItems) {
            if ($canvas->itemcget($_, -fill) eq $color) {
                push @items, getName($canvas,$_);
            }
        }
    }
    return @items;
}

sub colorMenu($)
{
    my ($canvas) = @_;
    if (defined $canvas) {
        my @allItems = ($canvas->find('withtag', 'rectangle'));
        foreach my $item (@allItems) {
            if ($canvas->itemcget($item, -fill) eq
                $cfg{'colors'}{'audiobgsel'}) {
                my $name = getName($canvas,$item);
                my @parts = split /\|/, $name;
                $parts[3] = 'NULL';
                $name = join '|', @parts;
                my $color = getColor(\$name);
                $canvas->itemconfigure($item,
                    -fill => $color,
                    -outline => $color,
                );

                my $text = getText($canvas,$item);
                $canvas->itemconfigure($text,
                    -fill => $cfg{'colors'}{'audiofg'},
                );
            }
        }
    }
}

sub gPlay($$)
{
    my ($tree,$canvas) = @_;

    if (defined $cfg{'general'}{'mediaplayer'} and
        -r $cfg{'general'}{'mediaplayer'} and
        -x $cfg{'general'}{'mediaplayer'}) {
        my @names = getCanvasNames($canvas,$cfg{'colors'}{'audiobgsel'});
        my $tmpdir = File::Spec->tmpdir();
        mkpath $tmpdir, 0, 0700;
        if ($@) {
            message('ERR',"Could not create directory $tmpdir for playlist!\n");
            return;
        }
        my $filename = "$tmpdir/sorune.m3u";
        if (createPlaylistFile($filename,\@names,$tree,$neurosHome) == -1) {
            return;
        }
        startPlayer($cfg{'general'}{'mediaplayer'},$filename);
        colorMenu($canvas);
    } else {
        message('ERR',"Check media player settings.\n");
    }
}

sub gPlaylist($$$$$)
{
    my ($main,$tree,$canvas,$bottom,$sync) = @_;
    my @names = getCanvasNames($canvas,$cfg{'colors'}{'audiobgsel'});
    my $fileCount = scalar ($tree->selectionGet);
    my @findFiles = ();

    if (!scalar @names and $fileCount == 0) {
        message('ERR',"Please make selection from file/audio browser first.\n");
        return -1;
    }

    if ($fileCount) {
        my $status = gAdd($main,$tree,$canvas,$bottom,\@findFiles);
        if ($status != 0) { return -1; }
    }

    if (!scalar @names and !scalar @findFiles) { return -1; }

    my $top = $main->Toplevel(-title => 'Create Playlist');
    my $f1 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f2 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f3 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');
    my $f4 = $top->Frame->pack(-expand => 1, -fill => 'x', -side => 'top');

    my $lab1 = $f1->Label(-text => 'Name',
        -width => 10,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left');

    my $entry = $f1->BrowseEntry(-textvariable => \$playlistName,
    )->pack(-expand => 1, -fill => 'x');
    $entry->insert('end', sort keys %playlistDb);

    my $lab2 = $f3->Label(-text => "Randomize",
        -width => 10,
        -justify => 'right',
        -anchor => 'e',
    )->pack(-side => 'left');

    my $cb = $f3->Checkbutton(-variable => \$random,
    )->pack(-side => 'left');

    my $okButton = $f4->Button(-text => 'Ok',
        -command => sub {
            if (!defined $playlistName or $playlistName eq "") {
                message('ERR',"Please enter playlist name.\n");
                return -1;
            }
            my @files = getFiles(\@names);
            my ($musicHome,$key) = (undef,undef);
            if (defined $cfg{'general'}{'musichome'}) {
                $musicHome = $cfg{'general'}{'musichome'};
            }
            foreach my $file (@findFiles) {
                my $neurosFile = $file;
                if ($file =~ /^$neurosHome/) {
                    $neurosFile =~ s/^$neurosHome/$neurosDrive/;
                } else {
                    $neurosFile = getNeurosFilename($musicHome,$file,
                        $neurosDrive);
                }
                if (defined $musicDb{$neurosFile} or
                    defined $recordDb{$neurosFile}) {
                    push @files, $neurosFile;
                }
            }
            my $count = scalar @files;

            fixTag(\$playlistName);
            if (Encode::is_utf8($playlistName)) {
                $key = Encode::encode_utf8($playlistName);
                Encode::from_to($key,'utf8','iso-8859-1');
            } else {
                $key = $playlistName;
            }

            if (defined $playlistDb{$key}) {
                message('INFO',
                    "Added $count songs to playlist $key.\n");
            } else {
                message('INFO',
                    "Created playlist $key with $count songs.\n");
            }
            push @{$playlistDb{$key}}, @files;
            relate(\%musicDb,\%playlistDb);
            if ($random) { randomizePlaylist(\%playlistDb,$key); }
            $top->update;
            %info = ();
            displayMenu($canvas,undef,undef);
            updateTreeIcons($tree,undef);
            $top->destroy;
            $syncNeeded = 1;
            if ($sync) { gSync($main,$tree,$canvas,$bottom); }
            return 0;
        }
    )->pack(-side => 'left', -fill => 'x', -padx => 25, -pady => 5);

    $top->bind('<Return>', sub { $okButton->invoke; });

    my $cancelButton = $f4->Button(-text => 'Cancel',
        -command => sub { $top->destroy; }
    )->pack(-side => 'right', -fill => 'x', -padx => 25, -pady => 5);

    scheme('toplevel' => $top, 'frame' => $f1, 'frame' => $f2,
        'frame' => $f3, 'frame' => $f4, 'label' => $lab1, 'label' => $lab2,
        'browseentry' => $entry, 'checkbutton' => $cb, 'button' => $okButton,
        'button' => $cancelButton);
    $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; });

    # set the focus correctly
    if (defined $playlistName and $playlistName ne "") {
        $okButton->focus;
    } else {
        $entry->focus;
    }

    relatePos($main,$top,300,100,1,0);
    $top->update;
    return 0;
}

sub getTreeFiles($$$$)
{
    my ($tree,$top,$halt,$statusRef) = @_;
    my @findFiles = ();

    foreach my $item ($tree->selectionGet) {
        if ($item =~ /^[A-Z]:$/i) { $item .= "\\"; }
        $item = ascii(File::Spec->rel2abs($item));
        $item =~ s/\\/\//g;
        if (!-d $item and !-f $item) {
            message('ERR',"Invalid selection: $item\n");
            next;
        }
        if (-d $item) {
            my @options = (no_chdir => 1, postprocess => sub {
                if (defined $statusRef) {
                    $$statusRef = "Scanning $File::Find::dir";
                }
                if (defined $top) { $top->update; }
            });
            if ($^O ne "MSWin32") { push @options, (follow => 1); }
            find({wanted => sub {
                if (/\.ogg$/i or /\.mp3$/i or /\.wma$/i or /\.wav$/i or
                    /\.m3u$/i) {
                    if (defined $cfg{'general'}{'recordingshome'}) {
                        if ($File::Find::dir !~
                            /^$cfg{'general'}{'recordingshome'}/) {
                            push @findFiles, $File::Find::name;
                        }
                    } else {
                        push @findFiles, $File::Find::name;
                    }
                }
            }, @options}, $item);
        } elsif (-f $item) {
            push @findFiles, $item;
        }
        if (defined $halt and $halt) { last; }
    }

    return sort @findFiles;
}

sub gDelete($$$)
{
    my ($tree,$canvas,$bottom) = @_;
    my $change = 0;
    my @names = getCanvasNames($canvas,$cfg{'colors'}{'audiobgsel'});
    my @findFiles = getTreeFiles($tree,undef,undef,undef);

    if (!scalar @names and !scalar @findFiles) {
        message('ERR',"Please make selection from file/audio browser first.\n");
        return;
    }

    my $musicHome = undef;
    if (defined $cfg{'general'}{'musichome'}) {
        $musicHome = $cfg{'general'}{'musichome'};
    }
    foreach my $file (@findFiles) {
        $file = ascii($file);
        if (defined $addedFiles{$file}) {
            delByKey(\%musicDb,\%playlistDb,$addedFiles{$file},$neurosHome,
                $neurosDrive);
            $change = 1;
        } elsif (defined $existFiles{$file}) {
            delByKey(\%musicDb,\%playlistDb,$existFiles{$file},$neurosHome,
                $neurosDrive);
            $change = 1;
        } else {
            my $neurosFile = getNeurosFilename($musicHome,$file,$neurosDrive);
            if (defined $musicDb{$neurosFile}) {
                delByKey(\%musicDb,\%playlistDb,$neurosFile,$neurosHome,
                    $neurosDrive);
                $change = 1;
            } elsif (defined $recordDb{$neurosFile}) {
                delByKey(\%recordDb,\%playlistDb,$neurosFile,$neurosHome,
                    $neurosDrive);
                $change = 1;
            }
        }
    }

    foreach my $name (@names) {
        my ($dbKey,$pdbKey,$root,$color,$unique,$key) = split /\|/, $name;
        if ($root == 1 or $root == 2) {
            if ($dbKey eq 'playlist') {
                %playlistDb = ();
                $change = 1;
            } elsif ($dbKey eq 'type') {
                foreach my $key (keys %recordDb) {
                    $recordDb{$key}{'delete'} = 1;
                    $change = 1;
                }
            } else {
                foreach my $key (keys %musicDb) {
                    $musicDb{$key}{'delete'} = 1;
                    $change = 1;
                }
            }
        } elsif ($dbKey eq 'playlist') {
            delPlaylist(\%musicDb,\%playlistDb,$neurosHome,$unique,0);
            $change = 1;
        } elsif ($pdbKey =~ /playlist,(.*)/) {
            removeFromPlaylist(\%musicDb,\%playlistDb,$1,$key,$neurosHome);
            $change = 1;
        } elsif ($dbKey eq 'type') {
            my @files = getFileData(\%recordDb,'type',$unique);
            foreach (@files) {
                if (defined $recordDb{$_}) {
                    $recordDb{$_}{'delete'} = 1;
                    $change = 1;
                }
            }
        } elsif ($dbKey ne 'title') {
            delByName(\%musicDb,\%playlistDb,$dbKey,$unique,$neurosHome);
            delByName(\%recordDb,\%playlistDb,$dbKey,$unique,$neurosHome);
            $change = 1;
        } else {
            delByKey(\%musicDb,\%playlistDb,$key,$neurosHome,$neurosDrive);
            delByKey(\%recordDb,\%playlistDb,$key,$neurosHome,$neurosDrive);
            $change = 1;
        }
    }

    if ($change) {
        %info = ();
        relate(\%musicDb,\%playlistDb);
        displayMenu($canvas,undef,undef);
        updateTreeIcons($tree,undef);
        configureStatusBar($bottom);
        message('INFO',"Sorune deletions complete.\n");
        $syncNeeded = 1;
    }
}

sub getLeftClick($$)
{
    my ($tree,$canvas) = @_;

    $tree->selectionClear;

    my $item = getCurrent($canvas);
    my $name = getName($canvas,$item);
    if (defined $name) {
        my $color = $canvas->itemcget($item, -fill);
        my $text = getText($canvas,$item);
        if ($color ne $cfg{'colors'}{'audiobgsel'}) {
            $canvas->itemconfigure($item,
                -fill => $cfg{'colors'}{'audiobgsel'},
                -outline => $cfg{'colors'}{'audiobgsel'},
            );
        } else {
            my $color = getColor(\$name);
            $canvas->itemconfigure($item,
                -fill => $color,
                -outline => $color
            );
        }
        setItemInfo($canvas);
    }
}

sub getDoubleLeftClick($$)
{
    my ($tree,$canvas) = @_;

    $tree->selectionClear;

    # Don't allow multiple double clicks
    $canvas->CanvasBind('<Double-Button-1>' => sub {});

    # Don't allow right clicks until menu is fully drawn
    $canvas->CanvasBind('<Button-3>' => sub {});

    my $item = getCurrent($canvas);
    my $name = getName($canvas,$item);
    if (defined $name) {
        my $color = getColor(\$name);
        $canvas->itemconfigure($item,
            -fill => $color,
            -outline => $color
        );
        displayMenu($canvas,$name,undef);
    }

    # Reset double click to normal
    $canvas->CanvasBind('<Double-Button-1>' =>
        sub {getDoubleLeftClick($tree,$canvas)});

    # Reset right click to normal
    $canvas->CanvasBind('<Button-3>' => sub {getRightClick($tree,$canvas)});
}

sub getRightClick($$)
{
    my ($tree,$canvas) = @_;

    $tree->selectionClear;

    # Don't allow multiple right clicks until menu is fully drawn
    $canvas->CanvasBind('<Button-3>' => sub {});

    if (scalar @currentCmd > 1) {
        my $current = pop @currentCmd; # discard the current command
        my @parts = split /\|/, $current;
        if ($parts[6] ne "undef") {
            displayMenu($canvas,undef,$parts[6]);
        } else {
            displayMenu($canvas,undef,undef);
        }
    }

    # Reset right click to normal
    $canvas->CanvasBind('<Button-3>' => sub {getRightClick($tree,$canvas)});
}

sub moveItem($$$)
{
    my ($menuList, $menuItemsRef, $direction) = @_;
    my @newMenuItems = @$menuItemsRef;
    my ($item) = ($menuList->curselection);
    if (defined $item) {
        my $menuItem = splice(@newMenuItems,$item,1);
        if ($direction == 0) {
            $item--;
        } else {
            $item++;
        }
        if ($item >= 0 and $item < @$menuItemsRef) {
            splice(@newMenuItems,$item,0,$menuItem);
            @$menuItemsRef = @newMenuItems;
            $menuList->delete(0,'end');
            $menuList->insert('end',@$menuItemsRef);
            $menuList->selectionSet($item,$item);
        }
    }
}

sub setItemInfo($)
{
    my ($canvas) = @_;
    if (defined $canvas) {
        my $item = getCurrent($canvas);
        my $name = getName($canvas,$item);
        if (defined $name) {
            if (scalar @currentItem > 0) {
                my $oldText = shift @currentItem;
                my $oldItem = shift @currentItem;
                if (ref($oldItem) eq "ARRAY") { $oldItem = @{$oldItem}[0]; }
                if (ref($oldText) eq "ARRAY") { $oldText = @{$oldText}[0]; }
                if (ref($item) eq "ARRAY") { $item = @{$item}[0]; }
                if ($oldItem != $item) {
                    my $oldColor = $canvas->itemcget($oldItem, -fill);
                    my $oldName = getName($canvas,$oldItem);
                    if (defined $oldColor and $oldColor eq getColor(\$oldName)) {
                        my $oldTextColor = $canvas->itemcget($oldText, -fill);
                        if ($oldTextColor ne $cfg{'colors'}{'audiofg'}) {
                            $canvas->itemconfigure($oldText,
                                -fill => $cfg{'colors'}{'audiofg'}
                            );
                        }
                    }
                } else {
                    push @currentItem, $oldText, $oldItem;
                    return;
                }
            }
            my $text = getText($canvas,$item);
            my $textColor = $canvas->itemcget($text, -fill);
            if ($textColor ne $cfg{'colors'}{'audiofgsel'}) {
                $canvas->itemconfigure($text,
                    -fill => $cfg{'colors'}{'audiofgsel'}
                );
            }
            push @currentItem, $text, $item;

            my ($dbKey,$pdbKey,$root,$color,$unique,$key) = split /\|/, $name;
            if ($root == 0) {
                my $dbRef = undef;
                if (defined $key and defined $musicDb{$key}) {
                    $dbRef = \%musicDb;
                } elsif (defined $key and defined $recordDb{$key}) {
                    $dbRef = \%recordDb;
                }
                if (defined $dbRef) {
                    $info0 = "$dbRef->{$key}{'title'}";
                    if (defined $dbRef->{$key}{'artist2'} and
                        $dbRef->{$key}{'artist2'} ne "Unknown") {
                        $info1 = $dbRef->{$key}{'artist2'};
                    } else {
                        $info1 = $dbRef->{$key}{'artist'};
                    }
                    $info1 .= " / $dbRef->{$key}{'album'}";
                    my $time = getTime($dbRef->{$key}{'length'});
                    (my $format = $key) =~ s/.*\.//;
                    $info2 = "$dbRef->{$key}{'genre'} / " . 
                        "$dbRef->{$key}{'date'} / $time / " .
                        int($dbRef->{$key}{'size'}/1024) . 
                        " KB / " . uc($format);
                } else {
                    $info0 = $unique;
                    $info1 = $key;
                    $info2 = "";
                }
            } else {
                $info0 = $unique;
                $info1 = $key;
                $info2 = "";
            }
            return;
        }
    }
    if (scalar @currentItem > 0) {
        my $oldText = shift @currentItem;
        my $oldItem = shift @currentItem;
        my $oldColor = $canvas->itemcget($oldItem, -fill);
        my $oldName = getName($canvas,$oldItem);
        if (defined $oldColor and $oldColor eq getColor(\$oldName)) {
            my $oldTextColor = $canvas->itemcget($oldText, -fill);
            if ($oldTextColor ne $cfg{'colors'}{'audiofg'}) {
                $canvas->itemconfigure($oldText,
                    -fill => $cfg{'colors'}{'audiofg'}
                );
            }
        }
    }
    ($info0,$info1,$info2) = ('','','');
}

sub setInfo($)
{
    my $right = shift;
    my $just = 'top';

    if (defined $info0Frame) { $info0Frame->destroy; }
    if (defined $info1Frame) { $info1Frame->destroy; }
    if (defined $info2Frame) { $info2Frame->destroy; }

    $info0Frame = $right->Frame->pack(-fill => 'x', -side => 'top');
    $info1Frame = $right->Frame->pack(-fill => 'x', -side => 'top');
    $info2Frame = $right->Frame->pack(-fill => 'x', -side => 'top');

    $info0Frame->configure(-background => $cfg{'colors'}{'mainbg'});
    $info1Frame->configure(-background => $cfg{'colors'}{'mainbg'});
    $info2Frame->configure(-background => $cfg{'colors'}{'mainbg'});

    if ($cfg{'general'}{'abjust'} eq 'left') { $just = 'left'; }

    $info0Frame->Label(
        -font => $fontBold,
        -textvariable => \$info0,
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => $just);
    $info1Frame->Label(
        -font => $font,
        -textvariable => \$info1,
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => $just);
    $info2Frame->Label(
        -font => $font,
        -textvariable => \$info2,
        -background => $cfg{'colors'}{'mainbg'},
        -foreground => $cfg{'colors'}{'mainfg'},
    )->pack(-side => $just);
}

sub displayMenu
{
    my ($canvas,$cmd,$newLoc) = @_;
    if (!defined $cmd) {
        if (scalar @currentCmd) {
            $cmd = pop @currentCmd;
        } else {
            return;
        }
    }

    my ($dbKey,$pdbKey,$root,$color,$unique,$key,$loc) = split /\|/, $cmd;
    if ($root != 0 or $dbKey ne 'title') {
        my @items = findItems($dbKey,$pdbKey,$root,$unique);
        if (scalar @items) {
            if (!defined $loc or $loc eq "undef") {
                my @loc = $canvas->yview;
                $loc = $loc[0];
                $canvas->yviewMoveto(0);
            }
            push @currentCmd, "$dbKey|$pdbKey|$root|$color|$unique|$key|$loc";
            drawMenus($canvas,\@items,$newLoc);
            setBcText($canvas);
        } else {
            # no items left, try up a level
            $cmd = pop @currentCmd;
            displayMenu($canvas,$cmd,$newLoc);
        }
    } else {
        colorMenu($canvas);
    }
}

sub setDirText($)
{
    my $dir = shift;
    if (defined $dir and $dir ne "") {
        $dirText->delete('1.0','end');
        $dirText->insert('end',$dir);
    }
}

sub setBcText($)
{
    my $canvas = shift;
    my $text = undef;
    my $sidx = '1.0';
    my $eidx;
    my $length;
    my $i;

    # remove existing history trail
    $bcText->delete('1.0','end');
    foreach ($bcText->tagNames) { $bcText->tagDelete($_); }

    # create history trail
    my $cmds = scalar @currentCmd;
    for ($i = 0; $i < ($cmds - 1); $i++) {
        my @parts = split /\|/, $currentCmd[$i];
        if (defined $text) {
            $bcText->insert('end',' > ');
            $sidx = $bcText->index("insert lineend");
        }
        $text = $parts[4];
        $length = length($text);
        $bcText->insert('end',$text);
        $eidx = "$sidx + ${length}c";
        $bcText->tagAdd("link_$text",$sidx,$eidx);
        $bcText->tagConfigure("link_$text", -underline => 1,
            -data => $currentCmd[$i]);
        $bcText->tagBind("link_$text", "<Button-1>" => sub {
            my $link = getLink();
            if (defined $link) {
                my $cmd;
                do { $cmd = pop @currentCmd; } until ($link eq $cmd);
                displayMenu($canvas,$link,undef);
            }
        });
        $bcText->tagBind("link_$text", "<Enter>" => sub {
            $bcText->configure(-cursor => "hand2");
        });
        $bcText->tagBind("link_$text", "<Leave>" => sub {
            $bcText->configure(-cursor => undef);
        });
    }

    # insert the last item in the history trail w/o link
    my @parts = split /\|/, $currentCmd[$i];
    if ($i > 0) { $bcText->insert('end',' > '); }
    $text = $parts[4];
    $bcText->insert('end',$text);
}

sub textCenter($$)
{
    my ($w, $sidx, $s) = @_;
    my $len = length($s);
    my $eidx = "$sidx + ${len}c";
    $w->insert('end',$s);
    $w->tagAdd("center_$s",$sidx,$eidx);
    $w->tagConfigure("center_$s", -justify => 'center');
    return $eidx;
}

sub getLink()
{
    (my $anchor = $bcText->index('current')) =~ s/^.*\.//;
    foreach my $tag ($bcText->tagNames) {
        next if ($tag eq "sel");
        my ($start,$end) = $bcText->tagRanges($tag);
        $start =~ s/^.*\.//;
        $end =~ s/^.*\.//;
        if ($anchor >= $start and $anchor <= $end) {
            return $bcText->tagCget($tag, -data);
        }
    }
    return undef;
}

sub drawMenus($$$)
{
    my ($canvas,$itemsRef,$newLoc) = @_;
    my ($x,$y1,$y2,$anchor);
    if ($cfg{'general'}{'abjust'} eq 'left') {
        ($x,$y1,$y2,$anchor) = (5,0,25,'w');
    } else {
        ($x,$y1,$y2,$anchor) = (200,0,25,'center');
    }
    my $itemCount = scalar @$itemsRef;
    my @scrollRegion = (0,0,400,$itemCount*25);

    $canvas->delete('all');
    if ($^O eq 'MSWin32') {
        $canvas->CanvasBind('<MouseWheel>', [sub {
            my $scroll = 0;
            if ($canvas->Height < $itemCount*25) { $scroll = 5; }
            $_[0]->yview(scroll => -($_[1]/120) * $scroll, 'units')
        }, Ev('D')]);
    } else {
        $canvas->CanvasBind('<Button-4>', sub {
            my $scroll = 0;
            if ($canvas->Height < $itemCount*25) { $scroll = 5; }
            $canvas->yviewScroll(-$scroll, 'units');
        });
        $canvas->CanvasBind('<Button-5>', sub {
            my $scroll = 0;
            if ($canvas->Height < $itemCount*25) { $scroll = 5; }
            $canvas->yviewScroll($scroll, 'units');
        });
    }
    $canvas->configure(-scrollregion => \@scrollRegion);

    for (my $i = 0; $i < $itemCount; $i++) {
        my ($dbKey,$pdbKey,$root,$color,$unique,$key,$loc) =
            split /\|/, $itemsRef->[$i];
        my $rect = $canvas->createRectangle(0,$y1,415,$y2,
            -fill => $color,
            -activefill => $cfg{'colors'}{'audiobgsel'},
            -outline => $color,
            -activeoutline => $cfg{'colors'}{'audiobgsel'},
            -updatecommand => [ \&setItemInfo, $canvas ],
            -tags => [ 'rectangle', $itemsRef->[$i] ],
        );
        $canvas->createText($x,$y2-12.5,
            -fill => $cfg{'colors'}{'audiofg'},
            -anchor => $anchor,
            -text => $unique,
            -state => 'disabled',
            -font => $font,
            -tags => [ "text-$rect" ],
        );
        $y1 += 25;
        $y2 += 25;
        if ($i and !defined $newLoc and $i % 100 == 0) {
            # for some reason, windows needs to be told the scrollregion
            # before each update or the scrollbar gets messed up
            $canvas->configure(-scrollregion => \@scrollRegion);
            $canvas->update;
        }
    }

    if (defined $newLoc) { $canvas->yviewMoveto($newLoc); }
    $canvas->configure(-scrollregion => \@scrollRegion);
    $canvas->update;
}

sub findItems($$$$)
{
    my ($dbKey,$pdbKey,$root,$unique) = @_;
    my ($songs,$playlists,$artists,$albums,$genres,$years,$recordings) =
        ("Songs","Playlists","Artists","Albums","Genres","Years","Recordings");
    if (defined $cfg{'nam'}{'songs'}) {
        $songs = $cfg{'nam'}{'songs'};
    }
    if (defined $cfg{'nam'}{'playlists'}) {
        $playlists = $cfg{'nam'}{'playlists'};
    }
    if (defined $cfg{'nam'}{'artists'}) {
        $artists = $cfg{'nam'}{'artists'};
    }
    if (defined $cfg{'nam'}{'albums'}) {
        $albums = $cfg{'nam'}{'albums'};
    }
    if (defined $cfg{'nam'}{'genres'}) {
        $genres = $cfg{'nam'}{'genres'};
    }
    if (defined $cfg{'nam'}{'years'}) {
        $years = $cfg{'nam'}{'years'};
    }
    if (defined $cfg{'nam'}{'recordings'}) {
        $recordings = $cfg{'nam'}{'recordings'};
    }

    getInfo();
    if ($dbKey eq 'main') {
        my $titleCount = $info{'all'}{'count'};
        my $titleTime = getTime($info{'all'}{'length'});
        my $playlistCount = $info{'playlist'}{'count'};
        my $artistCount = $info{'artist'}{'count'};
        my $albumCount = $info{'album'}{'count'};
        my $genreCount = $info{'genre'}{'count'};
        my $dateCount = $info{'date'}{'count'};
        my $typeCount = $info{'type'}{'count'};
        my @menuOrder = split /\s*,\s*/, $cfg{'nam'}{'menuorder'};
        my @items = ();

        foreach my $item (@menuOrder) {
            if ($item eq "Songs") {
                push @items, "title|undef|1|$cfg{'colors'}{'audiobg'}|$songs|$titleCount $songs ($titleTime)|undef";
            } elsif ($item eq "Playlists") {
                push @items, "playlist|undef|1|$cfg{'colors'}{'audiobg'}|$playlists|$playlistCount $playlists|undef";
            } elsif ($item eq "Artists") {
                if ($cfg{'nam'}{'genreartist'} eq "1") {
                    push @items,
                        "artist|undef|2|$cfg{'colors'}{'audiobg'}|$artists|$artistCount $artists|undef";
                } else {
                    push @items,
                        "artist|undef|1|$cfg{'colors'}{'audiobg'}|$artists|$artistCount $artists|undef";
                }
            } elsif ($item eq "Albums") {
                if ($cfg{'nam'}{'artistalbum'} eq "1" or
                    $cfg{'nam'}{'genrealbum'} eq "1") {
                    push @items,
                        "album|undef|2|$cfg{'colors'}{'audiobg'}|$albums|$albumCount $albums|undef";
                } else {
                    push @items,
                        "album|undef|1|$cfg{'colors'}{'audiobg'}|$albums|$albumCount $albums|undef";
                }
            } elsif ($item eq "Genres") {
                push @items, "genre|undef|1|$cfg{'colors'}{'audiobg'}|$genres|$genreCount $genres|undef";
            } elsif ($item eq "Years") {
                push @items, "date|undef|1|$cfg{'colors'}{'audiobg'}|$years|$dateCount $years|undef";
            } elsif ($item eq "Recordings") {
                push @items, "type|undef|1|$cfg{'colors'}{'audiobg'}|$recordings|$typeCount $recordings|undef";
            } else {
                message('ERR',"Invalid item $item in menu ordering.\n");
            }
        }
        return @items;
    } elsif ($root == 2) {
        my $artistCount = $info{'artist'}{'count'};
        my $albumCount = $info{'album'}{'count'};
        my $genreCount = $info{'genre'}{'count'};
        if ($dbKey eq 'album') {
            my @items = ("album|undef|1|$cfg{'colors'}{'audiobg'}|All|$albumCount $albums|undef");
            if ($cfg{'nam'}{'artistalbum'} eq "1") {
                push @items,
                    "artist|album|4|$cfg{'colors'}{'audiobg'}|$artists|$artistCount $artists|undef";
            }
            if ($cfg{'nam'}{'genrealbum'} eq "1") {
                push @items,
                    "genre|album|4|$cfg{'colors'}{'audiobg'}|$genres|$genreCount $genres|undef";
            }
            return @items;
        } elsif ($dbKey eq 'artist') {
            return (
                "artist|undef|1|$cfg{'colors'}{'audiobg'}|All|$artistCount $artists|undef",
                "genre|artist|4|$cfg{'colors'}{'audiobg'}|$genres|$genreCount $genres|undef"
            );
        }
    }
    my $dbRef;
    if ($dbKey eq 'type') {
        $dbRef = \%recordDb;
    } else {
        $dbRef = \%musicDb;
    }
    my @names = ();

    if ($root == 1) {
        if ($dbKey eq 'playlist') {
            foreach my $name (sort { lc($a) cmp lc($b) } keys %playlistDb) {
                my $count = $info{$dbKey}{$name}{'count'};
                my $time = getTime($info{$dbKey}{$name}{'length'});
                push @names,
                    "$dbKey|undef|0|$cfg{'colors'}{'audiobg'}|$name|$count $songs ($time)|undef"
            }
        } elsif ($dbKey eq 'title') {
            my @keys = getKeys('title',$root,"");
            foreach my $key (sort
                { lc($dbRef->{$a}{'title'}) cmp
                  lc($dbRef->{$b}{'title'}) } @keys) {
                my $name =
                    "title|undef|0|NULL|$dbRef->{$key}{'title'}|$key|undef";
                getColor(\$name);
                push @names, $name;
            }
        } elsif ($dbKey eq 'date') {
            foreach my $name (sort {sprintf("%04s",$b) cmp sprintf("%04s",$a) }
                getUniqueNames($dbRef,'date')) {
                my $count = $info{$dbKey}{$name}{'count'};
                my $time = getTime($info{$dbKey}{$name}{'length'});
                push @names,
                    "date|undef|0|$cfg{'colors'}{'audiobg'}|$name|$count $songs ($time)|undef";
            }
        } else {
            foreach my $name (sort {lc($a) cmp lc($b)}
                getUniqueNames($dbRef,$dbKey)) {
                my $count = $info{$dbKey}{$name}{'count'};
                my $time = getTime($info{$dbKey}{$name}{'length'});
                push @names,
                    "$dbKey|undef|0|$cfg{'colors'}{'audiobg'}|$name|$count $songs ($time)|undef"
            }
        }
    } elsif ($root == 3) {
        my @keys = ();
        if ($pdbKey eq 'album') {
            if ($unique =~ /^various/i) {
                @keys = sort {
                    lc($musicDb{$a}{'album'}) cmp
                    lc($musicDb{$b}{'album'})
                } getUniqueFileData($dbRef,$dbKey,$unique,'album');
            } else {
                @keys = sort {
                    lc($musicDb{$a}{'album'}) cmp
                    lc($musicDb{$b}{'album'})
                } getUniqueFileData($dbRef,$dbKey,$unique,'album');
                if ($cfg{'nam'}{'aasort'} eq '1') {
                    @keys = sort {
                        lc($musicDb{$a}{'date'}) cmp
                        lc($musicDb{$b}{'date'})
                    } @keys;
                } elsif ($cfg{'nam'}{'aasort'} eq '2') {
                    @keys = sort {
                        lc($musicDb{$b}{'date'}) cmp
                        lc($musicDb{$a}{'date'})
                    } @keys;
                }
            }
        } else {
            @keys = sort {
                lc($musicDb{$a}{$pdbKey}) cmp
                lc($musicDb{$b}{$pdbKey})
            } getUniqueFileData($dbRef,$dbKey,$unique,$pdbKey);
        }
        foreach my $key (@keys) {
            my $count = $info{$pdbKey}{$musicDb{$key}{$pdbKey}}{'count'};
            my $time = getTime($info{$pdbKey}{$musicDb{$key}{$pdbKey}}{'length'});
            push @names,
                "$pdbKey|undef|0|$cfg{'colors'}{'audiobg'}|" .
                "$musicDb{$key}{$pdbKey}|$count $songs ($time)|undef";
        }
    } elsif ($root == 4) {
        foreach my $name (sort {lc($a) cmp lc($b)}
            getUniqueNames($dbRef,$dbKey)) {
            my $count = $info{$dbKey}{$name}{'count'};
            my $time = getTime($info{$dbKey}{$name}{'length'});
            push @names,
                "$dbKey|$pdbKey|3|$cfg{'colors'}{'audiobg'}|$name|$count $songs ($time)|undef"
        }
    } else {
        if ($dbKey eq 'playlist') {
            if (defined $playlistDb{$unique}) {
                my @keys = @{$playlistDb{$unique}};
                foreach my $key (@keys) {
                    my $name =
                        "title|$dbKey,$unique|0|NULL|$dbRef->{$key}{'title'}|$key|undef";
                    getColor(\$name);
                    push @names, $name;
                }
            }
        } elsif ($dbKey eq 'album') {
            my @keys = getKeys('album',$root,$unique);
            foreach my $key (sort
                { $dbRef->{$a}{'tracknumber'} <=> $dbRef->{$b}{'tracknumber'} }
                @keys) {
                my $name =
                    "title|undef|0|NULL|$dbRef->{$key}{'title'}|$key|undef";
                getColor(\$name);
                push @names, $name;
            }
        } elsif ($dbKey ne 'title') {
            my @keys = getKeys($dbKey,$root,$unique);
            foreach my $key (sort
                { lc($dbRef->{$a}{'title'}) cmp lc($dbRef->{$b}{'title'}) }
                @keys) {
                my $name =
                    "title|undef|0|NULL|$dbRef->{$key}{'title'}|$key|undef";
                getColor(\$name);
                push @names, $name;
            }
        }
    }

    return @names;
}

sub getInfo()
{
    if (!defined $info{'all'}{'count'}) {
        $info{'all'}{'count'} = scalar keys %musicDb;
        $info{'playlist'}{'count'} = scalar keys %playlistDb;
        $info{'type'}{'count'} = scalar keys %recordDb;

        my @names = getUniqueNames(\%musicDb,'artist');
        $info{'artist'}{'count'} = scalar @names;

        @names = getUniqueNames(\%musicDb,'album');
        $info{'album'}{'count'} = scalar @names;

        @names = getUniqueNames(\%musicDb,'genre');
        $info{'genre'}{'count'} = scalar @names;

        @names = getUniqueNames(\%musicDb,'date');
        $info{'date'}{'count'} = scalar @names;
        foreach my $key (keys %musicDb) {
            (my $format = $key) =~ s/.*\.(.*)$/$1/;
            my $artist = $musicDb{$key}{'artist'};
            my $album = $musicDb{$key}{'album'};
            my $genre = $musicDb{$key}{'genre'};
            my $length = $musicDb{$key}{'length'};
            my $size = $musicDb{$key}{'size'};
            my $date = $musicDb{$key}{'date'};
            if (defined $musicDb{$key}{'playlist'}) {
                foreach my $list (@{$musicDb{$key}{'playlist'}}) {
                    $info{'playlist'}{$list}{'count'} += 1;
                    $info{'playlist'}{$list}{'length'} += $length;
                }
            }

            $format = uc($format);
            $info{$format}{'count'} += 1;
            $info{$format}{'length'} += $length;
            $info{$format}{'size'} += $size;
            $info{'all'}{'length'} += $length;
            $info{'all'}{'size'} += $size;
            $info{'artist'}{$artist}{'count'} += 1;
            $info{'artist'}{$artist}{'length'} += $length;
            $info{'album'}{$album}{'count'} += 1;
            $info{'album'}{$album}{'length'} += $length;
            $info{'genre'}{$genre}{'count'} += 1;
            $info{'genre'}{$genre}{'length'} += $length;
            $info{'date'}{$date}{'count'} += 1;
            $info{'date'}{$date}{'length'} += $length;
            if (defined $musicDb{$key}{'add'}) {
                $info{'all'}{'addsize'} += $size;
            }
            if (defined $musicDb{$key}{'delete'}) {
                $info{'all'}{'deletesize'} += $size;
            }
        }

        foreach my $key (keys %recordDb) {
            my $type = $recordDb{$key}{'type'};
            my $length = $recordDb{$key}{'length'};
            $info{'type'}{$type}{'count'} += 1;
            $info{'type'}{$type}{'length'} += $length;
        }
    }
}

sub getColor($)
{
    my $nameRef = shift;
    my ($dbKey,$pdbKey,$root,$color,$value,$key) = split /\|/, $$nameRef;

    if (defined $color and $color ne "NULL") {
        return $color;
    }

    $color = $cfg{'colors'}{'audiobg'};
    if (defined $musicDb{$key}) {
        if (defined $musicDb{$key}{'add'} and
            $musicDb{$key}{'add'} == 1) {
            $color = $cfg{'colors'}{'audiobgadd'};
        } elsif (defined $musicDb{$key}{'delete'} and
            $musicDb{$key}{'delete'} == 1) {
            $color = $cfg{'colors'}{'audiobgdel'};
        }
    } elsif (defined $recordDb{$key}) {
        if (defined $recordDb{$key}{'delete'} and
            $recordDb{$key}{'delete'} == 1) {
            $color = $cfg{'colors'}{'audiobgdel'};
        }
    }
    $$nameRef =~ s/\|NULL\|/\|$color\|/;

    return $color;
}

sub getKeys($$$)
{
    my ($dbKey,$root,$unique) = @_;
    my @keys = ();

    if ($root == 1 or $root == 2) {
        if ($dbKey eq 'playlist') {
            @keys = keys %playlistDb;
        } elsif ($dbKey eq 'type') {
            @keys = keys %recordDb;
        } else {
            @keys = keys %musicDb;
        }
    } else {
        if ($dbKey eq 'playlist') {
            foreach my $file (@{$playlistDb{$unique}}) {
                push @keys, $file;
            }
        } elsif ($dbKey eq 'type') {
            @keys = getFileData(\%recordDb,'type',$unique);
        } else {
            @keys = getFileData(\%musicDb,$dbKey,$unique);
            if ($dbKey eq 'artist') {
                push @keys, getFileData(\%musicDb,'artist2',$unique);
            }
        }
    }

    return @keys;
}

sub getFiles($)
{
    my $nameRef = shift;
    my @files = ();

    foreach my $name (@$nameRef) {
        my ($dbKey,$pdbKey,$root,$color,$unique,$filename) = split /\|/, $name;
        if ($root == 1 or $root == 2) {
            if ($dbKey eq 'playlist') {
                if (scalar @files) { next; } # all files already added
                foreach my $list (sort keys %playlistDb) {
                    foreach my $file (@{$playlistDb{$list}}) {
                        push @files, $file;
                    }
                }
            } elsif ($dbKey eq 'type') {
                @files = sort { $recordDb{$a}{'type'} cmp
                    $recordDb{$b}{'type'} } keys %recordDb;
            } else {
                @files = sort { $musicDb{$a}{'title'} cmp
                    $musicDb{$b}{'title'} } keys %musicDb;
            }
        } else {
            my @data;
            if ($dbKey eq 'playlist') {
                foreach my $file (@{$playlistDb{$unique}}) {
                    push @files, $file;
                }
            } elsif ($dbKey eq 'type') {
                @data = getFileData(\%recordDb,'type',$unique);
            } elsif ($dbKey ne 'title') {
                @data = getFileData(\%musicDb,$dbKey,$unique);
                if (!scalar @data) {
                    @data = getFileData(\%recordDb,$dbKey,$unique);
                }
            } else {
                push @files, $filename;
                next;
            }

            if ($dbKey eq 'type') {
                foreach my $key (sort
                    { $recordDb{$a}{'title'} cmp
                      $recordDb{$b}{'title'} }
                    @data) {
                    push @files, $key;
                }
            } elsif ($dbKey eq 'album' or $pdbKey eq 'album') {
                if ($unique =~ /various/i or $cfg{'nam'}{'aasort'} eq '0') {
                    foreach my $key (sort {
                        sprintf("%s-%s-%03d",
                            $musicDb{$a}{'artist'},
                            $musicDb{$a}{'album'},
                            $musicDb{$a}{'tracknumber'},
                        )
                        cmp
                        sprintf("%s-%s-%03d",
                            $musicDb{$b}{'artist'},
                            $musicDb{$b}{'album'},
                            $musicDb{$b}{'tracknumber'},
                        )
                    } @data) {
                        push @files, $key;
                    }
                } else {
                    if ($cfg{'nam'}{'aasort'} eq '1') {
                        foreach my $key (sort {
                            sprintf("%d-%s-%s-%03d",
                                $musicDb{$a}{'date'},
                                $musicDb{$a}{'artist'},
                                $musicDb{$a}{'album'},
                                $musicDb{$a}{'tracknumber'},
                            )
                            cmp
                            sprintf("%d-%s-%s-%03d",
                                $musicDb{$b}{'date'},
                                $musicDb{$b}{'artist'},
                                $musicDb{$b}{'album'},
                                $musicDb{$b}{'tracknumber'},
                            )
                        } @data) {
                            push @files, $key;
                        }
                    } elsif ($cfg{'nam'}{'aasort'} eq '2') {
                        foreach my $key (sort {
                            sprintf("%d-%s-%s-%03d",
                                $musicDb{$b}{'date'},
                                $musicDb{$a}{'artist'},
                                $musicDb{$a}{'album'},
                                $musicDb{$a}{'tracknumber'},
                            )
                            cmp
                            sprintf("%d-%s-%s-%03d",
                                $musicDb{$a}{'date'},
                                $musicDb{$b}{'artist'},
                                $musicDb{$b}{'album'},
                                $musicDb{$b}{'tracknumber'},
                            )
                        } @data) {
                            push @files, $key;
                        }
                    }
                }
            } elsif ($dbKey ne 'playlist') {
                foreach my $key (sort
                    { $musicDb{$a}{'title'} cmp
                      $musicDb{$b}{'title'} }
                    @data) {
                    push @files, $key;
                }
            }
        }
    }
    return @files;
}

sub createPlaylistFile($$$)
{
    my ($filename,$nameRef,$tree) = @_;
    my $fh = new FileHandle;
    my $pfh = new FileHandle;
    my @canvasFiles = getFiles($nameRef);
    my @findFiles = ();

    foreach my $item ($tree->selectionGet) {
        $item = ascii(File::Spec->rel2abs($item));
        $item =~ s/\\/\//g;
        if (!-d $item and !-f $item) {
            message('ERR',"Invalid selection: $item\n");
            next;
        }
        if (-d $item) {
            my @options = (no_chdir => 1);
            if ($^O ne "MSWin32") { push @options, (follow => 1); }
            find({wanted => sub {
                if (/\.ogg$/i or /\.mp3$/i or /\.wma$/i or /\.wav$/i or
                    /\.m3u$/i) {
                    push @findFiles, $File::Find::name;
                }
            }, @options}, $item);
        } elsif (-f $item) {
            if ($item =~ /\.ogg$/i or $item =~ /\.mp3$/i or
                $item =~ /\.wma$/i or $item =~ /\.wav$/i or
                $item =~ /\.m3u$/i) {
                push @findFiles, $item;
            }
        }
    }

    if (!scalar @canvasFiles and !scalar @findFiles) {
        message('ERR',"No music files found.\n");
        return -1;
    }

    if ($fh->open(">$filename")) {
        my $count = 0;
        print $fh "#EXTM3U\n";
        foreach my $file (@canvasFiles) {
            next if (defined ($musicDb{$file}{'add'}) and
                $musicDb{$file}{'add'} eq '1');
            if (defined $musicDb{$file}{'length'} and
                defined $musicDb{$file}{'artist'} and
                defined $musicDb{$file}{'title'}) {
                print $fh "#EXTINF:" . $musicDb{$file}{'length'} . "," .
                    $musicDb{$file}{'artist'} . " / " .
                    $musicDb{$file}{'title'} . "\n";
            }
            (my $localFile = $file) =~ s/$neurosDrive/$neurosHome/i;
            print $fh "$localFile\n";
            $count++;
        }
        foreach my $file (sort @findFiles) {
            if ($file =~ /\.m3u$/i) {
                if ($pfh->open($file)) {
                    while (<$pfh>) {
                        s/[\r\n]//g;
                        if (/^\s*#\s*EXTINF/) {
                            print $fh "$_\n";
                            next;
                        }
                        my $pFile = File::Spec->rel2abs($_,dirname($file));
                        $pFile =~ s/\\/\//g;
                        if ($^O eq "MSWin32" and $pFile !~ /^[A-Z]:/i) {
                            (my $drive = $file) =~ s/^([A-Za-z]):.*/\u$1:/;
                            $pFile = "$drive$pFile";
                        }
                        if (-r $pFile) {
                            print $fh "$pFile\n";
                            $count++;
                        }
                    }
                    $pfh->close;
                }
            } else {
                print $fh "$file\n";
                $count++;
            }
        }
        $fh->close;
        if ($count == 0) {
            message('ERR',"Please synchronize first.\n");
            return -1;
        }
        if ($count > 1) {
            message('INFO',"Playing $count songs.\n");
        } else {
            message('INFO',"Playing song.\n");
        }
        return 0;
    } else {
        message('ERR',"Could not open $filename: $!\n");
        return -1;
    }
}

sub startPlayer($$)
{
    my ($player,$filename) = @_;

    if (!-x $player) {
        message('ERR',"Could not find media player $player: $!\n");
        return;
    }

    if ($^O eq "MSWin32") {
        my $processObj;
        $player =~ s/\//\\/g;
        $filename =~ s/\//\\/g;
        my $ps = Win32::Process::Create($processObj, $player,
            "\"$player\" \"$filename\"", 0, DETACHED_PROCESS(), ".");
        if (!$ps) {
            message('ERR',"Could not start media player: $!\n");
        }
    } else {
        my $pid;
        FORK: {
            if ($pid = fork) {
                return;
            } elsif (defined $pid) {
                exec ($player,$filename);
            } elsif ($! == EAGAIN) {
                sleep 5;
                redo FORK;
            } else {
                message('ERR',"Could not start media player: $!\n");
            }
        }
    }
}

sub startBrowser($)
{
    my ($url) = @_;

    if (!defined $url or $url eq "") { return; }

    if ($^O eq "MSWin32") {
        my %values = ();
        my ($dir,$entry) = ("","SOFTWARE\\Clients\\StartMenuInternet");

        if ($::HKEY_LOCAL_MACHINE->Open($entry,$dir) == 0) { return; }
        $dir->GetValues(\%values);
        my $defaultApp = @{$values{''}}[2];

        $entry .= "\\$defaultApp\\shell\\open\\command";
        if ($::HKEY_LOCAL_MACHINE->Open($entry,$dir) == 0) { return; }
        $dir->GetValues(\%values);
        $defaultApp = @{$values{''}}[2];
        $defaultApp =~ s/^"(.*)"$/$1/;

        my $processObj;
        my $ps = Win32::Process::Create($processObj, $defaultApp,
            "\"$defaultApp\" $url", 0, DETACHED_PROCESS(), ".");
    } else {
        my $pid;
        FORK: {
            if ($pid = fork) {
                return;
            } elsif (defined $pid) {
                # Is there a better way to determine the browser?
                my $path = searchPath('firefox');
                if (defined $path) { exec ($path,$url); }
                $path = searchPath('mozilla');
                if (defined $path) { exec ($path,$url); }
                $path = searchPath('opera');
                if (defined $path) { exec ($path,$url); }
                $path = searchPath('netscape');
                if (defined $path) { exec ($path,$url); }
            } elsif ($! == EAGAIN) {
                sleep 5;
                redo FORK;
            }
        }
    }
}

sub applyFont($$$$$)
{
    my ($bottom,$tree,$canvas,$family,$size) = @_;
    $font = "$family $size";
    if ($size >= 5) { $size -= 2; }
    $fontSm = "$family $size";
    $tree->configure(-font => $font);
    $msg->configure(-font => $font);
    my $cmd = pop @currentCmd;
    displayMenu($canvas,$cmd,undef);
    configureStatusBar($bottom);
}

sub dirTree($$$$$)
{
    my ($main,$win,$cwd,$entryRef,$new) = @_;
    my ($top,$mode,$all);

    if ($new) {
        $top = $win->Toplevel(-title => 'Browse');
        $mode = 'browse';
        $all = 1;
    } else {
        $top = $win;
        $mode = 'extended';
        $all = 0;
    }

    my $tree = $top->Scrolled('Tree',
        -separator => '/',
        -exportselection => 1,
        -scrollbars => 'osoe',
        -height => 15,
        -width => 15,
        -selectmode => $mode,
        -drawbranch => 0,
        -indicator => 0,
    )->pack(-expand => 1, -fill => 'both', -side => 'top', -pady => 5);
    $tree->configure(-command => [ \&addPaths, $tree, $all, $new ]);
    if (!$new) { $tree->configure(-browsecmd => [ \&setDirText ]); }
    scheme('tree',$tree);

    my $style = $tree->ItemStyle('imagetext',
        -font => $font,
        -background => $cfg{'colors'}{'filebg'},
        -foreground => $cfg{'colors'}{'filefg'},
        -selectbackground => $cfg{'colors'}{'filebgsel'},
        -selectforeground => $cfg{'colors'}{'filefgsel'},
    );

    # Populate tree with drives
    if ($^O eq "MSWin32") {
        foreach my $drive (getWindowsDrives()) {
            chdir("$drive/");
            $tree->add($drive, -text => $drive, -style => $style,
                -image => 'harddrive');
        }
    } else {
        $tree->add("/", -text => "", -style => $style);
        addPaths($tree,$all,$new,"/");
    }

    # Populate tree with cwd
    if (defined $cwd and -r $cwd) {
        my @parts = split /\//, $cwd;
        my $count = scalar @parts;
        my $path = "";
        for (my $i = 0; $i < ($count - 1); $i++) {
            $path .= "$parts[$i]";
            addPaths($tree,$all,$new,$path);
            $path .= "/";
        }
        if ($tree->info('exists',$cwd)) {
            $tree->see($cwd);
            $tree->selectionSet($cwd);
        }
    }

    if ($new) {
        my $okButton = $top->Button(-text => 'Ok',
            -command => sub {
                my @items = ($tree->selectionGet);
                $$entryRef = shift @items;
                $top->destroy;
            }
        )->pack(-side => 'left', -padx => 25, -pady => 5);
        my $cancelButton = $top->Button(-text => 'Cancel',
            -command => sub { $top->destroy; return undef; }
        )->pack(-side => 'right', -padx => 25, -pady => 5);

        scheme('toplevel' => $top, 'button' => $okButton,
            'button' => $cancelButton);
        $top->protocol('WM_DELETE_WINDOW', sub { $top->destroy; return undef;});

        relatePos($win,$top,500,350,1,0);
        $top->update;
        $tree->focus;
        return undef;
    }

    return $tree;
}

sub updateTreeIcons($)
{
    my ($tree,$path) = @_;
    my $hashRef;
    my @paths = ();

    if (! defined $path) {
        %deletedFiles = ();
        %addedFiles = ();
        %existFiles = ();
        foreach my $key (keys %musicDb) {
            if (defined $musicDb{$key}{'localFile'}) {
                if (defined $musicDb{$key}{'delete'} and
                    $musicDb{$key}{'delete'} == 1) {
                    $hashRef = \%deletedFiles;
                } elsif (defined $musicDb{$key}{'add'} and
                    $musicDb{$key}{'add'} == 1) {
                    $hashRef = \%addedFiles;
                } else {
                    $hashRef = \%existFiles;
                }
                $hashRef->{$musicDb{$key}{'localFile'}} = $key;
                my @parts = split /\//, $musicDb{$key}{'localFile'};
                pop @parts;                # discard filename
                my $file = shift @parts;
                $hashRef->{$file} = 1;
                foreach my $part (@parts) {
                    $file .= "/$part";
                    $hashRef->{$file} = 1;
                }
            }
            (my $localFile = $key) =~ s/^[CD]:/$neurosHome/;
            $hashRef->{$localFile} = $key;
            my @parts = split /\//, $localFile;
            pop @parts;                # discard filename
            my $file = shift @parts;
            $hashRef->{$file} = 1;
            foreach my $part (@parts) {
                $file .= "/$part";
                $hashRef->{$file} = 1;
            }
        }
        @paths = getTreePaths($tree,undef);
    } else {
        @paths = getTreePaths($tree,$path);
        push @paths, $path;
    }

    foreach my $path (@paths) {
        $path = ascii($path);
        if ($^O eq "MSWin32" and $path =~ /^[A-Z]:$/) { next; }
        if (defined $deletedFiles{$path} and defined $addedFiles{$path}) {
            if (-d $path) {
                if ($tree->getmode($path) eq 'open') {
                    $tree->entryconfigure($path,-image =>
                        'folderopenplusminus');
                } else {
                    $tree->entryconfigure($path,-image =>
                        'folderclosedplusminus');
                }
            }
        } elsif (defined $deletedFiles{$path}) {
            if (-d $path) {
                if ($tree->getmode($path) eq 'open') {
                    $tree->entryconfigure($path,-image => 'folderopenminus');
                } else {
                    $tree->entryconfigure($path,-image => 'folderclosedminus');
                }
            } else {
                $tree->entryconfigure($path,-image => 'audiofileminus');
            }
        } elsif (defined $addedFiles{$path}) {
            if (-d $path) {
                if ($tree->getmode($path) eq 'open') {
                    $tree->entryconfigure($path,-image => 'folderopenplus');
                } else {
                    $tree->entryconfigure($path,-image => 'folderclosedplus');
                }
            } else {
                $tree->entryconfigure($path,-image => 'audiofileplus');
            }
        } elsif (defined $existFiles{$path}) {
            if (-d $path) {
                if ($tree->getmode($path) eq 'open') {
                    $tree->entryconfigure($path,-image => 'folderopenexist');
                } else {
                    $tree->entryconfigure($path,-image => 'folderclosedexist');
                }
            } else {
                $tree->entryconfigure($path,-image => 'audiofileexist');
            }
        } else {
            if (-d $path) {
                if ($tree->getmode($path) eq 'open') {
                    $tree->entryconfigure($path,-image => 'folderopen');
                } else {
                    $tree->entryconfigure($path,-image => 'folderclosed');
                }
            } else {
                if ($path =~ /\.mp3$/i or $path =~ /\.ogg$/i or 
                    $path =~ /\.wav$/i or $path =~ /\.wma$/i or
                    $path =~ /\.m3u$/i) {
                    $tree->entryconfigure($path,-image => 'audiofile');
                }
            }
        }
    }
}

1
