//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [] = CL_plot_ephem(pos_car, win_id,color_id,tick_steps,data_bounds,thickness)
// Plots ground tracks 
//
// Calling Sequence
// CL_plot_ephem(pos_car [,win_id,color_id,tick_steps,data_bounds,thickness])
//
// Description
// <itemizedlist><listitem>
// <p>Plots the ground track in longitude and latitude corresponding to the trajectory given in cartesian coordinates.</p>
// <p>The plot x-axis and y-axis are respectively the (spherical) longitude and latitude in degrees.</p>
// <p>These longitude and latitude are obtained from pos_car by: [lon; lat; r] = CL_co_car2sph(pos_car).</p>
// </listitem>
// <listitem>
// <p><b>Note</b>: </p>
// <p>Only one graphical entity is created. </p>
// <p>Call gce() to get it and possibly change graphical properties. </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// pos_car: Positions in cartesian coordinates: [X;Y;Z] (3xN)
// win_id: (optional) Figure Id - Id of the current window if omitted.
// color_id: (optional) Color index - Default is 2.
// tick_steps: (optional) Steps of the grid in longitude and latitude in degrees (2x1), or [] (automatic). Default: []. 
// data_bounds: (optional) Definition of the view area: [longmin, latmin; longmax, latmax] in degrees. Default is [-180, -90; 180, 90].
// thickness: (optional) Line thickness. Default is 1.
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_plot_earthMap
//
// Examples
// t0 = 21915; // initial time
// kep_t0 = [7070.e3; 0.001; CL_deg2rad(98); 0; 0; 0]; // orbit parameters
// step = 30 / 86400; // propagation step
// t = t0:step:t0+1; 
// kep_t = CL_ex_secularJ2(t0, kep_t0, t); // orbit propagation
//
// pos_eci = CL_oe_kep2car(kep_t);      // inertial position
// pos_ecf = CL_fr_convert("ECI", "ECF", t, pos_eci); // position in rotating frame
//
// scf();
// CL_plot_ephem(pos_ecf);


// Declarations:

// Code:

// CODE FOR COMPATIBILITY
if exists('pos_car_ter','local'); pos_car=pos_car_ter; end
if exists('winId','local'); win_id=winId; end
if exists('colorId','local'); color_id=colorId; end
if exists('tickInterval','local'); tick_steps=tickInterval; end
if exists('dataBounds','local'); data_bounds=dataBounds; end


if exists('win_id','local') then   
  f = scf(win_id)
else
  f = gcf();
end

if ~exists('color_id','local') then
  color_id = 1;
end

if ~exists('tick_steps','local') then
  tick_steps = []; // degrees
end

if ~exists('data_bounds','local') then
  data_bounds = [-180,-90;180,90]; // degrees
end

if ~exists('thickness','local'); thickness = 1; end

// error checking
if (size(data_bounds,1) <> 2 | size(data_bounds,2) <> 2)
  CL__error('Invalid size for data_bounds');
end
if (data_bounds(2,1)<=data_bounds(1,1) | data_bounds(2,1)>data_bounds(1,1)+360)
  CL__error('Invalid data bounds in longitude');
end 
if (data_bounds(2,2)<=data_bounds(1,2) | data_bounds(1,2) < -90 | data_bounds(2,2) > 90)
  CL__error('Invalid data bounds in latitude');
end 

if (tick_steps <> [])
  if (length(tick_steps) <> 2)
    CL__error('Invalid size for tick_steps');
  end 
  if (tick_steps(1) <= 0 | tick_steps(2) <= 0)
    CL__error('Invalid values for tick_steps');
  end 
end

// plot
immediate_drawing_save = f.immediate_drawing; // store field
f.immediate_drawing = "off"; 

// interval in longitude containing the view
bmin = (data_bounds(1,1)+data_bounds(2,1))/2 - 180; 
bmax = (data_bounds(1,1)+data_bounds(2,1))/2 + 180; 

// Passage de pos en coordonnees spheriques
pos_sph = CL_co_car2sph(pos_car);

// plot curve (never empty)
[x, y] = CL__plot_genxy(CL_rad2deg(pos_sph(1,:)), CL_rad2deg(pos_sph(2,:)), bmin, bmax);  
plot2d(x, y);
h = CL_g_select(gce(), "Polyline"); 
h.foreground = color_id; 
h.thickness = thickness; 


// adjustments (grid, databounds)
a = gca();
a.data_bounds = data_bounds;
a.tight_limits = "on";

bticks = [ tick_steps(1) * floor(data_bounds(1,:)/tick_steps(1)); 
           tick_steps(2) * ceil((data_bounds(2,:))/tick_steps(2)) ];

if (tick_steps <> [])
  bticks = [ tick_steps(1)*floor(data_bounds(1,:)/tick_steps(1)); 
             tick_steps(2)*ceil((data_bounds(2,:))/tick_steps(2)) ];

  x_ticks = bticks(1,1) : tick_steps(1) : bticks(2,1);
  y_ticks = bticks(1,2) : tick_steps(2) : bticks(2,2);

  // strsplit+msprintf : used to avoid "string"
  x_ticks_labels = strsplit( stripblanks( msprintf("%.8g ",x_ticks')) , " ")' ; 
  y_ticks_labels = strsplit( stripblanks( msprintf("%.8g ",y_ticks')) , " ")' ;

  a.x_ticks = tlist(["ticks", "locations", "labels"], x_ticks, x_ticks_labels ); 
  a.y_ticks = tlist(["ticks", "locations", "labels"], y_ticks, y_ticks_labels ); 
end

CL_g_stdaxes(a);
a.axes_visible = ["on", "on", "on"];

f.immediate_drawing = immediate_drawing_save; // restore field

endfunction
