//socket.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2014
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#define _LIBROAR_NOATTR_TO_STATIC /* ignore warnings for TO_STATIC functions */
#include "libroar.h"

#if defined(ROAR_SUPPORT_PROXY) && !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER)
#include <pwd.h>
#endif

#define MODE_LISTEN  ROAR_SOCKET_MODE_LISTEN
#define MODE_CONNECT ROAR_SOCKET_MODE_CONNECT

static int roar_socket_open_file  (int mode, const char * host, int port);

#ifdef ROAR_SUPPORT_PROXY
static int roar_socket_open_proxy (int mode, int type, const char * host, int port, const char * proxy_type);

static int roar_socket_open_socks4 (int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts);
static int roar_socket_open_socks4a(int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts);
static int roar_socket_open_socks4d(int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts);
static int roar_socket_open_socks4x(int mode, int fh, char host[4], int port, const char * app, size_t app_len, const char * user);

static int roar_socket_open_http   (int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts);

#ifdef ROAR_HAVE_BIN_SSH
static int roar_socket_open_ssh    (int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts);
#endif
#endif


#ifdef ROAR_TARGET_WIN32
void roar_socket_win32_init (void) {
 static int inited = 0;
 WSADATA wsadata;

 if ( !inited ) {
  WSAStartup(MAKEWORD(1,1) , &wsadata);
  inited++;
 }
}
#else
#define roar_socket_win32_init()
#endif

#ifdef ROAR_HAVE_LIBDNET
static int roar_socket_decnet_set_timeout (int fh, time_t sec, int_least32_t usec) {
 struct timeval timeout = {sec, usec};

 return setsockopt(fh, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout));
}
#endif

static int roar_socket_set_tos(int fh) {
#if defined(ROAR_HAVE_BSDSOCKETS) && !defined(ROAR_TARGET_WIN32)
 int opt = IPTOS_LOWDELAY;
 int ret;

 roar_err_clear_errno();
 ret = setsockopt(fh, IPPROTO_IP, IP_TOS, &opt, sizeof(int));
 if ( ret < 0 )
  roar_err_from_errno();

 return ret;
#else
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
#endif
}

#if 0
static int roar_socket_new_decnet_seqpacket (void) {
#ifdef ROAR_HAVE_LIBDNET
 int fh;

 fh = socket(AF_DECnet, SOCK_SEQPACKET, DNPROTO_NSP);

 roar_socket_decnet_set_timeout(fh, 300, 0);

 return fh;
#else
 return -1;
#endif
}
#endif


#if 0
static int roar_socket_new_ipxspx (void) {
 return -1;
}
#endif

int roar_socket_new        (int type) {
#ifdef ROAR_HAVE_BSDSOCKETS
 int sys_domain = -1, sys_type = -1, sys_protocol = 0;
 int fh;
#if defined(ROAR_HAVE_IPV4) && defined(TCP_NODELAY) && !defined(ROAR_TARGET_WIN32)
 int t   = 1;
#endif

 switch (type) {
  case ROAR_SOCKET_TYPE_NONE:
  case ROAR_SOCKET_TYPE_GENSTR:
    roar_err_set(ROAR_ERROR_INVAL);
    return -1;
   break;
  case ROAR_SOCKET_TYPE_FILE:
  case ROAR_SOCKET_TYPE_FORK:
    roar_err_set(ROAR_ERROR_PERM);
    return -1;
   break;
#ifdef ROAR_HAVE_IPV4
  case ROAR_SOCKET_TYPE_TCP:
    sys_domain = AF_INET; sys_type = SOCK_STREAM;
   break;
  case ROAR_SOCKET_TYPE_UDP:
    sys_domain = AF_INET; sys_type = SOCK_DGRAM;
   break;
#endif
#ifdef ROAR_HAVE_UNIX
  case ROAR_SOCKET_TYPE_UNIX:
    sys_domain = AF_UNIX; sys_type = SOCK_STREAM;
   break;
#endif
#ifdef ROAR_HAVE_LIBDNET
  case ROAR_SOCKET_TYPE_DECNET:
    sys_domain = AF_DECnet; sys_type = SOCK_STREAM; sys_protocol = DNPROTO_NSP;
   break;
#endif
#ifdef ROAR_HAVE_IPV6
  case ROAR_SOCKET_TYPE_TCP6:
    sys_domain = AF_INET6; sys_type = SOCK_STREAM;
   break;
  case ROAR_SOCKET_TYPE_UDP6:
    sys_domain = AF_INET6; sys_type = SOCK_DGRAM;
   break;
#endif
#ifdef ROAR_HAVE_IPX
  case ROAR_SOCKET_TYPE_IPX:
    sys_domain = AF_IPX; sys_type = SOCK_DGRAM; sys_protocol = AF_IPX;
   break;
#endif
  case ROAR_SOCKET_TYPE_IPXSPX:
  case ROAR_SOCKET_TYPE_LAT_SERVICE:
  case ROAR_SOCKET_TYPE_LAT_REVERSE_PORT:
  default:
    roar_err_set(ROAR_ERROR_AFNOTSUP);
    return -1;
   break;
 }

 roar_socket_win32_init();

 roar_err_clear_all();
 fh = socket(sys_domain, sys_type, sys_protocol);
 if ( fh == -1 ) {
  roar_err_update();
  return -1;
 }

 // do the extra work we need to do for some socket types:
 switch (type) {
#ifdef ROAR_HAVE_IPV4
  case ROAR_SOCKET_TYPE_TCP:
#if defined(TCP_NODELAY) && !defined(ROAR_TARGET_WIN32)
   setsockopt(fh, IPPROTO_TCP, TCP_NODELAY, &t, sizeof(int));
#endif
  case ROAR_SOCKET_TYPE_UDP:
#endif
#ifdef ROAR_HAVE_IPV6
  case ROAR_SOCKET_TYPE_TCP6:
  case ROAR_SOCKET_TYPE_UDP6:
#endif
#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6)
    roar_socket_set_tos(fh);
   break;
#endif
#ifdef ROAR_HAVE_LIBDNET
  case ROAR_SOCKET_TYPE_DECNET:
    roar_socket_decnet_set_timeout(fh, 300, 0);
   break;
#endif
 }

 return fh;
#else
 roar_err_set(ROAR_ERROR_NOSYS);
 return -1;
#endif
}


int roar_socket_nonblock(int fh, int state) {
#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER) && defined(ROAR_HAVE_FCNTL)
 int flags;

 if ( (flags = fcntl(fh, F_GETFL, 0)) == -1 ) {
  roar_err_from_errno();
  ROAR_ERR("roar_socket_nonblock(fh=%i, state=%i): Can not read flags: %s", fh, state, strerror(errno));
  ROAR_DBG("roar_socket_nonblock(fh=%i, state=%i) = -1", fh, state);
  return -1;
 }

 if ( !(flags & O_NONBLOCK) && state == ROAR_SOCKET_BLOCK )
  return 0;

 flags |= O_NONBLOCK;

 if ( state == ROAR_SOCKET_BLOCK )
  flags -= O_NONBLOCK;

 if ( fcntl(fh, F_SETFL, flags) == -1 ) {
  roar_err_from_errno();
  ROAR_ERR("roar_socket_nonblock(fh=%i, state=%i): Can not set flags: %s", fh, state, strerror(errno));
  ROAR_DBG("roar_socket_nonblock(fh=%i, state=%i) = -1", fh, state);
  return -1;
 }

 ROAR_DBG("roar_socket_nonblock(fh=%i, state=%i) = 0", fh, state);
 return 0;
#else
 ROAR_WARN("roar_socket_nonblock(*): no nonblocking IO support on win32, use a real OS");
 return -1;
#endif
}

int roar_socket_dup_udp_local_end (int fh) {
#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER) && defined(ROAR_HAVE_FCNTL)
 int                  n              = -1;
 int                  flags          = -1;
 struct sockaddr_in   socket_addr;
 socklen_t            len            = sizeof(struct sockaddr_in);

 if ( (flags = fcntl(fh, F_GETFL, 0)) == -1 ) {
  ROAR_WARN("roar_socket_dup_udp_local_end(fh=%i): Can not read flags: %s", fh, strerror(errno));
 }

 if ( getsockname(fh, (struct sockaddr *)&socket_addr, &len) == -1 ) {
  roar_err_from_errno();
  return -1;
 }

 if ( socket_addr.sin_family != AF_INET ) {
  roar_err_set(ROAR_ERROR_TYPEMM);
  return -1;
 }

 n = roar_socket_new(ROAR_SOCKET_TYPE_UDP);

 if ( n == -1 )
  return -1;

//  if ( mode_func(fh, (struct sockaddr *)&socket_addr, sizeof(struct sockaddr_in)) == -1 ) {
 if ( bind(n, (struct sockaddr *)&socket_addr, len) == -1 ) {
  roar_err_from_errno();
  close(n);
  return -1;
 }

 if ( flags != -1 ) {
  if ( fcntl(fh, F_SETFL, flags) == -1 ) {
   ROAR_WARN("roar_socket_dup_udp_local_end(fh=%i): Can not set flags: %s", fh, strerror(errno));
   return -1;
  }
 }


 return n;
#else
 ROAR_WARN("roar_socket_dup_udp_local_end(*): this function is not supported on win32, use a real OS");
 return -1;
#endif
}


#define _SCMR_CONTROLLEN (sizeof(struct cmsghdr) + sizeof(int))
int roar_socket_send_fh (int sock, int fh, char * mes, size_t len) {
#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER) && !defined(ROAR_OS_SUNOS) && !defined(ROAR_TARGET_OPENVMS)
 struct iovec     iov[1];
 struct msghdr    msg;
 char             cmptr_buf[_SCMR_CONTROLLEN];
 struct cmsghdr * cmptr = (struct cmsghdr *) cmptr_buf;
 char             localmes[1] = {0};

 ROAR_DBG("roar_socket_send_fh(sock=%i, fh=%i, mes=%p, len=%u) = ?", sock, fh, mes, len);

 if ( sock < 0 || fh < 0 )
  return -1;

 if ( len == 0 ) {
  len = 1;
  mes = localmes;
 }

 memset(&msg,  0, sizeof(msg));
 memset(cmptr, 0, _SCMR_CONTROLLEN);

 iov[0].iov_base = mes;
 iov[0].iov_len  = len;
 msg.msg_iov     = iov;
 msg.msg_iovlen  = 1;
 msg.msg_name    = NULL;
 msg.msg_namelen = 0;

 cmptr->cmsg_level        = SOL_SOCKET;
 cmptr->cmsg_type         = SCM_RIGHTS;
 cmptr->cmsg_len          = _SCMR_CONTROLLEN;
 msg.msg_control          = (caddr_t) cmptr;
 msg.msg_controllen       = _SCMR_CONTROLLEN;
 *(int *)CMSG_DATA(cmptr) = fh;

 return sendmsg(sock, &msg, 0);
#else
 ROAR_ERR("roar_socket_send_fh(*): There is no UNIX Domain Socket support in win32, download a real OS.");
 return -1;
#endif
}

int roar_socket_recv_fh (int sock,         char * mes, size_t * len) {
#if !defined(ROAR_TARGET_WIN32) && !defined(ROAR_TARGET_MICROCONTROLLER) && !defined(ROAR_OS_SUNOS) && !defined(ROAR_TARGET_OPENVMS)
 struct iovec     iov[1];
 struct msghdr    msg;
 char             cmptr_buf[_SCMR_CONTROLLEN];
 struct cmsghdr * cmptr = (struct cmsghdr *) cmptr_buf;
 char             localmes[1];
 size_t           locallen[1] = {1};

 ROAR_DBG("roar_socket_recv_fh(sock=%i, mes=%p, len=%p) = ?", sock, mes, len);

 if ( sock < 0 ) {
  ROAR_DBG("roar_socket_recv_fh(sock=%i, mes=%p, len=%p) = -1 // invalid socket", sock, mes, len);
  return -1;
 }

 if ( len == NULL ) {
  len = locallen;
  mes = localmes;
 }

 iov[0].iov_base = mes;
 iov[0].iov_len  = *len;
 msg.msg_iov     = iov;
 msg.msg_iovlen  = 1;
 msg.msg_name    = NULL;
 msg.msg_namelen = 0;

 msg.msg_control    = (caddr_t) cmptr;
 msg.msg_controllen = _SCMR_CONTROLLEN;

 if ( (*len = recvmsg(sock, &msg, 0)) == -1 ) {
  ROAR_DBG("roar_socket_recv_fh(sock=%i, mes=%p, len=%p) = -1 // can not read from socket", sock, mes, len);
  return -1;
 }

 if ( msg.msg_controllen  < _SCMR_CONTROLLEN ||
      cmptr->cmsg_len    != _SCMR_CONTROLLEN  ) {
  ROAR_DBG("roar_socket_recv_fh(sock=%i, mes=%p, len=%p) = -1 // control len is wrong", sock, mes, len);
  return -1;
 }

 ROAR_DBG("roar_socket_recv_fh(sock=%i, mes=%p, len=%p) = %i", sock, mes, len, *(int *)CMSG_DATA(cmptr));
 return *(int *)CMSG_DATA(cmptr);
#else
 ROAR_ERR("roar_socket_recv_fh(*): There is no UNIX Domain Socket support in win32, download a real OS.");
 return -1;
#endif
}

int roar_socket_listen  (int type, const char * host, int port) {
 return roar_socket_open(MODE_LISTEN, type, host, port);
}

int roar_socket_connect (int type, const char * host, int port) {
 const char * proxy_type = roar_env_get("ROAR_PROXY");

 ROAR_DBG("roar_socket_connect(host='%s', port=%i) = ?", host, port);

 if ( proxy_type == NULL || strcmp(proxy_type, "") == 0 ) {
  return roar_socket_open(MODE_CONNECT, type, host, port);
 } else {
#ifdef ROAR_SUPPORT_PROXY
  return roar_socket_open_proxy(MODE_CONNECT, type, host, port, proxy_type);
#else
  ROAR_ERR("roar_socket_connect(host='%s', port=%i): no support for proxy code (proxy_type=%s)", host, port, proxy_type);
  return -1;
#endif
 }
}


#ifdef ROAR_HAVE_LIBDNET
static int roar_socket_listen_decnet (const char * object, int num) {
 int fh = roar_socket_new(ROAR_SOCKET_TYPE_DECNET);
 struct sockaddr_dn bind_sockaddr;

 if ( fh == -1 )
  return -1;

 if ( object != NULL && !*object )
  object = NULL;

 if ( (object != NULL && num) || (object != NULL && !*object && !num) || (object == NULL && !num) ) {
  ROAR_WARN("roar_socket_listen_decnet(object='%s', num=%i): illegal address!", object, num);
  close(fh);
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 }

 memset((void*)&bind_sockaddr, 0, sizeof(struct sockaddr_dn));

 bind_sockaddr.sdn_family    = AF_DECnet;
 bind_sockaddr.sdn_flags     = 0;
 bind_sockaddr.sdn_objnum    = num;

 if ( num ) {
  bind_sockaddr.sdn_objnamel = 0;
 } else {
  bind_sockaddr.sdn_objnamel  = ROAR_HOST2LE16(strlen(object));
  if ( bind_sockaddr.sdn_objnamel > DN_MAXOBJL )
   bind_sockaddr.sdn_objnamel = DN_MAXOBJL;
  strncpy((char*)bind_sockaddr.sdn_objname, object, DN_MAXOBJL);
 }

 if ( bind(fh, (struct sockaddr *) &bind_sockaddr, sizeof(bind_sockaddr)) == -1 ) {
  roar_err_from_errno();
  close(fh);
  return -1;
 }

 if ( listen(fh, 8) == -1 ) {
  roar_err_from_errno();
  close(fh);
  return -1;
 }

 return fh;
}
#endif

const char * roar_socket_get_local_nodename(void) {
#ifdef ROAR_HAVE_LIBDNET
 static char node[16] = {0};
 struct dn_naddr      *binaddr;
 struct nodeent       *dp;

 if ( !node[0] ) {

 // TODO: This does this function not use getnodename()?

 // Those strange workarounds for the error codes are because those functions currently
 // don't set errno. This way we will use errno as soon as it starts to set it.

  roar_err_update();
  if ( (binaddr=getnodeadd()) == NULL) {
   roar_err_update();
   if ( roar_error == ROAR_ERROR_NONE )
    roar_err_set(ROAR_ERROR_NOENT);
   return NULL;
  }

  roar_err_update();
  if ( (dp=getnodebyaddr((char*)binaddr->a_addr, binaddr->a_len, AF_DECnet)) == NULL ) {
   roar_err_update();
   if ( roar_error == ROAR_ERROR_NONE )
    roar_err_set(ROAR_ERROR_NOENT);
   return NULL;
  }

  strncpy(node, dp->n_name, 15);
  node[15] = 0;
 }

 return node;
#else
 roar_err_set(ROAR_ERROR_AFNOTSUP);
 return NULL;
#endif
}

int roar_socket_open (int mode, int type, const char * host, int port) {
// int type = ROAR_SOCKET_TYPE_INET;
 int fh;
#ifdef ROAR_HAVE_IPX
#define _NEED_OBJ
#endif
#if defined(ROAR_HAVE_IPX) || defined(ROAR_HAVE_GETADDRINFO)
 int ret;
#endif
#ifdef ROAR_HAVE_IPX
 unsigned int ipx_port;
#endif
#ifdef ROAR_HAVE_UNIX
 int abstract = 0;
#endif
#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6) || defined(ROAR_HAVE_UNIX) || defined(ROAR_HAVE_IPX)
 union {
  struct sockaddr     sa;
#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6)
  struct sockaddr_in  in;
#endif
#ifdef ROAR_HAVE_UNIX
  struct sockaddr_un  un;
#endif
#ifdef ROAR_HAVE_IPV6
  struct sockaddr_in6 in6;
#endif
#ifdef ROAR_HAVE_IPX
  struct sockaddr_ipx ipx;
#endif
 } socket_addr;
 socklen_t addrlen;
#endif
#if (defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6)) && !defined(ROAR_HAVE_GETADDRINFO)
 struct hostent     * he = NULL;
#endif
 //unsigned int host_div = 0;
#ifdef ROAR_TARGET_WIN32
 int PASCAL (*mode_func)(SOCKET,const struct sockaddr*,int) = connect; // default is to connect
#else
 int (*mode_func)(int sockfd, const struct sockaddr *serv_addr, socklen_t addrlen) = connect; // default is to connect
#endif
#ifdef ROAR_HAVE_LIBDNET
#define _NEED_OBJ
 char * dnet_node_buf;
#endif
#ifdef _NEED_OBJ
 char obj[80];
 char * del;
#endif
#ifdef ROAR_HAVE_GETADDRINFO
 int af_guessed = 0;
 struct addrinfo hints, *res = NULL;
 char port_as_string[32];
#endif
#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6)
 int is_numerical = 1;
 const char * numptr;
#endif

 ROAR_DBG("roar_socket_open(mode=%i, type=%i, host='%s', port=%i) = ?", mode, type, host, port);

 if ( host == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 roar_err_set(ROAR_ERROR_UNKNOWN);

 if ( mode == MODE_LISTEN )
  mode_func = bind;

 if ( type == ROAR_SOCKET_TYPE_UNKNOWN ) {
#ifdef ROAR_HAVE_GETADDRINFO
  af_guessed = 1;
#endif
  type = ROAR_SOCKET_TYPE_INET;
  if ( *host == '/' ) {
   type = ROAR_SOCKET_TYPE_UNIX;
  } else if ( strcmp(host, "+fork") == 0 ) {
   type = ROAR_SOCKET_TYPE_FORK;
  } else if ( strcmp(host, "+abstract") == 0 ) {
   type = ROAR_SOCKET_TYPE_UNIX;
#ifdef ROAR_HAVE_UNIX
   abstract = 1;
#endif
  } else if ( strstr(host, "::") != NULL ) {
   type = ROAR_SOCKET_TYPE_DECNET;
  } else if ( host[strlen(host)-1] == ')' ) {
   type = ROAR_SOCKET_TYPE_IPX;
  }
 }


 ROAR_DBG("roar_socket_open(mode=%i, type=%i, host='%s', port=%i) = ?", mode, type, host, port);

 ROAR_DBG("roar_socket_open(*): type=%s, host='%s', port=%i",
             type == ROAR_SOCKET_TYPE_UNIX ? "UNIX" : "???", host, port);

 if ( type == ROAR_SOCKET_TYPE_DECNET ) {
#ifdef ROAR_HAVE_LIBDNET
   ROAR_DBG("roar_socket_open(*): nodename for DECnet: host(%p)=%s", host, host);
   dnet_node_buf = roar_mm_strdup(host);
   // roar_error is still set.
   if ( dnet_node_buf == NULL )
    return -1;

   host = dnet_node_buf;

   del = strstr(host, "::");
   ROAR_DBG("roar_socket_open(*): nodename for DECnet: del(%p)=%s", del, del);

   if ( del == NULL ) {
    ROAR_WARN("roar_socket_open(*): invalid nodename for DECnet: %s", host);
    roar_mm_free(dnet_node_buf);
    roar_err_set(ROAR_ERROR_INVAL);
    return -1;
   }

   *del = 0;

   if ( *(del+2) == '#' ) { // assume we have node::#num
    port = atoi(del+2);
   }

   if ( port ) {
    snprintf(obj, 7, "%i", port); // no need for snprintf() as dec(port) is smaller than obj[]
   } else {
    *obj = 0;
    strncat(obj, del+2, 79);
   }

  ROAR_DBG("roar_socket_open(*): obj='%s', port=%i", obj, port);

  if ( mode == MODE_LISTEN ) {
   fh = roar_socket_listen_decnet(obj, port);
   roar_mm_free(dnet_node_buf);
   return fh;
  } else {
   // There is nothing wrong in this case to use dnet_conn() so we do.
   ROAR_DBG("roar_socket_open(*): CALL dnet_conn('%s', '%s', SOCK_STREAM, 0 ,0 ,0 , 0)", dnet_node_buf, obj);
   fh = dnet_conn(dnet_node_buf, obj, SOCK_STREAM, 0, 0, 0, 0);
   ROAR_DBG("roar_socket_open(*): RET %i", fh);
   roar_mm_free(dnet_node_buf);
   return fh;
  }
#else
  roar_err_set(ROAR_ERROR_AFNOTSUP);
  return -1; // no decnet support
#endif
 }

#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6)
 memset(&socket_addr,    0, sizeof(socket_addr));
#endif


 if ( type == ROAR_SOCKET_TYPE_INET || type == ROAR_SOCKET_TYPE_INET6 ) {
#if defined(ROAR_HAVE_IPV4) || defined(ROAR_HAVE_IPV6)

  ROAR_DBG("roar_socket_open(*): type=INET|INET6, host='%s', port=%i", host, port);

  roar_socket_win32_init(); // we need to do this early as gethostbyname() requires this.

  ROAR_DBG("roar_socket_open(*): type=INET|INET6, host='%s', port=%i", host, port);

  if ( !!strcmp(host, "0.0.0.0") ) {
   for (numptr = host; is_numerical && *numptr != 0; numptr++)
    if ( ! ((*numptr >= '0' && *numptr <= '9') || *numptr == '.')  )
     is_numerical = 0;

   if ( is_numerical && roar_libroar_iswarn(NULL) ) {
    ROAR_WARN("roar_socket_open(*): Hostname \"%s\" is numerical. Do not use IP addresses directly. Use Hostnames.", host);
   }
  }

#ifdef ROAR_HAVE_GETADDRINFO
  memset(&hints, 0, sizeof(hints));
  hints.ai_socktype = SOCK_STREAM;
  if ( af_guessed ) {
   hints.ai_family   = AF_UNSPEC;
  } else {
   hints.ai_family   = type == ROAR_SOCKET_TYPE_INET ? AF_INET : AF_INET6;
  }

  snprintf(port_as_string, sizeof(port_as_string), "%i", port);

  ret = getaddrinfo(host, port_as_string, &hints, &res);
  if ( ret == EAI_SYSTEM ) {
   roar_err_from_errno();
   return -1;
  } else if ( ret != 0 ) {
   roar_err_convert(&roar_error, ROAR_ERROR_TYPE_ROARAUDIO, ret, ROAR_ERROR_TYPE_EAI);
   return -1;
  }

  if ( af_guessed ) {
   type = res->ai_family == AF_INET ? ROAR_SOCKET_TYPE_INET : ROAR_SOCKET_TYPE_INET6;
  }

  if ( type == ROAR_SOCKET_TYPE_INET ) {
   fh = roar_socket_new(ROAR_SOCKET_TYPE_TCP);
  } else {
   fh = roar_socket_new(ROAR_SOCKET_TYPE_TCP6);
  }

  memcpy(&(socket_addr.sa), res->ai_addr, res->ai_addrlen);
  addrlen = res->ai_addrlen;

  if ( res != NULL )
   freeaddrinfo(res);
#else
  if ( (he = gethostbyname(host)) == NULL ) {
   ROAR_ERR("roar_socket_open(*): Can\'t resolve host name '%s'",
                     host);
   roar_err_from_errno();
   return -1;
  }

   ROAR_DBG("roar_socket_open(*): he=%p", he);

   memcpy((struct in_addr *)&(socket_addr.in.sin_addr), he->h_addr, sizeof(struct in_addr));

   /* set the connect information */
   socket_addr.in.sin_family = AF_INET;
   socket_addr.in.sin_port   = ROAR_HOST2NET16(port);

  fh = roar_socket_new(ROAR_SOCKET_TYPE_TCP);
  addrlen = sizeof(struct sockaddr_in);
#endif

  ROAR_DBG("roar_socket_open(*): fh=%i", fh);

  if ( fh == -1 ) {
   ROAR_ERR("roar_socket_open(*): Can\'t create TCP socket: %s", strerror(errno));
   return -1;
  }

  ROAR_DBG("roar_socket_open(*) = ?");

  ROAR_DBG("roar_socket_open(*): fh=%i", fh);

   if ( mode_func(fh, (struct sockaddr *)&socket_addr.sa, addrlen) == -1 ) {
    ROAR_DBG("roar_socket_open(*): Can not connect/bind: %s", strerror(errno));
    roar_err_from_errno();
    close(fh);
    return -1;
   }

  ROAR_DBG("roar_socket_open(*): fh=%i", fh);

  // hey! we have a socket...
  ROAR_DBG("roar_socket_open(*) = ? // we have a socket :)");
#else
  ROAR_DBG("roar_socket_open(*) = -1 // no IPv4 or IPv6 support");
  roar_err_set(ROAR_ERROR_AFNOTSUP);
  return -1;
#endif
 } else if ( type == ROAR_SOCKET_TYPE_UNIX ) {
#ifdef ROAR_HAVE_UNIX
  socket_addr.un.sun_family = AF_UNIX;

  if ( abstract ) {
   memset(socket_addr.un.sun_path, 0, sizeof(socket_addr.un.sun_path));
   snprintf(socket_addr.un.sun_path+1, sizeof(socket_addr.un.sun_path)-1, "RoarAudio/UNIX/Abstract/%i", abstract);
  } else {
   strncpy(socket_addr.un.sun_path, host, sizeof(socket_addr.un.sun_path) - 1);
  }

  fh = roar_socket_new(ROAR_SOCKET_TYPE_UNIX);

  if ( mode_func(fh, (struct sockaddr *)&socket_addr.un, sizeof(struct sockaddr_un)) == -1 ) {
   ROAR_DBG("roar_socket_open(*): Can not connect/bind: %s", strerror(errno));
   roar_err_from_errno();
   close(fh);
   return -1;
  }
#else
  ROAR_ERR("roar_socket_open(*): There is no UNIX Domain Socket support in win32, download a real OS.");
  roar_err_set(ROAR_ERROR_AFNOTSUP);
  return -1;
#endif
 } else if ( type == ROAR_SOCKET_TYPE_IPX ) {
#ifdef ROAR_HAVE_IPX
  socket_addr.ipx.sipx_family = AF_IPX;

  obj[0] = 0;

  if ( (ret = sscanf(host, "%8x.%12s(%x)", &socket_addr.ipx.sipx_network, obj, &ipx_port)) < 2 ) {
   return -1;
   socket_addr.ipx.sipx_port = ipx_port;
  } else if ( ret == 2 ) {
   socket_addr.ipx.sipx_port = port; // Network Byte Order?
  }

  memset(socket_addr.ipx.sipx_node, 0, IPX_NODE_LEN);
  ret = strlen(obj);

  if ( ret % 2 )  // needs to be even at the moment
   return -1;

  fh = roar_socket_new(ROAR_SOCKET_TYPE_IPX);

  close(fh);
  roar_err_set(ROAR_ERROR_AFNOTSUP);
  return -1;
#else
  roar_err_set(ROAR_ERROR_AFNOTSUP);
  return -1;
#endif
 } else if ( type == ROAR_SOCKET_TYPE_FORK ) {
  roar_err_set(ROAR_ERROR_INVAL);
  return -1;
 } else if ( type == ROAR_SOCKET_TYPE_FILE ) {
  return roar_socket_open_file(mode, host, port);
 } else {
  roar_err_set(ROAR_ERROR_AFNOTSUP);
  return -1;
 }

 if ( mode == MODE_LISTEN ) {
#if defined(ROAR_HAVE_BSDSOCKETS) || defined(ROAR_TARGET_WIN32)
  if ( listen(fh, ROAR_SOCKET_QUEUE_LEN) == -1 ) {
   roar_err_from_errno();
   close(fh);
   return -1;
  }
#else
  roar_err_set(ROAR_ERROR_NOSYS);
  return -1;
#endif
 }

 return fh;
}

static int roar_socket_open_file  (int mode, const char * host, int port) {
#ifdef ROAR_HAVE_IO_POSIX
 int fh;

 if ( mode == MODE_LISTEN )
  return -1;

 if ( (fh = open(host, O_RDONLY, 0644)) == -1 ) {
  ROAR_ERR("roar_socket_open_file(*): Can not open file %s: %s", host, strerror(errno));
 }

 return fh;
#else
 return -1;
#endif
}

// --- [ PROXY CODE ] ---

#ifndef ROAR_HAVE_IO_POSIX
#ifdef  ROAR_SUPPORT_PROXY
#undef  ROAR_SUPPORT_PROXY
#endif
#endif

// generic proxy code:

#ifdef ROAR_SUPPORT_PROXY
static int roar_socket_open_proxy (int mode, int type, const char * host, int port, const char * proxy_type) {
 int    proxy_port = -1;
 char   proxy_host[ROAR_SOCKET_MAX_HOSTNAMELEN];
 const char * proxy_addr = NULL;
 int    i;
 int    fh = -1;
 const char * user = NULL, * pw = NULL, * opts = NULL;
 char * sep;
 int    no_fh = 0;
 char   proxy_addr_buf[1024];
 static struct passwd * passwd;
 int (* code)(int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts) = NULL;

 if ( passwd == NULL ) {
  passwd = getpwuid(getuid());
 }

// TODO: fix this in a good way
#ifndef ROAR_TARGET_MICROCONTROLLER
 if ( passwd != NULL )
  user = passwd->pw_name;
#endif

 if ( user == NULL )
  user = roar_env_get("USER");

 // TODO: change this so we support listen() proxys (ssh -R)
 if ( mode != MODE_CONNECT )
  return -1;

 if ( !strncmp(proxy_type, "socks", 5) ) {
  proxy_addr = roar_env_get("socks_proxy");

  proxy_port = 9050; // TOR's default port
 } else if ( !strcmp(proxy_type, "http") || !strcmp(proxy_type, "https") ) {
  proxy_port = 8080;

  if ( (proxy_addr = roar_env_get("http_proxy")) == NULL )
   proxy_addr = roar_env_get("https_proxy");

  if ( proxy_addr == NULL )
   return -1;

  if ( !strncmp(proxy_addr, "http://", 7) )
   proxy_addr += 7;
 } else if ( !strncmp(proxy_type, "ssh", 3) ) {
  proxy_port = 22;
  proxy_addr = roar_env_get("ssh_proxy");
  no_fh      = 1;
 }

 proxy_addr_buf[1023] = 0;
 strncpy(proxy_addr_buf, proxy_addr, 1023);
 proxy_addr = proxy_addr_buf;

 if ( (sep = strstr(proxy_type, "/")) != NULL )
  opts = sep+1;

 if ( proxy_addr == NULL )
  return -1;

 if ( (sep = strstr(proxy_addr, "@")) != NULL ) {
  *sep = 0;
  user = proxy_addr;
  proxy_addr = sep+1;

  if ( (sep = strstr(user, ":")) != NULL ) {
   *sep = 0;
   pw = sep+1;
  }
 }

 ROAR_DBG("roar_socket_open_proxy(*): proxy_type='%s', opts='%s', user='%s', pw=(not shown), proxy_addr='%s'", proxy_type, opts, user, proxy_addr);

 for (i = 0; proxy_addr[i] != 0 && proxy_addr[i] != ':' && i < (ROAR_SOCKET_MAX_HOSTNAMELEN - 1); i++)
  proxy_host[i] = proxy_addr[i];
 proxy_host[i] = 0;

 if ( i == 0 ) // no hostname found
  return -1;

 if ( proxy_addr[i] == ':' )
  proxy_port = atoi(&proxy_addr[i+1]);

 if ( ! no_fh ) {
  if ( (fh = roar_socket_open(mode, type, proxy_host, proxy_port)) == -1) {
   return -1;
  }
 }

 if ( !strcmp(proxy_type, "socks4a") ) { // for TOR, the only supported type at the moment
  code = roar_socket_open_socks4a;
 } else if ( !strcmp(proxy_type, "socks4d") ) { // DECnet
  code = roar_socket_open_socks4d;
 } else if ( !strcmp(proxy_type, "socks4") ) { // good old SOCKS4
  code = roar_socket_open_socks4;
 } else if ( !strcmp(proxy_type, "http") ) { // HTTP CONNECT
  code = roar_socket_open_http;
 } else if ( !strncmp(proxy_type, "ssh", 3) ) { // SSH...
#ifdef ROAR_HAVE_BIN_SSH
  code = roar_socket_open_ssh;
#else
  ROAR_ERR("roar_socket_open_proxy(*): No SSH support compiled in");
#endif
 } else {
  return -1; // unknown type
 }

 if ( code != NULL ) {
  if ( no_fh ) {
   fh = code(mode, fh, host, port, user, pw, opts);
  } else {
   if ( code(mode, fh, host, port, user, pw, opts) == -1 ) {
    close(fh);
    return -1;
   }
  }

  return fh;
 }

 close(fh);
 return -1;
}

// protocoll dependet proxy code:

static int roar_socket_open_socks4 (int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts) {
#ifndef ROAR_TARGET_MICROCONTROLLER
 struct hostent     * he;

 if ( (he = gethostbyname(host)) == NULL ) {
  ROAR_ERR("roar_socket_open_socks4(*): Can\'t resolve host name '%s'", host);
  return -1;
 }

 return roar_socket_open_socks4x(mode, fh, he->h_addr, port, NULL, 0, user);
#else
 return -1;
#endif
}

static int roar_socket_open_socks4a(int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts) {
 return roar_socket_open_socks4x(mode, fh, "\0\0\0\1", port, host, strlen(host)+1, user);
}

static int roar_socket_open_socks4d(int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts) {
 size_t len = strlen(host)+1;
 char * dp;

 if ( port == 0 ) {
  if ( (dp = strstr(host, "::")) == NULL )
   return -1;

  len--;
  *dp = 0;
  memmove(dp+1, dp+2, len - (dp-host) - 1);
 }

 return roar_socket_open_socks4x(mode, fh, "\0\2\0\0", port, host, len, user);
}

static int roar_socket_open_socks4x(int mode, int fh, char host[4], int port, const char * app, size_t app_len, const char * user) {
 char buf[9];
 int len;

 buf[0] = 0x04;
 buf[1] = mode == MODE_CONNECT ? 0x01 : 0x02;
 *((uint16_t*)&buf[2]) = htons(port);
 memcpy(buf+4, host, 4);

 if ( user == NULL ) {
  buf[8] = 0x00;
  len = 9;
 } else {
  len = 8;
 }

 if ( write(fh, buf, len) != len )
  return -1;

 if ( user != NULL ) {
  len = strlen(user) + 1;
  if ( write(fh, user, len) != len )
   return -1;
 }

 if ( app_len > 0 )
  if ( write(fh, app, app_len) != (ssize_t)app_len )
   return -1;

 if ( read(fh, buf, 8) != 8 )
  return -1;

 if ( buf[1] != 0x5a )
  return -1;

 return 0;
}

static int roar_socket_open_http   (int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts) {
 char buf[1024];
 int len;

 if ( port == 0 || host == NULL )
  return -1;

 if ( *host == '/' ) // AF_UNIX
  return -1;

 if ( (len = snprintf(buf, sizeof(buf), "CONNECT %s:%i HTTP/1.0\r\nUser-Agent: libroar\r\n\r\n", host, port)) == -1 )
  return -1;

 if ( write(fh, buf, len) != len )
  return -1;

 while ( (len = read(fh, buf, sizeof(buf))) ) {
  if ( len == sizeof(buf) ) { // overlong lion
   return -1;
  } else if ( len == 2 && buf[0] == '\r' && buf[1] == '\n' ) {
   break;
  } else if ( len == 1 && (buf[0] == '\r' || buf[0] == '\n') ) { // bad proxy or devel trying to debug ;)
   break;
  } else if ( len >= 4 && buf[len-4] == '\r' && buf[len-3] == '\n' && buf[len-2] == '\r' && buf[len-1] == '\n' ) {
   break;
  }
 }

 return 0;
}


#ifdef ROAR_HAVE_BIN_SSH
static int roar_socket_open_ssh    (int mode, int fh, const char * host, int port, const char * user, const char * pw, const char * opts) {
 const char * proxy_addr = roar_env_get("ssh_proxy");
 char * sep;
 char * bin_ssh;
 char   cmd[1024] = "", rcmd[1024] = "";
 int    proxy_port = 22;
 int    use_socat = 0;
 int r;
 int socks[2];

 if ( host == NULL )
  return -1;

 if ( *host == '/' )
  use_socat = 1;

 if ( mode == MODE_LISTEN )
  return -1;

 if ( proxy_addr == NULL )
  return -1;

 if ( opts != NULL ) {
  if ( !strcmp(opts, "socat") ) {
   use_socat = 1;
  } else if ( !strcmp(opts, "netcat") ) {
   use_socat = 0;
  } else {
   return -1;
  }
 }

 ROAR_DBG("roar_socket_open_ssh(*): proxy_addr='%s'", proxy_addr);

 if ( (sep = strstr(proxy_addr, "@")) != NULL )
  proxy_addr = sep+1;

 if ( (sep = strstr(proxy_addr, ":")) != NULL ) {
  *sep = 0;
  proxy_port = atoi(sep+1);
 }


 if ( !strcmp(host, "+fork") ) {
  strncpy(rcmd, "roard --no-listen --client-fh 0", 32);
 } else {
  if ( use_socat ) {
   if ( *host == '/' ) {
    snprintf(rcmd, sizeof(rcmd)-1, "socat stdio unix-connect:\"%s\"", host);
   } else {
    snprintf(rcmd, sizeof(rcmd)-1, "socat stdio tcp:\"%s\":%i", host, port);
   }
  } else {
   snprintf(rcmd, sizeof(rcmd)-1, "$(which netcat nc 2> /dev/null | grep -v \" \" | head -n 1) \"%s\" %i", host, port);
  }

  rcmd[sizeof(rcmd)-1] = 0;
 }

 ROAR_DBG("roar_socket_open_ssh(*): proxy_port=%i, user='%s', proxy_addr='%s'", proxy_port, user, proxy_addr);
 ROAR_DBG("roar_socket_open_ssh(*): rcmd: %s", rcmd);
 bin_ssh = roar_libroar_get_path("bin-ssh", 0, NULL, NULL);
 if ( bin_ssh == NULL )
  return -1;

 snprintf(cmd, sizeof(cmd)-1, "%s -p %i -l '%s' '%s' '%s'", bin_ssh, proxy_port, user, proxy_addr, rcmd);
 cmd[sizeof(cmd)-1] = 0;

 roar_mm_free(bin_ssh);


// TODO: get this more portable!
#ifdef AF_UNIX
 if ( socketpair(AF_UNIX, SOCK_STREAM, 0, socks) == -1 ) {
  return -1;
 }
#else
 return -1;
#endif

 r = roar_fork(NULL);

 if ( r == -1 ) { // error!
  ROAR_ERR("roar_socket_open_ssh(*): Can not fork: %s", strerror(errno));
  close(socks[0]);
  close(socks[1]);
  return -1;
 } else if ( r == 0 ) { // we are the child
  roar_watchdog_stop();

  close(socks[0]);

  close(ROAR_STDIN ); // we do not want roard to have any standard input
  close(ROAR_STDOUT); // STDOUT is also not needed, so we close it,
                      // but STDERR we keep open for error messages.

  dup2(socks[1], 0);
  dup2(socks[1], 1);

  execlp("sh", "sh", "-c", cmd, (_LIBROAR_GOOD_CAST char*)NULL);

  // we are still alive?
  ROAR_ERR("roar_socket_open_ssh(*): alive after exec(), that's bad!");
  ROAR_U_EXIT(1);
 } else { // we are the parent
  close(socks[1]);
  return socks[0];
 }
 return -1;
}
#endif

#endif // ROAR_SUPPORT_PROXY

//ll
